<?php
/**
 * Plugin Name: Custom BuddyBoss Group Tabs
 * Description: Allows group managers to add and edit custom tabs in BuddyBoss groups
 * Version: 1.0.0
 * Author: Mobinteg
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Check if BuddyBoss Platform is active
if (!function_exists('bp_is_active')) {
    return;
}

class Custom_BuddyBoss_Group_Tabs {
    
    public function __construct() {
        add_action('wp_ajax_save_custom_tab', array($this, 'save_custom_tab'));
        add_action('wp_ajax_delete_custom_tab', array($this, 'delete_custom_tab'));
        add_action('wp_ajax_get_custom_tab', array($this, 'get_custom_tab'));
    }

    public function get_existing_tabs($group_id) {
        $args = array(
            'post_type' => 'bpgtc_group_tab',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_bpgtc_tab_groups',
                    'value' => (string)$group_id,
                    'compare' => 'LIKE'
                )
            )
        );
        
        $tabs = get_posts($args);
        $formatted_tabs = array();
        
        foreach ($tabs as $tab) {
            $formatted_tabs[] = array(
                'id' => $tab->ID,
                'name' => $tab->post_title,
                'slug' => $tab->post_name,
                'is_active' => get_post_meta($tab->ID, '_bpgtc_tab_is_active', true),
                'position' => get_post_meta($tab->ID, '_bpgtc_tab_position', true),
                'subnav_items' => get_post_meta($tab->ID, '_bpgtc_subnav_items', true)
            );
        }
        
        return $formatted_tabs;
    }

    public function render_tabs_management() {
        $group_id = bp_get_current_group_id();
        $existing_tabs = $this->get_existing_tabs($group_id);
        $tab_visibility_options = [
            'all' => __('Qualquer pessoa', 'custom-group-tabs'),
            'logged_in' => __('Com sessão iniciada', 'custom-group-tabs'),
            'members_only' => __('Apenas membros', 'custom-group-tabs'),
            'admin_only' => __('Apenas organizadores', 'custom-group-tabs'),
            'mods_only' => __('Apenas moderadores', 'custom-group-tabs'),
        ];
        ?>
        <div class="custom-tabs-management">
            <div id="tabs-list-view">
                <h2><?php _e('Gerir separadores personalizados', 'custom-group-tabs'); ?></h2>
                <button type="button" id="add-new-tab" class="button button-primary"><?php _e('Adicionar novo separador', 'custom-group-tabs'); ?></button>
                
                <?php if (!empty($existing_tabs)): ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php _e('Nome do separador', 'custom-group-tabs'); ?></th>
                                <th><?php _e('Slug', 'custom-group-tabs'); ?></th>
                                <th><?php _e('Posição', 'custom-group-tabs'); ?></th>
                                <th><?php _e('Estado', 'custom-group-tabs'); ?></th>
                                <th><?php _e('Ações', 'custom-group-tabs'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($existing_tabs as $tab): ?>
                                <tr>
                                    <td><?php echo esc_html($tab['name']); ?></td>
                                    <td><?php echo esc_html($tab['slug']); ?></td>
                                    <td><?php echo esc_html($tab['position']); ?></td>
                                    <td><?php echo $tab['is_active'] === 'on' ? __('Ativo', 'custom-group-tabs') : __('Inativo', 'custom-group-tabs'); ?></td>
                                    <td>
                                        <button type="button" class="button edit-tab" data-tab-id="<?php echo esc_attr($tab['id']); ?>"><?php _e('Editar', 'custom-group-tabs'); ?></button>
                                        <button type="button" class="button delete-tab" data-tab-id="<?php echo esc_attr($tab['id']); ?>"><?php _e('Eliminar', 'custom-group-tabs'); ?></button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p><?php _e('Não foram encontrados separadores personalizados para este grupo.', 'custom-group-tabs'); ?></p>
                <?php endif; ?>
            </div>

            <div id="tab-form-view" style="display: none;">
                <h2><?php _e('Adicionar/Editar separador personalizado', 'custom-group-tabs'); ?></h2>
                <button type="button" id="back-to-list" class="button"><?php _e('← Voltar à lista', 'custom-group-tabs'); ?></button>
                <form id="add-custom-tab" method="post">
                    <input type="hidden" id="tab-id" name="tab_id" value="">
                    <div class="form-group">
                        <label for="tab-name"><?php _e('Etiqueta do separador', 'custom-group-tabs'); ?></label>
                        <input type="text" id="tab-name" name="tab_name" required>
                    </div>
                    <div class="form-group">
                        <label for="tab-slug"><?php _e('Slug do separador', 'custom-group-tabs'); ?></label>
                        <input type="text" id="tab-slug" name="tab_slug">
                    </div>
                    <div class="form-group">
                        <label for="tab-position"><?php _e('Posição do separador', 'custom-group-tabs'); ?></label>
                        <input type="number" id="tab-position" name="tab_position" value="100">
                    </div>
                    <div class="form-group">
                        <label><?php _e('Visibilidade do separador', 'custom-group-tabs'); ?></label><br>
                        <?php foreach ($tab_visibility_options as $key => $label): ?>
                            <label><input type="checkbox" name="tab_visible_for[]" value="<?php echo esc_attr($key); ?>" <?php echo ($key === 'all') ? 'checked' : ''; ?>> <?php echo esc_html($label); ?></label> &nbsp;
                        <?php endforeach; ?>
                    </div>
                    <div class="form-group">
                        <label><input type="checkbox" name="tab_is_default_component" value="1"> <?php _e('Definir como componente predefinido', 'custom-group-tabs'); ?></label>
                    </div>
                    <div class="form-group">
                        <label><input type="checkbox" name="tab_is_active" value="1" checked> <?php _e('O separador está ativo', 'custom-group-tabs'); ?></label>
                    </div>
                    <div class="form-group">
                        <label><?php _e('Sub-separadores', 'custom-group-tabs'); ?></label>
                        <div id="subtabs-container"></div>
                        <button type="button" id="add-subtab" class="button">+ <?php _e('Adicionar sub-separador', 'custom-group-tabs'); ?></button>
                    </div>
                    <button type="submit" class="button button-primary"><?php _e('Guardar separador', 'custom-group-tabs'); ?></button>
                </form>
            </div>

            <style>
            .custom-tabs-modal {
                display: none;
                position: fixed;
                z-index: 9999;
                left: 0;
                top: 0;
                width: 100vw;
                height: 100vh;
                background: rgba(0,0,0,0.7);
                align-items: center;
                justify-content: center;
                overflow: auto;
            }
            .custom-tabs-modal-content {
                background: #fff;
                padding: 32px 24px;
                border-radius: 10px;
                max-width: 900px;
                width: 98vw;
                margin: 0 auto;
                max-height: 60vh;
                overflow-y: auto;
                overflow-x: auto;
                position: relative;
                box-shadow: 0 8px 32px rgba(0,0,0,0.25);
                box-sizing: border-box;
            }
            .custom-tabs-modal-content .wp-list-table {
                max-width: 100%;
                width: 100%;
                table-layout: auto;
                display: block;
                overflow-x: auto;
                box-sizing: border-box;
            }
            .custom-tabs-modal-content .wp-list-table thead,
            .custom-tabs-modal-content .wp-list-table tbody {
                display: table;
                width: 100%;
                table-layout: auto;
            }
            .custom-tabs-modal-close {
                position: absolute;
                top: 16px;
                right: 16px;
                cursor: pointer;
                font-size: 22px;
                background: #fff;
                width: 32px;
                height: 32px;
                text-align: center;
                line-height: 32px;
                border-radius: 50%;
                z-index: 1;
            }
            .custom-tabs-management {
                max-width: 650px;
                margin: 0 auto;
                padding: 0;
                background: none;
                border-radius: 0;
                box-shadow: none;
            }
            .custom-tabs-management h2 {
                margin-bottom: 20px;
                color: #222;
                font-size: 26px;
                font-weight: 700;
            }
            table.wp-list-table.widefat.fixed.striped {
                position: relative;
            }
            .wp-list-table {
                width: 100%;
                border-collapse: collapse;
                background: #fff;
                border-radius: 8px;
                overflow: hidden;
                box-shadow: 0 2px 8px rgba(0,0,0,0.06);
            }
            .wp-list-table th, .wp-list-table td {
                padding: 12px 10px;
                text-align: left;
            }
            .wp-list-table th {
                background: #f7f7f7;
                font-weight: 600;
                color: #333;
                border-bottom: 1px solid #e0e0e0;
            }
            .wp-list-table tr:nth-child(even) {
                background: #fafbfc;
            }
            .wp-list-table tr:nth-child(odd) {
                background: #fff;
            }
            .wp-list-table td {
                color: #222;
                font-size: 15px;
            }
            .button, .button.button-primary {
                background: #2563eb;
                color: #fff;
                border: none;
                padding: 7px 18px;
                border-radius: 4px;
                cursor: pointer;
                font-size: 15px;
                font-weight: 500;
                margin-right: 6px;
                transition: background 0.2s;
            }
            .button.button-primary, .button.edit-tab {
                background: #2563eb;
            }
            .button.button-primary:hover, .button.edit-tab:hover {
                background: #1741a6;
            }
            .delete-tab {
                background: #dc3545 !important;
                color: #fff !important;
            }
            .delete-tab:hover {
                background: #b52a37 !important;
            }
            #add-new-tab {
                margin-bottom: 18px;
            }
            #back-to-list {
                margin-bottom: 18px;
            }
            .form-group {
                margin-bottom: 18px;
            }
            .form-group label {
                display: block;
                margin-bottom: 7px;
                font-weight: 500;
                color: #444;
            }
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group textarea {
                width: 100%;
                padding: 8px 12px;
                border: 1px solid #ddd;
                border-radius: 4px;
                font-size: 14px;
            }
            .form-group input[type="checkbox"] {
                margin-right: 8px;
            }
            .subtab-group {
                background: #f9f9f9;
                border: 1px solid #e0e0e0 !important;
                border-radius: 6px;
                padding: 15px !important;
                margin-bottom: 15px !important;
            }
            .subtab-group label {
                display: block;
                margin-bottom: 10px;
            }
            .subtab-group input[type="text"],
            .subtab-group textarea {
                width: 100%;
                margin-bottom: 10px;
            }
            .remove-subtab {
                background: #dc3545;
                color: white;
                border: none;
                padding: 5px 10px;
                border-radius: 4px;
                cursor: pointer;
            }
            .remove-subtab:hover {
                background: #c82333;
            }
            #add-subtab {
                background: #28a745;
                color: white;
                border: none;
                padding: 8px 15px;
                border-radius: 4px;
                cursor: pointer;
                margin-bottom: 20px;
            }
            #add-subtab:hover {
                background: #218838;
            }
            button[type="submit"] {
                background: #2563eb;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 4px;
                cursor: pointer;
                font-size: 16px;
            }
            button[type="submit"]:hover {
                background: #1741a6;
            }
            @media (max-width: 600px) {
                .custom-tabs-modal-content {
                    padding: 12px 4px;
                    max-width: 99vw;
                }
                .wp-list-table th, .wp-list-table td {
                    padding: 8px 4px;
                    font-size: 13px;
                }
            }
            </style>
            <script>
            jQuery(document).ready(function($) {
                // Show form view
                $('#add-new-tab').on('click', function() {
                    $('#tabs-list-view').hide();
                    $('#tab-form-view').show();
                    $('#add-custom-tab')[0].reset();
                    $('#tab-id').val('');
                    $('#subtabs-container').empty();
                    addSubtab();
                });

                // Back to list view
                $('#back-to-list').on('click', function() {
                    $('#tab-form-view').hide();
                    $('#tabs-list-view').show();
                });

                // Edit tab
                $('.edit-tab').on('click', function() {
                    var tabId = $(this).data('tab-id');
                    var $button = $(this);
                    $button.prop('disabled', true);
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'get_custom_tab',
                            tab_id: tabId,
                            group_id: '<?php echo $group_id; ?>',
                            nonce: '<?php echo wp_create_nonce('get_custom_tab'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                var tab = response.data;
                                $('#tab-id').val(tab.id);
                                $('#tab-name').val(tab.name);
                                $('#tab-slug').val(tab.slug);
                                $('#tab-position').val(tab.position);
                                
                                // Reset checkboxes
                                $('input[name="tab_visible_for[]"]').prop('checked', false);
                                $('input[name="tab_is_default_component"]').prop('checked', false);
                                $('input[name="tab_is_active"]').prop('checked', false);
                                
                                // Set checkboxes based on tab data
                                if (tab.visible_for) {
                                    tab.visible_for.forEach(function(value) {
                                        $('input[name="tab_visible_for[]"][value="' + value + '"]').prop('checked', true);
                                    });
                                }
                                if (tab.is_default_component === 'on') {
                                    $('input[name="tab_is_default_component"]').prop('checked', true);
                                }
                                if (tab.is_active === 'on') {
                                    $('input[name="tab_is_active"]').prop('checked', true);
                                }
                                
                                // Clear and rebuild subtabs
                                $('#subtabs-container').empty();
                                if (tab.subnav_items) {
                                    tab.subnav_items.forEach(function(subtab) {
                                        addSubtab(
                                            subtab.label,
                                            subtab.slug,
                                            subtab.content,
                                            subtab.position,
                                            subtab.is_active === 'on'
                                        );
                                    });
                                } else {
                                    addSubtab();
                                }
                                
                                $('#tabs-list-view').hide();
                                $('#tab-form-view').show();
                            } else {
                                alert(response.data || '<?php _e('Error loading tab data.', 'custom-group-tabs'); ?>');
                            }
                            $button.prop('disabled', false);
                        },
                        error: function() {
                            alert('<?php _e('Error loading tab data.', 'custom-group-tabs'); ?>');
                            $button.prop('disabled', false);
                        }
                    });
                });

                // Delete tab
                $('.delete-tab').on('click', function() {
                    if (!confirm('<?php _e('Tem a certeza de que pretende eliminar este separador?', 'custom-group-tabs'); ?>')) {
                        return;
                    }
                    var tabId = $(this).data('tab-id');
                    var $row = $(this).closest('tr');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'delete_custom_tab',
                            tab_id: tabId,
                            group_id: '<?php echo $group_id; ?>',
                            nonce: '<?php echo wp_create_nonce('delete_custom_tab'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $row.fadeOut(400, function() {
                                    $(this).remove();
                                    if ($('.wp-list-table tbody tr').length === 0) {
                                        location.reload();
                                    }
                                });
                            } else {
                                alert(response.data || '<?php _e('Error deleting tab.', 'custom-group-tabs'); ?>');
                            }
                        }
                    });
                });

                function renderRemoveButtons() {
                    var subtabs = $('#subtabs-container .subtab-group');
                    if (subtabs.length === 1) {
                        subtabs.find('.remove-subtab').hide();
                    } else {
                        subtabs.find('.remove-subtab').show();
                    }
                }

                function addSubtab(label = '', slug = '', content = '', position = 100, is_active = true) {
                    var checked = is_active ? 'checked' : '';
                    var html = '<div class="subtab-group" style="border:1px solid #ccc;padding:10px;margin-bottom:10px;">' +
                        '<label><?php _e('Etiqueta do sub-separador', 'custom-group-tabs'); ?> <input type="text" name="subtab_label[]" value="' + label + '" required></label><br>' +
                        '<label><?php _e('Slug do sub-separador', 'custom-group-tabs'); ?> <input type="text" name="subtab_slug[]" value="' + slug + '" required></label><br>' +
                        '<label><?php _e('Conteúdo do sub-separador', 'custom-group-tabs'); ?><textarea name="subtab_content[]" rows="3">' + content + '</textarea></label><br>' +
                        '<label><?php _e('Posição do sub-separador', 'custom-group-tabs'); ?> <input type="number" name="subtab_position[]" value="' + position + '"></label><br>' +
                        '<label><input type="checkbox" name="subtab_is_active[]" value="1" ' + checked + '> <?php _e('O sub-separador está ativo', 'custom-group-tabs'); ?></label><br>' +
                        '<button type="button" class="remove-subtab button">&times; <?php _e('Remover', 'custom-group-tabs'); ?></button>' +
                        '</div>';
                    var $el = $(html);
                    $('#subtabs-container').append($el);
                    renderRemoveButtons();
                }

                // Add one subtab by default if none exists
                if ($('#subtabs-container .subtab-group').length === 0) {
                    addSubtab();
                }

                $('#add-subtab').on('click', function(e) {
                    e.preventDefault();
                    addSubtab();
                });

                $(document).on('click', '.remove-subtab', function() {
                    $(this).closest('.subtab-group').remove();
                    renderRemoveButtons();
                });

                renderRemoveButtons();

                $('#add-custom-tab').on('submit', function(e) {
                    e.preventDefault();
                    
                    // Require at least one sub-tab
                    if ($('#subtabs-container .subtab-group').length < 1) {
                        alert('<?php _e('Adicione pelo menos um sub-separador.', 'custom-group-tabs'); ?>');
                        return;
                    }

                    // Require all sub-tab label and slug fields
                    var valid = true;
                    $('#subtabs-container .subtab-group').each(function() {
                        var label = $(this).find('input[name="subtab_label[]"]').val();
                        var slug = $(this).find('input[name="subtab_slug[]"]').val();
                        if (!label || !slug) {
                            valid = false;
                        }
                    });
                    if (!valid) {
                        alert('<?php _e('Todos os sub-separadores devem ter uma etiqueta e um slug.', 'custom-group-tabs'); ?>');
                        return;
                    }

                    var formData = new FormData(this);
                    formData.append('action', 'save_custom_tab');
                    formData.append('group_id', '<?php echo $group_id; ?>');
                    formData.append('nonce', '<?php echo wp_create_nonce('save_custom_tab'); ?>');

                    // Add subtab data
                    var subtabData = [];
                    $('#subtabs-container .subtab-group').each(function(i, el) {
                        var $el = $(el);
                        subtabData.push({
                            label: $el.find('input[name="subtab_label[]"]').val(),
                            slug: $el.find('input[name="subtab_slug[]"]').val(),
                            content: $el.find('textarea[name="subtab_content[]"]').val(),
                            position: $el.find('input[name="subtab_position[]"]').val(),
                            is_active: $el.find('input[name="subtab_is_active[]"]').is(':checked') ? '1' : ''
                        });
                    });
                    formData.append('subtab_data', JSON.stringify(subtabData));

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: formData,
                        processData: false,
                        contentType: false,
                        success: function(response) {
                            if (response.success) {
                                location.reload();
                            } else {
                                alert(response.data || '<?php _e('Error saving tab.', 'custom-group-tabs'); ?>');
                            }
                        }
                    });
                });
            });
            </script>
        </div>
        <?php
    }

    public function save_custom_tab() {
        check_ajax_referer('save_custom_tab', 'nonce');
        
        $group_id = intval($_POST['group_id']);
        if (!groups_is_user_admin(get_current_user_id(), $group_id)) {
            wp_send_json_error('Unauthorized');
        }

        $tab_id = !empty($_POST['tab_id']) ? intval($_POST['tab_id']) : 0;
        $tab_name = sanitize_text_field($_POST['tab_name']);
        $tab_slug = sanitize_title($_POST['tab_slug'] ?: $tab_name);
        $tab_position = isset($_POST['tab_position']) ? intval($_POST['tab_position']) : 100;
        $tab_visible_for = isset($_POST['tab_visible_for']) ? array_map('sanitize_text_field', (array)$_POST['tab_visible_for']) : ['all'];
        $tab_is_default_component = !empty($_POST['tab_is_default_component']) ? 'on' : '';
        $tab_is_active = !empty($_POST['tab_is_active']) ? 'on' : '';

        // Process sub-tabs data
        $subtab_data = isset($_POST['subtab_data']) ? json_decode(stripslashes($_POST['subtab_data']), true) : [];
        $subtabs = [];

        if (!empty($subtab_data)) {
            foreach ($subtab_data as $subtab) {
                if (empty($subtab['label']) || empty($subtab['slug'])) {
                    continue;
                }

                $subtabs[] = array(
                    'label' => sanitize_text_field($subtab['label']),
                    'slug' => sanitize_title($subtab['slug'] ?: $subtab['label']),
                    'content' => wp_kses_post($subtab['content']),
                    'position' => (string)intval($subtab['position']),
                    'is_active' => (!empty($subtab['is_active']) && $subtab['is_active'] === '1') ? 'on' : '',
                    'enabled_for' => $tab_visible_for,
                    'visible_for' => $tab_visible_for
                );
            }
        }

        if (empty($subtabs)) {
            wp_send_json_error('At least one sub-tab is required');
            return;
        }

        // Create/update the tab in the BuddyBoss Group Tabs Creator Pro post type
        if ($tab_id) {
            // Update existing tab
            $post_id = $tab_id;
            wp_update_post(array(
                'ID' => $post_id,
                'post_title' => $tab_name,
                'post_name' => $tab_slug
            ));
        } else {
            // Create new tab
            $args = array(
                'post_type' => 'bpgtc_group_tab',
                'name' => $tab_slug,
                'posts_per_page' => 1,
                'post_status' => 'publish'
            );
            
            $existing_post = get_posts($args);
            
            if (!empty($existing_post)) {
                $post_id = $existing_post[0]->ID;
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_title' => $tab_name,
                    'post_name' => $tab_slug
                ));
            } else {
                $post_id = wp_insert_post(array(
                    'post_title' => $tab_name,
                    'post_name' => $tab_slug,
                    'post_status' => 'publish',
                    'post_type' => 'bpgtc_group_tab'
                ));
            }
        }

        if (!is_wp_error($post_id) && $post_id) {
            // Add CMB2 nonce to $_POST for proper saving
            $_POST['_wpnonce'] = wp_create_nonce('update-post_' . $post_id);
            
            // Direct call to save_fields
            if (class_exists('CMB2')) {
                $cmb = cmb2_get_metabox('bpgtc_group_tab_metabox', $post_id);
                if ($cmb) {
                    $cmb->save_fields($post_id, 'post', array(
                        '_bpgtc_tab_is_active' => $tab_is_active,
                        '_bpgtc_tab_label' => $tab_name,
                        '_bpgtc_tab_slug' => $tab_slug,
                        '_bpgtc_tab_visible_for' => $tab_visible_for,
                        '_bpgtc_tab_groups' => array((string)$group_id),
                        '_bpgtc_tab_position' => $tab_position,
                        '_bpgtc_is_default_component' => $tab_is_default_component,
                        '_bpgtc_subnav_items' => $subtabs,
                        '_bpgtc_tab_enabled_for' => $tab_visible_for,
                        '_bpgtc_selected_group_types' => array()
                    ));
                }
            }
            
            // Save all meta data
            update_post_meta($post_id, '_bpgtc_tab_is_active', $tab_is_active);
            update_post_meta($post_id, '_bpgtc_tab_label', $tab_name);
            update_post_meta($post_id, '_bpgtc_tab_slug', $tab_slug);
            update_post_meta($post_id, '_bpgtc_tab_visible_for', $tab_visible_for);
            update_post_meta($post_id, '_bpgtc_tab_groups', array((string)$group_id));
            update_post_meta($post_id, '_bpgtc_tab_position', $tab_position);
            update_post_meta($post_id, '_bpgtc_is_default_component', $tab_is_default_component);
            update_post_meta($post_id, '_bpgtc_tab_enabled_for', $tab_visible_for);
            update_post_meta($post_id, '_bpgtc_selected_group_types', array());
            update_post_meta($post_id, '_bpgtc_subnav_items', $subtabs);
            
            // Clear BuddyBoss specific caches
            if (function_exists('bp_core_clear_nav_cache')) {
                bp_core_clear_nav_cache();
            }
            
            // Force refresh of BuddyBoss navigation
            if (class_exists('PressThemes\BPGTC\BuddyPress\Groups\Group_Tabs_Actions')) {
                $tabs_actions = new \PressThemes\BPGTC\BuddyPress\Groups\Group_Tabs_Actions();
                $tabs_actions->setup_nav();
            }

            // Force a hard refresh of the page to ensure navigation is rebuilt
            wp_send_json_success(array(
                'redirect' => bp_get_group_permalink(groups_get_group($group_id))
            ));
        }
        wp_send_json_error('Failed to save tab');
    }

    public function delete_custom_tab() {
        check_ajax_referer('delete_custom_tab', 'nonce');
        
        $group_id = intval($_POST['group_id']);
        if (!groups_is_user_admin(get_current_user_id(), $group_id)) {
            wp_send_json_error('Unauthorized');
        }

        $tab_id = intval($_POST['tab_id']);
        if ($tab_id) {
            wp_delete_post($tab_id, true);
        }

        wp_send_json_success();
    }

    public function get_custom_tab() {
        check_ajax_referer('get_custom_tab', 'nonce');
        
        $tab_id = intval($_POST['tab_id']);
        $group_id = intval($_POST['group_id']);
        
        if (!groups_is_user_admin(get_current_user_id(), $group_id)) {
            wp_send_json_error('Unauthorized');
        }

        $tab = get_post($tab_id);
        if (!$tab || $tab->post_type !== 'bpgtc_group_tab') {
            wp_send_json_error('Tab not found');
        }

        $tab_data = array(
            'id' => $tab->ID,
            'name' => get_post_meta($tab->ID, '_bpgtc_tab_label', true),
            'slug' => get_post_meta($tab->ID, '_bpgtc_tab_slug', true),
            'is_active' => get_post_meta($tab->ID, '_bpgtc_tab_is_active', true),
            'position' => get_post_meta($tab->ID, '_bpgtc_tab_position', true),
            'visible_for' => get_post_meta($tab->ID, '_bpgtc_tab_visible_for', true),
            'is_default_component' => get_post_meta($tab->ID, '_bpgtc_is_default_component', true),
            'subnav_items' => get_post_meta($tab->ID, '_bpgtc_subnav_items', true)
        );

        wp_send_json_success($tab_data);
    }
}

// Initialize the plugin
new Custom_BuddyBoss_Group_Tabs();

// Register the group extension on bp_init for reliability
add_action('bp_init', function() {
    if (class_exists('BP_Group_Extension')) {
        bp_register_group_extension('Custom_Group_Tabs_Extension');
    }
});

// Add this class at the end of the file, before plugin initialization
if (class_exists('BP_Group_Extension')) {
    class Custom_Group_Tabs_Extension extends BP_Group_Extension {
        public function __construct() {
            $args = array(
                'slug' => 'custom-tabs',
                'name' => __('Custom Tabs', 'custom-group-tabs'),
                'nav_item_position' => 105,
                'screens' => array(
                    'edit' => array('enabled' => true, 'name' => __('Custom Tabs', 'custom-group-tabs')),
                ),
            );
            parent::init($args);
        }

        public function edit_screen($group_id = null) {
            // Only group admins can access
            if (!groups_is_user_admin(get_current_user_id(), bp_get_current_group_id())) {
                echo '<div class="bp-feedback error">' . __('Não tem permissão para gerir separadores.', 'custom-group-tabs') . '</div>';
                return;
            }
            // The management UI is now only available via the modal (pencil button)
            echo '<p>' . __('Utilize o botão do lápis no cabeçalho do grupo para gerir os separadores personalizados.', 'custom-group-tabs') . '</p>';
        }

        public function edit_screen_save($group_id = null) {
            // No-op, AJAX handles saving
        }
    }
    bp_register_group_extension('Custom_Group_Tabs_Extension');
}

// Hook to render the form in the extension
add_action('custom_group_tabs_render_form', function() {
    $plugin = new Custom_BuddyBoss_Group_Tabs();
    $plugin->render_tabs_management();
});

// Add pencil button to group header actions for Organizers
add_action('bp_group_header_actions', function() {
    static $rendered = false;
    if ($rendered) return;
    $rendered = true;
    if (!bp_is_group()) return;
    $group_id = bp_get_current_group_id();
    $user_id = get_current_user_id();
    if (!groups_is_user_admin($user_id, $group_id)) return;
    ?>
    <style>
    .custom-tabs-pencil-btn {
        display: inline-block;
        margin-left: 8px;
        padding: 8px 20px !important;
        cursor: pointer;
        vertical-align: middle;
        background: none;
        border: none;
        font-size: 18px;
    }
    .custom-tabs-modal {
        display: none;
        position: fixed;
        z-index: 9999;
        left: 0;
        top: 0;
        width: 100vw;
        height: 100vh;
        background: rgba(0,0,0,0.4);
        align-items: center;
        justify-content: center;
        overflow: hidden;
    }
    .custom-tabs-modal-content {
        background: #fff;
        padding: 32px 24px;
        border-radius: 10px;
        max-width: 900px;
        width: 98vw;
        margin: 0 auto;
        max-height: 60vh;
        overflow-y: auto;
        position: relative;
        box-shadow: 0 8px 32px rgba(0,0,0,0.25);
    }
    .custom-tabs-modal-close {
        position: sticky;
        top: 0;
        right: 0;
        float: right;
        cursor: pointer;
        font-size: 20px;
        background: #fff;
        width: 24px;
        height: 24px;
        text-align: center;
        line-height: 24px;
        border-radius: 50%;
        margin-bottom: 10px;
        z-index: 1;
    }
    </style>
    <button class="custom-tabs-pencil-btn" id="customTabsPencilBtn" title="Manage Tabs">Gerir separadores</button>
    <div class="custom-tabs-modal" id="customTabsModal">
        <div class="custom-tabs-modal-content">
            <span class="custom-tabs-modal-close" id="customTabsModalClose">&times;</span>
            <?php
            $plugin = new Custom_BuddyBoss_Group_Tabs();
            $plugin->render_tabs_management();
            ?>
        </div>
    </div>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        var btn = document.getElementById('customTabsPencilBtn');
        var modal = document.getElementById('customTabsModal');
        var close = document.getElementById('customTabsModalClose');
        if(btn && modal && close) {
            btn.onclick = function() { 
                modal.style.display = 'flex'; 
                document.body.style.overflow = 'hidden';
            };
            close.onclick = function() { 
                modal.style.display = 'none'; 
                document.body.style.overflow = '';
            };
            window.onclick = function(event) { 
                if(event.target == modal) {
                    modal.style.display = 'none'; 
                    document.body.style.overflow = '';
                }
            };
        }
    });
    </script>
    <?php
}); 