<?php
/**
 * General functions file.
 *
 * @package BuddyBossApp
 */

use BuddyBossApp\AppSettings;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Auth\Common;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\NativeAppPage;
use BuddyBossApp\Tools\Logger;
use BuddyBossApp\ManageApp;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Return true or false based on private app is enabled or not.
 *
 * @return bool
 */
function bbapp_is_private_app_enabled() {
	$settings = \BuddyBossApp\ManageApp::instance()->get_settings();
	$enabled  = ( isset( $settings['private_app.enabled'] ) && $settings['private_app.enabled'] );

	return $enabled;
}

/**
 * Just a function to validate a URL.
 *
 * @param string $url Url.
 *
 * @return bool
 */
function bbapp_is_valid_url( $url ) {
	// Must start with http:// or https://.
	if ( 0 !== strpos( $url, 'http://' ) && 0 !== strpos( $url, 'https://' ) ) {
		return false;
	}

	// Must pass validation.
	if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
		return false;
	}

	return true;
}

/**
 * It's a alias for wp_remote_get but allows filters.
 *
 * @param string $url Url.
 * @param array  $args Arguments.
 *
 * @return array|WP_Error
 */
function bbapp_remote_get( $url, $args = array() ) {
	/**
	 * Filter rest url.
	 *
	 * @param string           $url      Rest url to get response.
	 * @param array            $args     Rest arguments.
	 */
	$url = apply_filters( 'bbapp_remote_get_url', $url, $args );

	/**
	 * Filters rest request arguments.
	 *
	 * @param array  $args Rest arguments.
	 * @param string $url  Rest url to get response.
	 */
	$args = apply_filters( 'bbapp_remote_get_args', $args, $url );

	$response = wp_remote_get( $url, $args );

	/**
	 * Filters remote response.
	 *
	 * @param WP_REST_Response $response Rest reponse.
	 * @param string           $url      Rest url to get response.
	 * @param array            $args     Rest arguments.
	 */
	$response = apply_filters( 'bbapp_remote_get_response', $response, $url, $args );

	return $response;
}

/**
 * It's a alias for wp_remote_post but allows filters.
 *
 * @param string $url  Url.
 * @param array  $args Arguments.
 *
 * @return array|WP_Error
 */
function bbapp_remote_post( $url, $args = array() ) {

	$url      = apply_filters( 'bbapp_remote_post_url', $url, $args );
	$args     = apply_filters( 'bbapp_remote_post_args', $args, $url );
	$response = wp_remote_post( $url, $args );
	$response = apply_filters( 'bbapp_remote_post_response', $response, $url, $args );

	return $response;
}

/**
 * Alias Function for accessing all apps.
 *
 * @return array|mixed|void
 */
function bbapp_get_app() {
	return \BuddyBossApp\ManageApp::instance()->get_app();
}

/**
 * Return the registered var from header
 *
 * @param string $name Name.
 *
 * @return bool
 */
function bbapp_get_var( $name ) {
	global $bbapp_var;

	if ( ! isset( $bbapp_var[ $name ] ) ) {
		return false;
	}

	return $bbapp_var[ $name ];
}

/**
 * Returns JWT jti.
 *
 * @param string $token Token.
 *
 * @return bool
 */
function bbapp_get_jwt_jti( $token ) {
	$token = explode( '.', $token );
	$token = (array) json_decode( base64_decode( $token[1] ) ); // phpcs:ignore

	if ( ! isset( $token['jti'] ) ) {
		$token['jti'] = false;
	}

	return $token['jti'];
}

/**
 * Download URL into WordPress attachment.
 *
 * @param string $url Media url.
 *
 * @return array|mixed|WP_Error
 */
function bbapp_download_to_media( $url ) {
	require_once ABSPATH . 'wp-admin/includes/file.php';

	$timeout_seconds = 5;
	$temp_file       = download_url( $url, $timeout_seconds );

	if ( ! is_wp_error( $temp_file ) ) {
		// Array based on $_FILE as seen in PHP file uploads.
		$file = array(
			'name'     => basename( $url ), // ex: wp-header-logo.png.
			'type'     => 'image/png',
			'tmp_name' => $temp_file,
			'error'    => 0,
			'size'     => filesize( $temp_file ),
		);

		$overrides = array(
			'test_form' => false,
			'test_size' => true,
		);

		// Move the temporary file into the uploads directory.
		$results = wp_handle_sideload( $file, $overrides );

		if ( ! empty( $results['error'] ) ) {
			return new \WP_Error( 'error_uploading', $results['error'], array( 'status' => 500 ) );
		} else {
			return $results;
			// Perform any actions here based in the above results.
		}
	}

	return $temp_file;
}

/**
 * Get Networks Options.
 *
 * @param int|string $option Option name.
 *
 * @return mixed|void
 */
function bbapp_get_network_option( $option ) {
	if ( bbapp()->is_network_activated() ) {
		$values = get_network_option( 1, $option );
	} else {
		$values = get_option( $option );
	}

	return $values;
}

/**
 * Update Network options
 *
 * @param int|string $option   Option name.
 * @param int|string $value    Option value.
 * @param null       $autoload If needs to autoload.
 *
 * @return bool
 */
function bbapp_set_network_option( $option, $value, $autoload = null ) {
	if ( bbapp()->is_network_activated() ) {
		return update_network_option( 1, $option, $value );
	} else {
		return update_option( $option, $value, $autoload );
	}
}

/**
 * Delete Network options
 *
 * @param string $option Option name.
 *
 * @return bool
 */
function bbapp_delete_network_option( $option ) {
	if ( bbapp()->is_network_activated() ) {
		return delete_network_option( 1, $option );
	} else {
		return delete_option( $option );
	}
}

/**
 * Return the BuddyBossApp Network Table.
 *
 * @param string $table_name Table name.
 *
 * @return string
 */
function bbapp_get_network_table( $table_name ) {
	$table_name = bbapp_get_global_db_prefix() . $table_name;

	return $table_name;
}

/**
 * Return the global db prefix.
 *
 * @return mixed
 */
function bbapp_get_global_db_prefix() {
	global $wpdb;
	static $bbapp_global_prefix;

	if ( ! isset( $bbapp_global_prefix ) ) {
		$bbapp_global_prefix = $wpdb->prefix;

		if ( is_multisite() ) {
			switch_to_blog( 1 );
			$bbapp_global_prefix = $wpdb->prefix;
			restore_current_blog();
		}
	}

	return $bbapp_global_prefix;
}

/**
 * Upgrade & fix the table data.
 */
function bbapp_table_fix_wp_bbapp_user_devices() {
	global $wpdb;

	$table = bbapp_get_network_table( 'bbapp_user_devices' );
	// Upgrade hashes of tokens.
	$wpdb->query( "UPDATE {$table} SET device_token_hash=SHA2(device_token,256) WHERE length(device_token_hash) < 64 " ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	// if found empty column change them to firebase as they probably old one.
	$wpdb->update( $table, array( 'push_type' => 'firebase' ), array( 'push_type' => '' ), array( '%s' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

/**
 * Upgrade & fix the table data.
 */
function bbapp_table_fix_remove_dublicates_device_tokens() {
	global $wpdb;

	$table = bbapp_get_network_table( 'bbapp_user_devices' );
	// Removes duplicates device tokens by device token. because there can be only 1 device token uniquely.
	$wpdb->query( "DELETE t1 FROM {$table} t1, {$table} t2 WHERE t1.id < t2.id AND t1.device_token = t2.device_token" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	// Remove dublicates auth tokens as there can be only 1 auth token per device and per push notification assigned.
	$wpdb->query( "DELETE t1 FROM {$table} t1, {$table} t2 WHERE t1.id < t2.id AND t1.user_id = t2.user_id AND t1.auth_token_hash = t2.auth_token_hash" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
}

/**
 * Delete all the notification device table rows.
 */
function bbapp_table_remove_all_device_tokens() {
	global $wpdb;

	$table = bbapp_get_network_table( 'bbapp_user_devices' );
	$exec  = $wpdb->query( "DELETE FROM {$table}" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

	if ( ! empty( $exec ) ) {
		Logger::instance()->add( 'all_device_tokens', sprintf( __( 'Deleted all device tokens.', 'buddyboss-app' ), $exec ) );
	}
}

/**
 * Converts a relative url scheme to current used one.
 *
 * @param string $url URL.
 *
 * @return string
 */
function bbapp_fix_url_scheme( $url ) {
	if ( '//' === substr( $url, 0, 2 ) ) {
		$url = str_replace( '//', '', $url );
	}

	$url = str_replace( array( 'http://', 'https://' ), '', $url );

	if ( is_ssl() ) {
		$url = "https://{$url}";
	} else {
		$url = "http://{$url}";
	}

	return $url;
}

/**
 * Get a list of all Multi Languages.
 *
 * @return array List of all multi languages.
 */
function bbapp_get_multi_languages_list() {
	$languages = array();

	/**
	 * Filters multi language list.
	 *
	 * @param array $languages Languages.
	 */
	return apply_filters( 'bbapp_multi_languages_list', $languages );
}

/**
 * Return whether string has lang string contains.
 *
 * @param string $string String.
 *
 * @return bool
 */
function bbapp_has_language( $string ) {
	$reg = '/<!--:-->/';

	if ( preg_match( $reg, $string ) ) {
		return true;
	}

	return false;
}

/**
 * Filter the language string based on given lang code.
 *
 * @param string $string            String.
 * @param bool   $lang_code         Language code.
 * @param bool   $default_lang_code Default language code.
 *
 * @return string
 */
function bbapp_get_language( $string, $lang_code = false, $default_lang_code = false ) {
	// Get Current Lang Code if not requested.
	if ( empty( $lang_code ) ) {
		$lang_code = bbapp_get_current_lang_code();
	}

	$has_language = bbapp_has_language( $string );

	// Return Actual string if not have language code.
	if ( empty( $has_language ) ) {
		return $string;
	}

	// Set Default lang code if true.
	if ( $default_lang_code ) {
		$lang_code = 'en';
	}

	// Get Current Lang Code.
	$current_lang_code = bbapp_get_current_lang_code();
	// Get Lang posh default lang code.
	$default_lang_code = bbapp_get_default_lang_code();

	// Check if Lang code match in string then user that lang code.
	if ( preg_match( "/:$current_lang_code/m", $string ) ) {
		$lang_code = $current_lang_code;
	} elseif ( preg_match( "/:$default_lang_code/m", $string ) ) {
		$lang_code = $default_lang_code;
	} elseif ( preg_match( '/<!--:/m', $string ) ) {
			$exploded_string          = explode( '-->', $string );
			$lang_code_string_explode = explode( '<!--:', $exploded_string[0] );
			$lang_code                = $lang_code_string_explode[1];
	}

	$exposed_strings = explode( '<!--:' . $lang_code . '-->', $string );

	if ( count( $exposed_strings ) > 1 ) {
		$more_strings      = $exposed_strings[1];
		$final_explode     = explode( '<!--:-->', $more_strings );
		$final_lang_string = $final_explode[0];
	} else {
		$final_lang_string = $exposed_strings[0];
	}

	/**
	 * Modify Language string.
	 *
	 * @param string $final_lang_string Modified string
	 * @param string $string          Actual string
	 * @param string $lang_code       Lang Code
	 *
	 * @return string $final_lang_string Modified string
	 */
	return apply_filters( 'bbapp_get_language_string', $final_lang_string, $string, $lang_code );
}

/**
 * Get Current Lang Code.
 *
 * @return string
 */
function bbapp_get_current_lang_code() {
	/**
	 * Modify Current Lang code.
	 *
	 * @param string $lang_code Language code.
	 *
	 * @return string
	 */
	return apply_filters( 'bbapp_get_current_lang_code', 'en' );
}

/**
 * Modify default Lang code.
 *
 * @return string
 */
function bbapp_get_default_lang_code() {
	/**
	 * Modify default Lang code.
	 *
	 * @param string $lang_code Language code.
	 *
	 * @return string
	 */
	return apply_filters( 'bbapp_get_default_lang_code', 'en' );
}

/**
 * Get default APP Page.
 *
 * @return int|bool
 */
function bbapp_get_default_app_page() {
	$bbapp_default_apppage = get_option( 'bbapp_default_apppage', false );
	$post                  = get_post( $bbapp_default_apppage );

	if ( empty( $bbapp_default_apppage ) || empty( $post ) ) {
		// Insert Default APP Page.
		$app_page = array(
			'post_title'  => __( 'App Editor Helper', 'buddyboss-app' ),
			'post_status' => 'publish',
			'post_type'   => 'app_page',
		);

		// Insert the post into the database.
		$app_page_id = wp_insert_post( $app_page );

		update_option( 'bbapp_default_apppage', $app_page_id, false );

		$bbapp_default_apppage = get_option( 'bbapp_default_apppage', false );
	} elseif ( ! empty( $bbapp_default_apppage ) && ! empty( $post ) && 'publish' !== $post->post_status ) {
		$bbapp_default_apppage_post = array(
			'ID'          => $bbapp_default_apppage,
			'post_status' => 'publish',
		);

		// Update the post into the database.
		wp_update_post( $bbapp_default_apppage_post );
	}

	if ( ! empty( $bbapp_default_apppage ) ) {
		return absint( $bbapp_default_apppage );
	}

	return $bbapp_default_apppage;
}

/**
 * Parse Gutenberg Content into App Page Parser.
 *
 * @param string $content Content.
 *
 * @return array $new_app_page_data
 */
function bbapp_parse_gutenberg_content( $content ) {
	$blocks = parse_blocks( $content );

	return bbapp_parse_inner_block( $blocks );
}

/**
 * Parse Blocks of App Editor.
 *
 * @param array $blocks Blocks.
 * @param array $parent_style Parent block style.
 *
 * @return array
 */
function bbapp_parse_inner_block( $blocks, $parent_style = array() ) {
	$new_app_page_data = \BuddyBossApp\NativeAppPage\ParseBlocks::parse_gutenberg_blocks( $blocks, $parent_style );

	return $new_app_page_data;
}

/**
 * Create excerpt form Blocks of App Editor
 * We will checking if any paragraph added in bbapp editor. Using first paragraph text as excerpt
 *
 * @param string $content Content.
 *
 * @return string|string[]
 */
function bbapp_parse_gutenberg_excerpt( $content ) {
	$content = wp_strip_all_tags( $content );
	$content = trim( $content );

	if ( empty( $content ) ) {
		return $content;
	}

	return wp_trim_words( $content, 55, '...' );
}

/**
 * Return the formatted ajax error.
 *
 * @param int    $code    Code status.
 * @param string $message Message.
 */
function bbapp_ajax_error_response( $code, $message ) {
	$data = array(
		'code'    => $code,
		'message' => $message,
	);
	wp_send_json_error( $data );
}

/**
 * Return the formatted ajax error.
 *
 * @param int    $code    Code status.
 * @param string $message Message.
 */
function bbapp_ajax_success_response( $code, $message ) {
	$data = array(
		'message' => $message,
		'code'    => $code,
	);

	wp_send_json_success( $data );
}

/**
 * Converts given text to camelcase based on given separator.
 *
 * @param string $input     Input text.
 * @param string $separator Separator.
 *
 * @return mixed
 */
function bbapp_camelize( $input, $separator = '_' ) {
	return str_replace( $separator, '', ucwords( $input, $separator ) );
}

/**
 * Returns the registered services.
 *
 * @return mixed
 */
function bbapp_get_push_services() {
	global $bbapp_push_notification_services;

	if ( ! is_array( $bbapp_push_notification_services ) ) {
		$bbapp_push_notification_services = array();
	}

	return $bbapp_push_notification_services;
}

/**
 * Returns the selected push service instance for app.
 *
 * @return bool|\BuddyBossApp\Notification\Services\ServiceAbstract
 */
function bbapp_get_app_push_instance() {
	// Get main network id's data for push notification.
	$settings      = \BuddyBossApp\Notification\Push::instance()->get_settings();
	$push_services = bbapp_get_push_services();

	/**
	 * Filters push notification service.
	 *
	 * @param string $service Push notification service.
	 */
	$settings['selected_service'] = apply_filters( 'bbapp_get_app_push_service', 'firebase' );

	if ( isset( $push_services[ $settings['selected_service'] ] ) ) {
		return $push_services[ $settings['selected_service'] ]['controller'];
	}

	return false;
}

/**
 * Returns the selected push service instance for app.
 *
 * @return bool|string
 */
function bbapp_get_app_push_firebase_key() {
	// Get main network id's data for push notification.
	$global_settings = ManageApp::instance()->get_settings( true );

	if ( ! empty( $global_settings['push.firebase_admin_key'] ) ) {
		return $global_settings['push.firebase_admin_key'];
	}

	if ( ! empty( $global_settings['push.firebase_server_key'] ) ) {
		return $global_settings['push.firebase_server_key'];
	}

	return false;
}

/**
 * Get the instance of Notification class.
 *
 * @return \BuddyBossApp\Notification\Notification
 */
function bbapp_notifications() {
	return \BuddyBossApp\Notification\Notification::instance();
}

/**
 * Helper function to send push notification to users.
 *
 * @param object|array $notification Notification.
 *
 * @return WP_Error
 */
function bbapp_send_push_notification( $notification ) {
	return \BuddyBossApp\Notification\Push::instance()->send( $notification );
}

/**
 * Return object of Auth Feature.
 *
 * @return mixed
 */
function bbapp_auth() {
	return \BuddyBossApp\Auth\Auth::instance();
}

/**
 * Returns multidimensional array unique for any single key index.
 * Reason : array_unique() is not intended to work on multi dimensional arrays.
 *
 * @param array      $multi_dimensional_array Multi dimensional array.
 * @param int|string $key                     Key.
 *
 * @return array
 */
function bbapp_unique_multidim_array( $multi_dimensional_array, $key ) {
	$temp_array = array();
	$i          = 0;
	$key_array  = array();

	foreach ( $multi_dimensional_array as $val ) {
		if ( ! in_array( $val[ $key ], $key_array, true ) ) {
			$key_array[ $i ]  = $val[ $key ];
			$temp_array[ $i ] = $val;
		}
		++$i;
	}

	return $temp_array;
}

/**
 * Tells if the site is loaded from InApp Browser or not.
 *
 * @param bool $set_cookie If needs to set cookie.
 *
 * @return bool|void
 */
function bbapp_is_loaded_from_inapp_browser( $set_cookie = false ) {
	if ( in_array( \BuddyBossApp\ClientCommon::instance()->get_page_mode( $set_cookie ), array( 'template' ), true ) || bbapp_get_page_mode_param() ) {
		return true;
	}
}

/**
 * Mobile page mode param used same as pagemode header
 *
 * @return false|mixed
 */
function bbapp_get_page_mode_param() {
	$mobile_page_mode_get = ( isset( $_GET['mobile-page-mode'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['mobile-page-mode'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

	if ( ! empty( $mobile_page_mode_get ) ) {
		return $mobile_page_mode_get;
	}

	return false;
}

/**
 * This function will allow developer to disable the default template load from BuddyBossApp App.
 * It's a API function provded to be used by 3rd party codebase.
 */
function bbapp_disable_default_inapp_browser_template() {
	add_filter(
		'bbapp_perform_pagemode_action',
		function ( $value ) {
			return false;
		}
	);
}

/**
 * Get Default Menu Icons.
 *
 * @param string $item_name Menu item.
 * @param string $type      Menu type.
 *
 * @return array
 */
function bbapp_get_menu_icon( $item_name, $type ) {
	switch ( $item_name ) {
		case 'home':
			$default_icon = array(
				'type' => 'buddyboss',
				'id'   => 'home',
			);
			break;
		case 'more':
			$default_icon = array(
				'type' => 'buddyboss',
				'id'   => 'grid-large',
			);
			break;
		case 'custom':
			$default_icon = array(
				'type' => 'buddyboss',
				'id'   => 'link',
			);
			break;
		default:
			$default_icon = array(
				'type' => 'buddyboss',
				'id'   => 'file',
			);
	}

	/**
	 * Get menu icon.
	 *
	 * @param array  $default_icon Menu default icon.
	 * @param string $item_name    Menu item.
	 * @param string $type         Menu type.
	 */
	return apply_filters( 'bbapp_get_menu_icon', $default_icon, $item_name, $type );
}

/**
 * Get current route URL.
 *
 * @return false|string
 */
function bbapp_get_current_route_url() {
	$current_path = add_query_arg( null, null );

	if ( false !== strpos( $current_path, 'wp-json/' ) ) {
		$current_path = explode( 'wp-json/', $current_path );
		$current_path = $current_path[1];

		// remove query vars.
		if ( strpos( $current_path, '?' ) !== false ) {
			$current_path = explode( '?', $current_path );
			$current_path = $current_path[0];
		}

		return trim( $current_path );

	}

	return false;
}

/**
 * Get default tab color.
 *
 * @return array
 */
function bbapp_get_default_tab_color() {

	$styling_obj     = new \BuddyBossApp\Styling();
	$styling_options = $styling_obj->get_options();

	$default_styling_option_data = $styling_obj->get_styling_colors();
	$default_styling_option      = isset( $default_styling_option_data['colors'] ) ? $default_styling_option_data['colors'] : array();
	$default_color               = isset( $default_styling_option['bottomTabsColor'] ) ? $default_styling_option['bottomTabsColor'] : '';
	$default_active_color        = isset( $default_styling_option['bottomTabsActiveColor'] ) ? $default_styling_option['bottomTabsActiveColor'] : '';

	// For Color Options.
	$styling_option_colors = array();

	if ( isset( $styling_options['styles'] ) ) {
		foreach ( $styling_options['styles'] as $style_key => $style_val ) {
			$style_keys                          = explode( '.', $style_key );
			$style_key                           = $style_keys[2];
			$styling_option_colors[ $style_key ] = $style_val;
		}
	}

	if ( isset( $styling_option_colors ) && ! empty( $styling_option_colors ) && is_array( $styling_option_colors ) ) {
		$default_color        = $styling_option_colors['bottomTabsColor'];
		$default_active_color = $styling_option_colors['bottomTabsActiveColor'];
	}

	return array(
		'default_color'        => $default_color,
		'default_active_color' => $default_active_color,
	);
}

/**
 * Get selected Icon Path.
 *
 * @param string      $icon_value Icon value.
 * @param string|bool $icon_style Icon style.
 *
 * @return string $selected Icon path.
 */
function bbapp_tabbar_get_selected_icon( $icon_value, $icon_style = false ) {
	$icons_path        = \BuddyBossApp\Common\IconPicker::instance()->icon_picker_app_icon_url();
	$custom_icons_path = \BuddyBossApp\Common\IconPicker::instance()->icon_picker_custom_icon_dir();
	$custom_icons_url  = \BuddyBossApp\Common\IconPicker::instance()->icon_picker_custom_icon_url();
	if ( 'lined' === $icon_style ) {
		$icon_style = 'outlined';
	}

	if ( false !== strpos( $icon_value, 'custom/' ) ) {
		// if it's custom uploaded icon.
		$selected_icon_path = $custom_icons_path . '/' . str_replace( 'custom/', '', $icon_value );
		$selected_icon      = $custom_icons_url . '/' . str_replace( 'custom/', '', $icon_value );

		if ( ! file_exists( $selected_icon_path ) ) {
			$selected_icon = "{$icons_path}/bbapp/file-outlined.png";
		}
	} elseif ( ! empty( $icon_value ) ) {
		if ( strpos( $icon_value, 'bbapp/' ) !== false && false !== $icon_style ) {
			$selected_icon = "{$icons_path}{$icon_value}-{$icon_style}.png";
		} else {
			$selected_icon = $icons_path . $icon_value;
		}
	} else {
		$selected_icon = $icons_path . "/file-{$icon_style}.png";
	}

	return $selected_icon;
}

/**
 * Fetch Forums though API for Admin Preview.
 *
 * @param int $user_id User ID.
 *
 * @return array
 */
function bbapp_get_profile_tabs( $user_id = 0 ) {
	$profile_tabs = get_transient( 'bbapp_preview_profile_tabs' );
	$show_tabs    = true;

	if ( empty( $profile_tabs ) ) {
		$tabs_request  = new WP_REST_Request( 'GET', '/buddyboss/v1/members/' . $user_id . '/detail' );
		$server        = rest_get_server();
		$item          = $server->dispatch( $tabs_request );
		$server_status = $item->get_status();
		$profile_tabs  = array();

		if ( 200 === $server_status ) {
			$profile_tabs = $item->get_data();
		}

		set_transient( 'bbapp_preview_profile_tabs', $profile_tabs, 12 * HOUR_IN_SECONDS );
	}

	if ( empty( $profile_tabs ) ) {
		$show_tabs = false;
	}

	return array( $profile_tabs, $show_tabs );
}

/**
 * Profile tabs icons.
 *
 * @return array
 */
function bbapp_profile_tabs_svg() {
	return array(
		'activities' => '<svg width="24" height="22" xmlns="http://www.w3.org/2000/svg"><g stroke="#000" stroke-width="1.5" fill="none" fill-rule="evenodd"><rect x=".75" y=".75" width="22.5" height="20.5" rx="4"/><path d="M5 12.199h2.349a.583.583 0 00.514-.308L9.32 9.167h0l2.33 7.333 2.515-11 2.174 6.306a.583.583 0 00.552.393H19" stroke-linecap="round" stroke-linejoin="round"/></g></svg>',
		'xprofile'   => '<svg width="20" height="24" xmlns="http://www.w3.org/2000/svg"><defs><path d="M15 16.09c5.523 0 10 2.443 10 5.455C25 24.558 25 27 15 27c-9.61 0-9.985-2.256-10-5.105v-.35c0-3.012 4.477-5.454 10-5.454zM15 3c3.37 0 6.111 2.691 6.111 6 0 3.308-2.74 6-6.11 6S8.89 12.309 8.888 9c0-3.309 2.741-6 6.111-6z" id="a"/></defs><path stroke="#000" stroke-width="1.5" d="M10 13.84c-5.164 0-9.25 2.23-9.25 4.704v.347c.018 3.37 1.41 4.359 9.25 4.359 4.538 0 7.104-.504 8.285-1.436.806-.637.965-1.3.965-3.269 0-2.475-4.086-4.704-9.25-4.704zM10 .75C7.041.75 4.64 3.109 4.64 6S7.042 11.25 10 11.25c2.959 0 5.361-2.359 5.361-5.25 0-2.892-2.402-5.25-5.36-5.25z" fill="none" fill-rule="evenodd"/></svg>',
		'friends'    => '<svg width="26" height="24" xmlns="http://www.w3.org/2000/svg"><g stroke="#000" stroke-width="1.5" fill="none" fill-rule="evenodd"><path d="M9.5 13.795c-4.888 0-8.75 2.126-8.75 4.476v.346c.018 3.197 1.331 4.133 8.75 4.133 4.302 0 6.732-.481 7.845-1.368.756-.602.905-1.23.905-3.11 0-2.351-3.863-4.477-8.75-4.477zm0-12.545c-2.789 0-5.056 2.245-5.056 5 .001 2.755 2.268 5 5.056 5 2.789 0 5.056-2.245 5.056-5s-2.267-5-5.056-5z"/><g stroke-linecap="round"><path d="M21.5 5v7M18 8.5h7"/></g></g></svg>',
		'groups'     => '<svg width="26" height="22" xmlns="http://www.w3.org/2000/svg"><g fill-rule="nonzero" stroke="#000" stroke-width="1.5" fill="none"><path d="M16.4 12.65c-4.612 0-8.25 2.005-8.25 4.208v.346c.02 2.988 1.254 3.864 8.25 3.864 4.067 0 6.362-.455 7.407-1.289.704-.56.843-1.148.843-2.92 0-2.203-3.639-4.208-8.25-4.208zm0-11.9c-2.62 0-4.75 2.112-4.75 4.704 0 2.593 2.13 4.705 4.75 4.705s4.75-2.112 4.75-4.704C21.15 2.862 19.02.75 16.4.75z"/><path d="M8.729 11.984a7.911 7.911 0 00-2.329-.333c-2.761 0-5 .788-5 2.432 0 1.56 0 2.839 4.276 2.965" stroke-linecap="round"/><path d="M6.4 9.505c1.407 0 2.55-1.134 2.55-2.523 0-1.39-1.143-2.523-2.55-2.523-1.407 0-2.55 1.134-2.55 2.523 0 1.389 1.143 2.523 2.55 2.523z"/></g></svg>',
		'courses'    => '<svg width="27" height="23" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path d="M-2-4h30v30H-2z"/><path d="M1.057 7.309C1.495 3.845 4.624 1 8.117 1h6.597c3.857 0 6.97 3.53 6.854 7.414-.112 3.753-3.388 6.646-7.142 6.646h-2.132c-.514 0-1.013.173-1.416.491l-3.644 2.876h0v-3.166c0-.13-.097-.24-.227-.257 0 0 0 0 0 0a6.857 6.857 0 01-5.95-7.695z" stroke="#000" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M14.096 18.514h1.868c.474 0 .933.163 1.3.463l3.225 2.629v-2.914a.23.23 0 01.201-.229c3.264-.417 5.728-3.493 5.29-7.065a6.557 6.557 0 00-.801-2.418" stroke="#000" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M7.857 7.06a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285zm3.429 0a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285zm3.428 0a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285z" fill="#000"/></g></svg>',
		'photos'     => '<svg width="24" height="20" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path d="M-3-5h30v30H-3z"/><g transform="translate(1 1)" stroke="#000" stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"><path d="M3.826 0h14.348A3.826 3.826 0 0122 3.826v10.522a3.826 3.826 0 01-3.826 3.826H3.826A3.826 3.826 0 010 14.348V3.826A3.826 3.826 0 013.826 0z"/><path d="M.29 11.03L3.3 8.069a2.87 2.87 0 014.025 0l2.658 2.615h0l1.781 1.752M10.335 14.192l3.948-3.953a2.87 2.87 0 014.041-.02l3.414 3.353h0"/><circle cx="12.196" cy="5.5" r="1.674"/></g></g></svg>',
		'forums'     => '<svg width="27" height="23" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path d="M-2-4h30v30H-2z"/><path d="M1.057 7.309C1.495 3.845 4.624 1 8.117 1h6.597c3.857 0 6.97 3.53 6.854 7.414-.112 3.753-3.388 6.646-7.142 6.646h-2.132c-.514 0-1.013.173-1.416.491l-3.644 2.876h0v-3.166c0-.13-.097-.24-.227-.257 0 0 0 0 0 0a6.857 6.857 0 01-5.95-7.695z" stroke="#000" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M14.096 18.514h1.868c.474 0 .933.163 1.3.463l3.225 2.629v-2.914a.23.23 0 01.201-.229c3.264-.417 5.728-3.493 5.29-7.065a6.557 6.557 0 00-.801-2.418" stroke="#000" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M7.857 7.06a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285zm3.429 0a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285zm3.428 0a1.143 1.143 0 110 2.285 1.143 1.143 0 010-2.285z" fill="#000"/></g></svg>',
	);
}

/**
 * Fetch Forums though API for Admin Preview.
 *
 * @return array
 */
function bbapp_get_forums() {
	$forums      = get_transient( 'bbapp_preview_forums' );
	$show_forums = true;

	if ( false === $forums || empty( $forums ) ) {
		$forum_request          = new WP_REST_Request( 'GET', '/buddyboss/v1/forums' );
		$forums_per_page        = ( function_exists( 'bbp_get_forums_per_page' ) ? bbp_get_forums_per_page() : get_option( '_bbp_forums_per_page', 15 ) );
		$query_args['per_page'] = $forums_per_page;
		$query_args['orderby']  = 'activity';
		$query_args['order']    = 'desc';

		$forum_request->set_query_params( $query_args );

		$server       = rest_get_server();
		$item         = $server->dispatch( $forum_request );
		$forum_status = $item->get_status();
		$forums       = array();
		if ( 200 === $forum_status ) {
			$forums = $item->get_data();
		}

		set_transient( 'bbapp_preview_forums', $forums, 12 * HOUR_IN_SECONDS );
	}

	if ( empty( $forums ) ) {
		$show_forums = false;
	}

	return array( $forums, $show_forums );
}

/**
 * Fetch Topics though API for Admin Preview.
 *
 * @return array
 */
function bbapp_get_topics() {
	$forum_request          = new WP_REST_Request( 'GET', '/buddyboss/v1/forums' );
	$query_args['per_page'] = 1;
	$query_args['orderby']  = 'activity';
	$query_args['order']    = 'desc';

	$forum_request->set_query_params( $query_args );
	$forum_request->set_query_params( $query_args );

	$server       = rest_get_server();
	$item         = $server->dispatch( $forum_request );
	$forum_status = $item->get_status();
	$show_topic   = true;
	$forums       = array();
	if ( 200 === $forum_status ) {
		$forums = $item->get_data();
	}

	if ( empty( $forums ) ) {
		$show_topic = false;
	}

	$first_forum = isset( $forums[0] ) ? $forums[0] : array();

	if ( empty( $first_forum ) ) {
		$show_topic = false;
	}

	$topic_id = isset( $first_forum['last_topic_id'] ) ? $first_forum['last_topic_id'] : 0;

	if ( empty( $topic_id ) ) {
		$show_topic = false;
	}

	$topics = get_transient( 'bbapp_preview_topics' );

	if ( $show_topic && ( false === $topics || empty( $topics ) ) ) {
		$topic_request                = new WP_REST_Request( 'GET', '/buddyboss/v1/topics/' . $topic_id );
		$topic_query_args['per_page'] = 1;
		$topic_request->set_query_params( $topic_query_args );

		$server     = rest_get_server();
		$topic_item = $server->dispatch( $topic_request );
		$status     = $topic_item->get_status();
		$topics     = array();
		if ( 200 === $status ) {
			$topics = $topic_item->get_data();
		}

		set_transient( 'bbapp_preview_topics', $topics, 12 * HOUR_IN_SECONDS );
	}

	if ( empty( $topics ) ) {
		$topics     = array();
		$show_topic = false;
	}

	$reply = array();

	if ( ! empty( $topic_id ) ) {
		$reply_request              = new WP_REST_Request( 'GET', '/buddyboss/v1/reply' );
		$reply_query_args['parent'] = $topic_id;

		$reply_request->set_query_params( $reply_query_args );

		$server     = rest_get_server();
		$reply_item = $server->dispatch( $reply_request );
		$status     = $reply_item->get_status();

		if ( 200 === $status ) {
			$reply = $reply_item->get_data();
		}
	}

	return array( $topics, $reply, $show_topic );
}

/**
 * Fetch Lesson though API for Admin Preview.
 *
 * @return array
 */
function bbapp_get_lessons() {
	$course_request         = new WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/courses' );
	$query_args['per_page'] = 1;
	$course_request->set_query_params( $query_args );
	$server        = rest_get_server();
	$course_item   = $server->dispatch( $course_request );
	$course_status = $course_item->get_status();
	$show_lesson   = true;
	$courses       = array();

	if ( 200 === $course_status ) {
		$courses = $course_item->get_data();
	}

	if ( empty( $courses ) ) {
		$show_lesson = false;
	}

	$first_course = isset( $courses[0] ) ? $courses[0] : array();

	if ( empty( $first_course ) ) {
		$show_lesson = false;
	}

	$first_course_id = isset( $first_course['id'] ) ? $first_course['id'] : 0;

	if ( empty( $first_course_id ) ) {
		$show_lesson = false;
	}

	$no_of_lessons = isset( $first_course['lessons'] ) ? count( $first_course['lessons'] ) : 0;
	$lessons       = get_transient( 'bbapp_preview_single_lesson' );
	if ( $show_lesson ) {
		if ( false === $lessons || empty( $lessons ) ) {
			$course_request          = new WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/lessons' );
			$query_args['course_id'] = $first_course_id;
			$course_request->set_query_params( $query_args );
			$server  = rest_get_server();
			$item    = $server->dispatch( $course_request );
			$status  = $item->get_status();
			$lessons = array();

			if ( 200 === $status ) {
				$lessons = $item->get_data();
			}

			set_transient( 'bbapp_preview_single_lesson', $lessons, 12 * HOUR_IN_SECONDS );
		}
	}

	if ( empty( $lessons ) ) {
		$lessons     = array();
		$show_lesson = false;
	}

	$first_lesson = isset( $lessons[0] ) ? $lessons[0] : array();

	if ( empty( $first_lesson ) ) {
		$show_lesson = false;
	}

	return array( $lessons, $show_lesson, $no_of_lessons );
}

/**
 * Get Styling Color by Key.
 *
 * @param string $key Key.
 *
 * @return mixed|string
 */
function bbapp_get_default_color( $key ) {
	// Read from JSON file.
	$default_styling_option_data = BuddyBossApp\Styling::instance()->get_styling_colors();
	$default_styling_option      = isset( $default_styling_option_data['colors'] ) ? $default_styling_option_data['colors'] : array();
	$default_icon_color          = isset( $default_styling_option[ $key ] ) ? $default_styling_option[ $key ] : '';
	// Get value from DB.
	$styling_options = BuddyBossApp\Styling::instance()->get_options();
	$values          = isset( $styling_options['styles'] ) ? $styling_options['styles'] : array();
	if ( isset( $values ) && ! empty( $values ) ) {
		$default_icon_color = $values[ 'styles.colors.' . $key ];
	}

	return $default_icon_color;
}

/**
 * Helper function to get app rest language for preview.
 *
 * @since 2.4.10
 * @return mixed
 */
function bbapp_get_rest_app_languages_for_preview() {
	$languages = BuddyBossApp\AppLanguages::instance();

	// Get the current app language.
	$app_selected_language = $languages->get_app_language();

	// Get language strings for the current language.
	$languages_data = $languages->get_languages_strings_by_language_code(
		array(
			'language_code' => $app_selected_language,
		)
	);

	if ( is_wp_error( $languages_data ) ) {
		return array();
	}

	// Handle array return type.
	if ( is_array( $languages_data ) ) {
		if ( ! empty( $languages_data[ $app_selected_language ] ) ) {
			return $languages_data[ $app_selected_language ];
		} elseif ( isset( $languages_data['en'] ) ) {
			return $languages_data['en'];
		}
	}

	// Worst case fallback - empty array.
	return array();
}

/**
 * Tells if user is seeing super admin page.
 * Cases are.
 * 1. Return true when plugin activated in network mode & it is network admin area
 * 2. Return true when plugin activated in normal mode it's wp-admin.
 *
 * @since 2.4.10
 * @return bool
 */
function bbapp_is_super_admin_page() {
	if ( bbapp()->is_network_activated() ) {
		if ( is_network_admin() ) {
			return true;
		}
	} elseif ( is_admin() ) {
		return true;
	}

	return false;
}

/**
 * Check if user is on admin page.
 * Cases are.
 * 1. Returns true when plugin activated in network mode & it is not network admin area.
 * 2. Returns true when plugin activated in normal mode and it's wp-admin page.
 *
 * @return bool
 */
function bbapp_is_admin_page() {
	if ( bbapp()->is_network_activated() ) {
		if ( is_network_admin() ) {
			return false;
		}
	}

	return is_admin();
}

/**
 * Return the super admin URL.
 *
 * @param string $path Path within site.
 *
 * @return string|void
 */
function bbapp_get_admin_url( $path = '' ) {
	if ( bbapp()->is_network_activated() && is_network_admin() ) { // links are diff for pages.
		$link = network_admin_url( $path );
	} else {
		$link = admin_url( $path );
	}

	return $link;
}

/**
 * Return the super admin URL.
 * If network mode is enabled and user viewing link on normal dashboard he will get direct to network admin.
 *
 * @param string $path Path within site.
 *
 * @return string|void
 */
function bbapp_get_super_admin_url( $path = '' ) {
	if ( bbapp()->is_network_activated() ) { // links are diff for pages.
		$link = network_admin_url( $path );
	} else {
		$link = admin_url( $path );
	}

	return $link;
}

/**
 * Tells weather the component is active or not.
 *
 * @param string $component_id Component id.
 *
 * @return bool
 */
function bbapp_is_active( $component_id ) {
	return BuddyBossApp\Components::instance()->is_active( $component_id );
}


/**
 * Return the iap upload path.
 *
 * @param string $relativepath Relative path for upload.
 *
 * @return string
 */
function bbapp_get_upload_full_path( $relativepath ) {
	static $upload_dir;

	if ( empty( $relativepath ) ) {
		return null;
	}

	if ( ! is_array( $upload_dir ) ) {
		if ( bbapp()->is_network_activated() ) { // Plugin is activated on network mode.
			switch_to_blog( 1 );
			$upload_dir = wp_upload_dir();
			restore_current_blog();
		} else {
			$upload_dir = wp_upload_dir();
		}
	}

	$path = trailingslashit( $upload_dir['basedir'] ) . $relativepath;

	return preg_replace( '#/+#', '/', $path );
}

/**
 * Render the register fields.
 *
 * @param object $field    Field Object.
 *
 * @return false|string
 */
function bbapp_render_registration_field( $field ) {
	if ( empty( $field ) ) {
		return '';
	}

	$type = $field->type;

	switch ( $type ) {
		case 'text':
		case 'email':
		case 'password':
		case 'textbox':
		case 'telephone':
		case 'url':
		case 'number':
		case 'datebox':
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', esc_html( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			$pass_icon_warp = ( 'password' === $type ) ? ' pass-icon-wrap' : '';
			printf( '<div class="app-field %1$s>', esc_attr( $pass_icon_warp ) );
			if ( 'password' === $type ) {
				echo '<i class="bb-icon-l bb-icon-eye-slash"></i>';
			} else {
				printf( '<span class="placeholder">%1$s %2$s</span>', esc_html__( 'Enter', 'buddyboss-app' ), esc_html( strtolower( $field->label ) ) );
			}
			echo '</div>';
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
		case 'gender':
		case 'radio':
			$options = $field->options;
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', esc_html( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			echo '<div class="app-field radio-icon-wrap">';
			$allowed_html           = wp_kses_allowed_html( 'post' );
			$allowed_html['svg']    = array(
				'width'  => true,
				'height' => true,
				'xmlns'  => true,
			);
			$allowed_html['g']      = array(
				'fill'      => true,
				'fill-rule' => true,
			);
			$allowed_html['circle'] = array(
				'stroke'       => true,
				'stroke-width' => true,
				'fill'         => true,
				'cx'           => true,
				'cy'           => true,
				'r'            => true,
			);
			foreach ( $options as $option ) {
				$is_default_option = '<svg width="22" height="22" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><circle stroke-opacity=".7" stroke="#FFF" stroke-width="1.5" cx="11" cy="11" r="10.25"/><circle cx="11" cy="11" r="7"/></g></svg>';
				if ( true === $option->is_default_option ) {
					$is_default_option = '<svg width="22" height="22" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><circle stroke="#FFF" stroke-width="1.5" fill="#FFF" cx="11" cy="11" r="10.25"/><circle fill="#385DFF" cx="11" cy="11" r="6"/></g></svg>';
				}

				printf( '<div class="app-field field-options">%1$s %2$s</div>', esc_html( $option->name ), wp_kses( $is_default_option, $allowed_html ) );
			}
			echo '</div>';
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
		case 'checkbox':
			$options = $field->options;
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', wp_kses_post( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			echo '<div class="app-field checkbox-icon-wrap">';
			$allowed_html['svg']    = array(
				'width'  => true,
				'height' => true,
				'xmlns'  => true,
			);
			$allowed_html['g']      = array(
				'fill'      => true,
				'fill-rule' => true,
			);
			$allowed_html['circle'] = array(
				'stroke'       => true,
				'stroke-width' => true,
				'fill'         => true,
				'cx'           => true,
				'cy'           => true,
				'r'            => true,
			);
			$allowed_html['rect']   = array(
				'stroke'       => true,
				'stroke-width' => true,
				'fill'         => true,
				'x'            => true,
				'y'            => true,
				'width'        => true,
				'height'       => true,
				'rx'           => true,
			);
			$allowed_html['path']   = array(
				'stroke'          => true,
				'stroke-width'    => true,
				'stroke-linecap'  => true,
				'stroke-linejoin' => true,
				'd'               => true,
			);
			foreach ( $options as $option ) {
				$is_default_option = '<svg width="22" height="22" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><circle stroke-opacity=".7" stroke="#FFF" stroke-width="1.5" cx="11" cy="11" r="10.25"/><circle cx="11" cy="11" r="7"/></g></svg>';
				if ( true === $option->is_default_option ) {
					$is_default_option = '<svg width="22" height="22" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><rect stroke="#FFF" stroke-width="1.5" fill="#FFF" x=".75" y=".75" width="20.5" height="20.5" rx="10.25"/><path stroke="#385DFF" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" d="M6.6 11.656l2.992 2.672 6.254-6.408"/></g></svg>';
				}
				printf( '<div class="app-field field-options">%1$s %2$s</div>', esc_html( $option->name ), wp_kses( $is_default_option, $allowed_html ) );
			}
			echo '</div>';
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
		case 'selectbox':
		case 'multiselectbox':
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', esc_html( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			echo '<div class="app-field selectbox-field">';
			printf( '%1$s %2$s', esc_html__( 'Select', 'buddyboss-app' ), esc_html( strtolower( $field->label ) ) );
			echo '</div>';
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
		case 'socialnetworks':
			$options = $field->options;
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', esc_html( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			foreach ( $options as $social ) {
				printf( '<div class="app-field social-field"><span class="placeholder">%1$s %2$s</span></div>', esc_html__( 'Enter', 'buddyboss-app' ), esc_html( strtolower( $social->name ) ) );
			}
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
		case 'textarea':
			echo '<div class="app-field-set bbapp-style-bodyText login-label regFieldTextColor color-attr">';
			printf( '<label>%1$s %2$s</label>', esc_html( $field->label ), ( true !== $field->required ? '(optional)' : '' ) );
			printf( '<div class="app-field field-textarea"><span class="placeholder">%1$s %2$s</span></div>', esc_html__( 'Enter', 'buddyboss-app' ), esc_html( strtolower( $field->label ) ) );
			if ( ! empty( $field->description ) ) {
				echo '<div class="app-field-description">' . wp_kses_post( $field->description ) . '</div>';
			}
			echo '</div>';
			break;
	}
}

/**
 * Render default images.
 *
 * @param string $type Image type.
 *
 * @return string
 */
function bbapp_render_default_images( $type ) {
	$image = '';

	switch ( $type ) {
		case 'group':
			$image = bbapp()->plugin_url . 'assets/img/default-group-img.png';
			break;
		case 'group-r':
			$image = bbapp()->plugin_url . 'assets/img/default-group-img-r.png';
			break;
		case 'forum':
			$image = bbapp()->plugin_url . 'assets/img/default-forum-img.png';
			break;
		case 'member':
			$image = bbapp()->plugin_url . 'assets/img/default-member-img.png';
			break;
	}

	return $image;
}

/**
 * Returns the bool value.
 *
 * @param string $google_font_key Google font key.
 *
 * @return mixed
 */
function is_google_api_valid( $google_font_key ) {
	$response = bbapp_remote_get( 'https://www.googleapis.com/webfonts/v1/webfonts?key=' . $google_font_key );

	if ( ! is_wp_error( $response ) ) {
		// Retrieve body.
		$response_body = wp_remote_retrieve_body( $response );
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_data = json_decode( $response_body );

		if ( 200 === (int) wp_remote_retrieve_response_code( $response ) ) {
			return true;
		}

		$response_message = isset( $response_data->error ) && isset( $response_data->error->message ) ? $response_data->error->message : wp_remote_retrieve_response_message( $response );

		return new WP_Error( 'google_font_json_error', $response_message, array( 'status' => $response_code ) );
	}

	return new WP_Error( 'google_font_json_error', __( 'There was a problem with the Vimeo API', 'buddyboss-app' ), array( 'status' => 400 ) );
}

/**
 * Check the vimeo auth code is valid or not.
 *
 * @param string $vimeo_auth_code Vimeo authentication code.
 *
 * @return bool|WP_Error
 *
 * @since 1.3.6
 */
function is_vimeo_api_valid( $vimeo_auth_code ) {
	$http_user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
	$args            = array(
		'user-agent' => isset( $request['user_agent'] ) ? $request['user_agent'] : $http_user_agent,
		'method'     => 'GET',
		'headers'    => array( 'Authorization' => 'Bearer ' . $vimeo_auth_code ),
	);
	$response        = wp_remote_request(
		'https://api.vimeo.com/oauth/verify',
		$args
	);

	if ( ! is_wp_error( $response ) ) {
		// Retrieve body.
		$response_body = wp_remote_retrieve_body( $response );
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_data = json_decode( $response_body );

		if ( 200 === (int) wp_remote_retrieve_response_code( $response ) ) {
			return true;
		}

		$response_message = isset( $response_data->error ) ? $response_data->error : wp_remote_retrieve_response_message( $response );

		return new WP_Error( 'vimeo_json_error', $response_message, array( 'status' => $response_code ) );
	}

	return new WP_Error( 'vimeo_json_error', __( 'There was a problem with the Vimeo API', 'buddyboss-app' ), array( 'status' => 400 ) );
}

if ( ! function_exists( 'array_key_last' ) ) {
	/**
	 * An array_key_last function Support for PHP < 7.3
	 *
	 * @param array $array Array to get the last key.
	 *
	 * @return null|int|string
	 */
	function array_key_last( $array ) {
		if ( ! is_array( $array ) || empty( $array ) ) {
			return null;
		}

		return array_keys( $array )[ count( $array ) - 1 ];
	}
}

if ( ! function_exists( 'array_key_first' ) ) {
	/**
	 * An array_key_first function Support for PHP < 7.3.
	 *
	 * @param array $arr Array to get the first key.
	 *
	 * @return null|int|string
	 */
	function array_key_first( array $arr ) {
		foreach ( $arr as $key => $unused ) {
			return $key;
		}

		return null;
	}
}

/**
 * Get app version core.
 *
 * @param bool $use_cache Use cached results.
 *
 * @return array App version code.
 */
function bbapp_get_app_core( $use_cache = true ) {
	$cache = get_transient( 'bbapp_get_app_core' );

	if ( ! empty( $cache ) && $use_cache ) {
		return $cache;
	}

	// Get response.
	$app_core_response = wp_remote_get( 'https://buddyboss.com/resources/wp-json/wp/v2/releases-app?per_page=1' );
	$app_core_data     = array();

	if ( ! is_wp_error( $app_core_response ) ) {

		// Retrieve body.
		$response_body = wp_remote_retrieve_body( $app_core_response );

		// Get version.
		if ( ! empty( $response_body ) ) {
			$app_code_data = json_decode( $response_body );
			if ( ! empty( $app_code_data[0] ) ) {
				$app_core_data['version'] = isset( $app_code_data[0]->title->rendered ) ? $app_code_data[0]->title->rendered : '';
				$app_core_data['link']    = isset( $app_code_data[0]->link ) ? $app_code_data[0]->link : '';
				$app_core_data['slug']    = isset( $app_code_data[0]->slug ) ? $app_code_data[0]->slug : '';
			}
		}
	}

	set_transient( 'bbapp_get_app_core', $app_core_data, 12 * HOUR_IN_SECONDS ); // 12 hours of cache.

	return $app_core_data;
}

if ( ! function_exists( 'bbapp_is_rest' ) ) {
	/**
	 * Check the request is REST or not.
	 *
	 * @return bool
	 */
	function bbapp_is_rest() {
		$request_url = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		return ( false !== strpos( $request_url, '/wp-json' ) );
	}
}

if ( ! function_exists( 'bbapp_delete_transients' ) ) {
	/**
	 * Delete transient based option name.
	 * Function will remove transient from DB when click on clear Transient button( Tools -> Transients ).
	 *
	 * @param string|array $bbapp_options transient options.
	 */
	function bbapp_delete_transients( $bbapp_options ) {
		global $wpdb;

		$sql_condition = '';

		if ( is_array( $bbapp_options ) && ! empty( $bbapp_options ) ) {
			foreach ( $bbapp_options as $option_key => $option_val ) {
				if ( 0 === $option_key ) {
					$sql_condition .= ' WHERE option_name LIKE "%' . $option_val . '%"';
				} else {
					$sql_condition .= ' OR option_name LIKE "%' . $option_val . '%"';
				}
			}
		} else {
			$sql_condition .= ' WHERE option_name LIKE "%' . $bbapp_options . '%"';
		}

		$sql       = 'SELECT option_name FROM ' . $wpdb->options;
		$sql      .= $sql_condition;
		$all_cache = $wpdb->get_col( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		if ( ! empty( $all_cache ) ) {
			foreach ( $all_cache as $cache_name ) {
				$cache_name = str_replace( '_site_transient_', '', $cache_name );
				$cache_name = str_replace( '_transient_', '', $cache_name );
				delete_transient( $cache_name );
				delete_site_transient( $cache_name );
			}
		}
	}
}

if ( ! function_exists( 'bbapp_generate_password' ) ) {
	/**
	 * Copy of wp_generate_password excluding the filter hook.
	 *
	 * @param int  $length              Password length.
	 * @param bool $special_chars       If special characters are allowed.
	 * @param bool $extra_special_chars If extra special characters are allowed.
	 *
	 * @return string
	 */
	function bbapp_generate_password( $length = 12, $special_chars = true, $extra_special_chars = false ) {
		$chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';

		if ( $special_chars ) {
			$chars .= '!@#$%^&*()';
		}

		if ( $extra_special_chars ) {
			$chars .= '-_ []{}<>~`+=,.;:/?|';
		}

		$password = '';

		for ( $i = 0; $i < $length; $i++ ) {
			$password .= substr( $chars, wp_rand( 0, strlen( $chars ) - 1 ), 1 );
		}

		return $password;
	}
}

if ( ! function_exists( 'bbapp_untrailingslashit' ) ) {
	/**
	 * Url un trailing slash it
	 *
	 * @param string $url Url.
	 *
	 * @sicne 1.3.7
	 *
	 * @return mixed|string
	 */
	function bbapp_untrailingslashit( $url ) {
		$parse_url = wp_parse_url( $url );

		if ( ! empty( $parse_url['query'] ) ) {
			$url = untrailingslashit( $url );
		}

		return $url;
	}
}

if ( ! function_exists( 'bbapp_tooltip' ) ) {
	/**
	 * Load tooltip html.
	 *
	 * @param string $text Tooltip text.
	 * @param string $html Html to add to tooltip.
	 *
	 * @since 1.4.1
	 */
	function bbapp_tooltip( $text, $html = '' ) {
		?>
		<div class="bbapp-tooltip">
			<span class="bbapp-tooltip-text" id="bbapp-span-tooltip" data-tooltip="<?php echo wp_kses_post( $text ); ?>"></span>
			<?php
			echo wp_kses_post( $html );
			?>
		</div>
		<?php
	}
}

/**
 * Function to get the user display name.
 *
 * @param int|string $user_id_or_username user id or username.
 *
 * @since 1.4.1
 *
 * @return false|mixed|void
 */
function bbaap_get_user_display_name( $user_id_or_username ) {
	if ( empty( $user_id_or_username ) ) {
		return false;
	}

	if ( function_exists( 'bp_core_get_user_displayname' ) && class_exists( 'BuddyPress' ) ) {
		$name = bp_core_get_user_displayname( $user_id_or_username );

		if ( ! is_numeric( $user_id_or_username ) ) {
			$user    = get_user_by( 'login', $user_id_or_username );
			$user_id = ! empty( $user->ID ) ? $user->ID : null;
		} else {
			$user_id = $user_id_or_username;
		}
	} elseif ( ! is_numeric( $user_id_or_username ) ) {
			$user_data = get_user_by( 'login', $user_id_or_username );
			$user_id   = ! empty( $user_data->ID ) ? $user_data->ID : null;
			$name      = get_the_author_meta( 'display_name', $user_id );
	} else {
		$user_id = $user_id_or_username;
		$name    = get_the_author_meta( 'display_name', $user_id );
	}

	/**
	 * Function to filter the display name.
	 *
	 * @since 1.4.1
	 *
	 * @param string $name    Display name for the user.
	 * @param int    $user_id ID of the user to check.
	 */
	return apply_filters( 'bbaap_get_user_display_name', trim( $name ), $user_id );
}

/**
 * Checking whether the app is enabled sandbox.
 *
 * @since 1.4.3
 *
 * @return bool
 */
function bbapp_is_sandbox_enabled() {
	$bbapp_info = \BuddyBossApp\ManageApp::instance()->get_app_info();

	if ( empty( $bbapp_info ) || is_wp_error( $bbapp_info ) || ! is_array( $bbapp_info ) || empty( $bbapp_info['is_sandbox_app'] ) ) {
		return false;
	}

	return 1 === (int) $bbapp_info['is_sandbox_app'] ? true : false;
}

if ( ! function_exists( 'bbapp_universal_links' ) ) {
	/**
	 * Call universal links functions for iOS and Android.
	 *
	 * @since 1.4.8
	 */
	function bbapp_universal_links() {
		bbapp_ios_universal_links();
		bbapp_android_app_links();
	}

	add_action( 'plugin_loaded', 'bbapp_universal_links' );
}

/**
 * Process the .p12 file to update the password.
 *
 * @param string $p12_file_path Path to the .p12 file.
 * @param string $password      New password for the .p12 file.
 *
 * @since 2.3.21
 * @return string|WP_Error Path to the updated .p12 file or WP_Error object.
 */
function bbapp_process_p12_file( $p12_file_path, $password ) {
	// Check if `shell_exec` is available.
	if ( ! function_exists( 'shell_exec' ) || ! is_callable( 'shell_exec' ) ) {
		return new WP_Error(
			'shell_exec_disabled',
			esc_html__( 'The shell_exec function is disabled on this server. Unable to process the .p12 file.', 'buddyboss-app' )
		);
	}

	// Ensure the p12 file exists.
	if ( ! file_exists( $p12_file_path ) ) {
		return new WP_Error(
			'file_not_found',
			esc_html__( 'The provided .p12 file does not exist.', 'buddyboss-app' )
		);
	}

	// Check if temp directory is writable.
	if ( ! is_writable( sys_get_temp_dir() ) ) {
		return new WP_Error(
			'temp_dir_not_writable',
			esc_html__( 'The temporary directory is not writable.', 'buddyboss-app' )
		);
	}

	// Generate temporary file paths.
	$decrypted_tmp_path = tempnam( sys_get_temp_dir(), 'key_decrypted' );
	$new_p12_tmp_path   = tempnam( sys_get_temp_dir(), 'key_new' ) . '.p12';

	// Step 1: Decrypt the .p12 file to a temporary decrypted file.
	$command1 = sprintf(
		'openssl pkcs12 -legacy -in %s -nodes -out %s -password pass:%s 2>&1',
		escapeshellarg( $p12_file_path ),
		escapeshellarg( $decrypted_tmp_path ),
		escapeshellarg( $password )
	);
	$output1  = shell_exec( $command1 );

	if ( ! file_exists( $decrypted_tmp_path ) ) {
		return new WP_Error(
			'openssl_error',
			esc_html__( 'Failed to decrypt the .p12 file.', 'buddyboss-app' ),
			array( 'output' => $output1 )
		);
	}

	// Step 2: Re-encrypt the decrypted file into a new .p12 file.
	$command2 = sprintf(
		'openssl pkcs12 -in %s -export -out %s -password pass:%s 2>&1',
		escapeshellarg( $decrypted_tmp_path ),
		escapeshellarg( $new_p12_tmp_path ),
		escapeshellarg( $password )
	);
	$output2  = shell_exec( $command2 );

	if ( ! file_exists( $new_p12_tmp_path ) ) {
		unlink( $decrypted_tmp_path ); // Clean up if re-encryption fails.

		return new WP_Error(
			'openssl_error',
			esc_html__( 'Failed to re-encrypt the .p12 file.', 'buddyboss-app' ),
			array( 'output' => $output2 )
		);
	}

	// Clean up the decrypted temporary file.
	unlink( $decrypted_tmp_path );

	// Return the path to the updated .p12 file.
	return $new_p12_tmp_path;
}

/**
 * Read the .p12 file to extract the team ID.
 *
 * @param array  $app_info                     App info array.
 * @param string $file_content                 Contents of the .p12 file.
 * @param string $signing_certificate_password Password for the .p12 file.
 *
 * @since 2.3.21
 * @return array Updated app info array.
 */
function bbapp_read_pack12_read( $app_info, $file_content, $signing_certificate_password ) {
	if ( openssl_pkcs12_read( $file_content, $p12cert, $signing_certificate_password ) ) {
		$cache_key = 'bbapp_apple_team_id';
		$cert      = ( isset( $p12cert['cert'] ) ? $p12cert['cert'] : '' );

		if ( ! empty( $cert ) && function_exists( 'openssl_x509_parse' ) ) {
			$cert_data = openssl_x509_parse( $cert );

			if ( ! empty( $cert_data ) && isset( $cert_data['subject'] ) ) {
				$app_info['team_id'] = ( isset( $cert_data['subject']['OU'] ) ? $cert_data['subject']['OU'] : '' );

				set_site_transient( $cache_key, $app_info, 24 * HOUR_IN_SECONDS );
			}
		}
	}

	return $app_info;
}

if ( ! function_exists( 'bbapp_ios_universal_links' ) ) {
	/**
	 * Create a json data with links to exlude from open in the app.
	 *
	 * E.g. /groups/
	 *
	 * @since 1.4.8
	 */
	function bbapp_ios_universal_links() {
		// Check if we are in apple-app-site-association.
		$request_url = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		if ( strpos( $request_url, 'apple-app-site-association' ) !== false ) {
			header( 'Content-type: application/json' );

			// Get app settings.
			$settings = \BuddyBossApp\ManageApp::instance()->get_app_settings();

			// If universal links is enabled.
			if ( isset( $settings['publish.ios.universal_link'] ) && ! empty( $settings['publish.ios.universal_link'] ) ) {
				// Set response array.
				$data         = array();
				$bbap_live_id = '';
				$bbap_dev_id  = '';

				if ( isset( $settings['publish.ios.namespace'] ) && ! empty( $settings['publish.ios.namespace'] ) ) {
					$bbap_live_id = $settings['publish.ios.namespace'];
				}

				if ( isset( $settings['publish.ios.dev.namespace'] ) && ! empty( $settings['publish.ios.dev.namespace'] ) ) {
					$bbap_dev_id = $settings['publish.ios.dev.namespace'];
				}

				// If buddle ID is not set then return.
				if ( empty( $bbap_live_id ) && empty( $bbap_dev_id ) ) {
					$data[] = __( 'There may be some missing data in your apple app site association config.', 'buddyboss-app' );
					// Print apple-app-site-association.
					echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
					exit;
				}

				$cache_key = 'bbapp_apple_team_id';
				$cached    = get_site_transient( $cache_key );
				$app_info  = array(
					'team_id' => '',
				);

				if ( false !== $cached ) {
					$app_info = $cached;
				} else {
					$signing_certificate          = $settings['publish.ios.signing_certificate'];
					$signing_certificate_password = $settings['publish.ios.signing_certificate_password'];
					if ( ! empty( $signing_certificate ) && ! empty( $signing_certificate_password ) && function_exists( 'openssl_pkcs12_read' ) ) {
						$upload    = wp_upload_dir();
						$file_path = $upload['basedir'] . $signing_certificate;

						if ( BuddyBossApp\Helpers\BBAPP_File::file_exists( $file_path ) ) {
							$file_content = BuddyBossApp\Helpers\BBAPP_File::get_contents( $file_path );

							$app_info = bbapp_read_pack12_read( $app_info, $file_content, $signing_certificate_password );
							if ( empty( $app_info['team_id'] ) ) {
								$updated_p12_path = bbapp_process_p12_file( $file_path, $signing_certificate_password );

								if ( is_wp_error( $updated_p12_path ) ) {
									$error_data = $updated_p12_path->get_error_data();
									$data[]     = $updated_p12_path->get_error_message();

									if ( isset( $error_data['output'] ) ) {
										$data[] = 'OpenSSL Output: ' . $error_data['output'];
									}

									if ( isset( $error_data['command'] ) ) {
										$data[] = 'OpenSSL Command: ' . $error_data['command'];
									}

									echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
									exit;
								}

								$file_content = BuddyBossApp\Helpers\BBAPP_File::get_contents( $updated_p12_path );
								$app_info     = bbapp_read_pack12_read( $app_info, $file_content, $signing_certificate_password );
								// Cleanup the updated .p12 file after processing
								unlink( $updated_p12_path );
							}
						}
					} else {
						// If signing certificate settings are not set then return.
						$data[] = __( 'There may be some missing data in your apple app site association config.', 'buddyboss-app' );
						// Print apple-app-site-association.
						echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
						exit;
					}
				}

				// Return if team ID is not set.
				if ( empty( $app_info['team_id'] ) ) {
					$data[] = __( 'There may be some missing data in your apple app site association config.', 'buddyboss-app' );
					// Print apple-app-site-association.
					echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
					exit;
				}

				$data['applinks']['apps'] = array();

				// For release iOS App.
				if ( ! empty( $bbap_live_id ) ) {
					$data['applinks']['details'][0]['appID']      = $app_info['team_id'] . '.' . $bbap_live_id;
					$data['applinks']['details'][0]['appIDs'][]   = $app_info['team_id'] . '.' . $bbap_live_id;
					$data['applinks']['details'][0]['paths']      = array();
					$data['applinks']['details'][0]['components'] = array();

					if ( isset( $settings['publish.ios.exclude_links'] ) && ! empty( $settings['publish.ios.exclude_links'] ) ) {
						$exclude_links = explode( "\r\n", $settings['publish.ios.exclude_links'] );

						// Loop through exclude links.
						if ( ! empty( $exclude_links ) ) {
							$components_array = array();
							$cnt              = 0;

							foreach ( $exclude_links as $link ) {
								// If it's URL remove the domain from it.
								if ( bbapp_is_valid_url( $link ) ) {
									$site_url = trailingslashit( trim( home_url() ) );
									$link     = trailingslashit( trim( $link ) );
									$link     = str_replace( $site_url, '/', $link );
								}

								$components_array[ $cnt ]['/']       = trailingslashit( $link ) . '*';
								$components_array[ $cnt ]['exclude'] = true;
								// Set paths array for versions lower than iOS 13.
								$data['applinks']['details'][0]['paths'] = array_merge( $data['applinks']['details'][0]['paths'], array( 'NOT ' . trailingslashit( $link ) . '*' ) );
								++$cnt;
							}

							// Set componenets array for iOS 13 and later.
							$data['applinks']['details'][0]['components'] = array_merge( $data['applinks']['details'][0]['components'], $components_array );
						}
					}

					// Default pass the current domain.
					$data['applinks']['details'][0]['paths']      = array_merge( $data['applinks']['details'][0]['paths'], array( '/*' ) );
					$bbapp_set_domain                             = array();
					$bbapp_set_domain[0]['/']                     = '/*';
					$data['applinks']['details'][0]['components'] = array_merge( $data['applinks']['details'][0]['components'], $bbapp_set_domain );
				}

				// For dev iOS App.
				if ( ! empty( $bbap_dev_id ) ) {
					$data['applinks']['details'][1]['appID']      = $app_info['team_id'] . '.' . $bbap_dev_id;
					$data['applinks']['details'][1]['appIDs'][]   = $app_info['team_id'] . '.' . $bbap_dev_id;
					$data['applinks']['details'][1]['paths']      = array();
					$data['applinks']['details'][1]['components'] = array();

					if ( isset( $settings['publish.ios.exclude_links'] ) && ! empty( $settings['publish.ios.exclude_links'] ) ) {

						$exclude_links = explode( "\r\n", $settings['publish.ios.exclude_links'] );

						// Loop through exclude links.
						if ( ! empty( $exclude_links ) ) {
							$components_array = array();
							$cnt              = 0;

							foreach ( $exclude_links as $link ) {

								// If it's URL remove the domain from it.
								if ( bbapp_is_valid_url( $link ) ) {
									$site_url = trailingslashit( trim( home_url() ) );
									$link     = trailingslashit( trim( $link ) );
									$link     = str_replace( $site_url, '/', $link );
								}

								$components_array[ $cnt ]['/']       = trailingslashit( $link ) . '*';
								$components_array[ $cnt ]['exclude'] = true;

								// Set paths array for versions lower than iOS 13.
								$data['applinks']['details'][1]['paths'] = array_merge( $data['applinks']['details'][1]['paths'], array( 'NOT ' . trailingslashit( $link ) . '*' ) );
								++$cnt;
							}

							// Set componenets array for iOS 13 and later.
							$data['applinks']['details'][1]['components'] = array_merge( $data['applinks']['details'][1]['components'], $components_array );
						}
					}

					// Default pass the current domain.
					$data['applinks']['details'][1]['paths'] = array_merge( $data['applinks']['details'][1]['paths'], array( '/*' ) );

					$bbapp_set_domain                             = array();
					$bbapp_set_domain[1]['/']                     = '/*';
					$data['applinks']['details'][1]['components'] = array_merge( $data['applinks']['details'][1]['components'], $bbapp_set_domain );
				}

				// Print apple-app-site-association.
				echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
				exit;
			} else {
				// Print apple-app-site-association.
				echo wp_json_encode( array( __( 'In order to view conent of this endpoint please enable iOS universal links.', 'buddyboss-app' ) ), JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
				exit;
			}
		}
	}
}

if ( ! function_exists( 'bbapp_android_app_links' ) ) {
	/**
	 * Create a json data with links to be open in the app.
	 *
	 * E.g. /groups/
	 *
	 * @since 1.4.8
	 *
	 * @return void
	 */
	function bbapp_android_app_links() {
		// Check if we are in assetlink.json.
		$request_url = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		if ( strpos( $request_url, 'assetlinks.json' ) !== false ) {
			header( 'Content-type: application/json' );

			// Get app settings.
			$settings = \BuddyBossApp\ManageApp::instance()->get_app_settings();

			// If universal links is enabled.
			if ( isset( $settings['publish.android.universal_link'] ) && ! empty( $settings['publish.android.universal_link'] ) ) {
				// Set response array.
				$data                            = array();
				$android_release_bundle_id       = '';
				$android_test_bundle_id          = '';
				$google_play_signing_certificate = '';

				if ( isset( $settings['publish.android.namespace'] ) && ! empty( $settings['publish.android.namespace'] ) ) {
					$android_release_bundle_id = $settings['publish.android.namespace'];
				}

				if ( isset( $settings['publish.android.dev.namespace'] ) && ! empty( $settings['publish.android.dev.namespace'] ) ) {
					$android_test_bundle_id = $settings['publish.android.dev.namespace'];
				}

				if ( isset( $settings['publish.android.google_play_signing_certificate'] ) && ! empty( $settings['publish.android.google_play_signing_certificate'] ) ) {
					$google_play_signing_certificate = $settings['publish.android.google_play_signing_certificate'];
				}

				$cert = get_option( 'android_sha256cert' );

				if ( ! empty( $cert ) ) {
					$android_test_data    = array();
					$android_release_data = array();

					// If release android namespace is not empty.
					if ( ! empty( $android_test_bundle_id ) ) {
						$android_test_data['relation']                           = array( 'delegate_permission/common.handle_all_urls' );
						$android_test_data['target']['namespace']                = 'android_app';
						$android_test_data['target']['package_name']             = $android_test_bundle_id;
						$android_test_data['target']['sha256_cert_fingerprints'] = array( $cert );
						$data[] = $android_test_data;
					}

					// If release android namespace is not empty.
					if ( ! empty( $android_release_bundle_id ) ) {
						$android_release_data['relation']                           = array( 'delegate_permission/common.handle_all_urls' );
						$android_release_data['target']['namespace']                = 'android_app';
						$android_release_data['target']['package_name']             = $android_release_bundle_id;
						$android_release_data['target']['sha256_cert_fingerprints'] = array( $cert );

						if ( ! empty( $google_play_signing_certificate ) ) {
							array_push( $android_release_data['target']['sha256_cert_fingerprints'], $google_play_signing_certificate );
						}

						$data[] = $android_release_data;
					}
				}

				if ( empty( $data ) ) {
					$data[] = __( 'There may be some missing data in your digital assetlinks statement.', 'buddyboss-app' );
				}

				// Print assetlinks.json.
				echo wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
				exit;
			} else {
				// Print assetlinks.json.
				echo wp_json_encode( array( __( 'In order to view conent of this endpoint please enable andorid app links.', 'buddyboss-app' ) ), JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
				exit;
			}
		}
	}
}

if ( ! function_exists( 'bbapp_generate_android_sha256cert' ) ) {
	/**
	 * Generate sha256 certificate for android.
	 *
	 * @since 1.4.8
	 *
	 * @return bool
	 */
	function bbapp_generate_android_sha256cert() {
		// Get app settings.
		$settings = \BuddyBossApp\ManageApp::instance()->get_app_settings();
		// Get keystore settings.
		$keystore_url     = ( isset( $settings['publish.android.keystore'] ) && ! empty( $settings['publish.android.keystore'] ) ) ? $settings['publish.android.keystore'] : '';
		$keystore_pwd     = ( isset( $settings['publish.android.keystore_pwd'] ) && ! empty( $settings['publish.android.keystore_pwd'] ) ) ? $settings['publish.android.keystore_pwd'] : '';
		$keystore_alias   = ( isset( $settings['publish.android.keystore_alias'] ) && ! empty( $settings['publish.android.keystore_alias'] ) ) ? $settings['publish.android.keystore_alias'] : '';
		$keystore_key_pwd = ( isset( $settings['publish.android.keystore_key_pwd'] ) && ! empty( $settings['publish.android.keystore_key_pwd'] ) ) ? $settings['publish.android.keystore_key_pwd'] : '';

		// Return is keystore settings ar not added.
		if ( empty( $keystore_url ) || empty( $keystore_pwd ) || empty( $keystore_alias ) || empty( $keystore_key_pwd ) ) {
			return true;
		}

		$upload_dir  = wp_upload_dir();
		$body_params = array(
			'key_password'   => sanitize_text_field( $keystore_key_pwd ),
			'store_password' => sanitize_text_field( $keystore_pwd ),
			'alias_name'     => sanitize_title_with_dashes( $keystore_alias ),
			'keystore_path'  => esc_url( $upload_dir['baseurl'] . $keystore_url ),
		);
		$app         = \BuddyBossApp\ManageApp::instance()->get_app();
		$do_request  = bbapp_remote_post(
			'https://bbapp-cert-services.buddyboss.com/api/keystore/sha',
			array(
				'method'  => 'POST',
				'timeout' => 45,
				'body'    => $body_params,
				'headers' => array(
					'bbapp_id'  => empty( $app ) ? false : $app['bbapp_app_id'],
					'bbapp_key' => \BuddyBossApp\ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		// Return if error.
		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return true;
		}

		$body = wp_remote_retrieve_body( $do_request );
		$body = json_decode( $body, true );

		// Return if error.
		if ( isset( $body['status'] ) && 'error' === $body['status'] && isset( $body['message'] ) ) {
			return true;
		}

		if ( ! empty( $body ) && 200 === wp_remote_retrieve_response_code( $do_request ) ) {
			$cert = ( isset( $body['data']['sha256'] ) && ! empty( $body['data']['sha256'] ) ) ? $body['data']['sha256'] : '';

			// If cert is not set then return.
			if ( empty( $cert ) ) {
				return true;
			}

			update_option( 'android_sha256cert', $cert );
		}
	}
}

/**
 * Whether platform notification legacy mode is enabled or disabled.
 *
 * @since 1.4.7
 *
 * @return bool
 */
function bbapp_is_platform_notification_preferences_legacy_mode_enabled() {
	$legacy_notification_mode = true; // by default it's enabled for all.

	if ( function_exists( 'bp_is_labs_notification_preferences_support_enabled' ) && bp_is_labs_notification_preferences_support_enabled() ) {
		$legacy_notification_mode = false; // if the lab available & enabled in same time then legacy not enabled.
	}

	return (bool) apply_filters( 'bbapp_platform_notification_preferences_legacy_mode_enabled', $legacy_notification_mode );
}

/**
 * Just a function to validate a URL.
 *
 * @param string $url URL.
 *
 * @since 1.4.8
 *
 * @return bool
 */
function bbapp_is_valid_link( $url ) {
	$response = wp_remote_get(
		$url,
		array(
			'sslverify' => false,
			'timeout'   => 15,
		)
	);

	if ( is_array( $response ) && ! is_wp_error( $response ) ) {
		$response_code = wp_remote_retrieve_response_code( $response );

		if ( 200 === $response_code ) {
			return true;
		} else {
			return false;
		}
	} else {
		return false;
	}
}

/**
 * Function to render tooltip html.
 *
 * @param string $html  HTML string.
 * @param string $class_name class name.
 *
 * @since 1.5.2.1
 * @return false|string
 */
function bbapp_custom_tooltip( $html, $class_name = '', $bb_icon = false, $bb_icon_args = array() ) {
	ob_start();
	?>
	<span class="bb-tooltip-container <?php echo esc_attr( $class_name ); ?>">
		<?php if ( true === $bb_icon ) { ?>
			<?php
			$args_default = array(
				'class' => '',
				'icon'  => '',
			);
			$bb_icon_args = wp_parse_args( $bb_icon_args, $args_default );
			printf( '<i class="%s bb-icon-f %s"></i>', esc_attr( $bb_icon_args['icon'] ), esc_attr( $bb_icon_args['class'] ) );
		} else {
			?>
			<span class="help-tip"></span>
		<?php } ?>
		<span class="bb-tooltip-persistent"><?php echo wp_kses_post( $html ); ?></span>
	</span>
	<?php
	return ob_get_clean();
}

/**
 * Function to get access control url.
 *
 * @since 1.5.2.1
 * @return string|void
 */
function bb_access_controls_enable_link() {
	if ( bbapp_is_active( 'access_controls' ) ) {
		$access_control_url = bbapp_get_admin_url( 'admin.php?page=bbapp-access-controls' );
	} else {
		$access_control_url = bbapp_get_admin_url( 'admin.php?page=bbapp-components' );
	}

	return $access_control_url;
}

/**
 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
 * Non-scalar values are ignored.
 *
 * @param string|array $var Data to sanitize.
 *
 * @since 1.6.3
 * @return string|array
 */
function bbapp_input_clean( $var ) {
	if ( is_array( $var ) ) {
		return array_map( 'bbapp_input_clean', $var );
	} else {
		return is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
	}
}

/**
 * Check is mobile app request or not.
 *
 * @since 1.6.3
 * @return boolean
 */
function bbapp_is_app_request() {
	$is_app_request = false;
	$all_headers    = \BuddyBossApp\Auth\Common::instance()->getallheaders();

	foreach ( $all_headers as $key => $value ) {
		if ( 'appid' === strtolower( $key ) ) {
			$is_app_request = true;
			break;
		}
	}

	return $is_app_request;
}

/**
 * Function to check if the site is connected as a secondary site or not.
 *
 * @since 1.6.7
 * @return bool
 */
function bbapp_is_connected_as_secondary_site() {
	$bbapp = ManageApp::instance()->get_app();
	if ( ! empty( $bbapp ) && ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'secondary' === $bbapp['bbapp_site_type'] ) {
		return true;
	}

	return false;
}

/**
 * Function to remove height and width from html.
 *
 * @param html $html HTML.
 *
 * @since 1.7.2
 * @return HTML
 */
function bbapp_remove_height_width( $html ) {
	// Remove height and width from html tag.
	$html = ! empty( $html ) ? preg_replace( '/(width|height)=["\'\d%\s]+/i', '', $html ) : $html;
	$html = ! empty( $html ) ? preg_replace( '/(.*?)style="width:(.*?);(.*?)/i', '$1style="', $html ) : $html; // Remove width from div tag style attribute.
	return $html;
}

/**
 * Merge user defined arguments into defaults array.
 *
 * This function is used throughout BuddyPress to allow for either a string or
 * array to be merged into another array. It is identical to wp_parse_args()
 * except it allows for arguments to be passively or aggressively filtered using
 * the optional $filter_key parameter. If no $filter_key is passed, no filters
 * are applied.
 *
 * @param string|array $args       Value to merge with $defaults.
 * @param array        $defaults   Array that serves as the defaults.
 * @param string       $filter_key String to key the filters from.
 *
 * @since 1.7.4
 *
 * @return array Merged user defined values with defaults.
 */
function bb_parse_args( $args, $defaults = array(), $filter_key = '' ) {

	// Setup a temporary array from $args.
	if ( is_object( $args ) ) {
		$r = get_object_vars( $args );
	} elseif ( is_array( $args ) ) {
		$r =& $args;
	} else {
		wp_parse_str( $args, $r );
	}

	// Passively filter the args before the parse.
	if ( ! empty( $filter_key ) ) {

		/**
		 * Filters the arguments key before parsing if filter key provided.
		 *
		 * This is a dynamic filter dependent on the specified key.
		 *
		 * @param array $r Array of arguments to use.
		 *
		 * @since 1.7.4
		 */
		$r = apply_filters( 'bb_before_' . $filter_key . '_parse_args', $r );
	}

	// Parse.
	if ( is_array( $defaults ) && ! empty( $defaults ) ) {
		$r = array_merge( $defaults, $r );
	}

	// Aggressively filter the args after the parse.
	if ( ! empty( $filter_key ) ) {

		/**
		 * Filters the arguments key after parsing if filter key provided.
		 *
		 * This is a dynamic filter dependent on the specified key.
		 *
		 * @param array $r Array of parsed arguments.
		 *
		 * @since 1.7.4
		 */
		$r = apply_filters( 'bb_after_' . $filter_key . '_parse_args', $r );
	}

	// Return the parsed results.
	return $r;
}

/**
 * Gets a value that has been cached using an incremented key.
 *
 * A utility function for use by query methods like BP_Activity_Activity::get().
 *
 * @param string $key   Unique key for the query. Usually a SQL string.
 * @param string $group Cache group. Eg 'bp_activity'.
 *
 * @since 1.7.4
 * @return array|bool False if no cached values are found, otherwise an array of IDs.
 * @see   bb_core_set_incremented_cache()
 */
function bb_core_get_incremented_cache( $key, $group ) {
	$cache_key = bb_core_get_incremented_cache_key( $key, $group );

	return wp_cache_get( $cache_key, $group );
}

/**
 * Gets the key to be used when caching a value using an incremented cache key.
 *
 * The $key is hashed with a component-specific incrementor, which is used to
 * invalidate multiple caches at once.
 *
 * @param string $key   Unique key for the query. Usually a SQL string.
 * @param string $group Cache group. Eg 'bp_activity'.
 *
 * @since 1.7.4
 *
 * @return string
 */
function bb_core_get_incremented_cache_key( $key, $group ) {
	$incrementor = bb_core_get_incrementor( $group );
	$cache_key   = md5( $key . $incrementor );

	return $cache_key;
}

/**
 * Caches a value using an incremented key.
 *
 * An "incremented key" is a cache key that is hashed with a unique incrementor,
 * allowing for bulk invalidation.
 *
 * Use this method when caching data that should be invalidated whenever any
 * object of a given type is created, updated, or deleted. This usually means
 * data related to object queries, which can only reliably cached until the
 * underlying set of objects has been modified. See, eg, BP_Activity_Activity::get().
 *
 * @param string $key   Unique key for the query. Usually a SQL string.
 * @param string $group Cache group. Eg 'bp_activity'.
 * @param array  $ids   Array of IDs.
 *
 * @since 1.7.4
 *
 * @return bool
 */
function bb_core_set_incremented_cache( $key, $group, $ids ) {
	$cache_key = bb_core_get_incremented_cache_key( $key, $group );

	return wp_cache_set( $cache_key, $ids, $group );
}

/**
 * Gets a group-specific cache incrementor.
 *
 * The incrementor is paired with query identifiers (like SQL strings) to
 * create cache keys that can be invalidated en masse.
 *
 * If an incrementor does not yet exist for the given `$group`, one will
 * be created.
 *
 * @param string $group Cache group. Eg 'bp_activity'.
 *
 * @since 1.7.4
 *
 * @return string
 */
function bb_core_get_incrementor( $group ) {
	$incrementor = wp_cache_get( 'incrementor', $group );
	if ( ! $incrementor ) {
		$incrementor = microtime();
		wp_cache_set( 'incrementor', $incrementor, $group );
	}

	return $incrementor;
}

/**
 * Determine which items from a list do not have cached values.
 *
 * @param array  $item_ids    ID list.
 * @param string $cache_group The cache group to check against.
 *
 * @since 1.7.4
 *
 * @return array
 */
function bb_get_non_cached_ids( $item_ids, $cache_group ) {
	$uncached = array();

	foreach ( $item_ids as $item_id ) {
		$item_id = (int) $item_id;
		if ( false === wp_cache_get( $item_id, $cache_group ) ) {
			$uncached[] = $item_id;
		}
	}

	return $uncached;
}

/**
 * Reset a group-specific cache incrementor.
 *
 * Call this function when all incrementor-based caches associated with a given
 * cache group should be invalidated.
 *
 * @param string $group Cache group. Eg 'bb_bookmark'.
 *
 * @since 1.7.4
 *
 * @return bool True on success, false on failure.
 */
function bb_core_reset_incrementor( $group ) {
	return wp_cache_delete( 'incrementor', $group );
}

/**
 * Convert the input date to RFC3339 format.
 *
 * @param string      $date_gmt Date GMT format.
 * @param string|null $date     Optional. Date object.
 *
 * @since 1.7.4
 *
 * @return string|null ISO8601/RFC3339 formatted datetime.
 */
function bb_rest_prepare_date_response( $date_gmt, $date = null ) {
	if ( isset( $date ) ) {
		return mysql_to_rfc3339( $date ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
	}

	if ( '0000-00-00 00:00:00' === $date_gmt ) {
		return null;
	}

	return mysql_to_rfc3339( $date_gmt ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
}

/**** Bookmarked global functions started */

use BuddyBossApp\BB_Bookmarks;

if ( ! function_exists( 'bb_get_bookmarks' ) ) {
	/**
	 * Retrieve bookmark.
	 *
	 * @param array $args                   {
	 *                                      An array of optional arguments.
	 *
	 * @type array|string $type                   Optional. Array or comma-separated list of bookmark types.
	 *                                            'post'.
	 * @type int          $item_id                Optional. Get bookmark site wise. Default current site ID.
	 *                                            Default: null.
	 * @type int          $user_id                Optional. If provided, results will be limited to bookmark.
	 *                                            Default: false.
	 * @type int          $per_page               Optional. Number of items to return per page of results.
	 *                                            Default: null (no limit).
	 *                                            }
	 *
	 * @param bool  $force_cache            Bypass the cache if true.
	 *
	 * @since 1.7.4
	 *
	 * @return array {
	 * @type array        $bookmarks              Array of bookmark objects returned by the
	 *                                            paginated query. (IDs only if `fields` is set to `id`.)
	 * @type int          $total                  Total count of all bookmarks matching non-
	 *                                            paginated query params.
	 *                                            }
	 */
	function bb_get_bookmarks( $args = array(), $force_cache = false ) {
		static $cache = array();

		$r = bb_parse_args(
			$args,
			array(
				'blog_id'  => get_current_blog_id(),
				'type'     => '',
				'item_id'  => '',
				'user_id'  => false,
				'per_page' => null,
				'page'     => null,
				'count'    => false,
			),
			'bb_get_bookmarks'
		);

		$cache_key = 'bb_get_bookmarks_' . md5( maybe_serialize( $r ) );
		if ( ! isset( $cache[ $cache_key ] ) || true === $force_cache ) {
			$bookmarks           = BB_Bookmarks::get( $r );
			$cache[ $cache_key ] = $bookmarks;
		} else {
			$bookmarks = $cache[ $cache_key ];
		}

		return $bookmarks;
	}
}

if ( ! function_exists( 'bb_get_user_bookmarks' ) ) {
	/**
	 * Retrieve user bookmark.
	 *
	 * @param array $args                   {
	 *                                      An array of optional arguments.
	 *
	 * @type array|string $type                   Optional. Array or comma-separated list of bookmark types.
	 *                                            'post'.
	 * @type int          $item_id                Optional. Get bookmark site wise. Default current site ID.
	 *                                            Default: null.
	 * @type int          $user_id                Optional. If provided, results will be limited to bookmark.
	 *                                            Default: Current user ID.
	 * @type int          $per_page               Optional. Number of items to return per page of results.
	 *                                            Default: null (no limit).
	 *                                            }
	 *
	 * @param bool  $force_cache            Bypass the cache if true.
	 *
	 * @since 1.7.4
	 *
	 * @return array {
	 * @type array        $bookmarks              Array of bookmark objects returned by the
	 *                                            paginated query. (IDs only if `fields` is set to `id`.)
	 * @type int          $total                  Total count of all bookmarks matching non-
	 *                                            paginated query params.
	 *                                            }
	 */
	function bb_get_user_bookmarks( $args = array(), $force_cache = false ) {
		$r = bb_parse_args(
			$args,
			array(
				'blog_id'  => get_current_blog_id(),
				'type'     => '',
				'item_id'  => '',
				'user_id'  => get_current_user_id(),
				'per_page' => null,
				'page'     => null,
				'count'    => false,
			),
			'bb_get_user_bookmarks'
		);

		return bb_get_bookmarks( $r, $force_cache );
	}
}

if ( ! function_exists( 'bb_get_bookmark_types' ) ) {
	/**
	 * Functions to get all registered bookmark types.
	 *
	 * @param string $type type string.
	 *
	 * @since 1.7.4
	 *
	 * @return array Return all bookmark types.
	 */
	function bb_get_bookmark_types( $type = '' ) {
		$bookmark_types = apply_filters( 'bb_bookmark_get_registered_types', array() );

		if ( ! empty( $type ) && isset( $bookmark_types[ $type ] ) ) {
			return $bookmark_types[ $type ];
		}

		return $bookmark_types;
	}
}

if ( ! function_exists( 'bb_add_bookmark' ) ) {
	/**
	 * Add bookmarks.
	 *
	 * @param array|string $args             {
	 *                                       An array of arguments.
	 *
	 * @type string        $type             The specific bookmark type, used for directory
	 *                                       filtering. 'post', etc.
	 * @type int|bool      $user_id          Optional. The ID of the user associated with the bookmark
	 *                                       item. May be set to false or 0 if the item is not related
	 *                                       to any user. Default: the ID of the currently logged-in user.
	 * @type int           $item_id          Optional. The ID of the associated item.
	 * @type int           $blog_id          Optional. The ID of the associated current blog id for multisite.
	 * @type string        $status           The status associated with
	 *                                       the bookmark item is active or deactivate.
	 * @type string        $date_recorded    Optional. The GMT time, in Y-m-d h:i:s format, when
	 *                                       }
	 *
	 * @since 1.7.4
	 *
	 * @return bool|WP_Error
	 */
	function bb_add_bookmark( $args ) {

		$r = bb_parse_args(
			$args,
			array(
				'type'          => false,
				'user_id'       => get_current_user_id(),
				'item_id'       => false,
				'blog_id'       => get_current_blog_id(),
				'status'        => 1,
				'date_recorded' => current_time( 'mysql', true ),
				'error_type'    => 'wp_error',
			),
			'bb_add_bookmark'
		);

		if ( false === $r['type'] ) {
			if ( 'wp_error' === $r['error_type'] ) {
				return new WP_Error( 'bb_bookmark_type_is_required', __( 'The type is required to create a bookmark.', 'buddyboss-app' ), array( 'status' => 400 ) );
			} else {
				return false;
			}
		}

		if ( false === $r['item_id'] ) {
			if ( 'wp_error' === $r['error_type'] ) {
				return new WP_Error( 'bb_bookmark_item_id_is_required', __( 'The item ID is required to create a bookmark.', 'buddyboss-app' ), array( 'status' => 400 ) );
			} else {
				return false;
			}
		}

		// Check if bookmark is existed or not?.
		$bookmarks = BB_Bookmarks::get(
			array(
				'type'    => $r['type'],
				'user_id' => $r['user_id'],
				'item_id' => $r['item_id'],
				'blog_id' => $r['blog_id'],
				'status'  => null,
				'cache'   => false,
			)
		);

		if ( ! empty( $bookmarks['bookmarks'] ) && ( ! empty( $bookmarks['total'] ) && $r['status'] === $bookmarks['bookmarks'][0]->status ) ) {
			if ( 'wp_error' === $r['error_type'] ) {
				return new WP_Error( 'bb_bookmark_create_exists', __( 'The bookmark is already exists.', 'buddyboss-app' ), array( 'status' => 404 ) );
			} else {
				return false;
			}
		}

		$bookmark_id                  = ! empty( $bookmarks['bookmarks'] ) && isset( $bookmarks['bookmarks'][0]->id ) ? $bookmarks['bookmarks'][0]->id : 0;
		$new_bookmarks                = new stdClass();
		$new_bookmarks->id            = $bookmark_id;
		$new_bookmarks->type          = $r['type'];
		$new_bookmarks->user_id       = $r['user_id'];
		$new_bookmarks->item_id       = $r['item_id'];
		$new_bookmarks->blog_id       = $r['blog_id'];
		$new_bookmarks->status        = $r['status'];
		$new_bookmarks->date_recorded = $r['date_recorded'];
		$new_bookmarks->error_type    = $r['error_type'];
		$new_bookmarks_created        = BB_Bookmarks::add( $new_bookmarks );

		// Return if not create a bookmark.
		if ( is_wp_error( $new_bookmarks_created ) || ! $new_bookmarks_created ) {
			$error_message = is_wp_error( $new_bookmarks_created ) ? $new_bookmarks_created->get_error_message() : __( 'There is an error while adding the bookmark.', 'buddyboss-app' );
			if ( 'wp_error' === $r['error_type'] ) {
				return new WP_Error(
					'bb_bookmark_invalid_item_request',
					$error_message,
					array(
						'status' => 400,
					)
				);
			} else {
				return false;
			}
		}

		/**
		 * Fires after create a new bookmark.
		 *
		 * @param array             $r                     Array of argument to create a new bookmark.
		 * @param int|bool|WP_Error $new_bookmarks_created The ID of new bookmark when it's true otherwise return error.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array( 'bb_create_bookmark', array( $r, $new_bookmarks_created ) );

		return $new_bookmarks_created;
	}
}

if ( ! function_exists( 'bb_delete_bookmark' ) ) {
	/**
	 * Delete a bookmark.
	 *
	 * @param int $bookmark_id ID of the bookmark to delete.
	 *
	 * @since 1.7.4
	 *
	 * @return bool True on success, false on failure.
	 */
	function bb_delete_bookmark( $bookmark_id ) {

		/**
		 * Fires before the deletion of a bookmark.
		 *
		 * @param int $bookmark_id ID of the bookmark to be deleted.
		 *
		 * @since 1.7.4
		 */
		do_action( 'bb_bookmarks_before_delete_bookmark', $bookmark_id );

		// Get the bookmark object.
		$bookmark = bb_bookmarks_get_bookmark( $bookmark_id );

		// Bail if bookmark cannot be deleted.
		if ( ! empty( $bookmark ) ) {
			if ( ! BB_Bookmarks::delete( $bookmark ) ) {
				return false;
			}
		}

		/**
		 * Fires after the deletion of a bookmark.
		 *
		 * @param int $bookmark_id ID of the bookmark that was deleted.
		 *
		 * @since 1.7.4
		 */
		do_action( 'bb_delete_bookmark', $bookmark_id );

		return true;
	}
}

if ( ! function_exists( 'bb_bookmarks_get_bookmark' ) ) {
	/**
	 * Fetch a single bookmark object.
	 *
	 * When calling up a bookmark object, you should always use this function instead
	 * of instantiating BB_Bookmarks directly, so that you will inherit cache
	 * support and pass through the bb_bookmarks_get_bookmark filter.
	 *
	 * @param int $bookmark_id ID of the bookmark.
	 *
	 * @since 1.7.4
	 *
	 * @return BB_Bookmarks $bookmark The bookmark object.
	 */
	function bb_bookmarks_get_bookmark( $bookmark_id ) {
		// Backward compatibility.
		if ( ! is_numeric( $bookmark_id ) ) {
			$r = bb_parse_args(
				$bookmark_id,
				array(
					'bookmark_id' => false,
				),
				'bookmarks_get_bookmark'
			);

			$bookmark_id = $r['bookmark_id'];
		}

		$bookmark = BB_Bookmarks::get_single_bookmark( $bookmark_id );

		/**
		 * Filters a single bookmark object.
		 *
		 * @param BB_Bookmarks $bookmark Single bookmark object.
		 *
		 * @since 1.7.4
		 */
		return apply_filters( 'bb_bookmarks_get_bookmark', $bookmark );
	}
}

if ( ! function_exists( 'bb_bookmarks_get_bookmark_by_item' ) ) {
	/**
	 * Fetch a single bookmark object.
	 *
	 * When calling up a bookmark object, you should always use this function instead
	 * of instantiating BB_Bookmarks directly, so that you will inherit cache
	 * support and pass through the bb_bookmarks_get_bookmark filter.
	 *
	 * @param string           $type     Type of the bookmark.
	 * @param int|string|array $item_ids Item ids of the bookmark.
	 * @param int              $user_id  User id.
	 *
	 * @since 1.7.4
	 *
	 * @return array $bookmark The bookmark object.
	 */
	function bb_bookmarks_get_bookmark_by_item( $type, $item_ids, $user_id = 0 ) {

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		if ( ! is_array( $item_ids ) ) {
			$item_ids = array( $item_ids );
		}

		// Get the bookmark.
		$all_bookmarks = bb_get_user_bookmarks(
			array(
				'type'          => $type,
				'include_items' => $item_ids,
				'blog_id'       => get_current_blog_id(),
				'user_id'       => $user_id,
				'count'         => false,
			)
		);
		$bookmarks     = ! empty( $all_bookmarks['bookmarks'] ) ? $all_bookmarks['bookmarks'] : array();

		/**
		 * Filters bookmarks object.
		 *
		 * @param BB_Bookmarks $bookmarks Bookmarks object.
		 *
		 * @since 1.7.4
		 */
		return apply_filters( 'bb_bookmarks_get_bookmark_by_type', $bookmarks );
	}
}

if ( ! function_exists( 'bb_delete_bookmarks_by_item' ) ) {
	/**
	 * Delete bookmark by item.
	 *
	 * @param string $type    Bookmark type.
	 * @param int    $item_id Bookmark item id.
	 *
	 * @since 1.7.4
	 *
	 * @return bool
	 */
	function bb_delete_bookmarks_by_item( $type, $item_id ) {

		// Get the bookmark.
		$all_bookmarks = bb_get_user_bookmarks(
			array(
				'type'    => $type,
				'item_id' => $item_id,
				'blog_id' => get_current_blog_id(),
				'fields'  => 'id',
				'count'   => false,
			)
		);
		$bookmarks     = ! empty( $all_bookmarks['bookmarks'] ) ? $all_bookmarks['bookmarks'] : array();

		/**
		 * Fires before the deletion of bookmarks.
		 *
		 * @param array  $bookmarks Array of bookmarks to delete.
		 * @param string $type      Type of the bookmark to delete.
		 * @param int    $item_id   Item ID of the bookmark to delete.
		 *
		 * @since 1.7.4
		 */
		do_action( 'bb_bookmarks_before_delete_item_bookmarks', $bookmarks, $type, $item_id );

		if ( ! empty( $bookmarks ) ) {
			foreach ( $bookmarks as $bookmark ) {
				bb_delete_bookmark( $bookmark );
			}
		}

		/**
		 * Fires after the deletion of bookmarks.
		 *
		 * @param array  $bookmarks Array of bookmarks to delete.
		 * @param string $type      Type of the bookmark to delete.
		 * @param int    $item_id   Item ID of the bookmark to delete.
		 *
		 * @since 1.7.4
		 */
		do_action( 'bb_bookmarks_after_delete_item_bookmarks', $bookmarks, $type, $item_id );

		return true;
	}
}

if ( ! function_exists( 'bb_bookmarks_update_bookmarks_status' ) ) {
	/**
	 * Update the bookmark item.
	 *
	 * @param string $type    Bookmark type.
	 * @param int    $item_id Bookmark item ID.
	 * @param int    $status  Bookmark item status.
	 * @param int    $blog_id Optional. Get bookmark site wise. Default current site ID.
	 *
	 * @since 1.7.4
	 *
	 * @return bool True on success, false on failure.
	 */
	function bb_bookmarks_update_bookmarks_status( $type, $item_id, $status, $blog_id = 0 ) {
		return BB_Bookmarks::update_status( $type, $item_id, $status, $blog_id );
	}
}

if ( ! function_exists( 'bb_bookmarks_has_items_sql_query' ) ) {
	/**
	 * Returns the SQL for getting the items which user has access to.
	 *
	 * @param array $args sql argument.
	 *
	 * @since 1.7.4
	 * @return bool
	 */
	function bb_bookmarks_has_items_sql_query( $args ) {
		global $wpdb;

		$default_args = array(
			'columns' => 'item_id',
			'type'    => false,
			'status'  => true,
			'user_id' => get_current_user_id(),
		);

		$args = wp_parse_args( $args, $default_args );

		if ( empty( $args['type'] ) || empty( $args['columns'] ) || empty( $args['user_id'] ) ) {
			return false;
		}

		$bookmark_tbl = BB_Bookmarks::get_bookmark_tbl();

		// phpcs:disable
		$query = $wpdb->prepare( "SELECT bm.{$args['columns']} FROM {$bookmark_tbl} as bm WHERE bm.type = %s AND bm.status = %d AND bm.user_id = %d", $args['type'], $args['status'], $args['user_id'] );

		// phpcs:enable
		return $query;
	}
}
/**** Bookmarked global functions ended ****/


/**
 * Get options for normal and multisite.
 *
 * @param string $option Option name.
 * @param int    $id     Current blog id.
 *
 * @since 1.7.5
 *
 * @return null|false|mixed
 */
function bbapp_get_option( $option, $id = 0 ) {
	if ( bbapp()->is_network_activated() ) {
		$id = (int) $id;

		if ( empty( $id ) ) {
			$id = get_current_blog_id();
		}
		$values = get_blog_option( $id, $option );
	} else {
		$values = get_option( $option );
	}

	return $values;
}

/**
 * Update options for normal and multisite.
 *
 * @param string                   $option Option name.
 * @param string|int|boolean|array $value  Option value.
 * @param null                     $autoload Optional. Whether to load the option when WordPress starts up.
 * @param int                      $id     Current blog id.
 *
 * @since 1.7.5
 *
 * @return bool|mixed
 */
function bbapp_set_option( $option, $value, $autoload = null, $id = 0 ) {
	if ( bbapp()->is_network_activated() ) {
		$id = (int) $id;

		if ( empty( $id ) ) {
			$id = get_current_blog_id();
		}

		return update_blog_option( $id, $option, $value );
	} else {
		return update_option( $option, $value, $autoload );
	}
}

/**
 * Delete options for normal and multisite.
 *
 * @param string $option Option name.
 * @param int    $id     Current blog id.
 *
 * @since 1.7.5
 *
 * @return bool|mixed
 */
function bbapp_delete_option( $option, $id = 0 ) {
	if ( bbapp()->is_network_activated() ) {
		$id = (int) $id;

		if ( empty( $id ) ) {
			$id = get_current_blog_id();
		}

		return delete_blog_option( $id, $option );
	} else {
		return delete_option( $option );
	}
}

/**
 * Get a count of unread notification items for a user from platform side with component.
 *
 * @since 1.7.6
 *
 * @param int $user_id ID of the user whose unread notifications are being
 *                     counted.
 * @return int Unread notification count.
 */
function bbapp_notifications_get_unread_notification_count( $user_id = 0 ) {
	if ( empty( $user_id ) ) {
		$user_id = ( bp_displayed_user_id() ) ? bp_displayed_user_id() : bp_loggedin_user_id();
	}

	$count = wp_cache_get( $user_id, 'bbapp_notifications_unread_count' );
	if ( false === $count ) {
		$registered_components   = bp_notifications_get_registered_components();
		$registered_components[] = 'bbapp';
		$count                   = BP_Notifications_Notification::get_total_count(
			array(
				'user_id'        => $user_id,
				'component_name' => $registered_components,
				'is_new'         => true,
			)
		);
		wp_cache_set( $user_id, $count, 'bbapp_notifications_unread_count' );
	}

	/**
	 * Filters the count of unread notification items for a user.
	 *
	 * @since 1.7.6
	 *
	 * @param int $count   Count of unread notification items for a user.
	 * @param int $user_id User ID for notifications count.
	 */
	return apply_filters( 'bbapp_notifications_get_total_notification_count', (int) $count, $user_id );
}

/**
 * Checking whether the app ui is enabled.
 *
 * @since 1.8.40
 *
 * @return bool
 */
function bbapp_is_app_ui_enabled() {
	$bbapp_info = \BuddyBossApp\ManageApp::instance()->get_app_info();

	if ( empty( $bbapp_info ) || is_wp_error( $bbapp_info ) || ! is_array( $bbapp_info ) || empty( $bbapp_info['experiments'] ) ) {
		return false;
	}

	return is_array( $bbapp_info['experiments'] ) && in_array( 'app-ui', $bbapp_info['experiments'] );
}

/**
 * Get the app info transient key.
 *
 * @since 1.8.80
 * @return string
 */
function bbapp_get_app_info_transient_key() {
	$cache_key = '_bbapp_get_app_info';

	if ( is_multisite() ) {
		$current_blog_id = get_current_blog_id();

		if ( bbapp()->is_network_activated() ) {
			$cache_key = '_bbapp_get_app_info_network_' . $current_blog_id;
		} elseif ( ! is_main_site() ) {
			$cache_key = '_bbapp_get_app_info_' . $current_blog_id;
		}
	}

	return $cache_key;
}

/**
 * Is share app link enabled.
 *
 * @since 2.0.70
 * @return bool
 */
function bbapp_is_share_app_link_enabled( $platform = '' ) {
	if ( AppSettings::instance()->get_setting_value( 'app_sharing.enabled' ) ) {

		$app_platform = ClientCommon::instance()->capture_header( 'appplatform' );

		if ( 'ios' === $app_platform && bbapp_is_ios_share_app_link_enabled() ) {
			return true;
		}

		if ( 'android' === $app_platform && bbapp_is_android_share_app_link_enabled() ) {
			return true;
		}
	}

	return false;
}

/**
 * Is ios share app link enabled.
 *
 * @since 2.0.70
 * @return bool
 */
function bbapp_is_ios_share_app_link_enabled() {
	if ( BuddyBossApp\AppStores\Apple::instance()->get_ios_store_app_id() && AppSettings::instance()->get_setting_value( 'app_sharing.ios.enabled' ) ) {
		return true;
	}

	return false;
}

/**
 * Is android share app link enabled.
 *
 * @since 2.0.70
 * @return bool
 */
function bbapp_is_android_share_app_link_enabled() {
	if ( BuddyBossApp\Admin\Configure::instance()->option( 'publish.android.namespace', false ) && AppSettings::instance()->get_setting_value( 'app_sharing.android.enabled' ) ) {
		return true;
	}

	return false;
}

/**
 * Retrieves a page given by its title.
 *
 * In case of more than one post having the same title, it will check the oldest
 * publication date
 *
 * @param string       $page_title
 * @param string       $output
 * @param string|array $post_type
 *
 * @since 2.2.20
 *
 * @return WP_Post|array|null WP_Post (or array) on success, or null on failure.
 */
function bbapp_get_page_by_title( $page_title, $output = OBJECT, $post_type = 'page' ) {
	$query = new WP_Query(
		array(
			'post_type'              => $post_type,
			'title'                  => $page_title,
			'post_status'            => 'all',
			'posts_per_page'         => 1,
			'update_post_term_cache' => false,
			'update_post_meta_cache' => false,
			'orderby'                => 'date',
			'order'                  => 'ASC',
		)
	);

	if ( ! empty( $query->post ) ) {
		$_post = $query->post;

		if ( ARRAY_A === $output ) {
			return $_post->to_array();
		} elseif ( ARRAY_N === $output ) {
			return array_values( $_post->to_array() );
		}

		return $_post;
	}

	return null;
}

/**
 * Get UTF8 substring.
 *
 * @param string $string String to get substring from.
 * @param int    $start Start position.
 * @param int    $length Length of the substring.
 *
 * @since 2.2.60
 * @return string
 */
function bbapp_utf8_substr( $string, $start, $length = null ) {
	// Ensure the string is UTF-8 encoded.
	$utf8_string = preg_split( '//u', $string, - 1, PREG_SPLIT_NO_EMPTY );

	// If length is not provided, get all characters from start position.
	if ( is_null( $length ) ) {
		return implode( '', array_slice( $utf8_string, $start ) );
	}

	// Return the substring with correct start and length.
	return implode( '', array_slice( $utf8_string, $start, $length ) );
}

/**
 * Collect BuddyBoss App plugin data.
 *
 * @param array $bbapp_telemetry_data Telemetry data.
 *
 * @return array
 * @since 2.2.70
 */
function bbapp_telemetry_data( $bbapp_telemetry_data ) {
	global $wpdb;

	// Get telemetry reporting level.
	$bbapp_telemetry_option = ManageApp::instance()->get_app_setting( 'app_telemetry' );
	$bbapp_telemetry_option = empty( $bbapp_telemetry_option ) ? 'anonymous' : $bbapp_telemetry_option;

	// Define the keys to collect.
	$key_to_collect = array(
		'bbapp_db_version',
		'bbapp_languages',
		'bbapp_active_components',
		'bbapp_settings' => array( // Nested settings for bbapp.
			'app_auth.enable_signup',
			'app_auth.signup_form',
			'private_app.enabled',
			'browser_auth.enabled',
			'support_email.enabled',
			'bugs_email.enabled',
			'app_rating.ios.enabled',
			'app_rating.android.enabled',
			'app_sharing.enabled',
			'app_smartbanner.ios.enabled',
			'app_smartbanner.android.enabled',
			'post_bookmark_enable',
			'post_share_enable',
			'related_post_enable',
			'push.skip_active_members',
			'iap.purchase_before_register',
			'iap.review_version.ios',
			'compatibility.custom_tab.profile',
			'compatibility.custom_tab.group',
			'compatibility.open_link_in',
			'cleartext_traffic.enabled',
			'learndash_course_downloading',
			'learndash_author_visible',
			'learndash_date_visible',
			'learndash_reader_app_compatibility',
			'logger.info_log.enabled',
			'logger.api_log.enabled',
			'logger.iap_log.enabled',
		),
	);

	$option_table = $wpdb->options;

	// Query the database for settings.
	$bbapp_settings = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		"SELECT option_name, option_value
		FROM {$option_table}
		WHERE option_name LIKE 'bbapp_%'
			OR option_name LIKE '_bbapp_%'"
	);

	$devices_table                                     = bbapp_get_network_table( 'bbapp_user_devices' );
	$bbapp_telemetry_data['total_ios_devices']         = $wpdb->get_var( "SELECT COUNT(*) FROM {$devices_table} WHERE `platform` = 'ios'" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$bbapp_telemetry_data['total_android_devices']     = $wpdb->get_var( "SELECT COUNT(*) FROM {$devices_table} WHERE `platform` = 'android'" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$bbapp_telemetry_data['list_unique_ios_model']     = $wpdb->get_var( "SELECT COUNT(DISTINCT `device_model`) FROM {$devices_table} WHERE `platform` = 'ios'" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$bbapp_telemetry_data['list_unique_android_model'] = $wpdb->get_var( "SELECT COUNT(DISTINCT `device_model`) FROM {$devices_table} WHERE `platform` = 'android'" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$bbapp_telemetry_data['bbapp_version']             = bbapp()->plugin_version;
	$bbapp_telemetry_data['bbapp_id']                  = ManageApp::instance()->get_app_id();
	$bbapp_telemetry_data['bbapp_key']                 = ManageApp::instance()->get_auth_app_key();

	// Process each setting and collect data.
	foreach ( $bbapp_settings as $setting ) {
		$option_name  = $setting->option_name;
		$option_value = maybe_unserialize( $setting->option_value );

		// Handle nested bbapp settings.
		if ( 'bbapp_settings' === $option_name && is_array( $option_value ) ) {
			// Filter and merge nested settings.
			$filtered_settings    = array_intersect_key( $option_value, array_flip( $key_to_collect['bbapp_settings'] ) );
			$bbapp_telemetry_data = array_merge( $bbapp_telemetry_data, $filtered_settings );
		} elseif ( in_array( $option_name, $key_to_collect, true ) ) {
			// Add top-level settings.
			$bbapp_telemetry_data[ $option_name ] = $option_value;
		}
	}

	if ( 'complete' === $bbapp_telemetry_option ) {
		$bbapp_telemetry_data['admin_email'] = get_option( 'admin_email' );
	}

	// Collect comprehensive multilingual analytics data.
	$bbapp_telemetry_data = bbapp_collect_multilingual_telemetry_data( $bbapp_telemetry_data );

	return $bbapp_telemetry_data;
}


/**
 * Collect comprehensive multilingual analytics data for telemetry.
 *
 * This function gathers detailed information about multilingual usage patterns:
 * - Which users have activated the multilingual feature
 * - What specific languages they have created or configured via WP Admin
 * - Usage statistics for different languages
 * - Integration with WPML and other multilingual plugins
 *
 * @param array $telemetry_data Current telemetry data array.
 *
 * @return array Enhanced telemetry data with multilingual analytics.
 * @since 2.4.10
 */
function bbapp_collect_multilingual_telemetry_data( $telemetry_data ) {

	// Initialize multilingual analytics data.
	$multilingual_data = array();

	// Check if multilingual feature is activated.
	$active_languages                                    = get_option( 'bbapp_active_languages', array() );
	$selected_language                                   = get_option( 'bbapp_selected_language', '' );
	$multilingual_data['multilingual_feature_activated'] = ! empty( $active_languages ) && count( $active_languages ) > 1;
	$multilingual_data['total_active_languages']         = is_array( $active_languages ) ? count( $active_languages ) : 0;

	// Get WordPress default language.
	$wp_app_locale                              = bbapp_wp_locale_to_app_locale();
	$multilingual_data['wp_default_language']   = $wp_app_locale;
	$multilingual_data['app_selected_language'] = $selected_language;

	// Collect active language details.
	if ( ! empty( $active_languages ) && is_array( $active_languages ) ) {
		$app_languages_instance = \BuddyBossApp\AppLanguages::instance();
		$supported_languages    = $app_languages_instance->get_app_supported_languages();

		$active_language_details = array();
		foreach ( $active_languages as $lang_code ) {
			$active_language_details[ $lang_code ] = array(
				'code'          => $lang_code,
				'name'          => isset( $supported_languages[ $lang_code ] ) ? $supported_languages[ $lang_code ] : $lang_code,
				'is_wp_default' => ( $lang_code === $wp_app_locale ),
				'is_selected'   => ( $lang_code === $selected_language ),
			);
		}
		$multilingual_data['active_languages'] = $active_language_details;
	}

	// Get build-specific language settings.
	$build_selected_languages = maybe_unserialize( \BuddyBossApp\AppSettings::instance()->get_setting_value( 'build_selected_languages' ) );
	if ( ! empty( $build_selected_languages ) && is_array( $build_selected_languages ) ) {
		$multilingual_data['build_selected_languages']       = $build_selected_languages;
		$multilingual_data['build_selected_languages_count'] = count( $build_selected_languages );
	}

	// Usage patterns - check if users are actually using multiple languages.
	$multilingual_data['multilingual_usage_patterns'] = array(
		'has_multiple_active_languages' => ! empty( $active_languages ) && count( $active_languages ) > 1,
		'has_custom_translations'       => ! empty( $multilingual_data['custom_translations'] ),
		'has_build_specific_languages'  => ! empty( $build_selected_languages ),
		'uses_non_english_primary'      => ! empty( $selected_language ) && 'en' !== $selected_language,
		'wp_locale_differs_from_app'    => $wp_app_locale !== $selected_language,
	);

	// Add all multilingual data to telemetry.
	$telemetry_data['multilingual_analytics'] = $multilingual_data;

	return $telemetry_data;
}

/**
 * Is supported LMS enabled.
 *
 * @since 2.2.80
 * @return bool True if supported LMS is enabled, false otherwise.
 */
function bbapp_supported_lms_enabled() {
	$is_ld_lms_enabled    = bbapp_is_learndash_enabled();
	$is_mp_lms_enabled    = bbapp_is_memberpress_lms_enabled();
	$is_tutor_lms_enabled = bbapp_is_tutor_lms_plugins_active( true );

	return $is_ld_lms_enabled || $is_mp_lms_enabled || $is_tutor_lms_enabled;
}

/**
 * Get the locked pages for the app license.
 *
 * @return array
 * @since 2.2.80
 */
function bbapp_lite_app_license_locked_pages() {
	$locked_pages = array(
		'groups',
		'members',
		'activity',
		'messages',
		'photos',
		'documents',
		'videos',
		'forums',
		'topics',
		'invites',
		'connections',
		'profile',
		'bb_logged_in_connections',
		'bb_logged_in_invites',
		'bb_logged_in_activity',
		'bb_logged_in_groups',
		'bb_logged_in_photos',
		'bb_logged_in_videos',
		'bb_logged_in_forums',
		'bb_logged_in_profile',
		'bb_logged_in_view',
		'view',
	);

	/**
	 * Filters the locked pages for the app license.
	 *
	 * @param array $locked_pages Array of locked pages.
	 *
	 * @since 2.2.80
	 */
	return apply_filters( 'bbapp_lite_app_license_locked_pages', $locked_pages );
}

/**
 * Get the locked quick link block for the app license.
 *
 * @return array
 * @since 2.2.80
 */
function bbapp_lite_app_license_locked_quick_link() {
	$locked_item['core_app_screen'] = array(
		'activity',
		'documents',
		'forums',
		'groups',
		'members',
		'messages',
		'photos',
		'topics',
		'videos',
		'profile',
	);

	$bb_users_links           = NativeAppPage::instance()->quick_link_user_links();
	$exclude_from_locked_item = array( 'notifications' );

	if ( ! empty( $bb_users_links ) ) {
		foreach ( $bb_users_links as $value ) {
			if ( in_array( $value->ID, $exclude_from_locked_item, true ) ) {
				continue;
			}
			$locked_item['link'][] = $value->ID;
		}
	}

	/**
	 * Filters the locked quick link item for the app license.
	 *
	 * @param array $locked_item Array of locked link slug.
	 *
	 * @since 2.2.80
	 */
	return apply_filters( 'bbapp_lite_app_license_locked_quick_link', $locked_item );
}

/**
 * Get the JWT token decoded.
 *
 * @param string $jwt_token JWT token.
 *
 * @return array
 *
 * @since 2.2.80
 */
function bbapp_decode_jwt_token( $jwt_token ) {
	// Split the JWT into header, payload, and signature.
	list( $header, $payload, $signature ) = explode( '.', $jwt_token );

	// Decode the header and payload.
	$decoded_header  = json_decode( base64_decode( strtr( $header, '-_', '+/' ) ), true );
	$decoded_payload = json_decode( base64_decode( strtr( $payload, '-_', '+/' ) ), true );

	// Check for JSON decode errors.
	if ( json_last_error() !== JSON_ERROR_NONE ) {
		return array();
	}

	return array(
		'header'    => $decoded_header,
		'payload'   => $decoded_payload,
		'signature' => $signature,
	);
}

/**
 * Verify if the app is in grace perid.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_grace_peroid() {
	$license_data_token = AppSettings::instance()->get_setting_value( 'license_data_token' );

	if ( empty( $license_data_token ) ) {
		$app_info = ManageApp::instance()->get_app_info( true );

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && ! empty( $app_info['app_licence_type'] ) && ! empty( $app_info['license_data'] ) ) {
			$license_data_token = $app_info['license_data'];

			ManageApp::instance()->update_license_data(
				array(
					'license_type'      => $app_info['app_licence_type'],
					'license_transient' => $app_info['app_licence_type'],
					'license_token'     => $license_data_token,
				)
			);
		}
	}

	$license_data = bbapp_decode_jwt_token( $license_data_token );
	$grace_period = $license_data['payload']['grace_period'] ?? null;

	return ! empty( $grace_period ) && $grace_period > time();
}

/**
 * Whether the app is a lite app.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_lite_app() {
	return 'lite' === AppSettings::instance()->get_setting_value( 'app_license_type' );
}

/**
 * Whether the app is a live app.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_live_app() {
	return 'live' === Common::instance()->get_app_type();
}

/**
 * Whether the app is a lite app.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_lite_live_app() {
	return bbapp_is_lite_app() && bbapp_is_live_app() && ! bbapp_is_grace_peroid();
}

/**
 * Check the menu item is locked or not.
 *
 * @param array $menu_item Menu item.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_menu_locked_page( $menu_item ) {
	if ( bbapp_is_lite_live_app() && ! empty( $menu_item['object'] ) && in_array( $menu_item['object'], bbapp_lite_app_license_locked_pages(), true ) ) {
		return true;
	}

	return false;
}

/**
 * Check whether leanrdash is enabled or not.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_learndash_enabled() {
	if ( ! function_exists( 'is_plugin_active' ) ) {
		include_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	return defined( 'LEARNDASH_VERSION' ) && is_plugin_active( 'sfwd-lms/sfwd_lms.php' );
}

/**
 * Check whether tutor lms is enabled or not.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_tutor_lms_enabled() {
	if ( ! function_exists( 'is_plugin_active' ) ) {
		include_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	return defined( 'TUTOR_VERSION' ) && is_plugin_active( 'tutor/tutor.php' );
}

/**
 * Check whether memberpress lms is enabled or not.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_memberpress_lms_enabled() {
	if ( ! function_exists( 'is_plugin_active' ) ) {
		include_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	return is_plugin_active( 'memberpress-courses/main.php' );
}

/**
 * Check whether BuddyBoss Platform is enabled or not.
 *
 * @return bool
 *
 * @since 2.2.80
 */
function bbapp_is_buddyboss_platform_enabled() {
	return defined( 'BP_PLATFORM_VERSION' );
}

if ( ! function_exists( 'bbapp_is_tutor_lms_plugins_active' ) ) {
	/**
	 * Check if Tutor LMS is active.
	 *
	 * @param bool $is_pro Check if Tutor LMS Pro is active.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	function bbapp_is_tutor_lms_plugins_active( $is_pro = false ) {
		if ( $is_pro ) {
			return defined( 'TUTOR_VERSION' ) && defined( 'TUTOR_PRO_VERSION' );
		}

		return defined( 'TUTOR_VERSION' );
	}
}

/**
 * Checks if RootsIO BedRock boilerplate is being used.
 *
 * @since 2.3.20
 *
 * @return bool
 */
function bbapp_is_bedrock_enabled() {
	$wp_root = ABSPATH;

	if ( strpos( $wp_root, 'web/wp/' ) !== false ) {
		$site_root = trailingslashit( rtrim( $wp_root, 'web/wp/' ) );
		$web_root  = trailingslashit( rtrim( $wp_root, 'wp/' ) );
	}

	/*
	 * Check if 'composer.json' exists in the root and contains 'roots/bedrock'.
	 * Also check for the presence of a '.env' file which is common in Bedrock.
	 */
	$composer_file = $site_root . 'composer.json';
	$env_file      = $site_root . '.env';
	if ( file_exists( $composer_file ) ) {
		$composer_content = file_get_contents( $composer_file );
		if ( file_exists( $env_file ) && ( strpos( $composer_content, 'roots/bedrock' ) !== false ) ) {
			return true;
		}
	}

	/*
	 * Check if WordPress core files are in the '/wp/' directory.
	 * Also if 'app' directory exists (where Bedrock stores themes and plugins).
	 */
	$wp_directory  = $wp_root . 'wp-load.php';
	$app_directory = $web_root . 'app';
	if ( file_exists( $wp_directory ) && is_dir( $app_directory ) ) {
		return true;
	}

	return false;
}

/**
 * Check if WPML is active.
 *
 * @since 2.4.10
 *
 * @return bool
 */
function bbapp_is_wpml_active() {
	return defined( 'ICL_SITEPRESS_VERSION' );
}

/**
 * Set the WPML language from the app language header
 *
 * @param string $language_code Optional language code. If not provided, it will be fetched from headers.
 *
 * @since 2.4.10
 *
 * @return string|bool The language code that was set, or false if no language was set
 */
function bbapp_set_wpml_language( $language_code = null ) {
	// Skip if WPML is not active.
	if ( ! bbapp_is_wpml_active() ) {
		return false;
	}

	// Get language code if not provided.
	if ( empty( $language_code ) ) {
		$language_code = bbapp_get_app_language_header();
	}

	// If no language code is provided, don't switch.
	if ( empty( $language_code ) ) {
		return false;
	}

	// Get WPML language code.
	$lang = bbapp_get_wpml_lang_code( $language_code );

	// If we can't convert to a valid WPML language code, don't switch.
	if ( empty( $lang ) ) {
		return false;
	}

	// Switch WPML language.
	global $sitepress;
	if ( $sitepress ) {
		$sitepress->switch_lang( $lang, true );
	}

	/**
	 * Switch language action.
	 *
	 * @param string $lang Language code.
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	do_action( 'wpml_switch_language', $lang );

	return $lang;
}

/**
 * Convert the WP locale to the app locale
 *
 * @param string $language_code Optional language code. If not provided, it will be fetched from headers.
 *
 * @since 2.4.10
 */
function bbapp_wp_locale_to_app_locale( $language_code = '' ) {
	static $bbapp_locales = null;

	if ( is_null( $bbapp_locales ) ) {
		$bbapp_locales = \BuddyBossApp\AppLanguages::instance()->bbapp_get_languages_locales();
		$bbapp_locales = array_flip( $bbapp_locales );
	}

	if ( empty( $language_code ) ) {
		$default_lang   = apply_filters( 'wpml_default_language', null );
		$default_locale = apply_filters( 'wpml_locale', $default_lang );

		if ( empty( $default_locale ) ) {
			$default_locale = get_locale();
		}

		if ( ! empty( $bbapp_locales[ $default_locale ] ) ) {
			$language_code = $bbapp_locales[ $default_locale ];
		} elseif ( in_array( $default_locale, $bbapp_locales, true ) ) {
			$language_code = $default_locale;
		} else {
			$language_code = 'en';
		}
	}

	// Step 1: Apply existing BuddyBoss language mappings first.
	$app_languages = \BuddyBossApp\AppLanguages::instance();
	$mapped_locale = $app_languages->mapped_language_code( strtolower( $language_code ) );

	// Step 2: Get PTC supported languages (lightweight, cached).
	$ptc_languages = $app_languages->get_ptc_supported_languages();

	// Step 3: Direct match - check if mapped locale is directly supported.
	if ( in_array( $mapped_locale, $ptc_languages, true ) ) {
		return $mapped_locale;
	}

	// Step 4: Handle common WordPress locale variants with static mapping.
	$language_variants = array(
		// English variants - all map to 'en'.
		'en_us'          => 'en',
		'en_au'          => 'en',
		'en_gb'          => 'en',
		'en_ca'          => 'en',
		'en_nz'          => 'en',
		'en_za'          => 'en',
		'en_ie'          => 'en',
		'en_in'          => 'en',
		'en_sg'          => 'en',
		'en_my'          => 'en',
		'en_ph'          => 'en',
		'en_jm'          => 'en',

		// Portuguese variants - based on ptc-langs.json content.
		'pt_br'          => 'pt-BR',  // if pt-BR exists in PTC, use it.
		'pt_pt'          => 'pt-PT',  // if pt-PT exists in PTC, use it.
		'pt_ao'          => 'pt',

		// Chinese variants.
		'zh_cn'          => 'zh-Hans',
		'zh_hans'        => 'zh-Hans',
		'zh_tw'          => 'zh-Hans',
		'zh_hk'          => 'zh-Hans',

		// French variants.
		'fr_ca'          => 'fr',
		'fr_fr'          => 'fr',
		'fr_be'          => 'fr',
		'fr_ch'          => 'fr',

		// German variants.
		'de_de'          => 'de',
		'de_ch'          => 'de',
		'de_at'          => 'de',
		'de_de_formal'   => 'de',
		'de_ch_informal' => 'de',

		// Spanish variants.
		'es_es'          => 'es',
		'es_mx'          => 'es',
		'es_ar'          => 'es',
		'es_co'          => 'es',
		'es_cl'          => 'es',
		'es_pe'          => 'es',
		'es_ve'          => 'es',
		'es_gt'          => 'es',
		'es_cr'          => 'es',
		'es_do'          => 'es',
		'es_ec'          => 'es',
		'es_uy'          => 'es',

		// Italian variants.
		'it_it'          => 'it',
		'it_ch'          => 'it',

		// Dutch variants.
		'nl_nl'          => 'nl',
		'nl_be'          => 'nl',

		// Arabic variants.
		'ar'             => 'ar',
		'ar_sa'          => 'ar',
		'ar_eg'          => 'ar',
		'ar_ma'          => 'ar',
		'ar_tn'          => 'ar',
		'ar_dz'          => 'ar',
		'ar_jo'          => 'ar',
		'ar_iq'          => 'ar',
		'ar_kw'          => 'ar',
		'ar_lb'          => 'ar',
		'ar_sy'          => 'ar',
		'ar_ye'          => 'ar',
		'ar_ae'          => 'ar',
		'ar_qa'          => 'ar',
		'ar_bh'          => 'ar',

		// Russian variants.
		'ru_ru'          => 'ru',
		'ru_ua'          => 'ru',

		// Japanese.
		'ja'             => 'ja',

		// Korean.
		'ko_kr'          => 'ko',

		// Norwegian variants.
		'nb_no'          => 'nb',
		'nn_no'          => 'nb',
		'no'             => 'nb',

		// Swedish variants.
		'sv_se'          => 'sv',
		'sv_fi'          => 'sv',

		// Danish.
		'da_dk'          => 'da',

		// Finnish.
		'fi'             => 'fi',
		'fi_fi'          => 'fi',

		// Polish.
		'pl_pl'          => 'pl',

		// Czech.
		'cs_cz'          => 'cs',

		// Slovak.
		'sk_sk'          => 'sk',

		// Hungarian.
		'hu_hu'          => 'hu',

		// Romanian.
		'ro_ro'          => 'ro',

		// Bulgarian.
		'bg_bg'          => 'bg',

		// Croatian.
		'hr'             => 'hr',
		'hr_hr'          => 'hr',

		// Serbian.
		'sr_rs'          => 'sr',

		// Slovenian.
		'sl_si'          => 'sl',

		// Estonian.
		'et'             => 'et',
		'et_ee'          => 'et',

		// Latvian.
		'lv'             => 'lv',
		'lv_lv'          => 'lv',

		// Lithuanian.
		'lt_lt'          => 'lt',

		// Greek.
		'el'             => 'el',
		'el_gr'          => 'el',

		// Turkish.
		'tr_tr'          => 'tr',

		// Hebrew.
		'he_il'          => 'he',

		// Hindi.
		'hi_in'          => 'hi',

		// Thai.
		'th'             => 'th',
		'th_th'          => 'th',

		// Vietnamese.
		'vi'             => 'vi',
		'vi_vn'          => 'vi',

		// Indonesian.
		'id_id'          => 'id',

		// Malay.
		'ms_my'          => 'ms',

		// Filipino.
		'fil'            => 'fil',

		// Ukrainian.
		'uk'             => 'uk',
		'uk_ua'          => 'uk',

		// Belarusian.
		'be_by'          => 'be',

		// Catalan.
		'ca'             => 'ca',
		'ca_es'          => 'ca',

		// Basque.
		'eu'             => 'eu',
		'eu_es'          => 'eu',

		// Galician.
		'gl_es'          => 'gl',

		// Welsh.
		'cy'             => 'cy',
		'cy_gb'          => 'cy',

		// Irish.
		'ga_ie'          => 'ga',

		// Scots Gaelic.
		'gd'             => 'gd',
		'gd_gb'          => 'gd',

		// Icelandic.
		'is_is'          => 'is',

		// Maltese.
		'mt_mt'          => 'mt',

		// Afrikaans.
		'af'             => 'af',
		'af_za'          => 'af',

		// Swahili.
		'sw'             => 'sw',
		'sw_ke'          => 'sw',

		// Amharic.
		'am'             => 'am',
		'am_et'          => 'am',

		// Bengali.
		'bn_bd'          => 'bn',
		'bn_in'          => 'bn',

		// Gujarati.
		'gu'             => 'gu',
		'gu_in'          => 'gu',

		// Punjabi.
		'pa_in'          => 'pa',

		// Tamil.
		'ta_in'          => 'ta',
		'ta_lk'          => 'ta',

		// Telugu.
		'te'             => 'te',
		'te_in'          => 'te',

		// Kannada.
		'kn'             => 'kn',
		'kn_in'          => 'kn',

		// Malayalam.
		'ml_in'          => 'ml',

		// Marathi.
		'mr'             => 'mr',
		'mr_in'          => 'mr',

		// Nepali.
		'ne_np'          => 'ne',

		// Urdu.
		'ur'             => 'ur',
		'ur_pk'          => 'ur',

		// Persian.
		'fa_ir'          => 'fa',

		// Azerbaijani.
		'az'             => 'az',
		'az_az'          => 'az',

		// Georgian.
		'ka_ge'          => 'ka',

		// Armenian.
		'hy'             => 'hy',
		'hy_am'          => 'hy',

		// Kazakh.
		'kk'             => 'kk',
		'kk_kz'          => 'kk',

		// Kyrgyz.
		'ky_kg'          => 'ky',

		// Mongolian.
		'mn'             => 'mn',
		'mn_mn'          => 'mn',

		// Uzbek.
		'uz_uz'          => 'uz',

		// Tajik.
		'tg'             => 'tg',
		'tg_tj'          => 'tg',

		// Kurdish.
		'ckb'            => 'ckb',
		'ku'             => 'ku',

		// Pashto.
		'ps'             => 'ps',
		'ps_af'          => 'ps',

		// Sindhi.
		'sd_pk'          => 'sd',

		// Sinhala.
		'si_lk'          => 'si',

		// Burmese.
		'my_mm'          => 'my',

		// Khmer.
		'km'             => 'km',
		'km_kh'          => 'km',

		// Lao.
		'lo'             => 'lo',
		'lo_la'          => 'lo',

		// Tibetan.
		'bo'             => 'bo',
		'bo_cn'          => 'bo',
	);

	$locale_lower = strtolower( $mapped_locale );
	if ( isset( $language_variants[ $locale_lower ] ) ) {
		$variant_target = $language_variants[ $locale_lower ];
		// Verify the target is supported in PTC.
		if ( in_array( $variant_target, $ptc_languages, true ) ) {
			return $variant_target;
		}
	}

	// Step 5: Extract base language from locale (e.g., 'pt' from 'pt_BR').
	$base_language = explode( '_', $locale_lower )[0];
	$base_language = explode( '-', $base_language )[0]; // Handle 'pt-BR' format too.

	if ( $base_language !== $locale_lower && in_array( $base_language, $ptc_languages, true ) ) {
		return $base_language;
	}

	// Step 6: Final fallback to English.
	return 'en';
}


/**
 * Get the WPML language code from the app locale
 *
 * @param string $language_code Optional language code. If not provided, it will be fetched from headers.
 *
 * @since 2.4.10
 */
function bbapp_get_wpml_lang_code( $language_code = '' ) {
	if ( empty( $language_code ) ) {
		$language_code = get_locale();
	}

	return bbapp_wp_locale_to_app_locale( $language_code );
}

/**
 * Get the app language header
 *
 * @since 2.4.10
 *
 * @return string The language code from the app language header
 */
function bbapp_get_app_language_header() {
	$app_language = \BuddyBossApp\ClientCommon::instance()->get_language_code_from_header();

	return $app_language;
}

