<?php
/**
 * Holds functionality to register notification user segment.
 *
 * @package BuddyBossApp\UserSegment
 */

namespace BuddyBossApp\UserSegment;

use BuddyBossApp\UserSegment;

/**
 * Abstract class for notification user segment registration.
 */
abstract class SegmentsAbstract {

	/**
	 * Array of filters.
	 *
	 * @var array $filters
	 */
	protected $filters = array();

	/**
	 * Array of fields.
	 *
	 * @var array $fields
	 */
	protected $fields = array();

	/**
	 * Filter index.
	 *
	 * @var bool $filter_index
	 */
	protected $filter_index = false;

	/**
	 * Filter data.
	 *
	 * @var array $filter_data
	 */
	protected $filter_data = array();

	/**
	 * Load method.
	 */
	public function load() {
		add_filter( 'bbapp_user_segment_add_fields', array( $this, 'bbapp_user_segment_add_fields' ), 10, 1 );
		add_filter( 'bbapp_user_segment_filter_users', array( $this, 'bbapp_user_segment_filter_users' ), 10, 3 );
		add_action( 'bbapp_user_segment_render_scripts', array( $this, 'bbapp_user_segment_render_scripts' ), 10 );
	}

	/**
	 * Return filter data.
	 *
	 * @return array
	 */
	public function get_filter_data() {
		return $this->filter_data;
	}


	/**
	 * Return the field data easily.
	 *
	 * @param string $name Name of the field.
	 *
	 * @return null
	 */
	public function get_filter_data_value( $name ) {

		if ( isset( $this->filter_data[ $name ][ $this->filter_index ] ) ) {
			return $this->filter_data[ $name ][ $this->filter_index ];
		}

		return null;
	}

	/**
	 * Render Scripts.
	 */
	public function bbapp_user_segment_render_scripts() {
		if ( method_exists( $this, 'render_script' ) ) {
			$this->render_script();
		}
	}

	/**
	 * Get the data from child class and hook it to original filter.
	 *
	 * @param array $r_users_ids Returned user.
	 * @param array $filter_data Filtered data based on condition.
	 * @param bool  $index       Inder for the user.
	 *
	 * @return mixed
	 */
	public function bbapp_user_segment_filter_users( $r_users_ids, $filter_data, $index ) {
		$this->filter_index = $index;
		$this->filter_data  = $filter_data;

		$users_id = $this->filter_users( $r_users_ids );

		if ( is_array( $users_id ) ) {
			return $users_id;
		}

		return $r_users_ids;
	}

	/**
	 * Function to add segment fields.
	 *
	 * @param array $ori_fields Original fields.
	 *
	 * @return array
	 */
	public function bbapp_user_segment_add_fields( $ori_fields ) {

		foreach ( $this->fields as $field_type => $fields ) {

			foreach ( $fields as $field_name => $field_args ) {
				$ori_fields[ $field_type ][ $field_name ] = $field_args;
			}
		}

		return $ori_fields;
	}

	/**
	 * Adds the Segment Group.
	 *
	 * @param string $group_name  Group name.
	 * @param string $group_label Group label.
	 * @param bool   $show_child  Show children or not.
	 */
	public function add_group( $group_name, $group_label, $show_child = false ) {
		UserSegment::instance()->setup_group( $group_name, $group_label, $show_child );
	}

	/**
	 * Helper function to add filters.
	 *
	 * @param string $group_name  Group name.
	 * @param string $filter_name Filter name.
	 * @param array  $fields      Array of fields.
	 * @param array  $args        Array of arguments.
	 */
	public function add_filter( $group_name, $filter_name, $fields = array(), $args = array() ) {
		UserSegment::instance()->setup_filter( $group_name, $filter_name, $fields, $args );
	}

	/**
	 * Adds the Field into Segment Filter.
	 *
	 * @param string $field_name Field name.
	 * @param string $field_type Field type.
	 * @param array  $args       Array of arguments.
	 */
	public function add_field( $field_name, $field_type, $args = array() ) {
		$this->fields[ $field_type ][ $field_name ] = $args;
	}

	/**
	 * Filter user based on condition.
	 *
	 * @param array $user_ids Array of user ids.
	 *
	 * @return mixed
	 */
	abstract public function filter_users( $user_ids );

	/**
	 * Script load method.
	 *
	 * @return mixed
	 */
	abstract public function render_script();
}
