<?php
/**
 * Holds user segment functionality for notifications.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'UserSegment' ) ) :

	/**
	 * User Segment is a feature which helps to create custom collection of users
	 * based on given information eg. by filters, custom IDs. etc.
	 */
	class UserSegment {

		/**
		 * Class instance.
		 *
		 * @var $instance
		 */
		private static $instance;

		/**
		 * Segment filter groups.
		 *
		 * @var array $filters_groups
		 */
		private $filters_groups = array();

		/**
		 * Segment filters.
		 *
		 * @var array $filters
		 */
		private $filters = array();

		/**
		 * Segment fields.
		 *
		 * @var array $fields
		 */
		private $fields = array();

		/**
		 * Third party integration.
		 *
		 * @var array $integrations
		 */
		private $integrations = array();

		/**
		 * User segment table.
		 *
		 * @var bool $user_segment_table
		 */
		private $user_segment_table = false;

		/**
		 * UserSegment constructor.
		 */
		public function __construct() {
		}

		/**
		 * Get the instance of class
		 *
		 * @return UserSegment
		 */
		public static function instance() {
			if ( ! isset( self::$instance ) ) {
				$class          = __CLASS__;
				self::$instance = new $class();
				self::$instance->setup_globals();
				self::$instance->hooks();
			}

			return self::$instance;
		}

		/**
		 * Setup globals
		 */
		public function setup_globals() {
			$this->filters_groups = array(
				'bb_platform'     => array(
					'label'      => __( 'BuddyBoss', 'buddyboss-app' ),
					'show_child' => true,
					'enabled'    => false,
				),
				'learndash'       => array(
					'label'      => __( 'LearnDash', 'buddyboss-app' ),
					'show_child' => true,
					'enabled'    => false,
				),
				'memberpress_lms' => array(
					'label'      => __( 'MemberPress LMS', 'buddyboss-app' ),
					'show_child' => true,
					'enabled'    => false,
				),
				'membership'      => array(
					'label'      => __( 'Membership', 'buddyboss-app' ),
					'show_child' => false,
					'enabled'    => false,
				),
				'access_controls' => array(
					'label'      => __( 'App Access Control', 'buddyboss-app' ),
					'show_child' => true,
					'enabled'    => false,
				),
			);

			$this->filters = array(
				'bb_platform'     => array(
					'user_profile_type' => array(
						'label'   => __( 'Profile Type', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_social_group' => array(
						'label'   => __( 'Social Group', 'buddyboss-app' ),
						'enabled' => false,
					),
				),
				'access_controls' => array(
					'user_access_group' => array(
						'label'   => __( 'Access Group', 'buddyboss-app' ),
						'enabled' => false,
					),
				),
				'learndash'       => array(
					'user_learndash_course' => array(
						'label'   => __( 'LearnDash Course', 'buddyboss-app' ),
						'enabled' => false,
					),
				),
				'memberpress_lms' => array(
					'user_memberpress_course' => array(
						'label'   => __( 'MemberPress Course', 'buddyboss-app' ),
						'enabled' => false,
					),
				),
				'membership'      => array(
					'user_learndash_group'   => array(
						'label'   => __( 'LearnDash Group', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_memberpress_level' => array(
						'label'   => __( 'MemberPress', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_pmpro_level'       => array(
						'label'   => __( 'Paid Memberships Pro', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_rcpro_level'       => array(
						'label'   => __( 'Restrict Content Pro', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_s2member_level'    => array(
						'label'   => __( 's2Member', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_wishlist_level'    => array(
						'label'   => __( 'WishList Member', 'buddyboss-app' ),
						'enabled' => false,
					),
					'user_wc_level'          => array(
						'label'   => __( 'WooCommerce Memberships', 'buddyboss-app' ),
						'enabled' => false,
					),
				),
			);
		}

		/**
		 * Call them only when they are required.
		 *
		 * @return false|void
		 */
		public function load_segments_classes() {
			static $loaded;

			if ( isset( $loaded ) ) {
				return false;
			}
			// Apply filters in cron jobs too as we are recalculating the segment.
			if ( ( ( is_admin() || is_network_admin() ) && current_user_can( 'manage_options' ) ) || wp_doing_cron() ) {
				$this->integrations = apply_filters( 'bbapp_user_segment_integrations', $this->integrations );
			}
		}

		/**
		 * Filters/hooks here.
		 */
		public function hooks() {
			$this->user_segment_table = bbapp_get_network_table( 'bbapp_user_segment' );
			add_action( 'wp_ajax_bbapp_user_segment', array( $this, 'ajax' ) );
			add_action( 'wp_ajax_nopriv_bbapp_user_segment', array( $this, 'ajax' ) );
			add_action( 'wp_ajax_bbapp_clear_user_segment', array( $this, 'clear_user_segment' ) );
		}

		/**
		 * Return the list of filters.
		 */
		public function get_fields() {
			return apply_filters( 'bbapp_user_segment_add_fields', array() );
		}

		/**
		 * Setup integration group
		 *
		 * @param string $group_name  Group name.
		 * @param string $group_label group lable.
		 * @param bool   $show_child  Show group's filter as sub-filter otherwise it'll all group filter merge with main filter list.
		 */
		public function setup_group( $group_name, $group_label, $show_child = false ) {
			$this->filters_groups[ $group_name ] = array(
				'label'      => $group_label,
				'show_child' => $show_child,
				'enabled'    => false,
			);
		}

		/**
		 * Setup integration
		 *
		 * @param string $group_name  Group name.
		 * @param string $filter_name Filter name.
		 * @param array  $fields      Fields.
		 * @param array  $args        Filter arguments.
		 */
		public function setup_filter( $group_name, $filter_name, $fields, $args ) {
			if ( isset( $this->filters[ $group_name ][ $filter_name ] ) ) {
				$this->filters_groups[ $group_name ]['enabled']          = true;
				$this->filters[ $group_name ][ $filter_name ]['enabled'] = true;
				$this->filters[ $group_name ][ $filter_name ]['fields']  = $fields;
			} elseif ( ! empty( $args ) ) {
				$this->filters_groups[ $group_name ]['enabled']          = true;
				$this->filters[ $group_name ][ $filter_name ]            = $args;
				$this->filters[ $group_name ][ $filter_name ]['enabled'] = true;
				$this->filters[ $group_name ][ $filter_name ]['fields']  = $fields;
			}
		}

		/**
		 * Return the shorttime expire datetime.
		 *
		 * @return false|int|string
		 */
		public function get_short_time_expiry_datetime() {
			// keep min expire here. once segment is in use we will extend it later.
			// Selecting users are minute work but still 1 hours just to be safe.
			$expire = strtotime( '+1 hour', time() );
			$expire = gmdate( 'Y-m-d H:i:s', $expire );

			return $expire;
		}

		/**
		 * User Segment Ajax.
		 */
		public function ajax() {
			global $wpdb;

			$this->load_segments_classes();

			$nonce_request = ( ! empty( $_REQUEST['bbapp_user_segment_nonce'] ) ) ? wp_unslash( $_REQUEST['bbapp_user_segment_nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! wp_verify_nonce( $nonce_request, 'bbapp_user_segment_nonce' ) ) {
				wp_send_json_error( esc_html__( 'Security error occurred. Please refresh the page and try again.', 'buddyboss-app' ) );
			}

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( esc_html__( 'You don\'t have permission to perform this action.', 'buddyboss-app' ) );
			}

			$filter_data = $this->get_filter_data_from_post_var();

			if ( empty( $filter_data ) ) {
				wp_send_json_error( esc_html__( 'No segment data to process.', 'buddyboss-app' ) );
			}

			$filter_data_md5 = md5( maybe_serialize( $filter_data ) );
			$transient_key   = "_bbapp_segment_data_{$filter_data_md5}";
			$process_data    = get_transient( $transient_key );

			if ( ! is_array( $process_data ) || ! isset( $_POST['process'] ) ) { // if it's not array it's means it's new.

				// Remove Expire Segments.
				$this->remove_expire_segments();

				$segment_id = $this->get_segment_id_from_post_var();

				if ( ! $segment_id ) {
					$segment_id = time() . wp_rand();
				}

				$process_data = array(
					'id'          => $segment_id,
					'all_filters' => array(),
					'queue'       => array(),
					'completed'   => array(),
				);

				$filters = isset( $filter_data['filter'] ) ? $filter_data['filter'] : array();

				foreach ( $filters as $filter_index => $filter ) {
					$process_data['all_filters'][ $filter_index ] = $filter;
					$process_data['queue'][ $filter_index ]       = $filter;
				}

				$first_process = true;

				/**
				 * Clean all exists entries from same segment_id is there is any.
				 */
				$this->remove_segment( $process_data['id'] );
			}

			// Get one filter from queue.
			$processing_filter       = false;
			$processing_filter_index = false;

			foreach ( $process_data['queue'] as $findex => $fname ) {
				$processing_filter       = $fname;
				$processing_filter_index = $findex;
				break;
			}

			$segment_group = $processing_filter . "[$processing_filter_index]";

			/**
			 * Filters segment user.
			 *
			 * @type array $users                   Segment users.
			 * @type array $filter_data             Segment filter data.
			 * @type int   $processing_filter_index Segment processing index.
			 */
			$user_ids = apply_filters( 'bbapp_user_segment_filter_users', array(), $filter_data, $processing_filter_index );

			/**
			 * Filter to use to change user segment count change during chunk.
			 *
			 * @param int $per_batch Per batch chunk.
			 *
			 * @since 1.7.7
			 */
			$user_segment_count = (int) apply_filters( 'bbapp_user_segment_count', 500 );

			$user_ids_chunks = array_chunk( $user_ids, $user_segment_count );
			$insert_query    = "INSERT INTO {$this->user_segment_table} (`user_id`, `segment_group`, `segment_id`, `created`, `expire`) VALUES ";
			$failed          = false;
			$expire          = $this->get_short_time_expiry_datetime();

			foreach ( $user_ids_chunks as $user_ids ) {
				$prepare_var            = array();
				$query_row_placeholders = array();

				foreach ( $user_ids as $user_id ) {
					$prepare_var[]            = $user_id; // user_id.
					$prepare_var[]            = $segment_group; // segment_group.
					$prepare_var[]            = $process_data['id']; // segment_id.
					$prepare_var[]            = current_time( 'mysql', 1 ); // created.
					$prepare_var[]            = $expire; // expire.
					$query_row_placeholders[] = '(%d ,%s ,%s ,%s ,%s)';
				}

				$query  = $insert_query . implode( ",\n ", $query_row_placeholders );
				$insert = $wpdb->query( $wpdb->prepare( $query, $prepare_var ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				if ( ! $insert ) {
					$failed = true;
					break;
				}
			}

			if ( $failed ) {
				// Clear all segments as one of filter found to be failed.
				$this->remove_segment( $process_data['id'] );

				/* translators: %s: Filter name. */
				wp_send_json_error( sprintf( esc_html__( 'Error while fetching users on %s filter.', 'buddyboss-app' ), $processing_filter ) );
			}

			// looks like ids are fetched successfully let's move forward.
			// Unset processed from queue.
			unset( $process_data['queue'][ $processing_filter_index ] );
			$process_data['completed'][ $processing_filter_index ] = $processing_filter;

			/**
			 * Save the state data.
			 */

			// we only need this data until all users are fetched from segments.
			// and if it's takes more than even 1 hours something is wrong.
			$expire = 1 * HOUR_IN_SECONDS;

			set_transient( $transient_key, $process_data, $expire );

			wp_send_json_success(
				array(
					'id'          => $process_data['id'],
					'completed'   => $process_data['completed'],
					'queue'       => $process_data['queue'],
					'all_filters' => $process_data['all_filters'],
				)
			);
		}

		/**
		 * Clear Segment
		 */
		public function clear_user_segment() {
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( esc_html__( 'You don\'t have permission to perform this action.', 'buddyboss-app' ) );
			}

			$segment_id = $this->get_segment_id_from_post_var();

			$this->remove_segment( $segment_id );
			$this->remove_segment( $segment_id . '_specific' );
		}

		/**
		 * Remove the segment by segment id.
		 *
		 * @param string $segment_id Segment id.
		 *
		 * @return false|int
		 */
		public function remove_segment( $segment_id ) {
			global $wpdb;

			$this->remove_expire_segments(); // Remove garbage also.

			return $wpdb->delete( $this->user_segment_table, array( 'segment_id' => $segment_id ), array( '%s' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		/**
		 * Nothing special just delete garbage from db.
		 *
		 * @return false|int
		 */
		public function remove_expire_segments() {
			global $wpdb;

			return $wpdb->query( $wpdb->prepare( "DELETE FROM {$this->user_segment_table} WHERE expire < %s", current_time( 'mysql', 1 ) ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		/**
		 * Simple function that's returns filter data from post var.
		 *
		 * @return array
		 */
		public function get_filter_data_from_post_var() {
			$return = array();

			foreach ( $_POST as $key => $value ) {
				if ( false !== strpos( $key, 'bbapp_segments_' ) ) {
					$key            = $this->trim_form_field_name( $key );
					$return[ $key ] = $value;
				}
			}

			return $return;
		}

		/**
		 * Returns the segment ID from post var.
		 *
		 * @return bool|string
		 */
		public function get_segment_id_from_post_var() {
			$segment_id_post = ( ! empty( $_POST['bbapp_segment_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_segment_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! empty( $segment_id_post ) ) {
				$id = sanitize_title( $segment_id_post );

				if ( empty( $id ) ) {
					return false;
				}

				return $id;
			}

			return false;
		}

		/**
		 * Return the filter fields form.
		 *
		 * @param bool $segment_id Pass the user segment ID. if false given then it will be auto generated.
		 */
		public function get_form( $segment_id = false ) {
			$this->load_segments_classes();

			include bbapp()->plugin_dir . 'views/push-notifications/form.php';
		}

		/**
		 * Get form by field name.
		 *
		 * @param string $field_name Field name.
		 *
		 * @return string
		 */
		public function get_form_field_name( $field_name ) {
			return 'bbapp_segments_' . $field_name;
		}

		/**
		 * Trim the form field identifier from field-name.
		 *
		 * @param array $form_field_name Segment form fields.
		 *
		 * @return string
		 */
		public function trim_form_field_name( $form_field_name ) {
			$form_field_name = explode( 'bbapp_segments_', $form_field_name );
			unset( $form_field_name[0] );

			return implode( 'bbapp_segments_', $form_field_name );
		}

		/**
		 * Template to show in notification segment.
		 */
		public function filter_templates() {
			$fields = $this->get_fields();

			/**
			 * We render all the possible fields registered by each segment groups as template.
			 * So they can be shown when individual segment filter is added.
			 */
			foreach ( $fields as $field_type => $field_single ) {
				$classname = 'BuddyBossApp\UserSegment\Fields\\' . $field_type;

				if ( method_exists( $classname, 'render' ) && method_exists( $classname, 'render_script' ) ) {
					foreach ( $field_single as $field_name => $field_args ) {
						echo '<script type="text/html" id="tmpl-bbapp_usegment_field_' . esc_attr( $field_name ) . '">';
						echo '<p class="description">' . esc_html( $field_args['title'] ) . '</p>';
						echo '<div id="bbapp_usegment_field_{{data._index}}" class="bbapp_usegment_field bbapp_usegment_field_' . esc_attr( $field_name ) . '">';
						$classname::render( $this->get_form_field_name( $field_name ), $field_args );
						echo '</div>';
						echo '</script>';
						$classname::render_script();
					}
				}
			}
		}

		/**
		 * Removes the user from the user segments.
		 *
		 * @param string $segment_id Segment id.
		 * @param int    $user_id    User id.
		 *
		 * @return bool|false|int
		 */
		public function remove_user( $segment_id, $user_id ) {
			global $wpdb;

			if ( empty( $segment_id ) || empty( $user_id ) ) {
				return false;
			}

			return $wpdb->delete( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$this->user_segment_table,
				array(
					'segment_id' => $segment_id,
					'user_id'    => $user_id,
				),
				array(
					'%s',
					'%d',
				)
			);
		}

		/**
		 * Add the user from the user segments.
		 *
		 * @param string $segment_id Segment id.
		 * @param int    $user_id    User id.
		 *
		 * @return bool|false|int
		 */
		public function add_user( $segment_id, $user_id ) {
			global $wpdb;

			if ( empty( $segment_id ) || empty( $user_id ) ) {
				return false;
			}

			if ( $this->has_user( $segment_id, $user_id ) ) {
				return true;
			}

			return $wpdb->insert(  //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$this->user_segment_table,
				array(
					'user_id'       => $user_id,
					'segment_group' => 'custom',
					'segment_id'    => $segment_id,
					'created'       => current_time( 'mysql', 1 ),
					'expire'        => $this->get_short_time_expiry_datetime(),
				)
			);
		}

		/**
		 * Check user is in segment or not.
		 *
		 * @param string $segment_id Segment id.
		 * @param int    $user_id    User id.
		 *
		 * @return array|bool|object|\stdClass
		 */
		public function has_user( $segment_id, $user_id ) {
			global $wpdb;

			if ( empty( $segment_id ) || empty( $user_id ) ) {
				return false;
			}

			$get = $wpdb->get_row( $wpdb->prepare( "SELECT DISTINCT user_id FROM {$this->user_segment_table} WHERE segment_id=%s AND user_id=%d", $segment_id, $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			return $get;
		}

		/**
		 * Return users based on filter data.
		 *
		 * @param string $segment_id                         Notification segment id.
		 * @param int    $page                               Number of page.
		 * @param int    $limit                              Number of items.
		 * @param bool   $include_user_only_has_device_token Include only registered devices.
		 *
		 * @return array
		 */
		public function get_users( $segment_id, $page = 1, $limit = 10, $include_user_only_has_device_token = false ) {
			global $wpdb;

			$page          = max( $page - 1, 0 );
			$offset        = $limit * $page;
			$users_ids     = array();
			$devices_table = bbapp_get_network_table( 'bbapp_user_devices' );
			$wp_user_capabilities = bbapp_get_global_db_prefix() . 'capabilities'; // this will force to collect user from correct blog.

			if ( ! $include_user_only_has_device_token ) {
				$query = $wpdb->prepare(
					"SELECT DISTINCT us.user_id FROM {$this->user_segment_table} as us,{$wpdb->users} as u, {$wpdb->usermeta} as um  WHERE us.segment_id=%s AND u.ID = us.user_id AND u.ID = um.user_id AND u.user_status = '0' AND (um.meta_key = %s) order by us.user_id ASC LIMIT %d OFFSET %d",
					$segment_id,
					$wp_user_capabilities,
					$limit,
					$offset
				);
			} else {
				$query = $wpdb->prepare(
					"SELECT DISTINCT us.user_id FROM {$this->user_segment_table} as us,{$wpdb->users} as u, {$wpdb->usermeta} as um , {$devices_table} as devices WHERE us.segment_id=%s AND u.ID = us.user_id AND u.id = um.user_id AND u.ID = devices.user_id AND u.user_status = '0' AND (um.meta_key = %s) order by us.user_id ASC LIMIT %d OFFSET %d",
					$segment_id,
					$wp_user_capabilities,
					$limit,
					$offset
				);
			}

			$results = $wpdb->get_results( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			if ( ! empty( $results ) ) {
				foreach ( $results as $result ) {
					$users_ids[] = $result->user_id;
				}
			}

			return $users_ids;
		}

		/**
		 * Return the total count of users in segments.
		 *
		 * @param string $segment_id                         Segment id.
		 * @param bool   $include_user_only_has_device_token Include on registered device users.
		 *
		 * @return int|null|string
		 */
		public function get_total_users( $segment_id, $include_user_only_has_device_token = false ) {
			global $wpdb;

			$devices_table = bbapp_get_network_table( 'bbapp_user_devices' );
			$wp_user_capabilities = bbapp_get_global_db_prefix() . 'capabilities'; // this will force to collect user from correct blog.

			if ( ! $include_user_only_has_device_token ) {
				$query = $wpdb->prepare(
					"SELECT DISTINCT count(us.user_id) as count FROM {$this->user_segment_table} as us, {$wpdb->users} as u, {$wpdb->usermeta} as um  WHERE us.segment_id=%s  AND u.ID = us.user_id AND u.ID = um.user_id AND u.user_status = '0' AND (um.meta_key = %s)",
					$segment_id,
					$wp_user_capabilities
				);
				$query = "SELECT count(ID) as count FROM {$wpdb->users} WHERE ID IN ({$query})";
			} else {
				$query = $wpdb->prepare(
					"SELECT DISTINCT us.user_id as count FROM {$this->user_segment_table} as us, {$wpdb->users} as u, {$wpdb->usermeta} as um , {$devices_table} as devices  WHERE us.segment_id=%s  AND u.ID = us.user_id AND u.ID = um.user_id AND u.user_status = '0' AND u.id = devices.user_id AND (um.meta_key = %s)",
					$segment_id,
					$wp_user_capabilities
				);
				$query = "SELECT count(ID) as count FROM {$wpdb->users} WHERE ID IN ({$query})";
			}

			$count = $wpdb->get_var( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			if ( empty( $count ) ) {
				return 0;
			}

			return $count;
		}

		/**
		 * Create User Segment.
		 *
		 * @param int   $push_id Push notification id.
		 * @param int   $segment_id Segment id.
		 * @param array $filter_data Segment Filters array..
		 *
		 * @since 1.7.6
		 *
		 * @return string|array
		 */
		public function bbapp_create_user_segment( $push_id, $segment_id, $filter_data ) {

			global $wpdb;
			// Load all Segment classes.
			$this->load_segments_classes();

			// Remove Expire Segments.
			$this->remove_expire_segments();

			$process_data = array(
				'id'          => $segment_id,
				'all_filters' => array(),
				'queue'       => array(),
				'completed'   => array(),
			);

			$filters = isset( $filter_data['filter'] ) ? $filter_data['filter'] : array();

			foreach ( $filters as $filter_index => $filter ) {
				$process_data['all_filters'][ $filter_index ] = $filter;
				$process_data['queue'][ $filter_index ]       = $filter;
			}

			/**
			 * Clean all exists entries from same segment_id if there is any.
			 */
			$this->remove_segment( $process_data['id'] );

			// Get one filter from queue.
			$processing_filter       = false;
			$processing_filter_index = false;
			if ( ! empty( $process_data['queue'] ) ) {
				foreach ( $process_data['queue'] as $findex => $fname ) {
					$processing_filter       = $fname;
					$processing_filter_index = $findex;

					$segment_group = $processing_filter . "[$processing_filter_index]";

					/**
					 * Filters segment user.
					 *
					 * @type array $users                   Segment users.
					 * @type array $filter_data             Segment filter data.
					 * @type int   $processing_filter_index Segment processing index.
					 *
					 * @since 1.7.6
					 */
					$segment_user_ids = apply_filters( 'bbapp_user_segment_filter_users', array(), $filter_data, $processing_filter_index );
					if ( ! empty( $segment_user_ids ) ) {
						/**
						 * Filter to use to change user segment count change during chunk.
						 *
						 * @param int $per_batch Per batch chunk.
						 *
						 * @since 1.7.7
						 */
						$segment_user_ids_count = (int) apply_filters( 'bbapp_segment_user_ids_count', 500 );
						$user_ids_chunks        = array_chunk( $segment_user_ids, $segment_user_ids_count );
						$insert_query           = "INSERT INTO {$this->user_segment_table} (`user_id`, `segment_group`, `segment_id`, `created`, `expire`) VALUES ";
						$failed                 = false;
						$expire                 = $this->get_short_time_expiry_datetime();

						foreach ( $user_ids_chunks as $user_ids ) {
							$prepare_var            = array();
							$query_row_placeholders = array();

							foreach ( $user_ids as $user_id ) {
								$prepare_var[]            = $user_id; // user_id.
								$prepare_var[]            = $segment_group; // segment_group.
								$prepare_var[]            = $process_data['id']; // segment_id.
								$prepare_var[]            = current_time( 'mysql', 1 ); // created.
								$prepare_var[]            = $expire; // expire.
								$query_row_placeholders[] = '(%d ,%s ,%s ,%s ,%s)';
							}

							$query  = $insert_query . implode( ",\n ", $query_row_placeholders );
							$insert = $wpdb->query( $wpdb->prepare( $query, $prepare_var ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

							if ( ! $insert ) {
								$failed = true;
								break;
							}
						}
						if ( $failed ) {
							// Clear all segments as one of filter found to be failed.
							$this->remove_segment( $process_data['id'] );
							/* translators: %s: Filter name. */
							\BuddyBossApp\Notification\Push::instance()->add_notification_log( $push_id, sprintf( esc_html__( 'Error while fetching users on %s filter.', 'buddyboss-app' ), $processing_filter ) );
							continue;
						}
						$process_data['completed'][ $processing_filter_index ] = $processing_filter;
					}
				}
			}
			return $process_data;
		}
	}
endif;
