<?php
/**
 * Allow automatic updates from BuddyBoss servers
 *
 * @since   BuddyBossApp 1.0.0
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Updater' ) ) :
	/**
	 * Load Updater class
	 * This class allows WordPress to Update Plugin from BuddyBoss Server.
	 *
	 * @since BuddyBossApp 1.0.0
	 */
	class Updater {

		/**
		 * License.
		 *
		 * @var mixed|string $license License.
		 */
		public $license;

		/**
		 * Api Url.
		 *
		 * @var string $api_url Api url.
		 */
		public $api_url;

		/**
		 * Plugin id.
		 *
		 * @var int $plugin_id
		 */
		public $plugin_id = 0;

		/**
		 * Plugin path
		 *
		 * @var string $plugin_path Plugin path.
		 */
		public $plugin_path;

		/**
		 * Plugin slug.
		 *
		 * @var array|mixed|string|string[] $plugin_slug Plugin slag.
		 */
		public $plugin_slug;

		/**
		 * Plugin transient name.
		 *
		 * @var string $bbapp_transient_name
		 */
		public $bbapp_transient_name;

		/**
		 * Plugin transient time.
		 *
		 * @var string $bbapp_transient_time
		 */
		public $bbapp_transient_time = 8 * HOUR_IN_SECONDS;

		/**
		 * Class constructor.
		 *
		 * @param string     $api_url     API url.
		 * @param string     $plugin_path Plugin path.
		 * @param string|int $plugin_id   Plugin id.
		 * @param string     $license     License.
		 */
		public function __construct( $api_url, $plugin_path, $plugin_id, $license = '' ) {
			$this->api_url     = $api_url;
			$this->plugin_path = $plugin_path;
			$this->license     = $license;
			$this->plugin_id   = $plugin_id;

			if ( strstr( $plugin_path, '/' ) ) {
				list ( $part1, $part2 ) = explode( '/', $plugin_path );
			} else {
				$part2 = $plugin_path;
			}

			$this->plugin_slug          = str_replace( '.php', '', $part2 );
			$this->bbapp_transient_name = 'bb_updates_' . $this->plugin_slug;

			add_filter( 'pre_set_site_transient_update_plugins', array( &$this, 'update_plugin' ) );
			add_filter( 'site_transient_update_plugins', array( &$this, 'check_for_package' ) );
			add_filter( 'plugins_api', array( &$this, 'plugins_api' ), 99, 3 );
			add_action( 'bbapp_every_day', array( $this, 'fetch_token_everyday' ) );
		}

		/**
		 * Pre set site transient.
		 *
		 * @param object $transient Transient data.
		 *
		 * @return mixed
		 */
		public function update_plugin( $transient ) {
			if ( ! isset( $transient->response ) ) {
				return $transient;
			}

			/**
			 * Get plugin version from transient. If transient return false then we will get plugin version from
			 * get_plugin_data function.
			 *
			 * @uses get_plugin_data()
			 */
			$current_version = isset( $transient->checked[ $this->plugin_path ] ) ? $transient->checked[ $this->plugin_path ] : false;

			if ( ! $current_version ) {
				$plugin_data = get_plugin_data( trailingslashit( WP_PLUGIN_DIR ) . $this->plugin_path, false, false );

				if ( ! empty( $plugin_data ) && isset( $plugin_data['Version'] ) ) {
					$current_version = $plugin_data['Version'];
				}
			}

			if ( ! $current_version ) {
				return $transient;
			}

			// Check if force check exists.
			$force_check = ! empty( $_GET['force-check'] ) ? true : false; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

			// Check if response exists then return existing transient.
			// Also check if force check exists then bypass transient.
			if ( ! $force_check ) {
				$response_transient = get_transient( $this->bbapp_transient_name );

				if ( ! empty( $response_transient ) ) {
					if ( isset( $response_transient->body ) ) {
						unset( $response_transient->body );
						$transient->no_update[ $this->plugin_path ] = $response_transient;
					} elseif ( $current_version === $response_transient->new_version ) {
						$transient->no_update[ $this->plugin_path ] = $response_transient;
						unset( $transient->response[ $this->plugin_path ] );
					} else {
						$transient->response[ $this->plugin_path ] = $response_transient;
					}

					$transient->last_checked = time();

					return $transient;
				}
			}

			/**
			 * Get plugin version from transient. If transient return false then we will get plugin version from
			 * get_plugin_data function.
			 *
			 * @uses get_plugin_data()
			 */
			$current_version = isset( $transient->checked[ $this->plugin_path ] ) ? $transient->checked[ $this->plugin_path ] : 0;

			if ( ! $current_version ) {
				$plugin_data = get_plugin_data( trailingslashit( WP_PLUGIN_DIR ) . $this->plugin_path, false, false );

				if ( ! empty( $plugin_data ) && isset( $plugin_data['Version'] ) ) {
					$current_version = $plugin_data['Version'];
				}
			}

			if ( ! $current_version ) {
				return $transient;
			}

			$apps_data    = ManageApp::instance()->get_apps_data();
			$token        = ! empty( $apps_data['token'] ) ? $apps_data['token'] : '';
			$request_data = array(
				'id'            => $this->plugin_id,
				'slug'          => $this->plugin_slug,
				'version'       => $current_version,
				'licence_stats' => $this->bbapp_get_license_stats( $this->plugin_path ),
				'token'         => $token,
				'app_id'        => ( ! empty( $apps_data['bbapp_app_id'] ) && ! empty( $apps_data['verified'] ) ) ? $apps_data['bbapp_app_id'] : '',
			);

			if ( ! empty( $this->license ) ) {
				$request_data['license'] = $this->license;
			}

			$raw_response = '';
			$request_data = $this->bbapp_validate_token_updated( $request_data );

			if ( ! empty( $request_data['token'] ) ) {
				$request_string = $this->request_call( 'update_check', $request_data );
				$raw_response   = wp_remote_post( $this->api_url, $request_string );
			}

			$response = null;

			if ( ! empty( $raw_response ) && ! is_wp_error( $raw_response ) && ( 200 === wp_remote_retrieve_response_code( $raw_response ) ) ) {
				if ( empty( wp_remote_retrieve_body( $raw_response ) ) ) {
					// If we have no update then we store response in $transient->no_update variable.
					$no_update_response                         = new \stdClass();
					$no_update_response->id                     = $this->plugin_id;
					$no_update_response->slug                   = $this->plugin_slug;
					$no_update_response->plugin                 = $this->plugin_path;
					$no_update_response->new_version            = $current_version;
					$no_update_response->body                   = wp_remote_retrieve_body( $raw_response );
					$transient->no_update[ $this->plugin_path ] = $no_update_response;

					set_transient( $this->bbapp_transient_name, $no_update_response, $this->bbapp_transient_time );
				}

				$response = maybe_unserialize( wp_remote_retrieve_body( $raw_response ) );
			}

			// Feed the candy.
			if ( is_object( $response ) && ! empty( $response ) ) {
				$transient->response[ $this->plugin_path ] = $response;

				// Set plugins data in transient for 8 hours to avoid multiple request to hit on server.
				set_transient( $this->bbapp_transient_name, $response, $this->bbapp_transient_time );

				$transient->last_checked = time();

				return $transient;
			}

			// If there is any same plugin from wordpress.org repository then unset it.
			if ( isset( $transient->response[ $this->plugin_path ] ) ) {
				if ( strpos( $transient->response[ $this->plugin_path ]->package, 'wordpress.org' ) !== false ) {
					unset( $transient->response[ $this->plugin_path ] );
				}
			}

			$transient->last_checked = time();

			return $transient;
		}

		/**
		 * Plugins api
		 *
		 * @param false|object|array $result The result object or array. Default false.
		 * @param string             $action The type of information being requested from the Plugin Installation API.
		 * @param object             $args   Plugin API arguments.
		 *
		 * @return mixed|\WP_Error
		 */
		public function plugins_api( $result, $action, $args ) {
			if ( ! isset( $args->slug ) || $args->slug !== $this->plugin_slug ) {
				return $result;
			}

			$plugin_info  = get_site_transient( 'update_plugins' );
			$apps_data    = ManageApp::instance()->get_apps_data();
			$token        = ! empty( $apps_data['token'] ) ? $apps_data['token'] : '';
			$request_data = array(
				'id'            => $this->plugin_id,
				'slug'          => $this->plugin_slug,
				// Current version.
				'version'       => ( isset( $plugin_info->checked ) ) ? $plugin_info->checked[ $this->plugin_path ] : 0,
				'licence_stats' => $this->bbapp_get_license_stats( $this->plugin_path ),
				'token'         => $token,
				'app_id'        => ( ! empty( $apps_data['bbapp_app_id'] ) && ! empty( $apps_data['verified'] ) ) ? $apps_data['bbapp_app_id'] : '',
			);

			if ( ! empty( $this->license ) ) {
				$request_data['license'] = $this->license;
			}

			$request_data   = $this->bbapp_validate_token_updated( $request_data );
			$request_string = $this->request_call( $action, $request_data );
			$raw_response   = wp_remote_post( $this->api_url, $request_string );

			if ( is_wp_error( $raw_response ) ) {
				$res = new \WP_Error( 'plugins_api_failed', __( 'An Unexpected HTTP Error occurred during the API request.</p> <p><a href="?" onclick="document.location.reload(); return false;">Try again</a>', 'buddyboss-app' ), $raw_response->get_error_message() );
			} else {
				$res = maybe_unserialize( wp_remote_retrieve_body( $raw_response ) );

				if ( false === $res ) {
					$res = new \WP_Error( 'plugins_api_failed', __( 'An unknown error occurred', 'buddyboss-app' ), $raw_response['body'] );
				}
			}

			return $res;
		}

		/**
		 * Request call.
		 *
		 * @param string $action Action.
		 * @param array  $args   Argument.
		 *
		 * @return array
		 */
		public function request_call( $action, $args ) {
			global $wp_version;

			return array(
				'body'       => array(
					'action'  => $action,
					'request' => maybe_serialize( $args ),
					'api-key' => md5( home_url() ),
				),
				'user-agent' => 'WordPress/' . $wp_version . '; ' . home_url(),
			);
		}

		/**
		 * Transient update plugins.
		 *
		 * @param object $transient Transient object data.
		 *
		 * @since 1.6.4
		 * @return mixed
		 */
		public function check_for_package( $transient ) {
			if ( 1133 === $this->plugin_id ) {
				/**
				 * Get plugin version from transient. If transient return false then we will get plugin version from.
				 * get_plugin_data function.
				 */
				$plugin_data = get_plugin_data( trailingslashit( WP_PLUGIN_DIR ) . $this->plugin_path, false, false );

				if ( ! empty( $plugin_data ) && isset( $plugin_data['Version'] ) ) {
					$current_version = $plugin_data['Version'];
				} else {
					$current_version = $transient->checked[ $this->plugin_path ] ?? 0;
				}

				$new_version = $transient->response[ $this->plugin_path ]->new_version ?? 0;

				if ( 0 === $new_version ) {
					$new_version = $transient->no_update[ $this->plugin_path ]->new_version ?? 0;
				}

				/**
				 * If current version and new version both not same.
				 */
				if ( $current_version !== $new_version ) {
					return $transient;
				}

				/**
				 * IF plugin is already exists in  no update list.
				 */
				if ( isset( $transient->no_update[ $this->plugin_path ] ) ) {
					return $transient;
				}

				if ( isset( $transient->response[ $this->plugin_path ] ) ) {
					// Remove plugin response.
					unset( $transient->response[ $this->plugin_path ] );
				}
			}

			return $transient;
		}

		/**
		 * Get the license stats.
		 *
		 * @since BuddyBoss [BBVERSION]
		 *
		 * @param string $main_file Plugin path.
		 *
		 * @return array
		 */
		public function bbapp_get_license_stats( $main_file = '' ) {
			global $wpdb;

			$stats = array(
				'site_url'            => get_bloginfo( 'wpurl' ),
				'wp_version'          => get_bloginfo( 'version' ),
				'locale'              => get_locale(),
				'php_version'         => PHP_VERSION,
				'server_architecture' => sprintf( '%s %s %s', php_uname( 's' ), php_uname( 'r' ), php_uname( 'm' ) ),
				//phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
				'web_server'          => ( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ?? '' ),
				'db_server_ver'       => $wpdb->get_var( 'SELECT VERSION()' ),
				'db_client_ver'       => $wpdb->__get( 'dbh' )->client_info,
				'db_charset'          => $wpdb->charset,
			);

			if ( is_multisite() ) {
				$stats['multisite'] = array(
					'is_multisite' => true,
					'active'       => ! empty( $main_file ) && function_exists( 'is_plugin_active_for_network' ) && is_plugin_active_for_network( $main_file ) ? 'networkwide' : 'sitewide',
				);
			}

			return $stats;
		}

		/**
		 * Function will validate token and update in DB.
		 *
		 * @param array $args Argument of args.
		 *
		 * @since [BBVERSION]
		 *
		 * @return array
		 */
		protected function bbapp_validate_token_updated( $args ) {
			if ( empty( $args['token'] ) ) {
				$args['token'] = $this->bbapp_refresh_token();
			} else {
				$validate_token_exp = $this->bbapp_validate_token_expiry( $args['token'] );

				if ( ! $validate_token_exp ) {
					$args['token'] = $this->bbapp_refresh_token();
				}
			}

			return $args;
		}

		/**
		 * Function to validate token expiry.
		 *
		 * @since 2.2.30
		 *
		 * @param string $token Token.
		 *
		 * @return bool
		 */
		protected function bbapp_validate_token_expiry( $token ) {
			if ( empty( $token ) ) {
				return false;
			}

			$token_parts = explode( '.', $token );

			if ( count( $token_parts ) < 2 ) {
				return false;
			}

			$payload = json_decode( base64_decode( $token_parts[1] ), true ); //phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

			if ( isset( $payload['exp'] ) ) {
				// The token should be considered valid if the current time is less than the expiry time.
				return time() < $payload['exp'];
			}

			return false;
		}

		/**
		 * Refresh token using dedicated token refresh endpoint.
		 *
		 * Enhanced with:
		 * - Runtime caching to prevent duplicate calls
		 * - Cache-based change detection
		 * - Smart throttling to minimize API calls
		 * - Conditional execution (only if connected)
		 *
		 * @since 2.10.0
		 *
		 * @return false|string Token on success, false on failure
		 */
		protected function bbapp_refresh_token() {
			// Runtime cache check - prevent duplicate calls within same request.
			if ( isset( $GLOBALS['app_refresh_token_runtime_cache'] ) ) {
				return $GLOBALS['app_refresh_token_runtime_cache'];
			}

			// Early bailout - not connected to App Center.
			if ( ! ManageApp::instance()->is_app_center_connected() && ! ManageApp::instance()->is_app_center_connected( 'secondary' ) ) {
				return false;
			}

			$apps_data       = ManageApp::instance()->get_apps_data();
			$bbapp_app_id    = ( ! empty( $apps_data['bbapp_app_id'] ) && ! empty( $apps_data['verified'] ) ) ? $apps_data['bbapp_app_id'] : '';
			$bbapp_app_key   = ! empty( $apps_data['bbapp_app_key'] ) ? $apps_data['bbapp_app_key'] : '';
			$bbapp_site_type = ! empty( $apps_data['bbapp_site_type'] ) ? $apps_data['bbapp_site_type'] : '';

			if ( empty( $bbapp_app_id ) || empty( $bbapp_app_key ) ) {
				return false;
			}

			// Get cached token state.
			$cache_key     = 'app_token_refresh_state';
			$cached_state  = get_site_transient( $cache_key );
			$last_refresh  = isset( $cached_state['timestamp'] ) ? $cached_state['timestamp'] : 0;
			$cached_token  = isset( $cached_state['token'] ) ? $cached_state['token'] : '';
			$cached_expiry = isset( $cached_state['expires_at'] ) ? $cached_state['expires_at'] : 0;

			// Smart throttling - Skip API call if token is still valid for more than 48 hours.
			$expires_within_48h = ( $cached_expiry - time() ) < ( 48 * HOUR_IN_SECONDS );
			$time_since_refresh = time() - $last_refresh;
			$force_refresh      = $time_since_refresh >= ( 7 * DAY_IN_SECONDS ); // Force refresh after 7 days.

			// If token is valid for more than 48 hours and not time for forced refresh, skip API call.
			if ( ! empty( $cached_token ) && ! $expires_within_48h && ! $force_refresh ) {
				$GLOBALS['app_refresh_token_runtime_cache'] = $cached_token;

				return $cached_token;
			}

			// Make API call to new lightweight token refresh endpoint.
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/refresh-token' );
			$url     = ManageApp::instance()->get_home_url();

			$response = bbapp_remote_post(
				$api_url,
				array(
					'timeout'  => 15,
					'blocking' => true,
					'cookies'  => array(),
					'body'     => array(
						'bbapp_id'  => $bbapp_app_id,
						'bbapp_key' => sha1( $bbapp_app_key ),
						'site_url'  => $url,
						'site_type' => $bbapp_site_type,
					),
				)
			);

			if ( ! is_wp_error( $response ) ) {
				$body          = wp_remote_retrieve_body( $response );
				$response_data = json_decode( $body, true );

				if ( is_array( $response_data ) && isset( $response_data['success'] ) && true === $response_data['success'] && ! empty( $response_data['license_data'] ) ) {
					$new_token = $response_data['license_data'];

					// Calculate expiry from token or default to 30 days.
					$expires_at = $this->bbapp_get_token_expiry( $new_token );
					if ( empty( $expires_at ) ) {
						$expires_at = time() + ( 30 * DAY_IN_SECONDS );
					}

					// Update apps_data with new token.
					$apps_data['token'] = $new_token;

					ManageApp::instance()->update_apps_data( $apps_data );

					// Update license data if provided.
					if ( ! empty( $response_data['license_type'] ) ) {
						ManageApp::instance()->update_license_data(
							array(
								'license_type'      => $response_data['license_type'],
								'license_transient' => $response_data['license_type'],
								'license_token'     => $new_token,
							)
						);
					}

					// Cache the token state for 30 days.
					$new_cache = array(
						'token'      => $new_token,
						'expires_at' => $expires_at,
						'timestamp'  => time(),
					);

					set_site_transient( $cache_key, $new_cache, 30 * DAY_IN_SECONDS );

					// Runtime cache.
					$GLOBALS['app_refresh_token_runtime_cache'] = $new_token;

					return $new_token;
				}
			}

			return false;
		}

		/**
		 * Get token expiry from JWT token payload.
		 *
		 * @since 2.3.40
		 *
		 * @param string $token JWT token.
		 *
		 * @return int|false Expiry timestamp or false
		 */
		protected function bbapp_get_token_expiry( $token ) {
			if ( empty( $token ) ) {
				return false;
			}

			$token_parts = explode( '.', $token );

			if ( count( $token_parts ) < 2 ) {
				return false;
			}

			$payload = json_decode( base64_decode( $token_parts[1] ), true ); //phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

			if ( isset( $payload['exp'] ) ) {
				return $payload['exp'];
			}

			return false;
		}

		/**
		 * Fetch token everyday.
		 *
		 * @since 2.2.30
		 */
		public function fetch_token_everyday() {
			$this->bbapp_refresh_token();}
	}
endif; // End class_exists check.
