<?php
/**
 * Holds fonts typography .
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use BuddyBossApp\Helpers\BBAPP_File;

/**
 * Typography class.
 */
class Typography {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return \BuddyBossApp\Typography
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Typography constructor.
	 */
	public function __construct() {
	}

	/**
	 * Check if user can upload fonts.
	 *
	 * @param int $user_id User id.
	 *
	 * @return mixed
	 */
	public function can_upload_fonts( $user_id ) {
		return Permissions::instance()->can_upload_fonts( $user_id );
	}

	/**
	 * Return the list of scales options for fonts.
	 *
	 * @return array
	 */
	public function get_fonts_scales_opts() {
		return range( 60, 140 );
	}

	/**
	 * Return font fields.
	 *
	 * @return array
	 */
	public function font_fields() {
		$fonts_fields                   = array();
		$fonts_fields['bodyText']       = array(
			'label' => __( 'Body Text', 'buddyboss-app' ),
			'desc'  => __( 'This text style will be used for general text throughout the app.', 'buddyboss-app' ),
			'data'  => 'font-body',
			'size'  => 16,
		);
		$fonts_fields['appHeaderTitle'] = array(
			'label' => __( 'Navigation Bar Titles', 'buddyboss-app' ),
			'desc'  => __( 'This text will be used for the screen title', 'buddyboss-app' ),
			'data'  => 'font-title',
			'size'  => 34,
		);
		$fonts_fields['appHeadings']    = array(
			'label' => __( 'Headings', 'buddyboss-app' ),
			'desc'  => __( 'This text style will be used for the title on default pages and many other screens throughout the app.', 'buddyboss-app' ),
			'data'  => 'font-headings',
			'size'  => 20,
		);
		$fonts_fields['appTabBar']      = array(
			'label' => __( 'Tab Bar', 'buddyboss-app' ),
			'desc'  => __( 'This text will be used in tab bar', 'buddyboss-app' ),
			'data'  => 'font-tabbar',
			'size'  => 10,
		);
		$fonts_fields['appMenus']       = array(
			'label' => __( 'Menus', 'buddyboss-app' ),
			'desc'  => __( 'This text will be used in menus', 'buddyboss-app' ),
			'data'  => 'font-menus',
			'size'  => 17,
		);

		return $fonts_fields;
	}

	/**
	 * Returns the branding options.
	 *
	 * @return array|mixed
	 */
	public function get_options() {
		$data = (array) get_option( 'bbapp_typography' );

		return $data;
	}

	/**
	 * Updates the branding options.
	 *
	 * @param string $options Typography options.
	 */
	public function set_options( $options ) {
		update_option( 'bbapp_typography', $options );
	}

	/**
	 * Get fonts Scales
	 *
	 * @return array|mixed
	 */
	public function get_fonts_scales() {
		$data = (array) get_option( 'bbapp_typography_font_scales' );
		$data = isset( $data ) ? $data : array();

		return $data;
	}

	/**
	 * Update fonts scales.
	 *
	 * @param array $scales Typography font scales options.
	 */
	public function set_fonts_scales( $scales ) {
		update_option( 'bbapp_typography_font_scales', $scales );
	}

	/**
	 * Return allowed fonts extensions.
	 *
	 * @return array
	 */
	public function allowed_fonts_ext() {
		return array(
			'ttf',
			'otf',
			'woff',
			'woff2',
		);
	}

	/**
	 * Returns the fonts upload dir.
	 * On Multi Site it's global for all SubSites. and only can be configure on main site.
	 *
	 * @return string
	 */
	public function get_fonts_upload_dir() {
		global $bbapp_get_fonts_upload;

		if ( isset( $bbapp_get_fonts_upload['dir'] ) ) {
			return $bbapp_get_fonts_upload['dir'];
		}

		if ( bbapp()->is_network_activated() ) {
			switch_to_blog( 1 );
		}

		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/fonts';

		BBAPP_File::create_dir( $dir );

		$path = trailingslashit( $dir ) . 'fonts.json';

		if ( ! file_exists( $path ) || filesize( $path ) <= 0 ) {
			$fp = fopen( $path, 'wb' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen
			fwrite( $fp, '[]' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fwrite
			fclose( $fp ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose
		}

		$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

		if ( bbapp()->is_network_activated() ) {
			restore_current_blog();
		}

		$bbapp_get_fonts_upload['dir'] = $dir;

		return $dir;
	}

	/**
	 * Returns the fonts upload url.
	 *
	 * @return string
	 */
	public function get_fonts_upload_url() {
		global $bbapp_get_fonts_upload;

		if ( isset( $bbapp_get_fonts_upload['url'] ) ) {
			return $bbapp_get_fonts_upload['url'];
		}

		if ( bbapp()->is_network_activated() ) {
			$curr_blog_id = get_current_blog_id();
			switch_to_blog( 1 );
		}

		$upload_dir = wp_upload_dir();
		$url        = $upload_dir['baseurl'] . '/bbapp/fonts';

		if ( bbapp()->is_network_activated() ) {
			switch_to_blog( $curr_blog_id );
		}

		$bbapp_get_fonts_upload['url'] = $url;

		return $url;
	}

	/**
	 * Get the specific font from installed fonts by the font name.
	 *
	 * @param string $fontname Font name.
	 *
	 * @return array
	 */
	public function get_installed_font( $fontname ) {
		$fonts = $this->get_installed_fonts();

		if ( isset( $fonts[ $fontname ] ) ) {
			return $fonts[ $fontname ];
		}

		return array();
	}

	/**
	 * Return the font URL
	 *
	 * @param string $font_file Font file.
	 *
	 * @return string
	 */
	public function get_font_url( $font_file ) {
		return self::instance()->get_fonts_upload_url() . "/{$font_file}";
	}

	/**
	 * Return typography.
	 *
	 * @return array|mixed
	 */
	public function get_bbapp_typography() {
		$typography_options         = $this->get_options();
		$default_typography_options = $this->font_fields();
		$typography_options         = array_replace_recursive( $default_typography_options, $typography_options );
		$return                     = array();

		foreach ( $default_typography_options as $key_path => $value ) {
			$user_value          = isset( $typography_options[ $key_path ] ) ? $typography_options[ $key_path ] : $value;
			$fontfamily          = isset( $user_value['family'] ) ? $user_value['family'] : null;
			$font                = $this->bbapp_get_installed_fonts_by_family( $fontfamily, 'Default-Regular' );
			$return[ $key_path ] = array(
				'family' => $fontfamily,
				'size'   => null,
				'scale'  => isset( $font['scale'] ) ? (int) $font['scale'] : 100,
			);
		}

		return $return;
	}


	/**
	 * Get font details based font name.
	 *
	 * @param string $font_key Font key.
	 *
	 * @return array|mixed
	 */
	public function bbapp_get_font_details( $font_key ) {
		$font_dir  = self::instance()->get_fonts_upload_dir();
		$get_fonts = array();

		if ( file_exists( trailingslashit( $font_dir ) . 'fonts.json' ) ) {
			$info = BBAPP_File::read_file( trailingslashit( $font_dir ) . 'fonts.json' );
			$info = json_decode( $info, true );

			foreach ( $info as $index => $fonts ) {
				if ( isset( $fonts[ $font_key ] ) ) {
					$get_fonts = $fonts[ $font_key ];
				}
			}
		}

		return $get_fonts;
	}

	/**
	 * Return the font by font family.
	 *
	 * @param string $family_name Font family name.
	 * @param bool   $default     Include default font.
	 *
	 * @return bool
	 */
	public function bbapp_get_installed_fonts_by_family( $family_name, $default = false ) {
		$installed_fonts = $this->get_installed_fonts();
		$return          = false;

		foreach ( $installed_fonts as $font_key_name => $font ) {
			if ( $font_key_name === $family_name ) {
				$return = $font;
				break;
			}
		}

		if ( $default && ! $return ) {
			return $this->bbapp_get_installed_fonts_by_family( $default );
		}

		return $return;
	}

	/**
	 * Get installed google fonts.
	 *
	 * @param bool $include_default include default fonts.
	 *
	 * @return array|mixed
	 */
	public function get_installed_fonts( $include_default = true ) {
		global $get_installed_fonts_cache;

		if ( isset( $get_installed_fonts_cache[ $include_default ] ) ) {
			return $get_installed_fonts_cache[ $include_default ];
		}

		$font_scales = $this->get_fonts_scales();
		$font_dir    = self::instance()->get_fonts_upload_dir();
		$font_url    = self::instance()->get_fonts_upload_url();
		$font_data   = array();

		if ( file_exists( trailingslashit( $font_dir ) . 'fonts.json' ) ) {
			$info = BBAPP_File::read_file( trailingslashit( $font_dir ) . 'fonts.json' );
			$info = json_decode( $info, true );

			foreach ( $info as $index => $fonts ) {
				$font_info = array();

				foreach ( $fonts as $font_key => $font ) {
					$font_file_exists = true;
					if ( is_array( $font ) && is_array( $font['fonts_family'] ) ) {
						foreach ( $font['fonts_family'] as $key => $font_item ) {
							if ( empty( $font_item ) ) {
								continue;
							}

							$ext                            = explode( '.', $font_item['file'] );
							$ext                            = strtoupper( end( $ext ) );
							$font_info[ $key ]['base_name'] = $font_item['file'];
							$font_info[ $key ]['ext']       = $ext;
							$font_info[ $key ]['name']      = $font_item['name'];
							$font_path                      = trailingslashit( $font_dir ) . $font_item['file'];

							/**
							 * Check if the font file exists.
							 * if any of font file is not found we will exclude that font from list.
							 */
							if ( ! file_exists( $font_path ) ) {
								$font_file_exists = false;
								continue;
							}

							$font_info[ $key ]['base64'] = null;
							$base64_path                 = trailingslashit( $font_dir ) . "{$font_item["file"]}.b64.txt";
							$base64_url                  = trailingslashit( $font_url ) . "{$font_item["file"]}.b64.txt";
							if ( ! file_exists( $base64_path ) ) {
							$font_base64 = BBAPP_File::read_file( trailingslashit( $font_dir ) . $font_item['file'] );
								$font_base64 = base64_encode( $font_base64 ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
							BBAPP_File::write_file( $base64_path, $font_base64 );
								if ( file_exists( $base64_path ) ) {
									$font_info[ $key ]['base64'] = $base64_url;
								}
							} else {
								$font_info[ $key ]['base64'] = $base64_url;
							}
						}
						if ( $font_file_exists ) {
							$font_data[ $font_key ]['name']         = $font['name'];
							$font_data[ $font_key ]['fonts_family'] = $font_info;
							$font_data[ $font_key ]['scale']        = (int) isset( $font_scales[ $font_key ] ) ? $font_scales[ $font_key ] : 100;
						}
					}
				}
			}
		}
		$info = $font_data;

		if ( $include_default ) {
			$info2['Default'] = array(
				'name'         => __( 'System Default', 'buddyboss-app' ),
				'fonts_family' => array(
					'regular' => array(
						'ext'     => '',
						'default' => true,
						'base64'  => '',
						'family'  => 'Default',
					),
				),
				'scale'        => (int) isset( $font_scales['Default'] ) ? $font_scales['Default'] : 100,
			);

			foreach ( $info as $font_key => $font ) {
				$info2[ $font_key ] = $font;
			}

			$info = $info2;
			unset( $info2 );
		}

		$get_installed_fonts_cache[ $include_default ] = $info;

		return $get_installed_fonts_cache[ $include_default ];
	}

	/**
	 * Return the Organized font data to be used on CI & API.
	 */
	public function get_fonts_data() {
		$installed_fonts = self::instance()->get_installed_fonts( false );
		$fonts           = array();

		foreach ( $installed_fonts as $font_key => $font ) {
			$fonts[ $font_key ] = array();

			foreach ( $font['fonts_family'] as $variant_name => $font_data ) {
				$parsed_variant       = $this->parse_variant_name( $variant_name );
				$fonts[ $font_key ][] = array(
					'name'   => $font_data['name'],
					'weight' => isset( $parsed_variant['weight'] ) ? $parsed_variant['weight'] : 'normal',
					'style'  => ( $parsed_variant['italic'] ) ? 'italic' : 'normal',
					'ext'    => $font_data['ext'],
					'base64' => $font_data['base64'],
				);
			}
		}

		return $fonts;
	}

	/**
	 * Parse Variant
	 *
	 * @param string $variant Font variant.
	 *
	 * @return array
	 */
	public function parse_variant_name( $variant ) {
		$retval = array(
			'weight' => false,
			'italic' => false,
		);

		if ( false !== strpos( $variant, 'italic' ) ) {
			$retval['italic'] = true;
			$variant          = str_replace( 'italic', '', $variant );
		}

		$retval['weight'] = ( is_numeric( $variant ) ) ? (int) $variant : $variant;

		return $retval;
	}
}
