<?php
/**
 * Holds log display functioanlity.
 *
 * @package BuddyBossApp\Tools\Logger
 */

namespace BuddyBossApp\Tools\Logger;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use \BuddyBossApp\Tools\Logger;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Log class.
 */
class LogList extends \WP_List_Table {

	/**
	 * Class constructor
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => __( 'Log', 'buddyboss-app' ), // Singular name of the listed records.
				'plural'   => __( 'Logs', 'buddyboss-app' ), // Plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
			)
		);
	}

	/**
	 * Get logs from database.
	 *
	 * @param string $types       Log type.
	 * @param int    $per_page    Items per page.
	 * @param int    $page_number Page number.
	 *
	 * @return mixed
	 */
	public static function get_logs( $types = '', $per_page = 5, $page_number = 1 ) {
		$args             = array();
		$order_by_request = ( ! empty( $_REQUEST['orderby'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$order_request    = ( ! empty( $_REQUEST['order'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $order_by_request ) ) {
			$args['orderby'] = $order_by_request;
		}

		if ( ! empty( $order_request ) ) {
			$args['order'] = $order_request;
		}

		$args['per_page']     = $per_page;
		$args['current_page'] = $page_number;
		$args['types']        = $types;
		$results              = Helpers::get_logs( $args );

		// Convert to array.
		foreach ( $results as $k => $v ) {
			$results[ $k ] = (array) $v;
		}

		return $results;
	}

	/**
	 * Returns the count of records in the database.
	 *
	 * @param string $types Log types.
	 *
	 * @return mixed
	 */
	public static function record_count( $types = '' ) {
		return Helpers::get_logs_count( $types );
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'created' => __( 'Date', 'buddyboss-app' ),
			'details' => __( 'Details', 'buddyboss-app' ),
		);
	}

	/**
	 * No items.
	 */
	public function no_items() {
		esc_html_e( 'No items found.', 'buddyboss-app' );
	}

	/**
	 * Column data.
	 *
	 * @param object $item        Current item.
	 * @param string $column_name Column name.
	 *
	 * @return string
	 */
	public function column_default( $item, $column_name ) {
		$date_format = get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' );

		switch ( $column_name ) {
			case 'created':
				$gmt_time = gmdate( $date_format, strtotime( $item['created'] ) ) . ' GMT';

				$return = $gmt_time;
				break;
			case 'details':
				$return = nl2br( $item['text'] );
				break;
			default:
				$return = 'N/A';
				break;
		}

		return $return;
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {
		/**
		 * Init column headers.
		 */
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		/* Process action */
		$this->process_actions();

		$per_page        = 20;
		$current_page    = $this->get_pagenum();
		$supported_types = Logger::get_logger_types();
		$supported_types = array_keys( $supported_types );
		$supported_types = array_filter( $supported_types );
		$filter_get      = ( ! empty( $_GET['log_filter'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['log_filter'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$types           = ( in_array( $filter_get, $supported_types, true ) ) ? $filter_get : $supported_types;
		$total_items     = self::record_count( $types );

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);

		$this->items = self::get_logs( $types, $per_page, $current_page );
	}

	/**
	 * Process table actions.
	 */
	public function process_actions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( "You don't have permission to access this page.", 'buddyboss-app' );
		}
	}

	/**
	 * Get column id.
	 *
	 * @param object $item Current item.
	 *
	 * @return string
	 */
	public function column_id( $item ) {
		$actions = array();

		return sprintf( '%1$s %2$s', $item['id'], $this->row_actions( $actions ) );
	}

	/**
	 * Get sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {
		return array(
			'created' => array( 'created', true ),
		);

	}
}
