<?php
/**
 * Holds log helper functions.
 *
 * @package BuddyBossApp\Tools\Logger
 */

namespace BuddyBossApp\Tools\Logger;

use \BuddyBossApp\Tools\Logger;

/**
 * Log helper class.
 */
class Helpers {

	/**
	 * Returns logs
	 *
	 * @param array $args Log get arguments.
	 *
	 * @return array
	 */
	public static function get_logs( $args ) {
		global $wpdb;

		$default_args = array(
			'per_page'     => false,
			'current_page' => 1,
			'orderby'      => 'id',
			'order'        => 'desc',
		);

		$args         = array_merge( $default_args, $args );
		$where_clause = '';

		// Type filter.
		if ( ! empty( $args['types'] ) ) {
			if ( is_array( $args['types'] ) ) {
				$args['types'] = implode( "','", $args['types'] );
			}

			$where_clause = array( "type IN ('" . $args['types'] . "')" );
		}

		if ( ! empty( $where_clause ) ) {
			$where_clause = 'WHERE ' . implode( 'AND', $where_clause );
		}

		$limit_clause = '';

		if ( $args['per_page'] ) {
			$args['per_page']     = (int) $args['per_page'];
			$args['current_page'] = (int) $args['current_page'];

			$limit_clause  = " LIMIT {$args["per_page"]} ";
			$limit_clause .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
		}

		$order_clause = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order_clause .= ' ORDER BY ' . esc_sql( $args['orderby'] );
			$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$table   = bbapp_get_network_table( Logger::instance()->table );
		$sql     = "SELECT * FROM {$table} {$where_clause} {$order_clause} {$limit_clause}";
		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return $results;
	}

	/**
	 * Return count of logs.
	 *
	 * @param string $types Log type.
	 *
	 * @return mixed
	 */
	public static function get_logs_count( $types = '' ) {
		global $wpdb;

		$table = bbapp_get_network_table( Logger::instance()->table );
		$sql   = "SELECT COUNT(*) FROM {$table}";

		if ( ! empty( $types ) ) {
			if ( is_array( $types ) ) {
				$types = implode( "','", $types );
			}

			$sql .= " WHERE type IN ('" . $types . "')";
		}

		return $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Clear of logs.
	 *
	 * @param string $log_type Log type.
	 *
	 * @return mixed
	 */
	public static function clear_logs( $log_type = '' ) {
		global $wpdb;

		if ( ! empty( $log_type ) ) {
			$table_name = bbapp_get_network_table( Logger::instance()->table );
			$status     = $wpdb->delete( $table_name, array( 'type' => $log_type ), array( '%s' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		} else {
			$table_name = bbapp_get_network_table( Logger::instance()->table );
			$status     = $wpdb->query( "TRUNCATE TABLE {$table_name}" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		return $status;
	}
}
