<?php
/**
 * Holds Background process log functionality.
 *
 * @package BuddyBossApp\Tools\Logger
 */

namespace BuddyBossApp\Tools\Logger;

use BuddyBossApp\Jobs;
use DateTime;
use DateTimeZone;
use Exception;
use mysqli_result;

/**
 * API logging class.
 */
class BgProcessLog {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Table name.
	 *
	 * @var string $table_name
	 */
	public $table_name = '';

	/**
	 * Allocate the start memory.
	 *
	 * @var string $start_memory_log
	 */
	private $start_memory_log = 0;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return BgProcessLog
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Initialize the logger by setting up hooks and loading necessary data.
	 *
	 * @since 2.0.80
	 */
	public function load() {
		global $wpdb;

		$this->table_name = "{$wpdb->prefix}bb_background_process_logs";

		$this->setup_hooks();
	}

	/**
	 * Setup hooks for logging actions.
	 *
	 * @since 2.0.80
	 */
	private function setup_hooks() {
		add_filter( 'bbapp_job_before_start', array( $this, 'record_parent_bg_process' ), 10, 2 );
		add_action( 'bbapp_job_completed', array( $this, 'record_parent_job_end' ), 10, 2 );
		add_action( 'bbapp_task_started', array( $this, 'record_child_task' ), 10, 2 );
		add_action( 'bbapp_task_finished', array( $this, 'record_child_task_end' ), 10 );
		add_action( 'wp', array( $this, 'schedule_log_cleanup' ) );
		// Re-schedule when update the timezone.
		add_action( 'update_option', array( $this, 'reschedule_event' ), 10, 3 );
		add_action( 'bbapp_clear_background_process_logs', array( $this, 'clear_background_process_logs' ) );
		add_action( 'init', array( $this, 'load_hourly_cron' ) );
	}

	/**
	 * Load hourly cron.
	 *
	 * @since 2.0.90
	 * @return void
	 */
	public function load_hourly_cron() {
		add_action( 'bbapp_every_hour', array( $this, 'clear_logs_hourly' ) );
	}
	/**
	 * Create parent background process.
	 *
	 * @param array  $args     Array of arguments.
	 * @param string $job_type Job type.
	 *
	 * @since 2.0.80
	 * @return array
	 */
	public function record_parent_bg_process( $args, $job_type ) {
		global $wpdb;

		$insert = $this->add_log(
			array(
				'parent'                 => 0,
				'component'              => $this->get_component_name( $job_type ),
				'bg_process_name'        => $job_type,
				'bg_process_from'        => 'app',
				'callback_function'      => maybe_serialize( $this->get_callbacks( $job_type ) ),
				'blog_id'                => get_current_blog_id(),
				'data'                   => 'N/A',
				'priority'               => 1,
				'process_start_date_gmt' => current_time( 'mysql', 1 ),
				'process_start_date'     => current_time( 'mysql' ),
				'process_end_date_gmt'   => '0000-00-00 00:00:00',
				'process_end_date'       => '0000-00-00 00:00:00',
			)
		);

		if ( $insert ) {
			$args['body']['parent_id'] = $wpdb->insert_id;

			// Start memory usage.
			$this->start_memory_log = memory_get_peak_usage( false );
		}

		return $args;
	}

	/**
	 * Add log.
	 *
	 * @param array $args Array of arguments.
	 *
	 * @since 2.0.80
	 * @return bool|int|mysqli_result|null
	 */
	public function add_log( $args ) {
		global $wpdb;

		$defaults = array(
			'process_id'             => 0,
			'component'              => 'core',
			'bg_process_from'        => 'app',
			'blog_id'                => get_current_blog_id(),
			'priority'               => 5,
			'process_start_date_gmt' => current_time( 'mysql', 1 ),
			'process_start_date'     => current_time( 'mysql' ),
			'process_end_date_gmt'   => '0000-00-00 00:00:00',
			'process_end_date'       => '0000-00-00 00:00:00',
		);

		$args = wp_parse_args( $args, $defaults );

		if ( empty( $args['component'] ) || empty( $args['bg_process_name'] ) || empty( $args['bg_process_from'] ) || empty( $args['callback_function'] ) || empty( $args['blog_id'] ) || empty( $args['priority'] ) || empty( $args['process_start_date_gmt'] ) || empty( $args['process_start_date'] ) || empty( $args['process_end_date_gmt'] ) || empty( $args['process_end_date'] ) ) {
			return false;
		}

		return $wpdb->insert( $this->table_name, $args ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Record end date and time for parent job.
	 *
	 * @param array $process   Processes.
	 * @param int   $parent_id Parent ID.
	 *
	 * @since 2.0.80
	 * @return bool|int|mysqli_result|null
	 */
	public function record_parent_job_end( $process, $parent_id ) {
		return $this->log_end_process( $parent_id );
	}

	/**
	 * Record child task.
	 *
	 * @param object $task      Task.
	 * @param int    $parent_id Parent ID.
	 *
	 * @since 2.0.80
	 * @return bool|int|mysqli_result|null
	 */
	public function record_child_task( $task, $parent_id ) {
		$process_name = ! empty( $task->type ) ? $task->type : 'N/A';
		$component    = $this->get_component_name( ! empty( $task->type ) ? $task->type : 'core' );

		$added_child_task = $this->add_log(
			array(
				'process_id'             => ( ! empty( $task->id ) ) ? $task->id : 0,
				'parent'                 => $parent_id,
				'component'              => $component,
				'bg_process_name'        => $process_name,
				'bg_process_from'        => 'app',
				'callback_function'      => maybe_serialize( $this->get_callbacks( $process_name ) ),
				'blog_id'                => get_current_blog_id(),
				'data'                   => maybe_serialize( ( ! empty( $task->data ) ) ? $task->data : 'N/A' ),
				'priority'               => ( ! empty( $task->priority ) ) ? $task->priority : 0,
				'process_start_date_gmt' => current_time( 'mysql', 1 ),
				'process_start_date'     => current_time( 'mysql' ),
				'process_end_date_gmt'   => '0000-00-00 00:00:00',
				'process_end_date'       => '0000-00-00 00:00:00',
			)
		);

		if ( $added_child_task ) {
			// Start memory usage.
			$this->start_memory_log = memory_get_peak_usage();
		}

		return $added_child_task;
	}

	/**
	 * Record end date and time for child task.
	 *
	 * @param object $task Task.
	 *
	 * @since 2.0.80
	 *
	 * @return bool|int|mysqli_result|null
	 */
	public function record_child_task_end( $task ) {
		return $this->log_end_child_process( $task->id );
	}

	/**
	 * Schedule log cleanup.
	 *
	 * @since 2.0.80
	 * @throws Exception Exception.
	 */
	public function schedule_log_cleanup() {
		// Check if the cron job is not already scheduled.
		$is_scheduled = wp_next_scheduled( 'bbapp_clear_background_process_logs' );

		// WP datetime.
		$final_date         = date_i18n( 'Y-m-d', strtotime( 'today' ) ) . ' 23:59:59';
		$local_datetime     = date_create( $final_date, wp_timezone() );
		$schedule_timestamp = $local_datetime->getTimestamp();

		// Schedule the cron job to run every Sunday at 12 AM.
		if ( ! $is_scheduled ) {
			wp_schedule_event( $schedule_timestamp, 'daily', 'bbapp_clear_background_process_logs' );
		} elseif ( $is_scheduled !== $schedule_timestamp ) {
			wp_clear_scheduled_hook( 'bbapp_clear_background_process_logs' );
			wp_schedule_event( $schedule_timestamp, 'daily', 'bbapp_clear_background_process_logs' );
		}
	}

	/**
	 * Re-Schedule event to clear the logs.
	 *
	 * @param string $option    Option name.
	 * @param string $old_value Old value.
	 * @param string $new_value New value.
	 *
	 * @since 2.0.80
	 *
	 * @return void
	 * @throws Exception Exception.
	 */
	public function reschedule_event( $option, $old_value = '', $new_value = '' ) {
		static $is_reschedule = false; // Avoid clearing multiple time.

		// Check if the updated option is 'timezone_string'.
		if ( ( 'timezone_string' === $option || 'gmt_offset' === $option ) && $old_value !== $new_value && ! $is_reschedule ) {
			wp_clear_scheduled_hook( 'bbapp_clear_background_process_logs' );
			$this->schedule_log_cleanup();

			$is_reschedule = true;
		}
	}

	/**
	 * Clear background process logs.
	 *
	 * @since 2.0.80
	 */
	public function clear_background_process_logs() {
		global $wpdb;

		$wpdb->query( "DELETE FROM {$this->table_name} WHERE process_start_date_gmt <= NOW() - INTERVAL 1 DAY" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Reduce the background process log table size.
	 *
	 * @since 2.0.90
	 *
	 * @return void
	 */
	public function clear_logs_hourly() {
		global $wpdb;

		$table_size = $this->get_bg_process_log_table_size();

		if ( $table_size > 1024 ) {
			$rows = $wpdb->get_row( "SELECT COUNT(id) as total_count FROM {$this->table_name}" );

			if ( ! empty( $rows ) && ! empty( $rows->total_count ) ) {
				// Average Row Size (bytes) = Total Table Size (bytes) / Total Number of Rows.
				$average_row_size_byte   = round( ( $table_size * ( 1024 * 1024 ) ) / $rows->total_count );
				$total_reduce_size_mb    = $table_size - 500;
				$total_reduce_size_bytes = round( $total_reduce_size_mb * ( 1024 * 1024 ) );

				// Rows to Delete = Size to Reduce (bytes) / Average Row Size (bytes).
				$rows_to_delete = round( $total_reduce_size_bytes / $average_row_size_byte );

				// Delete records.
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} ORDER BY id ASC LIMIT %d", $rows_to_delete ) );
			}
		}
	}

	/**
	 * Get the size of the background process log table.
	 *
	 * @since 2.0.90
	 *
	 * @return float|int|string
	 */
	public function get_bg_process_log_table_size() {
		global $wpdb;

		$table_size_bytes = $wpdb->get_var( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->prepare(
				'SELECT data_length + index_length AS table_size_bytes FROM information_schema.TABLES WHERE table_schema = %s AND table_name = %s',
				$wpdb->__get( 'dbname' ),
				$this->table_name
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		if ( ! empty( $table_size_bytes ) ) {
			// Convert bytes to megabytes.
			return round( $table_size_bytes / ( 1024 * 1024 ), 2 );
		}

		return 0;
	}

	/**
	 * Log end process.
	 *
	 * @param int $id ID.
	 *
	 * @since 2.0.80
	 * @return bool|int|mysqli_result|null
	 */
	public function log_end_process( $id ) {
		global $wpdb;

		// End memory usage.
		$get_memory_used = $this->get_memory_used();

		return $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$this->table_name,
			array(
				'process_end_date_gmt' => current_time( 'mysql', 1 ),
				'process_end_date'     => current_time( 'mysql' ),
				'memory'               => $get_memory_used,
			),
			array( 'id' => $id ),
			array(
				'%s',
				'%s',
				'%s',
			),
			array( '%d' )
		);
	}

	/**
	 * Log end child process.
	 *
	 * @param int $id ID.
	 *
	 * @since 2.0.80
	 * @return bool|int|mysqli_result|null
	 */
	public function log_end_child_process( $id ) {
		global $wpdb;

		// End memory usage.
		$get_memory_used = $this->get_memory_used();

		return $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$this->table_name,
			array(
				'process_end_date_gmt' => current_time( 'mysql', 1 ),
				'process_end_date'     => current_time( 'mysql' ),
				'memory'               => $get_memory_used,
			),
			array( 'process_id' => $id ),
			array(
				'%s',
				'%s',
				'%s',
			),
			array( '%d' )
		);
	}

	/**
	 * Get the component name.
	 *
	 * @param string $type Type.
	 *
	 * @since 2.0.80
	 *
	 * @return mixed|void|null
	 */
	public function get_component_name( $type ) {
		if ( empty( $type ) ) {
			return;
		}

		switch ( $type ) {
			case 'bb_ac_term_calc':
			case 'bb_store_meta_by_ids':
			case 'bb_ac_members_calc':
				$component = 'access-controls';
				break;
			case 'rm_dup_iap_orders_13':
			case 'rm_dup_devices_24':
				$component = 'db-update';
				break;
			case 'check_upload_status':
				$component = 'file-upload';
				break;
			case 'verify_iap_products':
				$component = 'iap';
				break;
			case 'check_publish_status':
				$component = 'publish';
				break;
			case 'check_build_status':
				$component = 'build';
				break;
			case 'ld_enrolled_course':
				$component = 'ld-course';
				break;
			case 'push_batch':
			case 'process_manual_push':
				$component = 'notifications';
				break;
			case 'bgp_log_delete':
				$component = 'bg-process';
				break;
			default:
				$component = 'core';
		}

		/**
		 * Filter the component name.
		 *
		 * @param string $component Component name.
		 * @param string $type      Type.
		 *
		 * @since 2.0.80
		 */
		return apply_filters( 'bbapp_bg_process_component_name', $component, $type );
	}

	/**
	 * Get the callbacks.
	 *
	 * @param string $job_type Job type.
	 *
	 * @since 2.0.80
	 *
	 * @return array
	 */
	public function get_callbacks( $job_type ) {
		global $wp_filter;

		// Retrieve all the functions attached to the current action.
		$hooked_functions = ( ! empty( $wp_filter[ 'bbapp_queue_task_' . $job_type ] ) ) ? $wp_filter[ 'bbapp_queue_task_' . $job_type ] : array();
		$callbacks        = array();

		// Check if there are functions attached to the current action.
		if ( $hooked_functions ) {
			// Loop through each priority level.
			foreach ( $hooked_functions as $priority => $functions ) {
				// Loop through each function attached to the current priority level.
				foreach ( $functions as $function ) {
					// Get the function name or method name.
					$function_name = $function['function'];
					$callbacks[]   = ( is_array( $function_name ) && is_callable( $function_name ) ) ? ( is_object( $function_name[0] ) ? get_class( $function_name[0] ) . '->' . $function_name[1] : $function_name[0] . '::' . $function_name[1] ) : $function_name;
				}
			}
		}

		return $callbacks;
	}

	/**
	 * Create the table.
	 *
	 * @since 2.0.80
	 * @return void
	 */
	public function create_table() {
		global $wpdb;

		$log_table_name  = "{$wpdb->prefix}bb_background_process_logs";
		$charset_collate = $wpdb->get_charset_collate();
		$has_table       = $wpdb->query( $wpdb->prepare( 'show tables like %s', $log_table_name ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( ! empty( $has_table ) ) {
			return;
		}

		$sql = "CREATE TABLE $log_table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            process_id bigint(20) NOT NULL,
            parent bigint(20) NULL,
            component varchar(55) NOT NULL,
            bg_process_name varchar(255) NOT NULL,
            bg_process_from varchar(255) NOT NULL,
            callback_function varchar(255) NULL,
            blog_id bigint(20) NOT NULL,
            data longtext NULL,
            priority bigint(10) NULL,
            memory varchar(20) DEFAULT 0 NULL,
            process_start_date_gmt datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            process_start_date datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            process_end_date_gmt datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            process_end_date datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            KEY  process_start_date_gmt (process_start_date_gmt)
            ) $charset_collate";

		dbDelta( $sql );
	}

	/**
	 * Get memory usages while running the background job.
	 *
	 * @since BuddyBoss 2.0.90
	 *
	 * @return string
	 */
	public function get_memory_used() {
		$start     = $this->start_memory_log;
		$end       = memory_get_peak_usage( false );
		$mem_usage = $end - $start;

		// Reset the variable.
		$this->start_memory_log = 0;

		if ( $mem_usage < 1024 ) {
			return $mem_usage . ' Bytes';
		} elseif ( $mem_usage < 1048576 ) {
			return round( $mem_usage / 1024, 2 ) . ' KB';
		} else {
			return round( $mem_usage / 1048576, 2 ) . ' MB';
		}
	}
}
