<?php
/**
 * Holds API logging functionality.
 *
 * @package BuddyBossApp\Tools\Logger
 */

namespace BuddyBossApp\Tools\Logger;

use BuddyBossApp\Tools\Logger;
use WP_Error;

/**
 * API logging class.
 */
class ApiLogger {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * If logging is enabled.
	 *
	 * @return bool
	 */
	public function is_enabled() {
		$settings = Logger::get_settings();

		return ( ! empty( $settings['logger_enabled'] ) && ! empty( $settings['logger_api_log_enabled'] ) );
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		if ( $this->is_enabled() ) {
			add_action( 'init', array( $this, 'create_log' ) );
		}
	}

	/**
	 * Create log.
	 */
	public function create_log() {
		if ( $this->is_rest_url() ) {
			$url = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( false !== strpos( $url, 'wp-json/' ) ) {
				$this->log( $url );
			}
		}
	}

	/**
	 * Log messages.
	 *
	 * @param string|array|object $text Log text.
	 *
	 * @return WP_Error|bool
	 */
	private function log( $text ) {
		if ( is_array( $text ) || is_object( $text ) ) {
			$text = wp_json_encode( $text );
		}

		$text                = ! empty( $text ) ? 'Rest :: ' . $text . "\n" : '';
		$access_token_server = ( ! empty( $_SERVER['HTTP_ACCESSTOKEN'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['HTTP_ACCESSTOKEN'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $access_token_server ) ) {
			$text = $text . 'AccessToken :: ' . $access_token_server . "\n";
		}

		$request_method_server = ( ! empty( $_SERVER['REQUEST_METHOD'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'POST' === $request_method_server && ! empty( $_POST ) ) {
			$text = $text . 'Data :: ' . wp_json_encode( $_POST ) . "\n\n";
		}

		Logger::instance()->add( 'api_log', $text );

		return true;
	}

	/**
	 * If URL is rest URL.
	 *
	 * @return bool
	 */
	private function is_rest_url() {
		$is_rest = false;
		if ( function_exists( 'rest_url' ) && ! empty( $_SERVER['REQUEST_URI'] ) ) {
			$rest_url_base      = get_rest_url( get_current_blog_id(), '/' );
			$rest_path          = trim( wp_parse_url( $rest_url_base, PHP_URL_PATH ), '/' );
			$request_uri_server = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$request_path       = trim( $request_uri_server, '/' );
			$is_rest            = ( strpos( $request_path, $rest_path ) === 0 );
		}

		return $is_rest;
	}
}
