<?php
/**
 * Smart Banner class.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Appearance\Branding;
use BuddyBossApp\Admin\Configure;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Library\Composer;

/**
 * Smart Banner class.
 */
class SmartBanner {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Permissions
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Returns the app id.
	 *
	 * @return mixed
	 */
	public function get_app_id() {
		return \BuddyBossApp\ManageApp::instance()->get_app_id();
	}
	/**
	 * Hooks can be added here.
	 *
	 * @return void
	 */
	public function load() {
		add_action( 'init', array( $this, 'init' ) );
	}

	/**
	 * Init smartbanner feature on WordPress init.
	 *
	 * @return bool|void
	 */
	public function init() {
		$mobile_detect = Composer::instance()->mobile_detect_instance()->Mobile_Detect();

		if ( ! $mobile_detect->isiOS() && ! $mobile_detect->isAndroidOS() ) {
			return;
		}

		$smartbanner_data = $this->get_smartbanner_data();

		if ( ! empty( $smartbanner_data ) ) {
			if ( ! empty( $smartbanner_data['google_activate'] ) || ! empty( $smartbanner_data['apple_activate'] ) ) {
				add_action( 'wp_head', array( $this, 'add_smart_banner' ) );
				add_action( 'wp_enqueue_scripts', array( $this, 'add_google_smart_banner_assets' ) );
			}
		}
	}

	/**
	 * Returns Smartbanner data.
	 *
	 * @return bool|array
	 */
	public function get_smartbanner_data() {
		$bbapp_global_settings             = ManageApp::instance()->get_settings();
		$smart_banner_logged_in_visibility = ! empty( $bbapp_global_settings['app_smartbanner.logged_in_only'] );
		if ( $smart_banner_logged_in_visibility && ! is_user_logged_in() ) {
			return false;
		}

		$is_bbapp_fallback         = function_exists( 'bbapp_is_loaded_from_inapp_browser' ) && bbapp_is_loaded_from_inapp_browser();
		$is_bbapp_mobile_view      = filter_input( INPUT_GET, 'mobile-view-content', FILTER_SANITIZE_NUMBER_INT );
		$is_bbapp_installer_screen = filter_input( INPUT_GET, 'bbapp_installer_screen', FILTER_SANITIZE_NUMBER_INT );
		if ( ! empty( $is_bbapp_installer_screen ) || ! empty( $is_bbapp_mobile_view ) || ! empty( $is_bbapp_fallback ) ) {
			return false;
		}

		$ios_smart_banner_activate     = ! empty( $bbapp_global_settings['app_smartbanner.ios.enabled'] ) ? $bbapp_global_settings['app_smartbanner.ios.enabled'] : false;
		$ios_app_id                    = Apple::instance()->get_ios_store_app_id();
		$android_smart_banner_activate = ! empty( $bbapp_global_settings['app_smartbanner.android.enabled'] ) ? $bbapp_global_settings['app_smartbanner.android.enabled'] : false;
		$android_application_id        = Configure::instance()->option( 'publish.android.namespace' );

		return array(
			'apple_activate'  => ! empty( $ios_smart_banner_activate ) && ! empty( $ios_app_id ),
			'google_activate' => ! empty( $android_smart_banner_activate ) && ! empty( $android_application_id ),
			'apple_id'        => $ios_app_id,
			'google_id'       => $android_application_id,
		);
	}

	/**
	 * Output smartbanner on frontend.
	 *
	 * @return void
	 */
	public function add_smart_banner() {
		$smartbanner_data = $this->get_smartbanner_data();
		if ( ( isset( $smartbanner_data['apple_activate'] ) && $smartbanner_data['apple_activate'] ) ) {
			echo '<meta name="apple-itunes-app" content="app-id=' . bbapp_input_clean( wp_unslash( $smartbanner_data['apple_id'] ) ) . '"/>' . "\n"; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
		if ( ( isset( $smartbanner_data['google_activate'] ) && $smartbanner_data['google_activate'] ) ) {
			echo '<meta name="google-play-app" content="app-id=' . bbapp_input_clean( wp_unslash( $smartbanner_data['google_id'] ) ) . '"/>' . "\n"; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		$site_title       = get_bloginfo( 'name' );
		$site_description = get_bloginfo( 'description' );
		echo '<meta name="smartbanner:title" content="' . bbapp_input_clean( wp_unslash( $site_title ) ) . '">' . "\n"; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '<meta name="smartbanner:author" content="' . bbapp_input_clean( wp_unslash( $site_description ) ) . '">' . "\n"; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		$this->output_app_icon();
	}

	/**
	 * Output icon on frontend
	 *
	 * @return bool
	 */
	public function output_app_icon() {
		$app_id   = $this->get_app_id();
		$app_icon = Branding::instance()->get_upload_file_info( 'app_icon_ios.png', $app_id );
		if ( ! $app_icon || ! isset( $app_icon['fileurl'] ) ) {
			return false;
		}
		$app_icon = $app_icon['fileurl']
		?>
		<link rel="icon" href="<?php echo esc_url( $app_icon ); ?>"/>
		<link rel="apple-touch-icon-precomposed" href="<?php echo esc_url( $app_icon ); ?>"/>
		<?php
	}

	/**
	 * Enqueue script & style for smartbanner logically.
	 *
	 * @return void
	 */
	public function add_google_smart_banner_assets() {
		wp_enqueue_script( bbapp()->plugin_prefix . '_jquery.detectIncognito' );
		wp_enqueue_script( bbapp()->plugin_prefix . '_jquery.smartbanner' );
	}
}
