<?php
/**
 * RestCDN class.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Settings;
use WP_REST_Request;
use WP_REST_Response;

/**
 * RestCDN class.
 */
class RestCDN {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Blog URL.
	 *
	 * @var $blog_url
	 */
	public $blog_url;

	/**
	 * CDN URL.
	 *
	 * @var $cdn_url
	 */
	public $cdn_url;

	/**
	 * Directory scope.
	 *
	 * @var $dirs
	 */
	public $dirs;

	/**
	 * Excludes extenstion.
	 *
	 * @var $excludes
	 */
	public $excludes;

	/**
	 * Relative URL.
	 *
	 * @var $relative
	 */
	public $relative;

	/**
	 * URL has https.
	 *
	 * @var $https
	 */
	public $https;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Add BuddyBoss App related cache hooks
	 */
	public function load() {
		$global_settings = Settings::instance()->get_settings();
		$cdn_enabled     = ( bbapp_is_active( 'performance' ) && ! empty( $global_settings['cdn.enabled'] ) ) ? $global_settings['cdn.enabled'] : false;
		if ( ! empty( $cdn_enabled ) && true === (bool) $cdn_enabled ) {
			add_action( 'rest_api_init', array( $this, 'run_actions' ), 10, 1 );
		}
	}

	/**
	 * Execute Actions
	 */
	public function run_actions() {
		$global_settings = Settings::instance()->get_settings();
		$this->blog_url  = get_option( 'home' );
		$this->cdn_url   = ! empty( $global_settings['cdn.url'] ) ? $global_settings['cdn.url'] : get_home_url();
		$this->dirs      = 'wp-content/uploads|wp-content/plugins';
		$this->excludes  = array( '.php' );
		$this->relative  = true;
		$this->https     = ( isset( $_SERVER['HTTPS'] ) && 'on' === (string) sanitize_text_field( wp_unslash( $_SERVER['HTTPS'] ) ) );
		add_filter( 'rest_post_dispatch', array( $this, 'rewrite_rest_post_dispatch' ), 10, 3 );
	}

	/**
	 * Override dispatching the request.
	 *
	 * @param WP_REST_Response $response Rest response.
	 * @param array            $handler  API handler.
	 * @param WP_REST_Request  $request  Rest request.
	 */
	public function rewrite_rest_post_dispatch( $response, $handler, $request ) {
		if ( is_array( $response->data ) ) {
			foreach ( $response->data as $key => $response_data ) {
				if ( is_array( $response_data ) ) {
					$response->data[ $key ] = $this->bbapp_rest_array( $response_data );
				} else {
					$response->data[ $key ] = $response_data;
				}
			}
		}

		// Re assign response data.
		return rest_ensure_response( $response );
	}

	/**
	 * Array response check the url and return the cdn url response.
	 *
	 * @param \WP_REST_Response|array $response_data Rest response.
	 *
	 * @return \WP_REST_Response
	 */
	public function bbapp_rest_array( $response_data ) {
		if ( ! is_array( $response_data ) && ! is_object( $response_data ) ) {
			return $response_data;
		}
		if ( is_object( $response_data ) ) {
			return $this->bbapp_rest_object( $response_data );
		}
		foreach ( $response_data as $key => $response ) {
			if ( '_links' === (string) $key ) {
				$response_data[ $key ] = $response;
				continue;
			}
			if ( is_array( $response ) ) {
				$response_data[ $key ] = $this->bbapp_rest_array( $response );
			}
			if ( is_object( $response ) ) {
				$response_data[ $key ] = $this->bbapp_rest_object( $response );
			}
			if ( ! is_array( $response ) && ! is_object( $response ) ) {
				if ( preg_match( $this->bbapp_url_regex(), $response ) ) {
					$response_data[ $key ] = $this->rewrite( $response );
				}
			}
		}

		return $response_data;
	}

	/**
	 * Object response check the url and return the cdn url response.
	 *
	 * @param \WP_REST_Response $response_object_data Rest response.
	 *
	 * @return mixed
	 */
	public function bbapp_rest_object( $response_object_data ) {
		foreach ( $response_object_data as $key => $response ) {
			if ( is_array( $response ) ) {
				$response_object_data->{$key} = $this->bbapp_rest_array( $response );
			}
			if ( is_object( $response ) ) {
				$response_object_data->{$key} = $this->bbapp_rest_object( $response );
			}
			if ( ! is_array( $response ) && ! is_object( $response ) ) {
				if ( preg_match( $this->bbapp_url_regex(), $response ) ) {
					$response_object_data->{$key} = $this->rewrite( $response );
				}
			}
		}

		return $response_object_data;
	}

	/**
	 * Exclude the asset.
	 *
	 * @param string $asset current asset.
	 *
	 * @return bool
	 */
	protected function exclude_asset( &$asset ) {
		// Excludes.
		foreach ( $this->excludes as $exclude ) {
			if ( stristr( $asset, $exclude ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Relative url.
	 *
	 * @param string $url a full url.
	 *
	 * @since   1.0.5
	 * @change  1.0.5
	 * @return  string  protocol relative url
	 */
	protected function relative_url( $url ) {
		return substr( $url, strpos( $url, '//' ) );
	}

	/**
	 * Rewrite url.
	 *
	 * @param string $asset current asset.
	 *
	 * @since   0.0.1
	 * @change  1.0.7
	 * @return  string  updated url if not excluded
	 */
	protected function rewrite_url( &$asset ) {
		if ( $this->exclude_asset( $asset[0] ) ) {
			return $asset[0];
		}

		$blog_url   = $this->relative_url( $this->blog_url );
		$subst_urls = array( 'http:' . $blog_url );

		// Rewrite both http and https URLs if we ticked 'enable CDN for HTTPS connections'.
		if ( $this->https ) {
			$subst_urls = array(
				'http:' . $blog_url,
				'https:' . $blog_url,
			);
		}

		// is it a protocol independent URL?
		if ( strpos( $asset[0], '//' ) === 0 ) {
			return str_replace( $blog_url, $this->cdn_url, $asset[0] );
		}

		// Check if not a relative path.
		if ( ! $this->relative || strstr( $asset[0], $blog_url ) ) {
			return str_replace( $subst_urls, $this->cdn_url, $asset[0] );
		}

		// Relative URL.
		return $this->cdn_url . $asset[0];
	}

	/**
	 * Get directory scope.
	 *
	 * @since   0.0.1
	 * @change  0.0.1
	 * @return  string  directory scope
	 */
	protected function get_dir_scope() {
		$input = explode( ',', $this->dirs );

		// Default.
		if ( '' === $this->dirs || count( $input ) < 1 ) {
			return 'wp\-content/upload';
		}

		return implode( '|', array_map( 'quotemeta', array_map( 'trim', $input ) ) );
	}

	/**
	 * Rewrite the Json.
	 *
	 * @param array $json Json Array.
	 *
	 * @return null|string|string[]
	 */
	public function rewrite( $json ) {
		// Get dir scope in regex format.
		$regex_rule = $this->bbapp_url_regex();

		// Call the cdn rewriter callback.
		return preg_replace_callback( $regex_rule, array( $this, 'rewrite_url' ), $json );
	}

	/**
	 * Regex match for URL.
	 *
	 * @return string
	 */
	public function bbapp_url_regex() {
		$dirs     = $this->get_dir_scope();
		$blog_url = $this->https
			? '(https?:|)' . $this->relative_url( quotemeta( $this->blog_url ) )
			: '(http:|)' . $this->relative_url( quotemeta( $this->blog_url ) );

		// Regex rule start.
		$regex_rule = '#';

		// Check if relative paths.
		if ( $this->relative ) {
			$regex_rule .= '(?:' . $blog_url . ')';
		} else {
			$regex_rule .= $blog_url;
		}

		// Regex rule end.
		$regex_rule .= '/(?:((?:' . $dirs . ')[^\"\')]+)|([^/\"\']+\.[^/\"\')]+))#';

		return $regex_rule;
	}

}
