<?php
/**
 * Holds App build publish functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\ClientCommon;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Admin\Configure;

/**
 * App build publish clas.
 */
class Publish {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Publish
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Publish constructor.
	 */
	public function __construct() {
	}

	/**
	 * Hooks/filter here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
	}

	/**
	 * Init function.
	 */
	public function load_init() {
	}

	/**
	 * Send Publish Request.
	 *
	 * @param string $build_json Build JSON data.
	 * @param array  $files     Files array.
	 *
	 * @return bool|array
	 */
	public function send_publish_request( $build_json, $files ) {
		$app = ManageApp::instance()->get_app();

		if ( empty( $app ) ) {
			return false;
		}

		$api_url = ClientCommon::instance()->get_center_api_url( 'v2', 'api-publish' );

		$post_fields = array(
			'action'    => 'publish-request',
			'data'      => $build_json,
			'bbapp_id'  => $app['bbapp_app_id'],
			'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
		);

		foreach ( $files as $file_key => $file ) {
			$post_fields[ 'file_' . md5( $file_key ) ] = new \CURLFILE( $file );
		}

		// Initialise the cURL var.
		$ch = curl_init();
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
		curl_setopt( $ch, CURLOPT_URL, $api_url );
		$post_data = $post_fields;
		curl_setopt( $ch, CURLOPT_POSTFIELDS, $post_data );
		$response = curl_exec( $ch );
		curl_close( $ch );

		if ( empty( $response ) ) {
			return false;
		}

		$body     = $response;
		$response = json_decode( $body, true );

		return $response;
	}

	/**
	 * Function to update app last version.
	 *
	 * @param boolean $force Boolean when force is true it will forcefully check instead of skipping.
	 *
	 * @return array|bool|mixed|\WP_Error
	 */
	public function update_last_versions( $force = false ) {
		$app = ManageApp::instance()->get_app();

		if ( empty( $app ) ) {
			return array();
		}

		if ( ! $force ) {
			$cache_key = '_bbapp_fetch_last_versions';
			$cached    = get_site_transient( $cache_key );

			if ( false !== $cached ) {
				return $cached;
			}
		}

		$bundle_id_ios     = Configure::instance()->option( 'publish.ios.namespace' );
		$bundle_id_android = Configure::instance()->option( 'publish.android.namespace' );
		$api_url           = ClientCommon::instance()->get_center_api_url( 'v2', 'api-publish' );

		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'           => 'app-versions',
					'android_bundleid' => $bundle_id_android,
					'ios_bundleid'     => $bundle_id_ios,
					'bbapp_id'         => $app['bbapp_app_id'],
					'bbapp_key'        => ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return false;
		}

		$body     = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = (array) json_decode( $body, true );

		if ( ! isset( $response['success'] ) || ! isset( $response['success'] ) ) {
			return false;
		}

		$build_options = $this->get_publish_options();

		// change the cast.
		$response['ios']     = (array) $response['ios'];
		$response['android'] = (array) $response['android'];

		// Update the versions info got from API.
		$build_options['app_version_ios']          = $response['ios']['version'];
		$build_options['app_version_android']      = $response['android']['version'];
		$build_options['app_version_code_ios']     = $response['ios']['version_code'];
		$build_options['app_version_code_android'] = $response['android']['version_code'];

		$this->update_publish_options( $build_options );

		set_site_transient(
			$cache_key,
			$response,
			30 * 60
		); // Cache it for 30min.

		return $do_request;
	}

	/**
	 * Checks for build status for app_id. & update the data accordingly.
	 *
	 * @param string $app_id   App id.
	 * @param string $platform App platform.
	 * @param string $env      App environment.
	 *
	 * @return bool
	 */
	public function cancel_publish( $app_id, $platform, $env ) {
		if ( ! in_array(
			$platform,
			array(
				'android',
				'ios',
			),
			true
		) || ! in_array( $env, array( 'publish' ), true ) ) {
			return false;
		}

		$app    = ManageApp::instance()->get_app();
		$builds = $this->get_publish_options();
		// check for each platform & it's env.
		$api_url      = ClientCommon::instance()->get_center_api_url( 'v2', 'api-publish' );
		$changes_made = false;
		$build        = $builds[ $platform ][ $env ];

		// We only allow pending with new substatus builds to be cancelled.
		if ( ! isset( $build['sub_status'] ) || 'new' !== $build['sub_status'] || ! isset( $build['status'] ) || 'pending' !== $build['status'] ) {
			return false;
		}

		// Do the api call.
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'     => 'publish-cancel',
					'publish_id' => $build['publish_id'],
					'bbapp_id'   => $app['bbapp_app_id'],
					'bbapp_key'  => ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return false;
		}

		$response = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = json_decode( $response, true );

		if ( ! isset( $response['status'] ) || ! isset( $response['sub_status'] ) ) {
			return false;
		}

		$builds[ $platform ][ $env ]['status']     = $response['status'];
		$builds[ $platform ][ $env ]['sub_status'] = $response['sub_status'];
		$builds[ $platform ]['builds'][]           = array(
			'publish_url' => $response['publish_url'],
			'status'      => $response['status'],
			'env'         => $response['env'],
			'time'        => time(),
		);

		$this->update_publish_options( $builds );

		return true;
	}

	/**
	 * Checks for build status for app_id. & update the data accordingly.
	 */
	public function check_publish_status() {
		$publish_options = $this->get_publish_options();

		if ( ! empty( $publish_options ) && empty( $publish_options['publish_requested'] ) ) {
			return;
		}

		$options = isset( $publish_options['publish_requested'] ) ? $publish_options['publish_requested'] : array();
		$api_url = ClientCommon::instance()->get_center_api_url( 'v2', 'api-publish' ); // check for each platform & it's env.

		if ( ! empty( $options ) ) {
			foreach ( $options as $platform => $publish ) {
				if ( empty( $publish['publish_id'] ) ) {
					continue;
				}

				if ( isset( $publish['publish_status'] ) ) {
					$publish_id = isset( $publish['publish_id'] ) ? $publish['publish_id'] : '';

					// Skip, if publish id is empty.
					if ( empty( $publish_id ) ) {
						continue;
					}

					delete_transient( 'get_publish_status_' . $publish_id );
					delete_site_transient( 'get_publish_status_' . $publish_id );
					$publish_status = \BuddyBossApp\Admin\Publish\Publish::instance()->get_publish_status( $publish_id );

					if ( ! empty( $publish_status ) ) {
						$options[ $platform ]['publish_status'] = $publish_status;
					}
				}
			}
		}

		$publish_options['publish_requested'] = $options;

		$this->update_publish_options( $publish_options );
	}

	/**
	 * Return build options
	 *
	 * @return mixed
	 */
	public function get_publish_options() {
		$options = bbapp_get_network_option( 'bbapp_publish' );

		if ( ! is_array( $options ) ) {
			$options = array();
		}

		if ( ! isset( $options ) ) {
			$options = array();
		}

		return $options;
	}

	/**
	 * Update build options.
	 *
	 * @param array $app_options App options.
	 *
	 * @return bool
	 */
	public function update_publish_options( $app_options ) {
		$options = bbapp_get_network_option( 'bbapp_publish' );

		if ( ! is_array( $options ) ) {
			$options = array();
		}

		$options = $app_options;

		bbapp_set_network_option( 'bbapp_publish', $options );

		return true;
	}

	/**
	 * Return the platform builds.
	 *
	 * @param string $app_id   App id.
	 * @param string $platform App platform.
	 *
	 * @return mixed
	 */
	public function get_builds( $app_id, $platform ) {
		if ( ! in_array( $platform, array( 'ios', 'android' ), true ) ) {
			return array();
		}

		$build_options = $this->get_publish_options();

		if ( ! isset( $build_options[ $platform ] ) || ! isset( $build_options[ $platform ]['builds'] ) ) {
			$build_options[ $platform ]['builds'] = array();
		}

		$builds = $build_options[ $platform ]['builds'];
		// format the build items.
		foreach ( $builds as $k => $v ) {
			$builds[ $k ] = $this->build_format( $v );
		}

		// Sort array to show new first.
		krsort( $builds );

		return $builds;
	}

	/**
	 * Publish format.
	 *
	 * @param array $data Build data.
	 *
	 * @return array
	 */
	public function build_format( $data ) {
		$default = array(
			'version'   => false,
			'build_url' => false,
			'status'    => 'failed',
			'env'       => false,
			'time'      => false,
		);

		return array_merge( $default, $data );
	}

	/**
	 * Get app version.
	 *
	 * @param string $platform App platform.
	 *
	 * @return string
	 */
	public function get_app_version( $platform ) {
		$build_options = $this->get_publish_options();
		$app_version   = $build_options[ "app_version_{$platform}" ];

		if ( empty( $app_version ) ) {
			// @todo: get this state value from customer center.
			$app_version = '0.0.0';
		}

		return $app_version;
	}

	/**
	 * Returns the app version code.
	 *
	 * @param string $platform App platform.
	 *
	 * @return string
	 */
	public function get_app_version_code( $platform ) {
		$build_options = $this->get_publish_options();
		$app_version   = $build_options[ "app_version_code_{$platform}" ];

		if ( empty( $app_version ) ) {
			// @todo: get this state value from customer center.
			$app_version = '0';
		}

		return $app_version;
	}

	/**
	 * Gets the next possible version.
	 *
	 * @param string $version App version.
	 *
	 * @return string
	 */
	public function get_next_version( $version ) {
		if ( '0.0.0' === $version ) {
			return '1.0.0';
		}

		$split_version = explode( '.', $version );
		$split_version[2] ++;

		if ( $split_version[2] > 250 ) {
			$split_version[1] ++;
			$split_version[2] = 0;
		}

		return implode( '.', $split_version );
	}

	/**
	 * Return the next possible version code.
	 *
	 * @param string $version_code App version code.
	 *
	 * @return mixed
	 */
	public function get_next_version_code( $version_code ) {
		$version_code ++;

		return $version_code;
	}

	/**
	 * Check if provided version is valid or not.
	 *
	 * @param string $app_version App version code.
	 *
	 * @return bool
	 */
	public function is_valid_app_version( $app_version ) {
		$allowed = array( '.' ); // you can add here more value, you want to allow.

		if ( ctype_digit( str_replace( $allowed, '', $app_version ) ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Return the toggle build min version support link.
	 *
	 * @param string $app_id             App id.
	 * @param string $platform           Platform.
	 * @param string $build_version_code Build version code.
	 *
	 * @return string
	 */
	public function get_build_min_version_code_toggle_link( $app_id, $platform, $build_version_code ) {
		return wp_nonce_url( bbapp_get_admin_url( "admin.php?page=bbapp-build&setting=all-builds&app_id={$app_id}&platform={$platform}&build_version_code={$build_version_code}&action=bbapp_min_version_build_toggle" ), 'bbapp_min_version_build_toggle_nonce', 'bbapp_min_version_build_toggle_nonce' );
	}

	/**
	 * Get publish term condition.
	 *
	 * @param bool $override_cache Overwrite cache.
	 *
	 * @return false
	 */
	public function get_publish_term_condition( $override_cache = true ) {
		$cache_key = 'bbapp_publish_term_condition';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			return $cached;
		} else {
			$api_url          = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-publish-terms-of-service' );
			$request          = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 45,
				)
			);
			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code && ! empty( $response ) && ! empty( $response['success'] ) ) {
				$data = ( isset( $response['terms_of_service'] ) ? $response['terms_of_service'] : '' );
				set_site_transient( $cache_key, $data, HOUR_IN_SECONDS );

				return $data;
			}
		}

		return false;
	}
}
