<?php
/**
 * BuddyBoss APP Courses Topics integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;
use LDLMS_Factory_Post;
use LearnDash_Settings_Section;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash Courses Topics Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class LD_Topics extends Integration_Abstract {


	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'sfwd-topic' );

		$purge_events = array(
			'save_post_sfwd-topic', // Called when lesson is created.
			'edit_post_sfwd-topic', // Called when lesson updated.
			'trashed_post', // Called when lesson trashed.
			'untrashed_post', // Called when lesson untrashed.
			'deleted_post', // Called when lesson deleted.
			'learndash_quiz_completed', // Called when quiz completed.
			'learndash_delete_user_data', // Called when user data removed.
			'learndash_assignment_uploaded', // When assignment Updated.

			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.
		);

		$this->purge_event( 'sfwd-topic', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		$purge_single_events = array(
			'save_post_sfwd-topic'                 => 1, // Called when lesson is created.
			'edit_post_sfwd-topic'                 => 1, // Called when lesson updated.
			'trashed_post'                         => 1, // Called when lesson trashed.
			'untrashed_post'                       => 1, // Called when lesson untrashed.
			'deleted_post'                         => 1, // Called when lesson deleted.
			'learndash_topic_completed'            => 1, // Called when lesson completed.
			'learndash_quiz_completed'             => 2, // Called when quiz completed.
			'learndash_delete_user_data'           => 1, // Called when user data removed.
			'learndash_assignment_uploaded'        => 2, // When assignment Updated.
			'learndash_mark_incomplete_process'    => 3, // Called when lesson/topic/quiz incompleted.

			// When topic's meta updated.
			'update_post_meta'                     => 4, // When `lesson_id` meta updated.

			// when course access expired need to update lesson has course access.
			'learndash_user_course_access_expired' => 2, // Called when course access expired.
			'learndash_update_course_access'       => 2, // Called when course access updated.

			// When Quiz lesson updated Quiz list & embed will be updated too.
			'save_post_sfwd-quiz'                  => 1, // Called when lesson is created.
			'edit_post_sfwd-quiz'                  => 1, // Called when lesson updated.

			// Added moderation support.
			'bp_suspend_user_suspended'            => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'          => 1, // Any User Unsuspended.
			'bp_moderation_after_save'             => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'           => 1, // Unhide activity when member unblocked.

			// Author embed.
			'profile_update'                       => 1, // User updated on site.
			'deleted_user'                         => 1, // User deleted on site.
			'xprofile_avatar_uploaded'             => 1, // User avatar photo updated.
			'bp_core_delete_existing_avatar'       => 1, // User avatar photo deleted.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active     = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
		$cache_support_learndash = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_learndash ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss-app/learndash/v1/topics/:id.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/topics',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-2: wp-json/buddyboss-app/learndash/v1/topics/<id>.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/topics/<id>',
				$cache_expiry_time,
				array(),
				false
			);
		}
	}

	/****************************** Topics Events *****************************/
	/**
	 * Called when topic is created
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_save_post_sfwd_topic( $topic_id ) {
		$this->purge_item_cache_by_item_id( $topic_id );
	}

	/**
	 * Called when topic updated
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_edit_post_sfwd_topic( $topic_id ) {
		$this->purge_item_cache_by_item_id( $topic_id );
	}

	/**
	 * Called when Quiz is created
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_save_post_sfwd_quiz( $quiz_id ) {
		$topic_ids = $this->get_topic_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * Called when quiz updated
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_edit_post_sfwd_quiz( $quiz_id ) {
		$topic_ids = $this->get_topic_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * Called when topic trashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_trashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-topic' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$topic_ids = $this->get_topic_ids_by_quiz_id( $object_id );

			if ( ! empty( $topic_ids ) ) {
				$this->purge_item_cache_by_item_ids( $topic_ids );
			}
		}
	}

	/**
	 * Called when topic untrashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_untrashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-topic' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$topic_ids = $this->get_topic_ids_by_quiz_id( $object_id );

			if ( ! empty( $topic_ids ) ) {
				$this->purge_item_cache_by_item_ids( $topic_ids );
			}
		}
	}

	/**
	 * Called when topic deleted
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_deleted_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-topic' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$topic_ids = $this->get_topic_ids_by_quiz_id( $object_id );

			if ( ! empty( $topic_ids ) ) {
				$this->purge_item_cache_by_item_ids( $topic_ids );
			}
		}
	}

	/**
	 * Called when topic completed
	 *
	 * @param array $topic_arr Topic Post.
	 */
	public function event_learndash_topic_completed( $topic_arr ) {
		$topic_id = $topic_arr['topic']->ID;
		$this->purge_item_cache_by_item_id( $topic_id );
	}

	/**
	 * Called when quiz completed
	 *
	 * @param array   $quizdata An array of quiz data.
	 * @param \WP_User $user     WP_User object.
	 */
	public function event_learndash_quiz_completed( $quizdata, $user ) {
		$topic_id = ! empty( $quizdata['topic'] ) ? $quizdata['topic']->ID : false;
		if ( $topic_id ) {
			$this->purge_item_cache_by_item_id( $topic_id );
		}
	}

	/**
	 * Called when user data removed
	 *
	 * @param int $user_id User ID.
	 */
	public function event_learndash_delete_user_data( $user_id ) {
		Cache::instance()->purge_by_component( 'sfwd-topic' );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking' );
	}

	/**
	 * When assignment Updated
	 *
	 * @param int   $assignment_post_id Newly created assignment post ID which the assignment is uploaded to.
	 * @param array $assignment_meta    Assignment meta data.
	 */
	public function event_learndash_assignment_uploaded( $assignment_post_id, $assignment_meta ) {
		$topic_id  = ! empty( $assignment_meta['lesson_id'] ) ? $assignment_meta['lesson_id'] : false;
		$post_type = get_post_type( $topic_id );
		if ( 'sfwd-topic' === $post_type ) {
			$this->purge_item_cache_by_item_id( $topic_id );
		}
	}

	/**
	 * When `lesson_id` meta updated
	 * For LearnDash Quiz, Topic id store as `Lesson_id`
	 *
	 * @param null|bool $meta_id     Whether to allow updating metadata for the given type.
	 * @param int       $object_id   ID of the object metadata is for.
	 * @param string    $meta_key    Metadata key.
	 * @param mixed     $_meta_value Metadata value. Must be serializable if non-scalar.
	 */
	public function event_update_post_meta( $meta_id, $object_id, $meta_key, $_meta_value ) {

		$post_type = get_post_type( $object_id );
		if ( 'lesson_id' === $meta_key && in_array( $post_type, array( 'sfwd-quiz' ), true ) ) {
			$old_topic = get_post_meta( $object_id, $meta_key, true );
			$this->purge_item_cache_by_item_id( $old_topic ); // Old topic id.
			$this->purge_item_cache_by_item_id( $_meta_value ); // New topic id.
		}
	}

	/**
	 * Called when course access expired
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_user_course_access_expired( $user_id, $course_id ) {
		$lessons = learndash_get_lesson_list( $course_id, array( 'num' => - 1 ) );

		if ( ! empty( $lessons ) ) {
			foreach ( $lessons as $lesson ) {
				$topics = $this->bbapp_ld_get_topics( $lesson->ID, $course_id );

				if ( ! empty( $topics ) ) {
					$this->purge_item_cache_by_item_ids( wp_list_pluck( $topics, 'ID' ) );
				}
			}
		}
	}

	/**
	 * Called when course access updated
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_update_course_access( $user_id, $course_id ) {
		$lessons = learndash_get_lesson_list( $course_id, array( 'num' => - 1 ) );

		if ( ! empty( $lessons ) ) {
			foreach ( $lessons as $lesson ) {
				$topics = $this->bbapp_ld_get_topics( $lesson->ID, $course_id );

				if ( ! empty( $topics ) ) {
					$this->purge_item_cache_by_item_ids( wp_list_pluck( $topics, 'ID' ) );
				}
			}
		}
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$topic_ids = $this->get_topic_ids_by_userid( $user_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$topic_ids = $this->get_topic_ids_by_userid( $user_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * Update cache for ld topic when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$topic_ids = $this->get_topic_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * Update cache for ld topic when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$topic_ids = $this->get_topic_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/****************************** Author Embed Support *****************************/
	/**
	 * User updated on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_profile_update( $user_id ) {
		$topic_ids = $this->get_topic_ids_by_userid( $user_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * User deleted on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_deleted_user( $user_id ) {
		$topic_ids = $this->get_topic_ids_by_userid( $user_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * User avatar photo updated
	 *
	 * @param int $user_id User ID.
	 */
	public function event_xprofile_avatar_uploaded( $user_id ) {
		$topic_ids = $this->get_topic_ids_by_userid( $user_id );

		if ( ! empty( $topic_ids ) ) {
			$this->purge_item_cache_by_item_ids( $topic_ids );
		}
	}

	/**
	 * User avatar photo deleted
	 *
	 * @param array $args Array of arguments used for avatar deletion.
	 */
	public function event_bp_core_delete_existing_avatar( $args ) {
		$user_id = ! empty( $args['item_id'] ) ? absint( $args['item_id'] ) : 0;

		if ( ! empty( $user_id ) ) {
			if ( isset( $args['object'] ) && 'user' === $args['object'] ) {
				$topic_ids = $this->get_topic_ids_by_userid( $user_id );

				if ( ! empty( $topic_ids ) ) {
					$this->purge_item_cache_by_item_ids( $topic_ids );
				}
			}
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get topic ids from user name.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return array
	 */
	private function get_topic_ids_by_userid( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) && function_exists( 'learndash_get_course_lessons_list' ) ) {
			$topic_ids  = array();
			$course_ids = learndash_get_user_course_access_list( $user_id );

			if ( ! empty( $course_ids ) ) {
				foreach ( $course_ids as $course_id ) {
					$course_lesson_ids = learndash_get_course_lessons_list( $course_id, $user_id );

					if ( ! empty( $course_lesson_ids ) ) {
						foreach ( array_column( $course_lesson_ids, 'id' ) as $course_lesson_id ) {
							$lesson_topic_ids = learndash_get_topic_list( $course_lesson_id, $course_id );
							$topic_ids        = array_merge( $topic_ids, array_column( $lesson_topic_ids, 'ID' ) );
						}
					}
				}
			}

			return $topic_ids;
		} else {
			global $wpdb;

			$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='sfwd-topic' AND post_author = %d", $user_id );

			return $wpdb->get_col( $sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}
	}

	/**
	 * Get topic ids from quiz id.
	 *
	 * @param int $quiz_id Quiz ID.
	 *
	 * @return array
	 */
	private function get_topic_ids_by_quiz_id( $quiz_id ) {
		global $wpdb;

		$topic_ids = array();
		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$quiz_id};%'";
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$object_ids = $wpdb->get_col( $sql );
			if ( ! empty( $object_ids ) ) {
				foreach ( $object_ids as $object_id ) {
					$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $object_id );
					$parent                 = $ld_course_steps_object->get_parent_step_id( $quiz_id, 'sfwd-topic' );
					if ( ! empty( $parent ) ) {
						$topic_ids[] = $parent;
					}
				}
			}
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'lesson_id'", $quiz_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$topic_ids = $wpdb->get_col( $sql );
		}

		return $topic_ids;
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param $topic_id
	 */
	private function purge_item_cache_by_item_id( $topic_id ) {
		Cache::instance()->purge_by_group( 'sfwd-topic_' . $topic_id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $topic_id ) );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $topic_ids Topic IDs.
	 *
	 * @since 2.0.70
	 * @return void
	 */
	private function purge_item_cache_by_item_ids( $topic_ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'sfwd-topic_' ), array( $this, 'prepare_topic_deeplink' ) );
	}

	/**
	 * Prepare activity deeplink.
	 *
	 * @param int $topic_id Topic ID.
	 *
	 * @return string
	 */
	public function prepare_topic_deeplink( $topic_id ) {
		return 'bbapp-deeplinking_' . get_permalink( $topic_id );
	}

	/**
	 * Get topic based on lesson and course id.
	 * Todo: Memberium autologin feature creating issue with learndash_get_topic_list function.
	 *
	 * @param     $lesson_id
	 * @param int       $course_id
	 *
	 * @return array|int[]|\WP_Post[]
	 */
	private function bbapp_ld_get_topics( $lesson_id, $course_id = 0 ) {
		$args   = array(
			'numberposts' => - 1,
			'post_type'   => learndash_get_post_type_slug( 'topic' ),
			'meta_query'  => array(
				'relation' => 'AND',
				array(
					'key'   => 'lesson_id',
					'value' => $lesson_id,
				),
				array(
					'key'   => 'course_id',
					'value' => $course_id,
				),
			),
		);
		$topics = get_posts( $args );
		if ( ! empty( $topics ) && ! is_wp_error( $topics ) ) {
			return $topics;
		}

		return array();
	}

	/**
	 * Called when topic incompleted.
	 *
	 * @param int $user_id User ID.
	 * @param int $course_id Course ID.
	 * @param int $step_id Topic ID.
	 * 
	 * @since 2.2.10
	 */
	public function event_learndash_mark_incomplete_process( $user_id, $course_id, $step_id ) {
		$step_post_type = get_post_type( $step_id );
		if ( 'sfwd-topic' === $step_post_type ) {
			$this->purge_item_cache_by_item_id( $step_id );
		}
	}
}
