<?php
/**
 * BuddyBoss APP Courses Quizzes integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash Courses Quizzes Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class LD_Quizzes extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'sfwd-quiz' );

		$purge_events = array(
			'save_post_sfwd-quiz', // Called when course is created.
			'edit_post_sfwd-quiz', // Called when course updated.
			'trashed_post', // Called when course trashed.
			'untrashed_post', // Called when course untrashed.
			'deleted_post', // Called when course deleted.
			'learndash_quiz_completed', // Called when quiz completed.
			'learndash_delete_user_data', // Called when user data removed.
			'learndash_essay_all_quiz_data_updated',
			'learndash_essay_quiz_data_updated',
			'learndash_quiz_resume_metadata_updated', // Called when quiz resume data updated.

			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.
		);

		$this->purge_event( 'sfwd-quiz', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		$purge_single_events = array(
			'save_post_sfwd-quiz'                    => 1, // Called when lesson is created.
			'edit_post_sfwd-quiz'                    => 1, // Called when lesson updated.
			'trashed_post'                           => 1, // Called when lesson trashed.
			'untrashed_post'                         => 1, // Called when lesson untrashed.
			'deleted_post'                           => 1, // Called when lesson deleted.
			'learndash_quiz_completed'               => 2, // Called when quiz completed.
			'learndash_delete_user_data'             => 1, // Called when user data removed.
			'learndash_essay_all_quiz_data_updated'  => 1, // When essay Updated.
			'learndash_essay_quiz_data_updated'      => 1, // When quiz data Updated.
			'learndash_quiz_resume_metadata_updated' => 5, // Called when quiz resume data updated.
			'learndash_mark_incomplete_process'      => 3, // Called when lesson/topic/quiz incompleted.

			// when course access expired need to update lesson has course access.
			'learndash_user_course_access_expired'   => 2, // Called when course access expired.
			'learndash_update_course_access'         => 2, // Called when course access updated.

			// Added moderation support.
			'bp_suspend_user_suspended'              => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'            => 1, // Any User Unsuspended.
			'bp_moderation_after_save'               => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'             => 1, // Unhide activity when member unblocked.

			// Author embed.
			'profile_update'                         => 1, // User updated on site.
			'deleted_user'                           => 1, // User deleted on site.
			'xprofile_avatar_uploaded'               => 1, // User avatar photo updated.
			'bp_core_delete_existing_avatar'         => 1, // User avatar photo deleted.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active     = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
		$cache_support_learndash = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_learndash ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss-app/learndash/v1/quiz.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/quiz',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-2: wp-json/buddyboss-app/learndash/v1/quiz/<id>.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/quiz/<id>',
				$cache_expiry_time,
				array(),
				false
			);
		}
	}

	/****************************** Quizzes Events *****************************/
	/**
	 * Called when Quiz is created
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_save_post_sfwd_quiz( $quiz_id ) {
		$this->purge_item_cache_by_item_id( $quiz_id );
	}

	/**
	 * Called when Quiz updated
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_edit_post_sfwd_quiz( $quiz_id ) {
		$this->purge_item_cache_by_item_id( $quiz_id );
	}

	/**
	 * Called when Quiz trashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_trashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );
		if ( 'sfwd-quiz' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when Quiz untrashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_untrashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );
		if ( 'sfwd-quiz' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when Quiz deleted
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_deleted_post( $object_id ) {
		$post_type = get_post_type( $object_id );
		if ( 'sfwd-quiz' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when quiz completed
	 *
	 * @param array   $quizdata An array of quiz data.
	 * @param \WP_User $user     WP_User object.
	 */
	public function event_learndash_quiz_completed( $quizdata, $user ) {
		$quiz_id = isset( $quizdata['quiz'] ) && isset( $quizdata['quiz']->ID ) ? $quizdata['quiz']->ID : false;
		if ( $quiz_id ) {
			$this->purge_item_cache_by_item_id( $quiz_id );
		}
	}

	/**
	 * Called when user data removed
	 *
	 * @param int $user_id User ID.
	 */
	public function event_learndash_delete_user_data( $user_id ) {
		Cache::instance()->purge_by_component( 'sfwd-quiz' );
		Cache::instance()->purge_by_component( 'bbapp-deeplinking' );
	}

	/**
	 * Called when user data removed
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_learndash_essay_all_quiz_data_updated( $quiz_id ) {
		$this->purge_item_cache_by_item_id( $quiz_id );
	}

	/**
	 * Called when user data removed
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_learndash_essay_quiz_data_updated( $quiz_id ) {
		$this->purge_item_cache_by_item_id( $quiz_id );
	}

	/**
	 * Fires when the quiz resume metadata is updated.
	 *
	 * @param bool                                         $changes_made  A flag indicating if changes were made.
	 * @param array{ quiz_started: int, results: mixed[] } $activity_data Activity data, including quiz started
	 *                                                                    timestamp and results.
	 * @param int                                          $quiz_id       Quiz ID.
	 * @param int                                          $course_id     Course ID.
	 * @param int                                          $user_id       User ID.
	 *
	 * @since 1.8.40
	 *
	 * @return void
	 */
	public function event_learndash_quiz_resume_metadata_updated( $changes_made, $activity_data, $quiz_id, $course_id, $user_id ) {
		if ( $changes_made ) {
			$this->purge_item_cache_by_item_id( $quiz_id );
		}
	}

	/**
	 * Called when course access expired
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_user_course_access_expired( $user_id, $course_id ) {
		$quiz_ids = $this->get_quiz_ids_by_course_id( $course_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * Called when course access updated
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_update_course_access( $user_id, $course_id ) {
		$quiz_ids = $this->get_quiz_ids_by_course_id( $course_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * Update cache for quiz when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$quiz_ids = $this->get_quiz_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * Update cache for quiz when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$quiz_ids = $this->get_quiz_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/****************************** Author Embed Support *****************************/
	/**
	 * User updated on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_profile_update( $user_id ) {
		$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * User deleted on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_deleted_user( $user_id ) {
		$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * User avatar photo updated
	 *
	 * @param int $user_id User ID.
	 */
	public function event_xprofile_avatar_uploaded( $user_id ) {
		$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

		if ( ! empty( $quiz_ids ) ) {
			$this->purge_item_cache_by_item_ids( $quiz_ids );
		}
	}

	/**
	 * User avatar photo deleted
	 *
	 * @param array $args Array of arguments used for avatar deletion.
	 */
	public function event_bp_core_delete_existing_avatar( $args ) {
		$user_id = ! empty( $args['item_id'] ) ? absint( $args['item_id'] ) : 0;

		if ( ! empty( $user_id ) ) {
			if ( isset( $args['object'] ) && 'user' === $args['object'] ) {
				$quiz_ids = $this->get_quiz_ids_by_userid( $user_id );

				if ( ! empty( $quiz_ids ) ) {
					$this->purge_item_cache_by_item_ids( $quiz_ids );
				}
			}
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get lesson ids from user name.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return array
	 */
	private function get_quiz_ids_by_userid( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) && function_exists( 'learndash_get_course_lessons_list' ) ) {
			$quiz_ids   = array();
			$course_ids = learndash_get_user_course_access_list( $user_id );

			if ( ! empty( $course_ids ) ) {
				foreach ( $course_ids as $course_id ) {
					$course_quiz_ids = $this->learndash_get_course_quizzes( $course_id );
					$quiz_ids        = $course_quiz_ids;
				}
			}

			return $quiz_ids;
		} else {
			global $wpdb;

			$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='sfwd-quiz' AND post_author = %d", $user_id );

			return $wpdb->get_col( $sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}
	}

	/**
	 * Get quiz ids from course id.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @since 1.8.20
	 *
	 * @return array|int[]|\WP_Post[]
	 */
	public function learndash_get_course_quizzes( $course_id = 0 ) {
		if ( ! empty( $course_id ) ) {
			$query_args = array(
				'post_type'      => 'sfwd-quiz',
				'posts_per_page' => - 1,
				'orderby'        => 'title',
				'order'          => 'ASC',
				'meta_key'       => 'course_id', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				'meta_value'     => $course_id, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				'meta_compare'   => '=',
				'fields'         => 'ids',
			);

			$query_results = new \WP_Query( $query_args );

			if ( ! empty( $query_results->posts ) ) {
				return $query_results->posts;
			}
		}

		return array();
	}

	/**
	 * Get Quiz ids from course id.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @return array
	 */
	private function get_quiz_ids_by_course_id( $course_id ) {
		global $wpdb;
		$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE meta_value = %d", $course_id );
		$sql .= " AND ( meta_key like 'ld_course_%' OR meta_key = 'course_id' )";

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->get_col( $sql );
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param $quiz_id
	 */
	private function purge_item_cache_by_item_id( $quiz_id ) {
		Cache::instance()->purge_by_group( 'sfwd-quiz_' . $quiz_id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $quiz_id ) );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $quiz_ids Array of ids.
	 *
	 * @since 2.0.70
	 */
	private function purge_item_cache_by_item_ids( $quiz_ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'sfwd-quiz_' ), array( $this, 'prepare_quiz_deeplink' ) );
	}

	/**
	 * Prepare activity deeplink.
	 *
	 * @param int $quiz_id Quiz ID.
	 *
	 * @return string
	 */
	public function prepare_quiz_deeplink( $quiz_id ) {
		return 'bbapp-deeplinking_' . get_permalink( $quiz_id );
	}

	/**
	 * Called when quiz incompleted.
	 *
	 * @param int $user_id User ID.
	 * @param int $course_id Course ID.
	 * @param int $step_id Quiz ID.
	 * 
	 * @since 2.2.10
	 */
	public function event_learndash_mark_incomplete_process( $user_id, $course_id, $step_id ) {
		$step_post_type = get_post_type( $step_id );
		if ( 'sfwd-quiz' === $step_post_type ) {
			$this->purge_item_cache_by_item_id( $step_id );
		}
	}
}
