<?php
/**
 * BuddyBoss APP Courses Lessons integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;
use LDLMS_Factory_Post;
use LearnDash_Settings_Section;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash Courses Lessons Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class LD_Lessons extends Integration_Abstract {


	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'sfwd-lessons' );

		$purge_events = array(
			'save_post_sfwd-lessons', // Called when lesson is created.
			'edit_post_sfwd-lessons', // Called when lesson updated.
			'trashed_post', // Called when lesson trashed.
			'untrashed_post', // Called when lesson untrashed.
			'deleted_post', // Called when lesson deleted.
			'learndash_quiz_completed', // Called when quiz completed.
			'learndash_delete_user_data', // Called when user data removed.
			'learndash_assignment_uploaded', // When assignment Updated.

			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.
		);

		$this->purge_event( 'sfwd-lessons', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		// @todo : Re-verify events for single lesson.
		$purge_single_events = array(
			'save_post_sfwd-lessons'               => 1, // Called when lesson is created.
			'edit_post_sfwd-lessons'               => 1, // Called when lesson updated.
			'trashed_post'                         => 1, // Called when lesson trashed.
			'untrashed_post'                       => 1, // Called when lesson untrashed.
			'deleted_post'                         => 1, // Called when lesson deleted.
			'learndash_lesson_completed'           => 1, // Called when lesson completed.
			'learndash_quiz_completed'             => 2, // Called when quiz completed.
			'learndash_delete_user_data'           => 1, // Called when user data removed.
			'learndash_assignment_uploaded'        => 2, // When assignment Updated.
			'learndash_mark_incomplete_process'    => 3, // Called when lesson/topic/quiz incompleted.

			// When Lesson's meta updated.
			'update_post_meta'                     => 4, // When `lesson_id` meta updated.

			// when course access expired need to update lesson has course access.
			'learndash_user_course_access_expired' => 2, // Called when course access expired.
			'learndash_update_course_access'       => 2, // Called when course access updated.

			// When topic lesson updated topic list will be updated too.
			'save_post_sfwd-topic'                 => 1, // Called when lesson is created.
			'edit_post_sfwd-topic'                 => 1, // Called when lesson updated.

			// When Quiz lesson updated Quiz list & embed will be updated too.
			'save_post_sfwd-quiz'                  => 1, // Called when lesson is created.
			'edit_post_sfwd-quiz'                  => 1, // Called when lesson updated.

			// Added moderation support.
			'bp_suspend_user_suspended'            => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'          => 1, // Any User Unsuspended.
			'bp_moderation_after_save'             => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'           => 1, // Unhide activity when member unblocked.

			// Author embed.
			'profile_update'                       => 1, // User updated on site.
			'deleted_user'                         => 1, // User deleted on site.
			'xprofile_avatar_uploaded'             => 1, // User avatar photo updated.
			'bp_core_delete_existing_avatar'       => 1, // User avatar photo deleted.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active     = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
		$cache_support_learndash = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_learndash ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss-app/learndash/v1/lessons.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/lessons',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-2: wp-json/buddyboss-app/learndash/v1/lessons/<id>.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/lessons/<id>',
				$cache_expiry_time,
				array(),
				false
			);
		}

	}

	/****************************** Lessons Events *****************************/
	/**
	 * Called when lesson is created
	 *
	 * @param int $lesson_id Lesson ID.
	 */
	public function event_save_post_sfwd_lessons( $lesson_id ) {
		$this->purge_item_cache_by_item_id( $lesson_id );
	}

	/**
	 * Called when lesson updated
	 *
	 * @param int $lesson_id Lesson ID.
	 */
	public function event_edit_post_sfwd_lessons( $lesson_id ) {
		$this->purge_item_cache_by_item_id( $lesson_id );
	}

	/**
	 * Called when lesson is created
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_save_post_sfwd_topic( $topic_id ) {
		$lesson_ids = $this->get_lesson_ids_by_topic_id( $topic_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Called when lesson updated
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_edit_post_sfwd_topic( $topic_id ) {
		$lesson_ids = $this->get_lesson_ids_by_topic_id( $topic_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Called when lesson is created
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_save_post_sfwd_quiz( $quiz_id ) {
		$lesson_ids = $this->get_lesson_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Called when lesson updated
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_edit_post_sfwd_quiz( $quiz_id ) {
		$lesson_ids = $this->get_lesson_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Called when lesson trashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_trashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_quiz_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_topic_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		}
	}

	/**
	 * Called when lesson untrashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_untrashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_quiz_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_topic_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		}
	}

	/**
	 * Called when lesson deleted
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_deleted_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_quiz_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$lesson_ids = $this->get_lesson_ids_by_topic_id( $object_id );

			if ( ! empty( $lesson_ids ) ) {
				$this->purge_item_cache_by_item_ids( $lesson_ids );
			}
		}
	}

	/**
	 * Called when lesson completed
	 *
	 * @param array $lesson_arr Lesson Post.
	 */
	public function event_learndash_lesson_completed( $lesson_arr ) {
		$lesson_id = $lesson_arr['lesson']->ID;
		$this->purge_item_cache_by_item_id( $lesson_id );
	}

	/**
	 * Called when quiz completed
	 *
	 * @param array   $quizdata An array of quiz data.
	 * @param \WP_User $user     WP_User object.
	 */
	public function event_learndash_quiz_completed( $quizdata, $user ) {
		$lesson_id = ! empty( $quizdata['lesson'] ) ? $quizdata['lesson']->ID : false;
		if ( $lesson_id ) {
			$this->purge_item_cache_by_item_id( $lesson_id );
		}
	}

	/**
	 * Called when user data removed
	 *
	 * @param int $user_id User ID.
	 */
	public function event_learndash_delete_user_data( $user_id ) {
		Cache::instance()->purge_by_component( 'sfwd-lessons' );
		Cache::instance()->purge_by_component( 'bbapp-deeplinking' );
	}

	/**
	 * When assignment Updated
	 *
	 * @param int   $assignment_post_id Newly created assignment post ID which the assignment is uploaded to.
	 * @param array $assignment_meta    Assignment meta data.
	 */
	public function event_learndash_assignment_uploaded( $assignment_post_id, $assignment_meta ) {
		$lesson_id = ! empty( $assignment_meta['lesson_id'] ) ? $assignment_meta['lesson_id'] : false;
		$post_type = get_post_type( $lesson_id );
		if ( 'sfwd-lessons' === $post_type ) {
			$this->purge_item_cache_by_item_id( $lesson_id );
		}
	}

	/**
	 * When `lesson_id` meta updated
	 *
	 * @param null|bool $meta_id     Whether to allow updating metadata for the given type.
	 * @param int       $object_id   ID of the object metadata is for.
	 * @param string    $meta_key    Metadata key.
	 * @param mixed     $_meta_value Metadata value. Must be serializable if non-scalar.
	 */
	public function event_update_post_meta( $meta_id, $object_id, $meta_key, $_meta_value ) {

		$post_type = get_post_type( $object_id );
		if ( 'lesson_id' === $meta_key && in_array( $post_type, array( 'sfwd-topic', 'sfwd-quiz' ), true ) ) {
			$old_lesson = get_post_meta( $object_id, $meta_key, true );
			$this->purge_item_cache_by_item_id( $old_lesson ); // Old lesson id.
			$this->purge_item_cache_by_item_id( $_meta_value ); // New lesson id.
		}
	}

	/**
	 * Called when course access expired
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_user_course_access_expired( $user_id, $course_id ) {
		$lessons = learndash_get_lesson_list( $course_id, array( 'num' => - 1 ) );

		if ( ! empty( $lessons ) ) {
			$this->purge_item_cache_by_item_ids( wp_list_pluck( $lessons, 'ID' ) );
		}
	}

	/**
	 * Called when course access updated
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_update_course_access( $user_id, $course_id ) {
		$lessons = learndash_get_lesson_list( $course_id, array( 'num' => - 1 ) );

		if ( ! empty( $lessons ) ) {
			$this->purge_item_cache_by_item_ids( wp_list_pluck( $lessons, 'ID' ) );
		}
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Update cache for lesson when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$lesson_ids = $this->get_lesson_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * Update cache for lesson when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$lesson_ids = $this->get_lesson_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/****************************** Author Embed Support *****************************/
	/**
	 * User updated on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_profile_update( $user_id ) {
		$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * User deleted on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_deleted_user( $user_id ) {
		$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * User avatar photo updated
	 *
	 * @param int $user_id User ID.
	 */
	public function event_xprofile_avatar_uploaded( $user_id ) {
		$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

		if ( ! empty( $lesson_ids ) ) {
			$this->purge_item_cache_by_item_ids( $lesson_ids );
		}
	}

	/**
	 * User avatar photo deleted
	 *
	 * @param array $args Array of arguments used for avatar deletion.
	 */
	public function event_bp_core_delete_existing_avatar( $args ) {
		$user_id = ! empty( $args['item_id'] ) ? absint( $args['item_id'] ) : 0;

		if ( ! empty( $user_id ) ) {
			if ( isset( $args['object'] ) && 'user' === $args['object'] ) {
				$lesson_ids = $this->get_lesson_ids_by_userid( $user_id );

				if ( ! empty( $lesson_ids ) ) {
					$this->purge_item_cache_by_item_ids( $lesson_ids );
				}
			}
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get lesson ids from user name.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return array
	 */
	private function get_lesson_ids_by_userid( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) && function_exists( 'learndash_get_course_lessons_list' ) ) {
			$lesson_ids = array();
			$course_ids = learndash_get_user_course_access_list( $user_id );

			if ( ! empty( $course_ids ) ) {
				foreach ( $course_ids as $course_id ) {
					$course_lesson_ids = learndash_get_course_lessons_list( $course_id, $user_id );
					$lesson_ids        = ! empty( $course_lesson_ids ) ? array_column( $course_lesson_ids, 'id' ) : $lesson_ids;
				}
			}

			return $lesson_ids;
		} else {
			global $wpdb;

			$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='sfwd-lessons' AND post_author = %d", $user_id );

			return $wpdb->get_col( $sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}
	}

	/**
	 * Get lesson ids from topic id.
	 *
	 * @param int $topic_id Topic ID.
	 *
	 * @return array
	 */
	private function get_lesson_ids_by_topic_id( $topic_id ) {
		global $wpdb;
		$lesson_ids = array();
		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$topic_id};%'";

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$object_ids = $wpdb->get_col( $sql );
			if ( ! empty( $object_ids ) ) {
				foreach ( $object_ids as $object_id ) {
					$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $object_id );
					$parent                 = $ld_course_steps_object->get_parent_step_id( $topic_id, 'sfwd-lessons' );
					if ( ! empty( $parent ) ) {
						$lesson_ids[] = $parent;
					}
				}
				$lesson_ids = array_unique( $lesson_ids );
			}
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'lesson_id'", $topic_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$lesson_ids = $wpdb->get_col( $sql );
		}

		return $lesson_ids;
	}

	/**
	 * Get lesson ids from quiz id.
	 *
	 * @param int $quiz_id Quiz ID.
	 *
	 * @return array
	 */
	private function get_lesson_ids_by_quiz_id( $quiz_id ) {
		global $wpdb;

		$lesson_ids = array();
		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$quiz_id};%'";
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$object_ids = $wpdb->get_col( $sql );
			if ( ! empty( $object_ids ) ) {
				foreach ( $object_ids as $object_id ) {
					$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $object_id );
					$parent                 = $ld_course_steps_object->get_parent_step_id( $quiz_id, 'sfwd-lessons' );
					if ( ! empty( $parent ) ) {
						$lesson_ids[] = $parent;
					}
				}
			}
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'lesson_id'", $quiz_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$lesson_ids = $wpdb->get_col( $sql );
		}

		return $lesson_ids;
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param $lesson_id
	 */
	private function purge_item_cache_by_item_id( $lesson_id ) {
		Cache::instance()->purge_by_group( 'sfwd-lessons_' . $lesson_id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $lesson_id ) );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $ids Item IDs.
	 *
	 * @since 2.0.70
	 */
	private function purge_item_cache_by_item_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'sfwd-lessons_' ), array( $this, 'prepare_lesson_deeplink' ) );
	}

	/**
	 * Prepare activity deeplink.
	 *
	 * @param int $lesson_id Lesson ID.
	 *
	 * @return string
	 */
	public function prepare_lesson_deeplink( $lesson_id ) {
		return 'bbapp-deeplinking_' . get_permalink( $lesson_id );
	}

	/**
	 * Called when lesson incompleted.
	 *
	 * @param int $user_id User ID.
	 * @param int $course_id Course ID.
	 * @param int $step_id Lesson ID.
	 * 
	 * @since 2.2.10
	 */
	public function event_learndash_mark_incomplete_process( $user_id, $course_id, $step_id ) {
		$step_post_type = get_post_type( $step_id );
		if ( 'sfwd-lessons' === $step_post_type ) {
			$this->purge_item_cache_by_item_id( $step_id );
		}
	}
}
