<?php
/**
 * BuddyBoss APP Courses integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;
use LDLMS_Factory_Post;
use LearnDash_Settings_Section;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash Courses Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class LD_Courses extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'sfwd-courses' );

		$purge_events = array(
			'save_post_sfwd-courses', // Called when course is created.
			'edit_post_sfwd-courses', // Called when course updated.
			'trashed_post', // Called when course trashed.
			'untrashed_post', // Called when course untrashed.
			'deleted_post', // Called when course deleted.

			// Support for "mycourses" sorting.
			'learndash_user_course_access_expired', // Called when course access expired.
			'learndash_update_course_access', // Called when course access updated.

			// Learndash Group Purchase.
			'ld_removed_group_access', // When user removed access for course group.
			'ld_added_group_access',   // When user enrolled access for course group.

			// Support for "my_progress" sorting.
			'learndash_topic_completed', // Called when topic completed.
			'learndash_lesson_completed', // Called when lesson completed.
			'learndash_course_completed', // Called when course completed.
			'learndash_quiz_completed', // Called when quiz completed.

			// Leardash setting update might change course list.
			'sfwd_options_reset', // Called when learndash setting updated.

			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.

		);

		$this->purge_event( 'sfwd-courses', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		$purge_single_events = array(
			'save_post_sfwd-courses'               => 1, // Called when course is created.
			'edit_post_sfwd-courses'               => 1, // Called when course updated.
			'trashed_post'                         => 1, // Called when course trashed.
			'untrashed_post'                       => 1, // Called when course untrashed.
			'deleted_post'                         => 1, // Called when course deleted.
			'learndash_user_course_access_expired' => 2, // Called when course access expired.
			'learndash_update_course_access'       => 2, // Called when course access updated.
			'learndash_topic_completed'            => 1, // Called when topic completed.
			'learndash_lesson_completed'           => 1, // Called when lesson completed.
			'learndash_course_completed'           => 1, // Called when course completed.
			'learndash_quiz_completed'             => 2, // Called when quiz completed.
			'learndash_delete_user_data'           => 1, // When user data removed.
			'learndash_mark_incomplete_process'    => 3, // Called when lesson/topic/quiz incompleted.

			// When Course Builder data updated.
			'update_post_meta'                     => 4, // When `ld_course_steps` OR `course_id` meta updated.

			// When lesson course updated lesson list will be updated too.
			'save_post_sfwd-lessons'               => 1, // Called when lesson is created.
			'edit_post_sfwd-lessons'               => 1, // Called when lesson updated.

			// When Quiz course updated Quiz list & embed will be updated too.
			'save_post_sfwd-quiz'                  => 1, // Called when lesson is created.
			'edit_post_sfwd-quiz'                  => 1, // Called when lesson updated.

			// Added moderation support.
			'bp_suspend_user_suspended'            => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'          => 1, // Any User Unsuspended.
			'bp_moderation_after_save'             => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'           => 1, // Unhide activity when member unblocked.

			// Author embed.
			'profile_update'                       => 1, // User updated on site.
			'deleted_user'                         => 1, // User deleted on site.
			'xprofile_avatar_uploaded'             => 1, // User avatar photo updated.
			'bp_core_delete_existing_avatar'       => 1, // User avatar photo deleted.

			// Learndash Group Purchase.
			'ld_removed_group_access'              => 2, // When user removed access for course group.
			'ld_added_group_access'                => 2, // When user enrolled access for course group.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active     = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
		$cache_support_learndash = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_learndash ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss-app/learndash/v1/courses.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/courses',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-2: wp-json/buddyboss-app/learndash/v1/courses/<id>.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/courses/<id>',
				$cache_expiry_time,
				array(),
				false
			);
		}
	}

	/****************************** Courses Events *****************************/
	/**
	 * Called when course is created
	 *
	 * @param int $course_id Course ID.
	 */
	public function event_save_post_sfwd_courses( $course_id ) {
		$this->purge_item_cache_by_item_id( $course_id );
	}

	/**
	 * Called when course updated
	 *
	 * @param int $course_id Course ID.
	 */
	public function event_edit_post_sfwd_courses( $course_id ) {
		$this->purge_item_cache_by_item_id( $course_id );
	}

	/**
	 * Called when lesson is created
	 *
	 * @param int $lesson_id Lessson ID.
	 */
	public function event_save_post_sfwd_lessons( $lesson_id ) {
		$course_ids = $this->get_course_ids_by_lesson_id( $lesson_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when lesson updated
	 *
	 * @param int $lesson_id Course ID.
	 */
	public function event_edit_post_sfwd_lessons( $lesson_id ) {
		$course_ids = $this->get_course_ids_by_lesson_id( $lesson_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when quiz is created
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_save_post_sfwd_quiz( $quiz_id ) {
		$course_ids = $this->get_course_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when quiz updated
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_edit_post_sfwd_quiz( $quiz_id ) {
		$course_ids = $this->get_course_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when course trashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_trashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when course untrashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_untrashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when course deleted
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_deleted_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_item_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}
	}

	/**
	 * Called when course access expired
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_user_course_access_expired( $user_id, $course_id ) {
		$this->purge_item_cache_by_item_id( $course_id );
	}

	/**
	 * Called when course access updated
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_update_course_access( $user_id, $course_id ) {
		$this->purge_item_cache_by_item_id( $course_id );
	}

	/**
	 * Called when the user is removed from group access meta.
	 *
	 * @param int $user_id  User ID.
	 * @param int $group_id Group ID.
	 */
	public function event_ld_removed_group_access( $user_id, $group_id ) {
		$course_ids = learndash_get_groups_courses_ids( $user_id, array( $group_id ) );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when the user is added to group access meta.
	 *
	 * @param int $user_id  User ID.
	 * @param int $group_id Group ID.
	 */
	public function event_ld_added_group_access( $user_id, $group_id ) {
		$course_ids = learndash_get_groups_courses_ids( $user_id, array( $group_id ) );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Called when topic completed
	 *
	 * @param array $topic_arr Topic Post.
	 */
	public function event_learndash_topic_completed( $topic_arr ) {
		$this->purge_item_cache_by_item_id( $topic_arr['course']->ID );
	}

	/**
	 * Called when lesson completed
	 *
	 * @param array $lesson_arr Lesson Post.
	 */
	public function event_learndash_lesson_completed( $lesson_arr ) {
		$this->purge_item_cache_by_item_id( $lesson_arr['course']->ID );
	}

	/**
	 * Called when course completed
	 *
	 * @param array $course_arr Course Post.
	 */
	public function event_learndash_course_completed( $course_arr ) {
		$this->purge_item_cache_by_item_id( $course_arr['course']->ID );
	}

	/**
	 * Called when quiz completed
	 *
	 * @param array    $quizdata An array of quiz data.
	 * @param \WP_User $user     WP_User object.
	 */
	public function event_learndash_quiz_completed( $quizdata, $user ) {
		$course_id = ! empty( $quizdata['course'] ) ? $quizdata['course']->ID : false;
		if ( $course_id ) {
			$this->purge_item_cache_by_item_id( $course_id );
		}
	}

	/**
	 * When user data removed
	 *
	 * @param int $user_id User ID.
	 */
	public function event_learndash_delete_user_data( $user_id ) {
		Cache::instance()->purge_by_component( 'sfwd-courses' );
		Cache::instance()->purge_by_component( 'bbapp-deeplinking' );
	}

	/**
	 * When `ld_course_steps` OR `course_id` meta updated
	 *
	 * @param null|bool $meta_id     Whether to allow updating metadata for the given type.
	 * @param int       $object_id   ID of the object metadata is for.
	 * @param string    $meta_key    Metadata key.
	 * @param mixed     $_meta_value Metadata value. Must be serializable if non-scalar.
	 */
	public function event_update_post_meta( $meta_id, $object_id, $meta_key, $_meta_value ) {
		$post_type = get_post_type( $object_id );

		if ( 'ld_course_steps' === $meta_key && 'sfwd-courses' === $post_type ) {
			$this->purge_item_cache_by_item_id( $object_id );
		}

		if ( 'course_id' === $meta_key && in_array( $post_type, array( 'sfwd-lessons', 'sfwd-topic', 'sfwd-quiz' ), true ) ) {
			$old_course = get_post_meta( $object_id, $meta_key, true );
			$this->purge_item_cache_by_item_id( $old_course ); // Old course id.
			$this->purge_item_cache_by_item_id( $_meta_value ); // New course id.
		}
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$course_ids = $this->get_course_ids_by_userid( $user_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$course_ids = $this->get_course_ids_by_userid( $user_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Update cache for course when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$course_ids = $this->get_course_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * Update cache for course when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$course_ids = $this->get_course_ids_by_userid( $bp_moderation->item_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/****************************** Author Embed Support *****************************/
	/**
	 * User updated on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_profile_update( $user_id ) {
		$course_ids = $this->get_course_ids_by_userid( $user_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}

		// Clear course, lesson, topics and quiz cache when update courses from profile in the admin side.
		if ( isset( $_POST['user_progress'] ) && ( isset( $_POST['user_progress'][ $user_id ] ) ) && ( ! empty( $_POST['user_progress'][ $user_id ] ) ) ) {
			$this->bbapp_ld_profile_update_clear_cache( $_POST['user_progress'], $user_id );
		}
	}

	/**
	 * Function will clear the cache for course, lesson, topics and quiz when update profile.
	 *
	 * @param $user_progress
	 * @param $user_id
	 */
	public function bbapp_ld_profile_update_clear_cache( $user_progress, $user_id ) {
		$user_progress = (array) json_decode( stripslashes( $user_progress[ $user_id ] ) );
		$user_progress = json_decode( wp_json_encode( $user_progress ), true );
		$cache_groups  = array();

		if ( ( isset( $user_progress['course'] ) ) && ( ! empty( $user_progress['course'] ) ) ) {
			// Course Data.
			foreach ( $user_progress['course']  as $course_id => $course_data ) {
				$cache_groups[] = 'sfwd-courses_' . $course_id;
				$cache_groups[] = 'sfwd-courses-details_' . $course_id;
				// Get lesson ids which checked in the profile section.
				// When we change course status from complete to progress and update profile,
				// at that time we will not get selected lesson ids. So, below code will find those selected ids.
				$lesson_ids = array();

				if ( ( isset( $course_data['lessons'] ) ) && ( ! empty( $course_data['lessons'] ) ) ) {
					$lessons = learndash_course_get_lessons( $course_id, array( 'per_page' => 0 ) );
					$lessons = wp_list_pluck( $lessons, 'ID' );

					if ( ! empty( $lessons ) && ! empty( array_keys( $course_data['lessons'] ) ) ) {
						$lesson_ids = array_diff( $lessons, array_keys( $course_data['lessons'] ) );
					}
				}

				// Clear cache for the lessons and topics.
				if ( ! empty( $lesson_ids ) ) {
					foreach ( $lesson_ids  as $lesson_id ) {
						$cache_groups[] = 'sfwd-lessons_' . $lesson_id;
						$course_topics  = learndash_course_get_topics( $course_id, $lesson_id );
						$course_topics  = wp_list_pluck( $course_topics, 'ID' );

						if ( ! empty( $course_topics ) ) {
							foreach ( $course_topics as $topic_id ) {
								$cache_groups[] = 'sfwd-topic_' . $topic_id;
							}
						}
					}
				}
			}
		}

		// Quiz Data.
		if ( ( isset( $user_progress['quiz'] ) ) && ( ! empty( $user_progress['quiz'] ) ) ) {
			foreach ( $user_progress['quiz']  as $course_id => $quiz_data ) {
				foreach ( array_keys( $quiz_data ) as $quiz_id ) {
					$cache_groups[] = 'sfwd-quiz_' . $quiz_id;
				}
			}
		}

		$this->purge_by_user_groups( $user_id, $cache_groups );
	}

	/**
	 * User deleted on site
	 *
	 * @param int $user_id User ID.
	 */
	public function event_deleted_user( $user_id ) {
		$course_ids = $this->get_course_ids_by_userid( $user_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * User avatar photo updated
	 *
	 * @param int $user_id User ID.
	 */
	public function event_xprofile_avatar_uploaded( $user_id ) {
		$course_ids = $this->get_course_ids_by_userid( $user_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_item_ids( $course_ids );
		}
	}

	/**
	 * User avatar photo deleted
	 *
	 * @param array $args Array of arguments used for avatar deletion.
	 */
	public function event_bp_core_delete_existing_avatar( $args ) {
		$user_id = ! empty( $args['item_id'] ) ? absint( $args['item_id'] ) : 0;

		if ( ! empty( $user_id ) ) {
			if ( isset( $args['object'] ) && 'user' === $args['object'] ) {
				$course_ids = $this->get_course_ids_by_userid( $user_id );

				if ( ! empty( $course_ids ) ) {
					$this->purge_item_cache_by_item_ids( $course_ids );
				}
			}
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get course ids from user name.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_userid( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) ) {
			return learndash_get_user_course_access_list( $user_id );
		} else {
			global $wpdb;

			$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='sfwd-courses' AND post_author = %d", $user_id );

			return $wpdb->get_col( $sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}
	}

	/**
	 * Get course ids from lesson id.
	 *
	 * @param int $lesson_id Lesson ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_lesson_id( $lesson_id ) {
		global $wpdb;

		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$lesson_id};%'";

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'course_id'", $lesson_id );

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		}

		return $course_ids;
	}

	/**
	 * Get course ids from quiz id.
	 *
	 * @param int $quiz_id Quiz ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_quiz_id( $quiz_id ) {
		global $wpdb;

		$course_ids = array();
		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$quiz_id};%'";

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$object_ids = $wpdb->get_col( $sql );
			if ( ! empty( $object_ids ) ) {
				foreach ( $object_ids as $object_id ) {
					$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $object_id );
					$parent                 = $ld_course_steps_object->get_parent_step_id( $quiz_id, 'sfwd-courses' );
					if ( empty( $parent ) ) {
						$course_ids[] = $object_id;
					}
				}
			}
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'course_id'", $quiz_id );

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		}

		return $course_ids;
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param $course_id
	 */
	private function purge_item_cache_by_item_id( $course_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses_' . $course_id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $course_id ) );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $ids Item IDs.
	 *
	 * @since 2.0.70
	 */
	private function purge_item_cache_by_item_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'sfwd-courses_' ), array( $this, 'prepare_course_deeplink' ) );
	}

	/**
	 * Prepare activity deeplink.
	 *
	 * @param int $course_id Course ID.
	 *
	 * @return string
	 */
	public function prepare_course_deeplink( $course_id ) {
		return 'bbapp-deeplinking_' . get_permalink( $course_id );
	}

	/**
	 * Purge cache by user groups.
	 *
	 * @param int   $user_id     User ID.
	 * @param array $group_names Group Names.
	 *
	 * @since 2.0.70
	 *
	 * @return void
	 */
	public function purge_by_user_groups( $user_id, $group_names ) {
		if ( empty( $user_id ) || empty( $group_names ) ) {
			return;
		}

		Cache::instance()->purge_user_groups( (int) $user_id, $group_names );
	}

	/**
	 * Called when lesson/topic/quiz incompleted
	 *
	 * @param int $user_id User ID.
	 * @param int $course_id Course ID.
	 * @param int $step_id Lesson/Topic/Quiz ID.
	 * 
	 * @since 2.2.10
	 */
	public function event_learndash_mark_incomplete_process( $user_id, $course_id, $step_id ) {
		$this->purge_item_cache_by_item_id( $course_id );
	}
}
