<?php
/**
 * BuddyBoss APP Courses Details integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;
use LDLMS_Factory_Post;
use LearnDash_Settings_Section;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash Courses Details Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class LD_Courses_Details extends Integration_Abstract {


	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'sfwd-courses-details' );

		$purge_single_events = array(
			'save_post_sfwd-courses'               => 1, // Called when course is created.
			'edit_post_sfwd-courses'               => 1, // Called when course updated.
			'trashed_post'                         => 1, // Called when course trashed.
			'untrashed_post'                       => 1, // Called when course untrashed.
			'deleted_post'                         => 1, // Called when course deleted.
			'learndash_user_course_access_expired' => 2, // Called when course access expired.
			'learndash_update_course_access'       => 2, // Called when course access updated.
			'learndash_topic_completed'            => 1, // Called when topic completed.
			'learndash_lesson_completed'           => 1, // Called when lesson completed.
			'learndash_course_completed'           => 1, // Called when course completed.
			'learndash_quiz_completed'             => 2, // Called when quiz completed.
			'learndash_delete_user_data'           => 1, // When user data removed.
			'learndash_mark_incomplete_process'    => 3, // Called when lesson/topic/quiz incompleted.

			// When lesson course updated lesson list will be updated too.
			'save_post_sfwd-lessons'               => 1, // Called when lesson is created.
			'edit_post_sfwd-lessons'               => 1, // Called when lesson updated.

			// When lesson topic updated lesson list will be updated too.
			'save_post_sfwd-topic'                 => 1, // Called when lesson is created.
			'edit_post_sfwd-topic'                 => 1, // Called when lesson updated.

			// When Quiz course updated Quiz list & embed will be updated too.
			'save_post_sfwd-quiz'                  => 1, // Called when lesson is created.
			'edit_post_sfwd-quiz'                  => 1, // Called when lesson updated.

			// Added moderation support.
			'bp_suspend_user_suspended'            => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'          => 1, // Any User Unsuspended.
			'bp_moderation_after_save'             => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'           => 1, // Unhide activity when member unblocked.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active     = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
		$cache_support_learndash = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_learndash ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-2: wp-json/buddyboss-app/learndash/v1/courses/<id>/details.
			$this->cache_endpoint(
				'buddyboss-app/learndash/v1/courses/<id>/details',
				$cache_expiry_time,
				array(),
				false
			);
		}
	}

	/****************************** Courses Details Events *****************************/
	/**
	 * Called when course is created
	 *
	 * @param int $course_id Course ID.
	 */
	public function event_save_post_sfwd_courses( $course_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when course updated
	 *
	 * @param int $course_id Course ID.
	 */
	public function event_edit_post_sfwd_courses( $course_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when lesson is created
	 *
	 * @param int $lesson_id Lesson ID.
	 */
	public function event_save_post_sfwd_lessons( $lesson_id ) {
		$course_ids = $this->get_course_ids_by_lesson_id( $lesson_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when lesson updated
	 *
	 * @param int $lesson_id Lesson ID.
	 */
	public function event_edit_post_sfwd_lessons( $lesson_id ) {
		$course_ids = $this->get_course_ids_by_lesson_id( $lesson_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when topic is created
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_save_post_sfwd_topic( $topic_id ) {
		$course_ids = $this->get_course_ids_by_topic_id( $topic_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when topic updated
	 *
	 * @param int $topic_id Topic ID.
	 */
	public function event_edit_post_sfwd_topic( $topic_id ) {
		$course_ids = $this->get_course_ids_by_topic_id( $topic_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when quiz is created
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_save_post_sfwd_quiz( $quiz_id ) {
		$course_ids = $this->get_course_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when quiz updated
	 *
	 * @param int $quiz_id Quiz ID.
	 */
	public function event_edit_post_sfwd_quiz( $quiz_id ) {
		$course_ids = $this->get_course_ids_by_quiz_id( $quiz_id );

		if ( ! empty( $course_ids ) ) {
			$this->purge_item_cache_by_course_ids( $course_ids );
		}
	}

	/**
	 * Called when course trashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_trashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$course_ids = $this->get_course_ids_by_topic_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $object_id );
		}
	}

	/**
	 * Called when course untrashed
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_untrashed_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$course_ids = $this->get_course_ids_by_topic_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $object_id );
		}
	}

	/**
	 * Called when course deleted
	 *
	 * @param int $object_id Post ID.
	 */
	public function event_deleted_post( $object_id ) {
		$post_type = get_post_type( $object_id );

		if ( 'sfwd-lessons' === $post_type ) {
			$course_ids = $this->get_course_ids_by_lesson_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-topic' === $post_type ) {
			$course_ids = $this->get_course_ids_by_topic_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-quiz' === $post_type ) {
			$course_ids = $this->get_course_ids_by_quiz_id( $object_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		} elseif ( 'sfwd-courses' === $post_type ) {
			Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $object_id );
		}
	}

	/**
	 * Called when course access expired
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_user_course_access_expired( $user_id, $course_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when course access updated
	 *
	 * @param int $user_id   User ID.
	 * @param int $course_id Course ID.
	 */
	public function event_learndash_update_course_access( $user_id, $course_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when topic completed
	 *
	 * @param array $topic_arr Topic Post.
	 */
	public function event_learndash_topic_completed( $topic_arr ) {
		$course_id = $topic_arr['course']->ID;
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when lesson completed
	 *
	 * @param array $lesson_arr Lesson Post.
	 */
	public function event_learndash_lesson_completed( $lesson_arr ) {
		$course_id = $lesson_arr['course']->ID;
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when course completed
	 *
	 * @param array $course_arr Course Post.
	 */
	public function event_learndash_course_completed( $course_arr ) {
		$course_id = $course_arr['course']->ID;
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}

	/**
	 * Called when quiz completed
	 *
	 * @param array   $quizdata An array of quiz data.
	 * @param \WP_User $user     WP_User object.
	 */
	public function event_learndash_quiz_completed( $quizdata, $user ) {
		$course_id = ! empty( $quizdata['course'] ) ? $quizdata['course']->ID : false;
		if ( $course_id ) {
			Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
		}
	}

	/**
	 * When user data removed
	 *
	 * @param int $user_id User ID.
	 */
	public function event_learndash_delete_user_data( $user_id ) {
		Cache::instance()->purge_by_component( 'sfwd-courses-details' );
	}

	/******************************* Moderation Support ******************************/

	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) ) {
			$course_ids = learndash_get_user_course_access_list( $user_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		if ( function_exists( 'learndash_get_user_course_access_list' ) ) {
			$course_ids = learndash_get_user_course_access_list( $user_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		}
	}

	/**
	 * Update cache for course when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		if ( function_exists( 'learndash_get_user_course_access_list' ) ) {
			$course_ids = learndash_get_user_course_access_list( $bp_moderation->item_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		}
	}

	/**
	 * Update cache for course when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		if ( function_exists( 'learndash_get_user_course_access_list' ) ) {
			$course_ids = learndash_get_user_course_access_list( $bp_moderation->item_id );

			if ( ! empty( $course_ids ) ) {
				$this->purge_item_cache_by_course_ids( $course_ids );
			}
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get course ids from lesson id.
	 *
	 * @param int $lesson_id Lesson ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_lesson_id( $lesson_id ) {
		global $wpdb;

		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$lesson_id};%'";
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'course_id'", $lesson_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		}

		return $course_ids;
	}

	/**
	 * Get course ids from topic id.
	 *
	 * @param int $topic_id Topic ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_topic_id( $topic_id ) {
		global $wpdb;

		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$topic_id};%'";
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'course_id'", $topic_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		}

		return $course_ids;
	}

	/**
	 * Get course ids from quiz id.
	 *
	 * @param int $quiz_id Quiz ID.
	 *
	 * @return array
	 */
	private function get_course_ids_by_quiz_id( $quiz_id ) {
		global $wpdb;

		$course_ids = array();
		if ( LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Builder', 'shared_steps' ) === 'yes' ) {
			// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
			$sql  = $wpdb->prepare( "SELECT DISTINCT post_id FROM {$wpdb->postmeta} WHERE `meta_key` = 'ld_course_steps'" );
			$sql .= " AND `meta_value` LIKE '%i:{$quiz_id};%'";
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$object_ids = $wpdb->get_col( $sql );
			if ( ! empty( $object_ids ) ) {
				foreach ( $object_ids as $object_id ) {
					$ld_course_steps_object = LDLMS_Factory_Post::course_steps( $object_id );
					$parent                 = $ld_course_steps_object->get_parent_step_id( $quiz_id, 'sfwd-courses' );
					if ( empty( $parent ) ) {
						$course_ids[] = $object_id;
					}
				}
			}
		} else {
			$sql = $wpdb->prepare( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'course_id'", $quiz_id );
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$course_ids = $wpdb->get_col( $sql );
		}

		return $course_ids;
	}

	/**
	 * Purge item cache by course ids.
	 *
	 * @param array $ids Course IDs.
	 *
	 * @since 2.0.70
	 */
	private function purge_item_cache_by_course_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'sfwd-courses-details_' ) );
	}

	/**
	 * Called when lesson/topic/quiz incompleted.
	 *
	 * @param int $user_id User ID.
	 * @param int $course_id Course ID.
	 * @param int $step_id Lesson/Topic/Quiz ID.
	 * 
	 * @since 2.2.10
	 */
	public function event_learndash_mark_incomplete_process( $user_id, $course_id, $step_id ) {
		Cache::instance()->purge_by_group( 'sfwd-courses-details_' . $course_id );
	}
}
