<?php
/**
 * BuddyBoss Bookmark integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * App Blog Posts Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class BB_Bookmarks extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @since 1.7.4
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'bb-bookmarks' );

		$purge_events = array(
			'bb_create_bookmark', // When new bookmark created.
			'bb_delete_bookmark', // When bookmark deleted.
			'bb_bookmarks_after_save', // When bookmark add/update.
			'bb_bookmarks_after_delete_bookmark', // When bookmark deleted.
			'deleted_user', // when user deleted.

			// Added moderation support.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.
		);

		$this->purge_event( 'bb-bookmarks', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		/**
		 * Support for single items purge¬
		 */
		$purge_single_events = array(
			'bb_create_bookmark'                        => 2, // When bookmark created.
			'bb_delete_bookmark'                        => 1, // When bookmark created.
			'bb_bookmarks_after_save'                   => 1, // When bookmark add/update.
			'bb_bookmarks_after_delete_bookmark'        => 2, // When bookmark deleted.
			'bb_bookmarks_after_update_bookmark_status' => 3, // When bookmark status update.

			// Add Author Embed Support.
			'profile_update'                            => 1, // User updated on site.
			'deleted_user'                              => 1, // User deleted on site.
			'xprofile_avatar_uploaded'                  => 1, // User avatar photo updated.

			// Added moderation support.
			'bp_moderation_after_save'                  => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'                => 1, // Unhide activity when member unblocked.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active      = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                 = Helper::instance()->get_app_settings( 'cache_post_bookmarks', 'buddyboss-app' );
		$cache_support_blog_posts = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_blog_posts ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss/v1/bookmarks.
			$this->cache_endpoint(
				'buddyboss/v1/bookmarks',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-2: wp-json/buddyboss/v1/bookmarks/<id>.
			$this->cache_endpoint(
				'buddyboss/v1/bookmarks/<id>',
				$cache_expiry_time,
				array(),
				false
			);

			// Endpoint-3: wp-json/buddyboss/v1/bookmark-types/?<type>.
			$this->cache_endpoint(
				'buddyboss/v1/bookmark-types/<type>',
				$cache_expiry_time,
				array(),
				false
			);
		}

	}

	/******************************** Bookmark Events ********************************/
	/**
	 * When Bookmark created.
	 *
	 * @param array $args        Array of Bookmark data to create.
	 * @param int   $bookmark_id Bookmark ID.
	 *
	 * @since 1.7.4
	 */
	public function event_bb_create_bookmark( $args, $bookmark_id ) {
		$type    = ! empty( $args['type'] ) ? $args['type'] : '';
		$item_id = ! empty( $args['item_id'] ) ? $args['item_id'] : '';

		if ( ! empty( $type ) && ! empty( $item_id ) ) {
			$this->purge_item_cache_by_type( $type, $item_id );
		}

		$this->purge_item_cache_by_item_id( $bookmark_id );
		// Clear bookmark type cache.
		Cache::instance()->purge_by_group( 'bb-bookmarks' );
		Cache::instance()->purge_by_group( 'blog-post' );
	}

	/**
	 * When Bookmark created.
	 *
	 * @param int $bookmark_id Bookmark ID.
	 *
	 * @since 1.7.4
	 */
	public function event_bb_delete_bookmark( $bookmark_id ) {
		$this->purge_item_cache_by_item_id( $bookmark_id );
		// Clear bookmark type cache.
		Cache::instance()->purge_by_group( 'bb-bookmarks' );
		Cache::instance()->purge_by_group( 'blog-post' );
	}

	/**
	 * When Bookmark created/updated.
	 *
	 * @param object $bookmark Bookmark object.
	 *
	 * @since 1.7.4
	 */
	public function event_bb_bookmarks_after_save( $bookmark ) {
		if ( ! empty( $bookmark->id ) ) {
			$this->purge_item_cache_by_item_id( $bookmark->id );
			$this->purge_item_cache_by_type( $bookmark->type, $bookmark->item_id );
		}
	}

	/**
	 * When Bookmark deleted.
	 *
	 * @param object $bookmark    Bookmark object.
	 * @param int    $bookmark_id Bookmark ID.
	 *
	 * @since 1.7.4
	 */
	public function event_bb_bookmarks_after_delete_bookmark( $bookmark, $bookmark_id ) {
		$this->purge_item_cache_by_item_id( $bookmark_id );
		$this->purge_item_cache_by_type( $bookmark->type, $bookmark->item_id );
	}

	/**
	 * Clear cache while updating the status of the bookmarks.
	 *
	 * @param string $type    Type bookmark item.
	 * @param int    $item_id The bookmark item ID.
	 * @param string $status Bookmark status..
	 *
	 * @since 1.7.4
	 */
	public function event_bb_bookmarks_after_update_bookmark_status( $type, $item_id, $status ) {
		if ( empty( $type ) || empty( $item_id ) ) {
			return;
		}

		$bookmark_ids = bb_get_bookmarks(
			array(
				'type'    => $type,
				'item_id' => $item_id,
				'fields'  => 'id',
				'status'  => $status,
			),
			true
		);

		if ( ! empty( $bookmark_ids['bookmarks'] ) ) {
			$this->purge_item_cache_by_item_ids( $bookmark_ids['bookmarks'] );
		}

		$this->purge_item_cache_by_type( $type, $item_id );
		// Clear bookmark type cache.
		Cache::instance()->purge_by_group( 'bb-bookmarks' );
		Cache::instance()->purge_by_group( 'blog-post' );
	}

	/****************************** Author Embed Support *****************************/
	/**
	 * User updated on site
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.7.4
	 */
	public function event_profile_update( $user_id ) {
		$this->purge_item_cache_by_user_id( $user_id );
	}

	/**
	 * User deleted on site
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.7.4
	 */
	public function event_deleted_user( $user_id ) {
		$this->purge_item_cache_by_user_id( $user_id );
	}

	/**
	 * User avatar photo updated
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.7.4
	 */
	public function event_xprofile_avatar_uploaded( $user_id ) {
		$this->purge_item_cache_by_user_id( $user_id );
	}


	/******************************* Moderation Support ******************************/

	/**
	 * Update cache for blog post/bookmark when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$bookmarks = $this->get_bookmark_by_userid( $bp_moderation->item_id );
		$ids       = array();

		if ( ! empty( $bookmarks ) ) {
			foreach ( $bookmarks as $bookmark ) {
				$ids[] = $bookmark->id;

				$this->purge_item_cache_by_type( $bookmark->type, $bookmark->item_id );
			}

			$this->purge_item_cache_by_item_ids( $ids );
		}
	}

	/**
	 * Update cache for blog post/bookmark when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$bookmarks = $this->get_bookmark_by_userid( $bp_moderation->item_id );
		$ids       = array();

		if ( ! empty( $bookmarks ) ) {
			foreach ( $bookmarks as $bookmark ) {
				$ids[] = $bookmark->id;

				$this->purge_item_cache_by_type( $bookmark->type, $bookmark->item_id );
			}

			$this->purge_item_cache_by_item_ids( $ids );
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get Bookmarks ids from user ID.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.7.4
	 *
	 * @return array|mixed
	 */
	private function get_bookmark_by_userid( $user_id ) {
		$all_bookmark = bb_get_bookmarks(
			array(
				'user_id' => $user_id,
				'status'  => null,
			)
		);

		if ( ! empty( $all_bookmark['bookmarks'] ) ) {
			return $all_bookmark['bookmarks'];
		}

		return array();
	}

	/**
	 * Purge item cache by user id.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.7.4
	 */
	private function purge_item_cache_by_user_id( $user_id ) {
		$bookmarks = $this->get_bookmark_by_userid( $user_id );
		$ids       = array();

		if ( ! empty( $bookmarks ) ) {
			foreach ( $bookmarks as $bookmark ) {
				$ids[] = $bookmark->id;

				$this->purge_item_cache_by_type( $bookmark->type, $bookmark->item_id );
			}

			$this->purge_item_cache_by_item_ids( $ids );
		}
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param int $bookmark_id Bookmark ID.
	 *
	 * @since 1.7.4
	 */
	private function purge_item_cache_by_item_id( $bookmark_id ) {
		Cache::instance()->purge_by_group( 'bb-bookmarks_' . $bookmark_id );
	}

	/**
	 * Purge item cache by type.
	 *
	 * @param string $type    Type bookmark item.
	 * @param int    $item_id The bookmark item ID.
	 *
	 * @since 1.8.00
	 */
	private function purge_item_cache_by_type( $type, $item_id ) {
		if ( 'post' === $type ) {
			Cache::instance()->purge_by_group( 'blog-post_' . $item_id );
		}
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $ids Array of bookmark ids.
	 *
	 * @since 2.0.70
	 *
	 * @return void
	 */
	private function purge_item_cache_by_item_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'bb-bookmarks_' ) );
	}
}
