<?php
/**
 * BuddyBoss APP Posts Comments integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * App Posts Comments Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class BB_App_Posts_Comments extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'post_comment' );

		$purge_events = array(
			'wp_insert_comment', // Called when post_comment is saved in backend.
			'edit_comment', // Called when post_comment is updated in backend.
			'deleted_comment', // Called when post_comment is delete in backend.
			'trashed_comment', // Called when post_comment is delete in backend.
			'untrashed_comment', // Called when post_comment is revert-delete in backend.
			'wp_set_comment_status', // Called when post_comment is revert-delete in backend.

			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_suspend_comment_suspended',   // Any Comment Suspended.
			'bp_suspend_comment_unsuspended', // Any Comment Unsuspended.
			'bp_moderation_after_save',     // Hide activity when member blocked.
			'bb_moderation_after_delete',    // Unhide activity when member unblocked.
		);

		$this->purge_event( 'post_comment', $purge_events );

		// @todo : Re-verify events for single post_comment
		$purge_single_events = array(
			'wp_insert_comment'              => 2, // Called when post_comment is saved in backend.
			'edit_comment'                   => 2, // Called when post_comment is updated in backend.
			'deleted_comment'                => 2, // Called when post_comment is delete in backend.
			'trashed_comment'                => 2, // Called when post_comment is delete in backend.
			'untrashed_comment'              => 2, // Called when post_comment is revert-delete in backend.
			'wp_set_comment_status'          => 2, // Called when post_comment is revert-delete in backend.

			// Added moderation support.
			'bp_suspend_user_suspended'      => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'    => 1, // Any User Unsuspended.
			'bp_suspend_comment_suspended'   => 1, // Any Comment Suspended.
			'bp_suspend_comment_unsuspended' => 1, // Any Comment Unsuspended.
			'bp_moderation_after_save'       => 1, // Hide activity when member blocked.
			'bb_moderation_after_delete'     => 1, // Unhide activity when member unblocked.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active   = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings              = Helper::instance()->get_app_settings( 'cache_post_comment', 'buddyboss-app' );
		$cache_support_comment = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_comment ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp/v2/comments.
			$this->cache_endpoint(
				'wp/v2/comments',
				$cache_expiry_time,
				array(
					'unique_id'       => 'id',
					'exclude_context' => 'embed',
				),
				true
			);

			// Endpoint-2: wp/v2/comments/<id>.
			$this->cache_endpoint(
				'wp/v2/comments/<id>',
				$cache_expiry_time,
				array(),
				false
			);
		}
	}

	/**
	 * When new comment created.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 */
	public function event_wp_insert_comment( $comment_id, $comment ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * When comment updated.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 */
	public function event_edit_comment( $comment_id, $comment ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * When comment deleted.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 */
	public function event_deleted_comment( $comment_id, $comment ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * When comment trashed.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 */
	public function event_trashed_comment( $comment_id, $comment ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * When comment un-trashed.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 */
	public function event_untrashed_comment( $comment_id, $comment ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * When comment un-trashed.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment_status    Comment object.
	 */
	public function event_wp_set_comment_status( $comment_id, $comment_status ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$this->clear_user_comments_cache( $user_id );
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$this->clear_user_comments_cache( $user_id );
	}

	/**
	 * Suspended Comment ID.
	 *
	 * @param int $comment_id Comment ID.
	 */
	public function event_bp_suspend_comment_suspended( $comment_id ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $comment_id Comment ID.
	 */
	public function event_bp_suspend_comment_unsuspended( $comment_id ) {
		Cache::instance()->purge_by_group( 'post_comment_' . $comment_id );
	}

	/**
	 * Update cache for comments when member blocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bp_moderation_after_save( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$this->clear_user_comments_cache( $bp_moderation->item_id );
	}

	/**
	 * Update cache for comments when member unblocked.
	 *
	 * @param \BP_Moderation $bp_moderation Current instance of moderation item. Passed by reference.
	 */
	public function event_bb_moderation_after_delete( $bp_moderation ) {
		if ( empty( $bp_moderation->item_id ) || empty( $bp_moderation->item_type ) || 'user' !== $bp_moderation->item_type ) {
			return;
		}

		$this->clear_user_comments_cache( $bp_moderation->item_id );
	}

	/**
	 * Clear cache from user comments.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 1.8.20
	 */
	public function clear_user_comments_cache( $user_id ) {
		$comments = get_comments(
			array(
				'user_id' => $user_id,
				'fields'  => 'ids',
			)
		);

		if ( empty( $comments ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $comments, array( 'post_comment_' ) );
	}
}
