<?php
/**
 * BuddyBoss APP Directory block
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Directory block integration class for cache management.
 *
 * @package BuddyBossApp\Performance
 */
class BB_App_Directory_Blocks extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'directory-block' );

		$purge_events = array(
			'save_post',
			'edit_post',
			'trashed_post',
			'untrashed_post',
			'deleted_post',

			'added_post_meta',
			'update_post_meta',

			'bb_create_bookmark', // When new bookmark created.
			'bb_bookmarks_before_delete_bookmark', // When before bookmark delete.
			'bb_bookmarks_after_save', // When bookmark add/update.
			'bb_bookmarks_after_delete_bookmark', // When bookmark deleted.

			'bp_suspend_user_suspended',
			'bp_suspend_user_unsuspended',
			'bp_moderation_after_save',
			'bb_moderation_after_delete',
		);

		$this->purge_event( 'directory-block', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		$purge_single_events = array(
			'save_post'                                 => 1,
			'edit_post'                                 => 1,
			'trashed_post'                              => 1,
			'untrashed_post'                            => 1,
			'deleted_post'                              => 1,

			'added_post_meta'                           => 2,
			'update_post_meta'                          => 2,

			'bb_create_bookmark'                        => 2, // When bookmark created.
			'bb_bookmarks_before_delete_bookmark'       => 1, // When before bookmark delete.
			'bb_bookmarks_after_save'                   => 1, // When bookmark add/update.
			'bb_bookmarks_after_delete_bookmark'        => 2, // When bookmark deleted.
			'bb_bookmarks_after_update_bookmark_status' => 3, // When bookmark status update.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active            = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                       = Helper::instance()->get_app_settings( 'cache_app_page', 'buddyboss-app' );
		$cache_support_directory_blocks = isset( $is_component_active ) && isset( $settings ) && $is_component_active && $settings;

		if ( $cache_support_directory_blocks ) {
			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists( 'BuddyBoss\Performance\Cache', 'cache_expiry' ) ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			// Endpoint-1: wp-json/buddyboss-app/v1/directory-block.
			$this->cache_endpoint(
				'buddyboss-app/v1/directory-block',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);
		}
	}

	/****************************** Events *****************************/

	/**
	 * Event: save_post
	 *
	 * @param int $post_id ID of the post that was saved.
	 *
	 * @since 2.3.40
	 */
	public function event_save_post( $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: edit_post
	 *
	 * @param int $post_id ID of the post that was edited.
	 *
	 * @since 2.3.40
	 */
	public function event_edit_post( $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: trashed_post
	 *
	 * @param int $post_id ID of the post that was trashed.
	 *
	 * @since 2.3.40
	 */
	public function event_trashed_post( $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: untrashed_post
	 *
	 * @param int $post_id ID of the post that was untrashed.
	 *
	 * @since 2.3.40
	 */
	public function event_untrashed_post( $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: deleted_post
	 *
	 * @param int $post_id ID of the post that was deleted.
	 *
	 * @since 2.3.40
	 */
	public function event_deleted_post( $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: added_post_meta
	 *
	 * @param int $meta_id ID of the meta data entry.
	 * @param int $post_id ID of the post the meta data is for.
	 *
	 * @since 2.3.40
	 */
	public function event_added_post_meta( $meta_id, $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}

	/**
	 * Event: event_update_post_meta
	 *
	 * @param int $meta_id ID of the meta data entry.
	 * @param int $post_id ID of the post the meta data is for.
	 *
	 * @since 2.3.40
	 */
	public function event_update_post_meta( $meta_id, $post_id ) {
		$this->purge_item_cache_by_item_id( $post_id );
	}


	/******************************** Bookmark Events ********************************/
	/**
	 * When Bookmark created.
	 *
	 * @param array $args        Array of Bookmark data to create.
	 * @param int   $bookmark_id Bookmark ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bb_create_bookmark( $args, $bookmark_id ) {
		$item_id = ! empty( $args['item_id'] ) ? $args['item_id'] : '';

		$this->purge_item_cache_by_item_id( $item_id );
	}

	/**
	 * When Bookmark deleted.
	 *
	 * @param int $bookmark_id Bookmark ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bb_bookmarks_before_delete_bookmark( $bookmark_id ) {
		// Get the bookmark object.
		$bookmark = bb_bookmarks_get_bookmark( $bookmark_id );

		if ( ! empty( $bookmark->item_id ) ) {
			$this->purge_item_cache_by_item_id( $bookmark->item_id );
		}
	}

	/**
	 * When Bookmark created/updated.
	 *
	 * @param object $bookmark Bookmark object.
	 *
	 * @since 2.3.40
	 */
	public function event_bb_bookmarks_after_save( $bookmark ) {
		if ( ! empty( $bookmark->item_id ) ) {
			$this->purge_item_cache_by_item_id( $bookmark->item_id );
		}
	}

	/**
	 * When Bookmark deleted.
	 *
	 * @param object $bookmark    Bookmark object.
	 * @param int    $bookmark_id Bookmark ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bb_bookmarks_after_delete_bookmark( $bookmark, $bookmark_id ) {
		$this->purge_item_cache_by_item_id( $bookmark->item_id );
	}

	/**
	 * Clear cache while updating the status of the bookmarks.
	 *
	 * @param string $type    Type bookmark item.
	 * @param int    $item_id The bookmark item ID.
	 * @param string $status  Bookmark status..
	 *
	 * @since 2.3.40
	 */
	public function event_bb_bookmarks_after_update_bookmark_status( $type, $item_id, $status ) {
		if ( empty( $type ) || empty( $item_id ) ) {
			return;
		}

		$this->purge_item_cache_by_item_id( $item_id );
	}

	/******************************* Moderation Support ******************************/

	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$post_ids = $this->get_post_ids_by_user_id( $user_id );

		$this->purge_item_cache_by_item_ids( $post_ids );
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$post_ids = $this->get_post_ids_by_user_id( $user_id );

		$this->purge_item_cache_by_item_ids( $post_ids );
	}

	/**
	 * Suspended Comment ID.
	 *
	 * @param int $comment_id Comment ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bp_suspend_comment_suspended( $comment_id ) {
		$comment = get_comment( $comment_id );

		if ( ! empty( $comment->comment_post_ID ) ) {
			$this->purge_item_cache_by_item_id( $comment->comment_post_ID );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $comment_id Comment ID.
	 *
	 * @since 2.3.40
	 */
	public function event_bp_suspend_comment_unsuspended( $comment_id ) {
		$comment = get_comment( $comment_id );

		if ( ! empty( $comment->comment_post_ID ) ) {
			$this->purge_item_cache_by_item_id( $comment->comment_post_ID );
		}
	}


	/****************************** Author Embed Support *****************************/

	/*********************************** Functions ***********************************/

	/**
	 * Get posts ids from user ID.
	 *
	 * @param int $user_id User ID.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	private function get_post_ids_by_user_id( $user_id ) {
		global $wpdb;

		$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_author = %d", $user_id );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->get_col( $sql );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $ids Array of ids.
	 *
	 * @since 2.3.40
	 *
	 * @return void
	 */
	private function purge_item_cache_by_item_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'directory-block_' ) );
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param int $item_id Item ID.
	 *
	 * @since 2.3.40
	 */
	private function purge_item_cache_by_item_id( $item_id ) {
		Cache::instance()->purge_by_group( 'directory-block_' . $item_id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $item_id ) );
	}
}
