<?php
/**
 * BuddyBoss APP Blog Posts integration.
 *
 * @package BuddyBossApp\Performance
 */

namespace BuddyBossApp\Performance\Integration;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * App Blog Posts Integration Class.
 *
 * @package BuddyBossApp\Performance
 */
class BB_App_Blog_Posts extends Integration_Abstract {

	/**
	 * Add(Start) Integration
	 *
	 * @return mixed|void
	 */
	public function set_up() {
		$this->register( 'blog-post' );

		$purge_events = array(
			'save_post_post', // Called when blog-post is saved in backend.
			'edit_post_post', // Called when blog-post is updated in backend.
			// Added moderation support.
			'bp_suspend_user_suspended',   // Any User Suspended.
			'bp_suspend_user_unsuspended', // Any User Unsuspended.
			'bp_suspend_comment_suspended',   // Any Comment Suspended.
			'bp_suspend_comment_unsuspended', // Any Comment Unsuspended.
		);

		$this->purge_event( 'blog-post', $purge_events );
		$this->purge_event( 'bbapp-deeplinking', $purge_events );

		// @todo : Re-verify events for single blog-post
		$purge_single_events = array(
			'save_post_post'                 => 1, // Called when blog-post is saved in backend.
			'edit_post_post'                 => 1, // Called when blog-post is updated in backend.
			// Added moderation support.
			'bp_suspend_user_suspended'      => 1, // Any User Suspended.
			'bp_suspend_user_unsuspended'    => 1, // Any User Unsuspended.
			'bp_suspend_comment_suspended'   => 1, // Any Comment Suspended.
			'bp_suspend_comment_unsuspended' => 1, // Any Comment Unsuspended.
		);

		$this->purge_single_events( $purge_single_events );

		// NOTE : Getting admin settings to toggle api cache.
		$is_component_active      = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );
		$settings                 = Helper::instance()->get_app_settings( 'cache_blog_post', 'buddyboss-app' );
		$cache_support_blog_posts = isset( $is_component_active ) && isset( $settings ) ? ( $is_component_active && $settings ) : false;

		if ( $cache_support_blog_posts ) {

			// Check if the cache_expiry static method exists and call it, or get the value from an instance.
			$cache_expiry_time = method_exists('BuddyBoss\Performance\Cache', 'cache_expiry') ? Cache::cache_expiry() : Cache::instance()->month_in_seconds;

			$this->cache_endpoint(
				'wp/v2/posts',
				$cache_expiry_time,
				array(
					'unique_id' => 'id',
				),
				true
			);

			// Endpoint-1: wp-json/wp/v2/posts.
			$this->cache_endpoint(
				'wp/v2/posts/<id>',
				$cache_expiry_time,
				array(),
				false
			);

			$this->cache_endpoint(
				'wp/v2/posts/archive',
				$cache_expiry_time,
				array(
					'unique_id'         => 'id',
					'purge_deep_events' => array_keys( $purge_single_events ),
				),
				false
			);

			$this->cache_endpoint(
				'wp/v2/posts/<post_id>/related-posts',
				$cache_expiry_time,
				array(
					'unique_id'         => 'id',
					'purge_deep_events' => array_keys( $purge_single_events ),
				),
				true
			);
		}
	}

	/**
	 * When blog post created
	 *
	 * @param int $blog_post_id Blog Post ID.
	 */
	public function event_save_post_post( $blog_post_id ) {
		$this->purge_item_cache_by_item_id( $blog_post_id );
		Cache::instance()->purge_by_endpoint( "/wp-json/wp/v2/posts/{$blog_post_id}/related-posts" );
	}

	/**
	 * When blog post updated
	 *
	 * @param int $blog_post_id Blog Post ID.
	 */
	public function event_edit_post_post( $blog_post_id ) {
		$this->purge_item_cache_by_item_id( $blog_post_id );
		Cache::instance()->purge_by_endpoint( "/wp-json/wp/v2/posts/{$blog_post_id}/related-posts" );
	}

	/******************************* Moderation Support ******************************/
	/**
	 * Suspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_suspended( $user_id ) {
		$post_ids = $this->get_post_ids_by_user_id( $user_id );

		$this->purge_item_cache_by_item_ids( $post_ids );
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $user_id User ID.
	 */
	public function event_bp_suspend_user_unsuspended( $user_id ) {
		$post_ids = $this->get_post_ids_by_user_id( $user_id );

		$this->purge_item_cache_by_item_ids( $post_ids );
	}

	/**
	 * Suspended Comment ID.
	 *
	 * @param int $comment_id Comment ID.
	 */
	public function event_bp_suspend_comment_suspended( $comment_id ) {
		$comment = get_comment( $comment_id );

		if ( ! empty( $comment->comment_post_ID ) ) {
			$this->purge_item_cache_by_item_id( $comment->comment_post_ID );
		}
	}

	/**
	 * Unsuspended User ID.
	 *
	 * @param int $comment_id Comment ID.
	 */
	public function event_bp_suspend_comment_unsuspended( $comment_id ) {
		$comment = get_comment( $comment_id );

		if ( ! empty( $comment->comment_post_ID ) ) {
			$this->purge_item_cache_by_item_id( $comment->comment_post_ID );
		}
	}

	/*********************************** Functions ***********************************/
	/**
	 * Get posts ids from user ID.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return array
	 */
	private function get_post_ids_by_user_id( $user_id ) {
		global $wpdb;

		$sql = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='post' AND post_author = %d", $user_id );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->get_col( $sql );
	}

	/**
	 * Purge item cache by item id.
	 *
	 * @param $id
	 */
	private function purge_item_cache_by_item_id( $id ) {
		Cache::instance()->purge_by_group( 'blog-post_' . $id );
		Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $id ) );
	}

	/**
	 * Purge item cache by item ids.
	 *
	 * @param array $ids Array of ids.
	 *
	 * @since 2.0.70
	 *
	 * @return void
	 */
	private function purge_item_cache_by_item_ids( $ids ) {
		if ( empty( $ids ) ) {
			return;
		}

		Cache::instance()->purge_by_group_names( $ids, array( 'blog-post_' ), array( $this, 'prepare_blog_post_deeplink' ) );
	}

	/**
	 * Prepare activity deeplink.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return string
	 */
	public function prepare_blog_post_deeplink( $post_id ) {
		return 'bbapp-deeplinking_' . get_permalink( $post_id );
	}
}
