<?php
/**
 * BuddyBoss Performance.
 *
 * @package BuddyBoss\App\Performance
 */

namespace BuddyBossApp\Performance;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Performance;
use BuddyBossApp\Performance\Integration\BB_App_Blog_Posts;
use BuddyBossApp\Performance\Integration\BB_App_Categories;
use BuddyBossApp\Performance\Integration\BB_App_Core;
use BuddyBossApp\Performance\Integration\BB_App_Directory_Blocks;
use BuddyBossApp\Performance\Integration\BB_App_Pages;
use BuddyBossApp\Performance\Integration\BB_App_Posts_Comments;
use BuddyBossApp\Performance\Integration\BB_Bookmarks;
use BuddyBossApp\Performance\Integration\BBAPP_DeepLinking;
use BuddyBossApp\Performance\Integration\LD_Courses;
use BuddyBossApp\Performance\Integration\LD_Courses_Details;
use BuddyBossApp\Performance\Integration\LD_Lessons;
use BuddyBossApp\Performance\Integration\LD_Quizzes;
use BuddyBossApp\Performance\Integration\LD_Topics;

/**
 * BuddyBoss APP Performance class.
 */
class BuddyBossApp_Performance {

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Add BuddyBoss App related cache hooks
	 */
	public function load() {
		// Check if BuddyBoss Platform plugin installed and meet min requirement.
		$bbplatform_version = $this->get_installed_plugins_version( 'buddyboss-platform/bp-loader.php' );
		if ( ! empty( $bbplatform_version ) && version_compare( $bbplatform_version, '1.5.7.3', '<' ) ) {
			add_action( 'admin_notices', array( $this, 'min_supported_plugin_admin_notices' ) );

			return;
		}

		add_action( 'rest_cache_loaded', array( $this, 'cache_loaded' ) );
		add_filter( 'performance_group_purge_actions', array( $this, 'get_group_purge_actions' ), 10, 2 );
		add_filter( 'performance_purge_url', array( $this, 'bbapp_performance_purge_url' ), 10, 2 );
		add_filter( 'performance_purge_components_flag', array( $this, 'performance_purge_components_flag' ) );
		add_filter( 'rest_cache_pre_current_user_id', array( $this, 'app_rest_auth' ), 2 );
		add_action( 'bb_purge_symlink_cache_after', array( $this, 'purge_deeplink_cache' ) );
		add_action( 'performance_cache_purge', array( $this, 'purge_deeplink_cache' ) );
		add_action( 'performance_purge_components_after', array( $this, 'purge_deeplink_cache' ) );
		add_filter( 'rest_post_disprepare_header_cache', array( $this, 'fallback_headers' ) );
	}

	/**
	 * Loas BuddyBoss App cache classes
	 */
	public function cache_loaded() {

		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-pages.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-posts-comments.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-blog-posts.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-categories.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-core.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-bookmarks.php';
		require_once dirname( __FILE__ ) . '/integrations/class-bb-app-directory-block.php';
		BB_App_Pages::instance();
		BB_App_Posts_Comments::instance();
		BB_App_Blog_Posts::instance();
		BB_App_Categories::instance();
		BB_App_Core::instance();
		BB_Bookmarks::instance();
		BB_App_Directory_Blocks::instance();

		if ( Performance::mu_is_plugin_active( 'sfwd-lms/sfwd_lms.php' ) ) {
			require_once dirname( __FILE__ ) . '/integrations/class-ld-courses.php';
			require_once dirname( __FILE__ ) . '/integrations/class-ld-courses-details.php';
			require_once dirname( __FILE__ ) . '/integrations/class-ld-lessons.php';
			require_once dirname( __FILE__ ) . '/integrations/class-ld-topics.php';
			require_once dirname( __FILE__ ) . '/integrations/class-ld-quizzes.php';
			LD_Courses::instance();
			LD_Courses_Details::instance();
			LD_Lessons::instance();
			LD_Topics::instance();
			LD_Quizzes::instance();
		}

		require_once dirname( __FILE__ ) . '/integrations/class-deeplinking.php';
		BBAPP_DeepLinking::instance();
	}

	/**
	 * Get Purge actions by group
	 *
	 * @param array  $actions Actions list to perform.
	 * @param string $group   Cache Group name.
	 *
	 * @return mixed|array
	 */
	public function get_group_purge_actions( $actions, $group ) {
		switch ( $group ) {
			case 'wordpress':
				$actions = array( 'app_page', 'blog_post', 'post_comment', 'categories', 'bb-bookmarks' );

				$custom_supports = self::get_custom_setting();
				if ( ! empty( $custom_supports ) ) {
					foreach ( $custom_supports as $custom_support ) {
						$actions[] = $custom_support['slug'];
					}
				}
				break;
			case 'learndash':
				$actions = array( 'sfwd-courses', 'sfwd-lessons', 'sfwd-topic', 'sfwd-quiz' );
				break;
			case 'bbapp':
				$actions = array( 'app_core' );
				break;
		}

		return $actions;
	}

	/**
	 * Filter to update url of the purge cache.
	 *
	 * @param string $url Performance purge URL.
	 *
	 * @return mixed|string|void
	 */
	public function bbapp_performance_purge_url( $url ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['page'] ) && 'bbapp-settings' === $_GET['page'] && isset( $_GET['setting'] ) && 'cache_support' === $_GET['setting'] ) {
			$url = admin_url( 'admin.php?page=bbapp-settings&setting=cache_support' );
		}

		return $url;
	}

	/**
	 * Check if the performance module is enabled in App Plugin.
	 *
	 * @param bool $flag Flag if the performance module is enabled.
	 *
	 * @return bool
	 */
	public function performance_purge_components_flag( $flag ) {
		return function_exists( 'bbapp_is_active' ) && bbapp_is_active( 'performance' ) ? true : $flag;
	}

	/**
	 * Get the Pre User ID from BuddyBoss APP JWT Token.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return int|void
	 */
	public function app_rest_auth( $user_id ) {
		$header    = $this->get_all_headers();
		$jwt_token = false;

		if ( ! empty( $header ) ) {
			foreach ( $header as $k => $v ) {
				if ( strtolower( $k ) === 'accesstoken' ) {
					$jwt_token = $v;
					break;
				}
			}
		}

		if ( $jwt_token ) {
			$token = explode( '.', $jwt_token );
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
			$token = (array) json_decode( base64_decode( $token[1] ) );

			if ( isset( $token['data'] ) && isset( $token['data']->user ) && isset( $token['data']->user->id ) ) {
				$user_id = $token['data']->user->id;

				// Check if there is any switch user.
				$switch_data = get_user_meta( $user_id, '_bbapp_jwt_switch_user', true );
				$switch_data = ( ! is_array( $switch_data ) ) ? array() : $switch_data;
				$jti         = ( isset( $token['jti'] ) ) ? $token['jti'] : false;

				// if switch user is found for current access token pass it.
				if ( $jti && isset( $switch_data[ $jti ] ) && is_numeric( $switch_data[ $jti ] ) ) {
					return (int) $switch_data[ $jti ];
				}

				// End Switch user logic's.
				return $user_id;
			}
		}
	}

	/**
	 * Purge the cache after deeplinking
	 *
	 * @since 2.4.31
	 */
	public function purge_deeplink_cache() {
		Cache::instance()->purge_by_component( 'bbapp-deeplinking' );
	}

	/**
	 * Add app headers.
	 *
	 * @param array $disallow_headers Disallowed headers.
	 *
	 * @return array
	 */
	public function fallback_headers( $disallow_headers ) {
		$app_headers = array(
			'bbapp-logged-in',
			'bbapp-unread-notifications',
		);

		return array_merge( $disallow_headers, $app_headers );
	}

	public function get_installed_plugins_version( $pluginFile ) {
		$version = 0;
		if ( file_exists( WP_PLUGIN_DIR . '/' . $pluginFile ) ) {

			// Check we have access to get_plugins()
			if ( ! function_exists( 'get_plugin_data' ) ) {
				require ABSPATH . 'wp-admin/includes/plugin.php';
			}

			$data    = \get_plugin_data( WP_PLUGIN_DIR . '/' . $pluginFile, true, false );
			$version = $data['Version'];
		}

		return $version;
	}

	/**
	 * Add admin notice if non supported activated
	 */
	public function min_supported_plugin_admin_notices() {
		$bbplatform_version = $this->get_installed_plugins_version( 'buddyboss-platform/bp-loader.php' );
		if ( ! Performance::mu_is_plugin_active( 'buddyboss-platform/bp-loader.php' ) && ! empty( $bbplatform_version ) && version_compare( $bbplatform_version, '1.5.7.3', '<' ) ) {
			printf(
			/* translators: 1. div classes. 2. notice message. 3. Admin update page url. 4. link text.  */
				'<div class="%1$s"><p>%2$s <a href="%3$s">%4$s</a></p></div>',
				esc_attr( 'notice notice-info' ),
				__( 'The BuddyBoss App plugin requires a minimum BuddyBoss Platform version of v1.5.7.3.', 'buddyboss-app' ),
				esc_url( admin_url( 'update-core.php' ) ),
				__( 'Please update BuddyBoss Platform.', 'buddyboss-app' )
			);
		}
	}

	/**
	 * Get list of setting for custom supports
	 * @return mixed|void
	 */
	public static function get_custom_setting(){
		return apply_filters( 'bbapp_admin_performance_custom_setting', array() );
	}

	/**
	 * Get Headers.
	 *
	 * @return array|false|string
	 */
	public function get_all_headers() {
		if ( function_exists( 'getallheaders' ) ) {
			return getallheaders();
		}

		if ( ! is_array( $_SERVER ) ) {
			return array();
		}

		$headers = array();

		foreach ( $_SERVER as $name => $value ) {
			if ( substr( $name, 0, 5 ) === 'HTTP_' ) {
				$headers[ str_replace( ' ', '-', ucwords( strtolower( str_replace( '_', ' ', substr( $name, 5 ) ) ) ) ) ] = $value;
			}
		}

		return $headers;
	}
}
