<?php
/**
 * BuddyBoss App notification user list.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Render list of user notification.
 *
 * Class UserNotificationList
 */
class UserNotificationList extends \WP_List_Table {

	/**
	 * User id.
	 *
	 * @var int $user_id
	 */
	public $user_id;

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'singular' => __( 'User Notification', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'User Notifications', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // does this table support ajax?
				'screen'   => 'user_notification', // load push data in custom screen.
			)
		);

		add_action( 'admin_footer', array( &$this, 'admin_header' ) );
	}

	/**
	 * Style list table
	 */
	public function admin_header() {
		$page = ( isset( $_GET['page'] ) ) ? esc_attr( bbapp_input_clean( wp_unslash( $_GET['page'] ) ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( 'bbapp-user-push' !== $page ) {
			return;
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'type'         => __( 'Type', 'buddyboss-app' ),
			'notification' => __( 'Notification', 'buddyboss-app' ),
			'device'       => __( 'Device', 'buddyboss-app' ),
			'status'       => __( 'Status', 'buddyboss-app' ),
		);

		return $columns;
	}

	/**
	 * Render a column when no column specific method exist.
	 *
	 * @param object $item        Table column item.
	 * @param string $column_name Table column name.
	 *
	 * @return mixed
	 */
	public function column_default( $item, $column_name ) {
		$title = '';
		if ( 'manual_push_notification' === $item->type ) {
			$push  = Push::instance()->get_notification( $item->item_id );
			$title = $push->secondary_text;
		}
		switch ( $column_name ) {
			case 'type':
				echo ! empty( $item->type ) && ( 'manual_push_notification' === $item->type ) ? esc_html__( 'Manual', 'buddyboss-app' ) : esc_html__( 'Automatic', 'buddyboss-app' );
				break;
			case 'notification':
				if ( 'manual_push_notification' === $item->type ) {
					echo sprintf( '<a class="row-title" href="admin.php?page=bbapp-notification&action=view_push&push_id=%1$s">%2$s</a>', absint( $item->item_id ), esc_html( $title ) );
				} else {
					global $bbapp_register_push_type;
					if ( isset( $bbapp_register_push_type[ $item->type ] ) ) {
						printf( '<span class="bbapp-notification-type">%s</span>', esc_html( $bbapp_register_push_type[ $item->type ]['admin_label'] ) ); // Custom type.
					} else {
						printf( '<span class="bbapp-notification-type">%1$s</span>', esc_html__( 'Unspecified Notification', 'buddyboss-app' ) );
					}
				}
				break;
			case 'device':
				echo ! empty( $item->device ) ? esc_html( Push::instance()->get_device_string( $item->device ) ) : esc_html( Push::instance()->get_device_string( $item->platform ) );
				break;
			case 'status':
				$status = $item->status;

				/* translators: %1$s is the date and %2$ is the time */
				$date_time_format = sprintf( _x( '%1$s @ %2$s', 'date_at_time', 'buddyboss-app' ), get_option( 'date_format' ), get_option( 'time_format' ) );

				if ( 1 === (int) $status ) {
					echo sprintf( '<strong>%s</strong> <br /><span>%s</span>', esc_html__( 'Sent', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item->processed_at, $date_time_format ) ) );
				} else {
					$nice_error = Push::instance()->nice_error_codes();
					$nice_error = isset( $nice_error[ $status ] ) ? $nice_error[ $status ] : $status[0];
					echo sprintf( '<strong>%s</strong> <br /><span>%s </span>%s', esc_html__( 'Failed', 'buddyboss-app' ), esc_html__( 'Reason:', 'buddyboss-app' ), esc_html( $nice_error ) );
				}

				break;
		}
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {

		$this->_column_headers = $this->get_column_info();

		$per_page     = $this->get_items_per_page( 'per_page', 10 );
		$current_page = $this->get_pagenum();

		$this->items = $this->get_push_notifications( $per_page, $current_page );
		$total_items = $this->get_push_notifications_count();

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);
	}

	/** Text displayed when no user app data is available */
	public function no_items() {
		esc_html_e( 'No push notifications found.', 'buddyboss-app' );
	}

	/**
	 * Get the user ID.
	 *
	 * Look for $_GET['user_id']. If anything else, force the user ID to the
	 * current user's ID so they aren't left without a user to edit.
	 *
	 * @return int
	 */
	private function get_user_id() {
		if ( ! empty( $this->user_id ) ) {
			return $this->user_id;
		}

		$this->user_id = (int) get_current_user_id();

		// We'll need a user ID when not on self profile.
		if ( ! empty( $_GET['user_id'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->user_id = (int) $_GET['user_id']; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		return $this->user_id;
	}

	/**
	 * Get push notification.
	 *
	 * @param int $per_page    Notification in per page.
	 * @param int $page_number Current page number.
	 *
	 * @return null|array|object
	 */
	public function get_push_notifications( $per_page = 10, $page_number = 1 ) {
		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications_history' );
		$sql        = $wpdb->prepare( "SELECT * FROM {$table_name} WHERE user_id = %d AND type != 'silent_push'", $this->get_user_id() ); //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $_REQUEST['orderby'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$orderby = 'processed_at';

			$sql .= ' ORDER BY ' . esc_sql( $orderby );
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' ASC'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		} else {
			$sql .= ' ORDER BY processed_at';
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' DESC'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		$sql   .= " LIMIT $per_page";
		$sql   .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;
		$result = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

		return $result;
	}

	/**
	 * Get push notification count
	 *
	 * @return null|int
	 */
	public function get_push_notifications_count() {
		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications_history' );
		return $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM {$table_name} WHERE user_id = %d AND type != 'silent_push'", $this->get_user_id() ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}
}
