<?php
/**
 * Silent notification.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

use BuddyBossApp\AccessControls\AccessRule;

/**
 * Silent notification.
 */
class SilentNotification {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance = null;

	/**
	 * Class construct.
	 *
	 * @since 2.1.20
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @since 2.1.20
	 * @return SilentNotification
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Initialize actions/filters
	 *
	 * @since 2.1.20
	 * @return void
	 */
	public function init() {
		add_action( 'bb_access_controls_group_add_members_after', array( $this, 'bb_access_controls_add_members' ), 10, 2 );
		add_action( 'bb_access_controls_condition_group_remove_users', array( $this, 'bb_access_controls_member_remove' ), 10, 2 );
	}

	/**
	 * Add members to group.
	 *
	 * @param int   $group_id Group ID.
	 * @param array $user_ids User IDs.
	 *
	 * @since 2.1.20
	 * @return void
	 */
	public function bb_access_controls_add_members( $group_id, $user_ids ) {
		$silent_data = $this->get_access_rules_data_by_id( $group_id );

		$this->send_silent_notification( $user_ids, $silent_data );
	}

	/**
	 * Remove member from group.
	 *
	 * @param int   $group_id Group ID.
	 * @param array $user_ids User IDs.
	 *
	 * @since 2.1.20
	 * @return void
	 */
	public function bb_access_controls_member_remove( $group_id, $user_ids ) {
		$silent_data = $this->get_access_rules_data_by_id( $group_id );

		$this->send_silent_notification( $user_ids, $silent_data );
	}

	/**
	 * Get access rules data by group id.
	 *
	 * @param int $group_id Group ID.
	 *
	 * @since 2.1.20
	 * @return array
	 */
	public function get_access_rules_data_by_id( $group_id ) {
		$silent_data  = array();
		$access_rules = AccessRule::instance()->get_access_rules( array( 'include_group_id' => $group_id ) );

		if ( ! empty( $access_rules['count'] ) ) {
			$access_object_map = array(
				'app_page'          => 'app_pages',
				'page'              => 'pages',
				'default_post'      => 'posts',
				'post_term'         => 'posts',
				'post'              => 'posts',
				'default_ld_course' => 'learndash_courses',
				'sfwd-courses_term' => 'learndash_courses',
				'sfwd-courses'      => 'learndash_courses',
			);
			$access_object     = array();

			foreach ( $access_rules['results'] as $access_rule ) {
				if ( isset( $access_object_map[ $access_rule['item_type'] ] ) ) {
					$access_object[] = $access_object_map[ $access_rule['item_type'] ];
				}
			}

			$access_object_encode = ! empty( $access_object ) ? wp_json_encode( array_values( array_unique( $access_object ) ) ) : '';
			$silent_data          = array(
				'component'     => 'access_control',
				'access_object' => $access_object_encode,
			);
		}

		return $silent_data;
	}

	/**
	 * Send silent notification.
	 *
	 * @param array $user_ids    User IDs.
	 * @param array $silent_data Silent data.
	 *
	 * @since 2.1.20
	 * @return void
	 */
	public function send_silent_notification( $user_ids, $silent_data = array() ) {
		if ( empty( $silent_data ) ) {
			return;
		}

		// Send silent notification.
		$push_args = array(
			'primary_text'                 => '',
			'secondary_text'               => '',
			'user_ids'                     => $user_ids,
			'sent_as'                      => 1,
			'type'                         => 'silent_push',
			'filter_users_by_subscription' => false,
			'data'                         => array(
				'silent' => true,
			),
			'normal_notification'          => false,

		);

		$push_args['data']['silent_data'] = $silent_data;
		bbapp_send_push_notification( $push_args );
	}
}
