<?php
/**
 * Service Abstract.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification\Services;

use BuddyBossApp\Tools\Logger;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Notification\Push;

/**
 * This is abstract notification service integration class.
 * Extending this class will provide all logical feature to push notification.
 *
 * Class ServiceAbstract
 */
abstract class ServiceAbstract {

	/**
	 * Push notification service name.
	 *
	 * @var string $service_name
	 */
	protected $service_name = false;

	/**
	 * Push notification service label.
	 *
	 * @var string $service_label
	 */
	protected $service_label = false;

	/**
	 * Setup Service.
	 *
	 * @param string $service_name  Service name.
	 * @param string $service_label Service lable.
	 */
	public function setup( $service_name, $service_label ) {
		$this->service_name  = $service_name;
		$this->service_label = $service_label;
		$this->register_service();
	}

	/**
	 * Register Service..
	 */
	public function register_service() {
		global $bbapp_push_notification_services;

		if ( ! isset( $bbapp_push_notification_services ) || ! is_array( $bbapp_push_notification_services ) ) {
			$bbapp_push_notification_services = array();
		}

		$bbapp_push_notification_services[ $this->service_name ] = array(
			'label'      => $this->service_label,
			'controller' => $this,
		);
	}

	/**
	 * Get service name.
	 */
	public function get_service_name() {
		return $this->service_name;
	}

	/**
	 * Get service label.
	 */
	public function get_service_label() {
		return $this->service_label;
	}

	/**
	 * How many push to sent per batch.
	 *
	 * @return int
	 */
	public function max_per_batch() {
		return 1000;
	}

	/**
	 * Weather to group same push content in one batch.
	 *
	 * @return bool
	 */
	public function do_group() {
		return true;
	}

	/**
	 * Push type is saved on database along with push token.
	 */
	abstract public function push_type();

	/**
	 * Render setting method.
	 */
	abstract public function render_settings();

	/**
	 * Admih initialize method.
	 */
	abstract public function init_admin();

	/**
	 * Function to send the push notification.
	 *
	 * @param array $push_batch Push notification batch.
	 */
	abstract public function send_push( $push_batch);

	/**
	 * Runs on init admin if on app pages of selected push service
	 */
	public function service_init_admin() {
		$this->init_admin();
	}

	/**
	 * Send Push Notification.
	 *
	 * @param array $push_batch Push notification batch.
	 *
	 * @return bool
	 */
	public function service_send_push( $push_batch ) {
		if ( empty( $push_batch ) ) {
			return false;
		}

		if ( is_array( $push_batch ) ) {
			foreach ( $push_batch as $k => $push ) {
				if ( ! is_array( $push_batch[ $k ]['data']['data'] ) ) {
					$push_batch[ $k ]['data']['data'] = array();
				}

				$push_batch[ $k ]['data']['data'] = $this->filter_notification_data_key_items( $push_batch[ $k ]['data']['data'] );
			}
		}

		$push_receipts = $this->send_push( $push_batch );

		if ( is_wp_error( $push_receipts ) ) {
			$error = __( 'Unknown Error', 'buddyboss-app' );

			if ( is_wp_error( $push_receipts ) ) {
				$error = $push_receipts->get_error_message();
			}

			// Log it.
			$logmsg = sprintf( '%1$s <b>%2$s</b> %3$s. <b>%4$s</b> %5$s ', esc_html__( 'Error while sending push notification.', 'buddyboss-app' ), esc_html__( 'Error :', 'buddyboss-app' ), $error, esc_html__( 'Push Service :', 'buddyboss-app' ), $this->service_name );

			Logger::instance()->add( 'push_notification', $logmsg );
			Logger::instance()->add( 'push_notification', var_export( $push_receipts, true ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_var_export

			return false;
		} else {
			/**
			 * Add all push notification into the history table.
			 */
			$this->save_push_notification_history( $push_batch, $push_receipts );

			return true;
		}
	}

	/**
	 * Save push data in push history table.
	 *
	 * @param array $push_batch    Push notification batch.
	 * @param array $push_receipts Push notification receipts.
	 */
	public function save_push_notification_history( $push_batch, $push_receipts ) {

		if ( is_array( $push_receipts ) ) {
			global $wpdb;
			$table_name   = "{$wpdb->prefix}bbapp_push_notifications_history";
			$insert_query = "INSERT INTO {$table_name} (`type`,`status`, `data`, `user_id`, `processed_at`, `item_id`, `secondary_id`, `device`, `platform`,`blog_id`) VALUES ";
			$entries      = array();

			foreach ( $push_batch as $push ) {
				$push_recipient_status = array_key_exists( $push['device']['token'], $push_receipts ) ? $push_receipts[ $push['device']['token'] ] : 0;
				$status                = ( $this->is_support_legacy_enabled() ) ? ( ! empty( $push_recipient_status['status'] ) ? $push_recipient_status['status'] : 0 ) : ( ( array_key_exists( $push['device']['token'], $push_receipts ) && ! empty( $push_receipts[ $push['device']['token'] ]['data'] ) && ! empty( $push_receipts[ $push['device']['token'] ]['data']->name ) ) ? 1 : 0 );
				$data                  = ! empty( $push_recipient_status['data'] ) ? $push_recipient_status['data'] : array();

				$entries[] = array(
					$push['type'], // type.
					$status, // status.
					wp_json_encode( $data ), // data.
					$push['user_id'], // user_id.
					current_time( 'mysql', 1 ), // processed_at.
					$push['data']['data']['item_id'], // item_id.
					$push['data']['data']['secondary_item_id'], // secondary_id.
					$push['device']['device_model'], // device.
					$push['device']['platform'], // platform.
					$push['data']['blog_id'], // blog_id.
				);
			}

			$prepare_var            = array();
			$query_row_placeholders = array();
			foreach ( $entries as $entry ) {
				foreach ( $entry as $e ) {
					$prepare_var[] = $e;
				}
				$query_row_placeholders[] = '(%s,%d,%s,%d,%s,%d,%d,%s,%s,%d)';
			}
			$insert_query .= implode( ",\n ", $query_row_placeholders );
			$sql_prepare   = $wpdb->prepare( $insert_query, $prepare_var ); //phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

			if ( ! $wpdb->query( $sql_prepare ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
				Logger::instance()->add( 'push_notification_history', __( 'Error while saving the push histories into the database.', 'buddyboss-app' ) );
			}
		}
	}


	/**
	 * Returns the push settings of specific app.
	 *
	 * @param string $key Settings key.
	 *
	 * @return bool
	 */
	public function get_settings( $key ) {

		$push_settings = Push::instance()->get_settings();

		if ( isset( $push_settings[ $key ] ) ) {
			return $push_settings[ $key ];
		}

		return false;

	}

	/**
	 * Formats the topic according to requirement of bbapp core app.
	 *
	 * @param string $topic Notification topic.
	 *
	 * @return string
	 */
	public function notification_topic_format( $topic ) {
		$bbapp_app_id = ManageApp::instance()->get_app_id();
		$topic        = "/topics/{$bbapp_app_id}_{$topic}_" . get_current_blog_id();
		return $topic;
	}

	/**
	 * Filter Notification extra data key using whitelist to avoid unnecessary data to be posted to push server.
	 *
	 * @param array $data Filter parameters.
	 *
	 * @return mixed
	 */
	public function filter_notification_data_key_items( $data ) {

		$white_list = array(
			'notification_type',
			'priority',
			'primary_text',
			'site_id',
			// Notification Items.
			'action',
			'item_id',
			'author_id',
			'secondary_item_id',
			'namespace',
			'badge_count',
			'link',
			'notification_id',
			'component_name',
			'silent',
			'silent_data',
		);

		$white_list = apply_filters( 'bbapp_push_service_filter_notification_data_whitelist', $white_list );

		foreach ( $data as $k => $v ) {
			if ( ! in_array( $k, $white_list, true ) ) {
				unset( $data[ $k ] );
			}
		}

		return $data;
	}

	/**
	 * Tells Weather Current Page is Push Setting Page.
	 *
	 * @return bool
	 */
	public function is_push_setting_page() {
		return ( isset( $_GET['page'] ) && 'bbapp-notification' === bbapp_input_clean( wp_unslash( $_GET['page'] ) ) && isset( $_GET['setting'] ) && 'settings' === bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}
}
