<?php
/**
 * Holds the Firebase functionality for push notifications.
 *
 * @package BuddyBossApp\Notification\Services
 */

namespace BuddyBossApp\Notification\Services\Legacy;

use BuddyBossApp\ManageApp;
use BuddyBossApp\Notification\Push;
use BuddyBossApp\Tools\Logger;

/**
 * Legacy firebase code.
 */
class Legacy {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return Legacy
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Loads once when instance registered.
	 */
	public function hooks() {
	}

	/**
	 * Returns the Firebase Server Key.
	 *
	 * @return bool
	 */
	public function get_server_key() {

		// Get main network id's data for push notification.
		$global_settings = ManageApp::instance()->get_settings( true );
		if ( empty( $global_settings['push.firebase_server_key'] ) ) {
			return false;
		}

		return $global_settings['push.firebase_server_key'];
	}

	/**
	 * Function to get the firebase endpoint.
	 *
	 * @return string
	 */
	public function get_firebase_endpoint() {
		return 'https://fcm.googleapis.com/fcm/send';
	}

	/**
	 * Formats the topic according to requirement of bbapp core app.
	 *
	 * @param string $topic Notification topic.
	 *
	 * @return string
	 */
	public function notification_topic_format( $topic ) {
		$bbapp_app_id = ManageApp::instance()->get_app_id();
		$topic        = "/topics/{$bbapp_app_id}_{$topic}_" . get_current_blog_id();
		return $topic;
	}

	/**
	 * Function to send the push notification.
	 *
	 * @param array $push_batch Push notification batch.
	 *
	 * @return array
	 */
	public function send_push( $push_batch ) {
		/**
		 * As in firebase service we are sending push in group mode.
		 * That's means in push batch we are receiving same content push notification.
		 * So we can pick the first notification from Push batch for content and collect device from each push to prepare single request.
		 * === badge_count ===
		 * As we are in firebase service sending same push content in single request,
		 * so we cannot send unique badge count to each user instead we will send 1 in this case.
		 */

		$first_push_from_batch = $push_batch[0]['data'];

		$primary_text   = html_entity_decode( $first_push_from_batch['primary_text'], ENT_QUOTES, 'UTF-8' );
		$secondary_text = html_entity_decode( $first_push_from_batch['secondary_text'], ENT_QUOTES, 'UTF-8' );

		// Prepare the push data from first notification on batch.
		$push_data = array(
			'notification' => array(
				'title' => $primary_text,
				'body'  => $secondary_text,
				'sound' => 'default',
			),
		);

		if ( isset( $first_push_from_batch['sound'] ) && $first_push_from_batch['sound'] ) {
			$push_data['notification']['sound'] = $first_push_from_batch['sound'];
		}

		$headers = array( 'Authorization' => 'key=' . $this->get_server_key() );

		if ( isset( $first_push_from_batch['topic'] ) && ! empty( $first_push_from_batch['topic'] ) ) {
			$push_data['to'] = $this->notification_topic_format( $first_push_from_batch['topic'] );
		} else {
			$push_data['registration_ids'] = array();

			// Add Devices.
			foreach ( $push_batch as $push ) {
				if ( 'firebase' === $push['device']['push_type'] ) { // make sure it's correct device type.
					$push_data['registration_ids'][] = $push['device']['token'];
				}
			}
		}

		/**
		 * Set the notification extra information data.
		 * These data can be the item ID or text anything.
		 */

		if ( isset( $first_push_from_batch['data'] ) && ! empty( $first_push_from_batch['data'] ) ) {

			$push_data['data'] = array();

			foreach ( $first_push_from_batch['data'] as $k => $v ) {
				if ( is_string( $v ) || is_int( $v ) ) {
					$push_data['data'][ $k ] = $v;
				}
			}
		}

		$push_data['content_available'] = true;
		$push_data['priority']          = ( isset( $first_push_from_batch['priority'] ) && ! empty( $first_push_from_batch['priority'] ) ) ? $first_push_from_batch['priority'] : 'normal';
		$response                       = $this->post( $push_data, $headers );

		if ( strpos( $response['body'], 'INVALID_KEY' ) ) {
			$invalid_msg = __( 'The Firebase Key is invalid.', 'buddyboss-app' );
			if ( ! empty( $push_data['data'] ) && ! empty( $push_data['data']['item_id'] ) ) {
				Push::instance()->add_notification_log( $push_data['data']['item_id'], $invalid_msg );
			}
			Logger::instance()->add( 'push_notification', $invalid_msg );
		}

		if ( ! empty( $response['response']['code'] ) && 200 !== (int) $response['response']['code'] ) {
			new \WP_Error( 'error', __( 'There was an error while sending push notification through the Firebase API.', 'buddyboss-app' ) );
		}

		$devices_results      = array();
		$k                    = 0;
		$response_body        = wp_remote_retrieve_body( $response );
		$response_body_decode = json_decode( $response_body, true );

		if ( $response_body_decode['results'] ) {
			foreach ( $response_body_decode['results'] as $result ) {
				$status = 1;
				$reason = isset( $result['message_id'] ) ? $result['message_id'] : '';
				if ( isset( $result['error'] ) ) {
					$status = $this->get_error_status_code( $result['error'] );
					$reason = $result['error'];
				}
				$devices_results[ $push_data['registration_ids'][ $k ] ] = array(
					'status' => $status,
					'reason' => $reason,
				);
				$k ++;
			}
		}

		return $devices_results;
	}

	/**
	 * Helper function to post data to Firebase.
	 *
	 * @param array $data    Push notification data.
	 * @param array $headers Push notification headers.
	 *
	 * @return array|\WP_Error
	 */
	public function post( $data, $headers = array() ) {
		$url                     = $this->get_firebase_endpoint();
		$headers['Content-Type'] = 'application/json';
		$data                    = wp_json_encode( $data );

		return wp_safe_remote_post(
			$url,
			array(
				'headers'     => $headers,
				'body'        => $data,
				'method'      => 'POST',
				'data_format' => 'body',
			)
		);
	}

	/**
	 * Get error status code.
	 *
	 * @param int|string $error Error code from the Google Firebase.
	 *
	 * @return false|int|string
	 */
	public function get_error_status_code( $error ) {
		$firebase_code_and_error = $this->firebase_error_code_map();

		return array_search( $error, $firebase_code_and_error, true );
	}


	/**
	 * List of firebase errors
	 *
	 * @ref https://firebase.google.com/docs/cloud-messaging/http-server-ref
	 * Error code left & Firebase Error Right Map.
	 * @return string[]
	 */
	public function firebase_error_code_map() {
		return array(
			'2'  => 'MissingRegistration',
			'3'  => 'InvalidRegistration',
			'4'  => 'NotRegistered',
			'5'  => 'InvalidPackageName',
			'6'  => 'MismatchSenderId',
			'7'  => 'InvalidParameters',
			'8'  => 'MessageTooBig',
			'9'  => 'InvalidDataKey',
			'10' => 'InvalidTtl',
			'11' => 'Unavailable',
			'12' => 'InternalServerError',
			'13' => 'DeviceMessageRateExceeded',
			'14' => 'TopicsMessageRateExceeded',
			'15' => 'InvalidApnsCredential',
		);
	}

}
