<?php
/**
 * Holds the Firebase functionality for push notifications.
 *
 * @package BuddyBossApp\Notification\Services
 */

namespace BuddyBossApp\Notification\Services;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\Notification\Services\Legacy\Legacy;
use BuddyBossApp\ManageApp;

/**
 * Firebase class for the push notification to send to Android devices.
 */
class Firebase extends ServiceAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return Firebase
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->init();
			self::$instance->setup( 'firebase', 'Firebase' );
		}

		return self::$instance;
	}

	/**
	 * Loads once when instance registered.
	 */
	public function init() {
	}

	/**
	 * How many push to sent per batch.
	 *
	 * @return int
	 */
	public function max_per_batch() {
		return 1000;
	}

	/**
	 * Weather to group same push content in one batch.
	 *
	 * @return bool
	 */
	public function do_group() {
		if ( $this->is_support_legacy_enabled() ) {
			return true;
		}

		return false;
	}

	/**
	 * Push type is saved on database along with push token.
	 *
	 * @return string
	 */
	public function push_type() {
		return 'firebase';
	}

	/**
	 * Returns the Firebase Server Key.
	 *
	 * @return bool
	 */
	public function get_admin_key() {
		$json_file = Configure::instance()->option( 'push.firebase_admin_key' );
		if ( ! empty( $json_file ) ) {
			return bbapp_get_upload_full_path( $json_file );
		}
		return false;
	}

	/**
	 * Admin init method.
	 *
	 * @return void
	 */
	public function init_admin() {
	}

	/**
	 * Function to get the firebase endpoint.
	 *
	 * @return string
	 */
	public function get_firebase_endpoint() {
		return 'https://fcm.googleapis.com' . $this->get_firebease_route();
	}

	/**
	 * Get firebase route.
	 *
	 * @since 1.7.6
	 * @return string
	 */
	public function get_firebease_route() {
		$project_id = bbapp_get_option( 'bbapp_firebease_project_id', get_current_network_id() );

		return "/v1/projects/{$project_id}/messages:send";
	}

	/**
	 * Notification topic format.
	 *
	 * @param string $topic Topic.
	 *
	 * @since 1.7.6
	 *
	 * @return string
	 */
	public function notification_topic_format( $topic ) {
		$bbapp_app_id = ManageApp::instance()->get_app_id();
		$topic        = $bbapp_app_id . '_' . $topic . '_' . get_current_blog_id();

		return $topic;
	}

	/**
	 * Function to send the push notification.
	 *
	 * @param array $push_batch Push notification batch.
	 *
	 * @return array
	 * @throws \Google_Exception Google Exception.
	 */
	public function send_push( $push_batch ) {
		if ( $this->is_support_legacy_enabled() ) {
			return Legacy::instance()->send_push( $push_batch );
		}

		return $this->send_push_v1( $push_batch );
	}

	/**
	 * Function to send the push notification.
	 *
	 * @param array $push_batch Push notification batch.
	 *
	 * @since 1.7.6
	 * @return array
	 * @throws \Google_Exception Google Exception.
	 */
	public function send_push_v1( $push_batch ) {
		/**
		 * As in firebase service we are sending push in group mode.
		 * That's means in push batch we are receiving same content push notification.
		 * So we can pick the first notification from Push batch for content and collect device from each push to prepare single request.
		 * === badge_count ===
		 * As we are in firebase service sending same push content in single request,
		 * so we cannot send unique badge count to each user instead we will send 1 in this case.
		 */
		$first_push_from_batch = $push_batch[0]['data'];
		$is_silent             = isset( $first_push_from_batch['data']['silent'] ) && $first_push_from_batch['data']['silent'];
		$primary_text          = html_entity_decode( $first_push_from_batch['primary_text'], ENT_QUOTES, 'UTF-8' );
		$secondary_text        = html_entity_decode( $first_push_from_batch['secondary_text'], ENT_QUOTES, 'UTF-8' );

		// Prepare the push data from first notification on batch.
		$push_data['message']                                     = array(
			'data'         => array(),
			'notification' => array(
				'title' => $primary_text,
				'body'  => $secondary_text,
			),
		);
		$push_data['message']['apns']['payload']['aps']['sound']  = 'default';
		$push_data['message']['android']['notification']['sound'] = 'default';

		if ( isset( $first_push_from_batch['sound'] ) && $first_push_from_batch['sound'] ) {
			$push_data['message']['apns']['payload']['aps']['sound']  = $first_push_from_batch['sound'];
			$push_data['message']['android']['notification']['sound'] = $first_push_from_batch['sound'];
		}

		/**
		 * Set the notification extra information data.
		 * These data can be the item ID or text anything.
		 */
		if ( ! empty( $first_push_from_batch['data'] ) ) {
			foreach ( $first_push_from_batch['data'] as $k => $v ) {
				if ( is_string( $v ) || is_int( $v ) ) {
					$push_data['message']['data'][ $k ] = (string) $v;
				}
			}
		}

		/**
		 * Set the silent notification data.
		 */
		if ( $is_silent && ! empty( $first_push_from_batch['data']['silent_data'] ) ) {
			foreach ( $first_push_from_batch['data']['silent_data'] as $k => $v ) {
				if ( is_string( $v ) || is_int( $v ) ) {
					$push_data['message']['data'][ $k ] = (string) $v;
				}
			}
		}

		$auth            = $this->generate_auth_key();
		$headers         = array( 'Authorization' => 'Bearer ' . $auth );
		$devices_results = array();

		if ( ! empty( $first_push_from_batch['topic'] ) ) {
			$push_data['message']['topic'] = $this->notification_topic_format( $first_push_from_batch['topic'] ); // Todo: we will implement later.
		}

		if ( ! empty( $push_batch ) ) {
			// Single device token push trigger.
			if ( 1 === count( $push_batch ) ) {
				if ( ! empty( $push_batch[0]['device'] ) && 'firebase' === $push_batch[0]['device']['push_type'] ) {
					$push_data['message']['token'] = $push_batch[0]['device']['token'];
					$push_data                     = $this->set_user_badge_count( $push_data, $push_batch[0]['user_id'] );

					if ( $is_silent ) {
						$push_data = $this->set_silent_notification_data( $push_data );
					}

					$devices_results = $this->process_single_push( $push_data, $headers );
				}
			} else {
				foreach ( $push_batch as $push ) {
					$is_silent                                     = isset( $push['data']['silent'] ) && $push['data']['silent'];
					$push_data['message']['notification']['title'] = html_entity_decode( $push['data']['primary_text'], ENT_QUOTES, 'UTF-8' );
					$push_data['message']['notification']['body']  = html_entity_decode( $push['data']['secondary_text'], ENT_QUOTES, 'UTF-8' );

					if ( ! empty( $push['data'] ) ) {
						foreach ( $push['data'] as $k => $v ) {
							if ( is_string( $v ) || is_int( $v ) ) {
								$push_data['message']['data'][ $k ] = (string) $v;
							}
						}
					}

					/**
					 * Set the silent notification data.
					 */
					if ( $is_silent && ! empty( $push['data']['silent_data'] ) ) {
						foreach ( $push['data']['silent_data'] as $k => $v ) {
							if ( is_string( $v ) || is_int( $v ) ) {
								$push_data['message']['data'][ $k ] = (string) $v;
							}
						}
					}

					if ( ! empty( $push['topic'] ) ) {
						$push_data['message']['topic'] = $this->notification_topic_format( $push['topic'] ); // Todo: we will implement later.
					}

					if ( ! empty( $push['device'] ) && 'firebase' === $push['device']['push_type'] ) {
						$push_data['message']['token'] = $push['device']['token'];
						$push_data                     = $this->set_user_badge_count( $push_data, $push['user_id'] );

						if ( $is_silent ) {
							$push_data = $this->set_silent_notification_data( $push_data );
						}

						$results                                     = $this->process_single_push( $push_data, $headers );
						$devices_results[ $push['device']['token'] ] = is_array( $results ) ? current( $results ) : $results;
					}
				}
			}
		}

		return $devices_results;
	}

	/**
	 * Set silent notification data.
	 * @param array $push_data Push data.
	 *
	 * @since 2.1.20
	 * @return array
	 */
	public function set_silent_notification_data( $push_data ) {
		unset( $push_data['message']['notification'] );
		unset( $push_data['message']['apns']['payload']['aps'] );

		$push_data['message']['apns']['payload']['aps']['content-available'] = 1;
		$push_data['message']['apns']['payload']['apns-priority']            = 5;

		return $push_data;
	}

	/**
	 * Set user badge count.
	 *
	 * @param array      $push_data Push data.
	 * @param int|string $user_id   User id.
	 *
	 * @since 1.7.6
	 *
	 * @return array
	 */
	public function set_user_badge_count( $push_data, $user_id ) {
		$push_data['message']['data']['badge_count']                           = 1;
		$push_data['message']['apns']['payload']['aps']['badge']               = 1;
		$push_data['message']['apns']['payload']['aps']['content-available']   = 1;
		$push_data['message']['android']['notification']['notification_count'] = 1;

		if ( function_exists( 'bbapp_notifications' ) ) {
			$notification_count = bbapp_notifications()->get_notification_count( $user_id );

			$push_data['message']['data']['badge_count']             = $notification_count;
			$push_data['message']['apns']['payload']['aps']['badge'] = (int) $notification_count;
		}

		return $push_data;
	}

	/**
	 * Process single push notification.
	 *
	 * @param array $data    Prepared push data.
	 * @param array $headers Push header.
	 *
	 * @since 1.7.6
	 *
	 * @return array|\WP_Error
	 */
	public function process_single_push( $data, $headers = array() ) {
		$response = $this->single_post( $data, $headers );
		if ( is_wp_error( $response ) ) {
			return new \WP_Error( 'error', __( 'There was an error while sending push notification through the Firebase API.', 'buddyboss-app' ) );
		}

		$response_body   = wp_remote_retrieve_body( $response );
		$response_result = json_decode( $response_body, true );

		$devices_results[ $data['message']['token'] ] = array( 'data' => $response_result );

		return $devices_results;
	}

	/**
	 * Process batch push notification.
	 *
	 * @param string $data    Prepared push html.
	 * @param array  $headers Push header.
	 * @param array  $devices Push devices.
	 *
	 * @since 1.7.6
	 *
	 * @return array|\WP_Error
	 */
	public function process_batch_push( $data, $headers = array(), $devices = array() ) {
		$response = $this->batch_post( $data, $headers );
		if ( is_wp_error( $response ) ) {
			return new \WP_Error( 'error', __( 'There was an error while sending push notification through the Firebase API.', 'buddyboss-app' ) );
		}

		// Get the response body as a string.
		$response_body = wp_remote_retrieve_body( $response );

		// Extract the JSON data from the response.
		$batch_responses = explode( '--batch_', $response_body );
		$json_responses  = array();

		foreach ( $batch_responses as $batch_response ) {
			if ( strpos( $batch_response, 'Content-Type: application/http' ) !== false ) {
				$json_response_start = strpos( $batch_response, '{' );
				$json_response_end   = strrpos( $batch_response, '}' );
				$json_response       = substr( $batch_response, $json_response_start, $json_response_end - $json_response_start + 1 );
				$json_responses[]    = json_decode( $json_response );
			}
		}
		$devices_results = array();
		if ( ! empty( $json_responses ) ) {
			foreach ( $devices as $key => $device ) {
				$devices_results[ $device ] = array( 'data' => $json_responses[ $key ] );
			}
		}

		return $devices_results;
	}

	/**
	 * Helper function to post data to Firebase.
	 *
	 * @param array $data    Push notification data.
	 * @param array $headers Push notification headers.
	 *
	 * @since 1.7.6
	 *
	 * @return array|\WP_Error
	 */
	public function single_post( $data, $headers = array() ) {
		$url                     = $this->get_firebase_endpoint();
		$headers['Content-Type'] = 'application/json';
		$data                    = wp_json_encode( $data );

		return wp_safe_remote_post(
			$url,
			array(
				'headers' => $headers,
				'body'    => $data,
				'method'  => 'POST',
			)
		);
	}

	/**
	 * Helper function to post data to Firebase.
	 *
	 * @param string $data    Push notification data.
	 * @param array  $headers Push notification headers.
	 *
	 * @since 1.7.6
	 *
	 * @return array|\WP_Error
	 */
	public function batch_post( $data, $headers = array() ) {
		$headers['Content-Type']   = 'multipart/mixed; boundary=subrequest_boundary';
		$headers['Content-Length'] = strlen( $data );

		return wp_remote_post(
			'https://fcm.googleapis.com/batch',
			array(
				'headers' => $headers,
				'body'    => $data,
				'method'  => 'POST',
			)
		);
	}

	/**
	 * Render setting method.
	 *
	 * @return void
	 */
	public function render_settings() {
		// TODO: Implement render_settings() method.
	}

	/**
	 * Is legacy support enable or not.
	 *
	 * @since 1.7.6
	 * @return bool
	 */
	public function is_support_legacy_enabled() {
		$firebase_admin_key = Configure::instance()->option( 'push.firebase_admin_key' );
		if ( empty( $firebase_admin_key ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Generate auth key.
	 *
	 * @since 1.7.6
	 * @return false|mixed
	 * @throws \Google_Exception Google Exception.
	 */
	public function generate_auth_key() {
		$json_file = $this->get_admin_key();
		if ( ! empty( $json_file ) && file_exists( $json_file ) ) {
			$client = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Client();
			$client->setAuthConfig( $json_file );
			$client->addScope( 'https://www.googleapis.com/auth/cloud-platform' );

			$access_token = $client->fetchAccessTokenWithAssertion();

			return $access_token['access_token'];
		}

		return false;
	}

	/**
	 * Validate auth token.
	 *
	 * @param string $token Current token.
	 *
	 * @since 1.7.6
	 *
	 * @return null|array|string
	 * @throws \Google_Exception Google Exception.
	 */
	public function validate_auth_token( $token ) {

		$json_file = $this->get_admin_key();
		if ( ! empty( $json_file ) && file_exists( $json_file ) ) {
			$client = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Client();
			$client->setAuthConfig( $json_file );
			$client->addScope( 'https://www.googleapis.com/auth/cloud-platform' );

			$client->setAccessToken( $token );

			if ( $client->isAccessTokenExpired() ) {
				$client->fetchAccessTokenWithAssertion();
			}

			return $client->getAccessToken();
		}

		return '';
	}
}
