<?php
/**
 * Push notification.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

// Exit if accessed directly.
use BuddyBossApp\Jobs;
use BuddyBossApp\Tools\Logger;
use BuddyBossApp\ManageApp;
use BuddyBossApp\UserSegment;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Push' ) ) :
	/**
	 * Main push notification class.
	 *
	 * Class Push
	 */
	class Push {

		/**
		 * The single instance of the class.
		 *
		 * @var null $instance
		 */
		private static $instance;

		/**
		 * Push notification queue table name.
		 *
		 * @var string $push_queue_table
		 */
		private $push_queue_table;

		/**
		 * Users can perform per batch on manual push.
		 *
		 * @var int $queue_users_per_manual_push_batch
		 */
		private $queue_users_per_manual_push_batch = 0; // how many user to perform per batch on manual push.

		/**
		 * Get the instance of class
		 *
		 * @return Push
		 */
		public static function instance() {
			if ( ! isset( self::$instance ) ) {
				$class          = __CLASS__;
				self::$instance = new $class();
				self::$instance->load();
			}

			return self::$instance;
		}

		/**
		 * Class construct
		 */
		public function __construct() {
		}

		/**
		 * Load push notification hooks.
		 */
		public function load() {

			/**
			 * Filter to use to change queue users per manual push batch.
			 *
			 * @param int $per_batch Per batch.
			 *
			 * @since 1.7.7
			 */
			$this->queue_users_per_manual_push_batch = (int) apply_filters( 'bbapp_queue_users_per_manual_push_batch', 200 );
			$this->push_queue_table = bbapp_get_network_table( 'bbapp_push_queue' );
			add_action( 'bbapp_queue_task_push_batch', array( $this, 'do_push_batch' ) );
			add_action( 'bbapp_queue_task_process_manual_push', array( $this, 'do_process_manual_push' ) );
			// Run once a day to clear expire device tokens.
			add_action( 'bbapp_every_day', array( $this, 'delete_expire_device_tokens' ) );
			add_action( 'bbapp_every_day', array( $this, 'delete_sent_notification' ) );
		}

		/**
		 * Delete the expire device token.
		 * This suppose to run every day.
		 */
		public function delete_expire_device_tokens() {
			global $wpdb;

			$table = bbapp_get_network_table( 'bbapp_user_devices' );

			/**
			 * Filters the device expiry delete duration.
			 *
			 * @param int $device_expiry Device expiry time duration in days.
			 *
			 * @since 1.7.7
			 */
			$device_expiry = apply_filters( 'bbapp_device_expiry', 90 ); // Days.

			$device_expiry            = ( ! empty( $device_expiry ) && $device_expiry > 0 ) ? (int) $device_expiry : absint( $device_expiry );
			$bbapp_device_expiry_days = "-$device_expiry days";
			$date                     = gmdate( 'Y-m-d H:i:s' );
			$expiry_date              = gmdate( 'Y-m-d H:i:s', strtotime( $bbapp_device_expiry_days, strtotime( $date ) ) );
			$device_tokens_deleted    = $wpdb->query( $wpdb->prepare( "DELETE FROM {$table} WHERE date_updated < %s && user_id != 0", $expiry_date ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( ! empty( $device_tokens_deleted ) ) {
				/* translators: %s: device token*/
				Logger::instance()->add( 'info_log', sprintf( esc_html__( 'Deleted %s expired device tokens.', 'buddyboss-app' ), $device_tokens_deleted ) );
			}

			/**
			 * Filters guest device expiry delete duration.
			 *
			 * @param int $guest_device_expiry Guest device expiry time duration in days.
			 *
			 * @since 1.7.7
			 */
			$guest_device_expiry = apply_filters( 'bbapp_guest_device_expiry', 90 ); // Days.

			$guest_device_expiry         = ( ! empty( $guest_device_expiry ) && $guest_device_expiry > 0 ) ? (int) $guest_device_expiry : absint( $guest_device_expiry );
			$guest_device_expiry_days    = "-$guest_device_expiry days";
			$guest_expiry_date           = gmdate( 'Y-m-d H:i:s', strtotime( $guest_device_expiry_days, strtotime( $date ) ) );
			$guest_device_tokens_deleted = $wpdb->query( $wpdb->prepare( "DELETE FROM {$table} WHERE date_updated < %s && user_id = 0", $guest_expiry_date ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( ! empty( $guest_device_tokens_deleted ) ) {
				/* translators: %s: device token*/
				Logger::instance()->add( 'info_log', sprintf( esc_html__( 'Deleted %s expired guest device tokens.', 'buddyboss-app' ), $guest_device_tokens_deleted ) );
			}
		}

		/**
		 * Delete sent notifications from queue.
		 */
		public function delete_sent_notification() {
			global $wpdb;

			$delete = $wpdb->delete( $this->push_queue_table, array( 'sent' => 1 ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			if ( ! empty( $delete ) ) {
				/* translators: %s: device token*/
				Logger::instance()->add( 'info_log', sprintf( esc_html__( 'Deleted %s sent push notifications in queue.', 'buddyboss-app' ), $delete ) );
			}

			/**
			 * Filters the notification expiry delete duration.
			 *
			 * @param int $days Device expiry time duration in days.
			 *
			 * @since 2.1.70
			 */
			$days = apply_filters( 'bbapp_unsent_notification_days', 1 ); // Days.

			$bbapp_unsent_notification_days = "-$days days";
			$date                           = gmdate( 'Y-m-d H:i:s' );
			$expiry_date                    = gmdate( 'Y-m-d H:i:s', strtotime( $bbapp_unsent_notification_days, strtotime( $date ) ) );

			// Delete unsent push notifications after 1 day.
			$delete = $wpdb->query( $wpdb->prepare( "DELETE FROM {$this->push_queue_table} WHERE sent=0 AND created < %s", $expiry_date ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( ! empty( $delete ) ) {
				/* translators: %s: device token*/
				Logger::instance()->add( 'info_log', sprintf( esc_html__( 'Deleted %s unsent push notifications in queue.', 'buddyboss-app' ), $delete ) );
			}

			$queue_count = $wpdb->get_var( "SELECT COUNT(id) FROM $this->push_queue_table" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( ! $queue_count ) {
				$wpdb->query( "TRUNCATE TABLE $this->push_queue_table" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				/* translators: %1ss: Push queue table name.*/
				Logger::instance()->add( 'info_log', sprintf( esc_html__( 'Truncated %1$s due to no entries', 'buddyboss-app' ), $this->push_queue_table ) );
			}
		}

		/**
		 * Sends the push notification to device.
		 * it's preferable to only pass limited no. of users eg. 500 each request.
		 *
		 * @param array $notification  {
		 *                             'primary_text'         (required) Primary Text
		 *                             'secondary_text'       (required) Secondary Text
		 *                             'user_ids'             (required) Users ID to send Push Notifications.
		 *                             'device_ids'           (required) Users device IDs to send Push Notifications.
		 *                             'sent_as'              (optional) default 1,
		 *                             'data'                 (optional) default array(),
		 *                             'agent'                (optional) default 'event',
		 *                             'topic'                (optional) default '',
		 *                             'priority'             (optional) default 9,
		 *                             'blog_id'              (optional) default get_current_blog_id(),
		 *                             'type'                 (optional) default 'manual_push_notification' Push Type
		 *                             'subscription_type'    (optional) default 'manual_push_notification' replaced by push_type, deprecated 1.3.3
		 *                             'push_notification_id' (optional) // Notification Push ID
		 *                             'normal_notification'  (optional) // true or false
		 *                             'normal_notification_data'  (optional) // Param should be same as send_normal() function.
		 *                             }.
		 *
		 * @return WP_Error | bool
		 */
		public function send( $notification ) {
			global $wpdb;
			$defaults = array(
				'primary_text'                 => '',
				'secondary_text'               => '',
				'user_ids'                     => array(), // User ids.
				'device_ids'                   => array(), // Device table ids.
				'sent_as'                      => 1,
				'data'                         => array(), // support link, site_id etc.
				'topic'                        => '',
				'priority'                     => 9,
				'blog_id'                      => get_current_blog_id(),
				'push_notification_id'         => false,
				'item_id'                      => false,
				'secondary_item_id'            => false,
				'type'                         => 'custom', // custom type is a default type.
				'filter_users_by_subscription' => true,
				'normal_notification'          => false,
				'normal_notification_data'     => array(
					'component_name'    => 'bbapp', // will be used for normal push.
					'component_action'  => 'manual_push', // Required.
					'item_id'           => false,
					'secondary_item_id' => false,
				),
			);
			// @deprecated
			if ( 'manual_push_notification' === $notification['type'] ) {
				$defaults['agent'] = 'manual';
			} else {
				$defaults['agent'] = 'event';
			}

			/**
			 * Validate Link if it's not valid Remove it.
			 */
			if ( isset( $notification['data'] ) && isset( $notification['data']['link'] ) && ! bbapp_is_valid_url( $notification['data']['link'] ) ) {
				unset( $notification['data']['link'] );
			}

			// When normal_notification_data is empty array then unset it so default can be used.
			if ( empty( $notification['normal_notification_data'] ) ) {
				unset( $notification['normal_notification_data'] );
			}

			$notification = wp_parse_args( $notification, $defaults );

			/**
			 * Deprecated param support.
			 */
			if ( ! empty( $notification['subscription_type'] ) ) {
				$notification['type'] = $notification['subscription_type'];
			}

			// When Push Notification ID is given & normal_notification is true then add Notification ID as item_id and sender id as secondary_item_id.
			if ( ! empty( $notification['push_notification_id'] ) && $notification['normal_notification'] ) {
				$notification['normal_notification_data']['item_id']           = $notification['push_notification_id'];
				$notification['normal_notification_data']['secondary_item_id'] = isset( $notification['sent_as'] ) ? $notification['sent_as'] : 0;
			}

			/**
			 * Use this flag to filter push notification to sent to user's devices.
			 *
			 * @param bool  $flag         Send push notification.
			 * @param array $notification Notification data.
			 */
			$flag = apply_filters( 'send_notification_send_push', true, $notification );

			if ( ! $flag ) {
				return false;
			}

			// Topic or users one of information is required.
			if ( empty( $notification['topic'] ) && ( empty( $notification['user_ids'] ) && empty( $notification['device_ids'] ) ) ) {
				return new WP_Error( 'error', __( 'No Users Selected.', 'buddyboss-app' ) );
			}

			/**
			 * Sanitize & Validate the user_ids array
			 */
			if ( is_array( $notification['user_ids'] ) ) {
				$fixed = array();
				foreach ( $notification['user_ids'] as $uid ) {
					if ( (int) $uid > 0 ) {
						$fixed[] = $uid;
					}
				}
				$notification['user_ids'] = $fixed;
			}

			/**
			 * Sending to users id.
			 */
			$devices_ids = array();
			if ( ! empty( $notification['user_ids'] ) || ! empty( $notification['device_ids'] ) ) {
				/**
				 * Get devices based on user ids.
				 */
				if ( ! empty( $notification['user_ids'] ) ) {
					// Filter out the user based on subscription settings they have.
					if ( true === $notification['filter_users_by_subscription'] ) {
						$notification['user_ids'] = Notification::instance()->filter_user_ids_by_subscription( $notification['user_ids'], $notification['type'] );
					}

					$user_devices = $this->get_devices_for_users( $notification['user_ids'], '1.0.0' );

					if ( empty( $user_devices ) ) {
						return new WP_Error( 'error', __( 'Sorry selected users found to have no registered device or they are not subscribed to notifications.', 'buddyboss-app' ) );
					}
				}

				/**
				 * Get devices based on device ids.
				 */
				if ( ! empty( $notification['device_ids'] ) ) {

					/**
					 * Sending to device ids.
					 */
					$devices_ids = $this->get_devices_by_ids( $notification['device_ids'], '1.0.0' );

					if ( empty( $devices_ids ) ) {
						return new WP_Error( 'error', __( 'Sorry selected users found to have no registered device or they are not subscribed to notifications.', 'buddyboss-app' ) );
					}
				}
			} else {

				if ( empty( $notification['topic'] ) ) {
					return new WP_Error( 'error', __( 'Error sending push - Not a valid recipient method given.', 'buddyboss-app' ) );
				}
			}

			$entries = array();

			if ( isset( $user_devices ) || ! empty( $devices_ids ) || ! empty( $notification['topic'] ) ) {
				$notification['data']['site_id']           = $notification['blog_id'];
				$notification['data']['item_id']           = ! empty( $notification['item_id'] ) ? $notification['item_id'] : $notification['normal_notification_data']['item_id'];
				$notification['data']['secondary_item_id'] = $notification['secondary_item_id'];

				// Creates the normal notification.
				$notification_ids = $this->process_normal_notification( $notification );

				$push_data = array(
					'primary_text'         => $notification['primary_text'],
					'secondary_text'       => $notification['secondary_text'],
					'sent_as'              => $notification['sent_as'],
					'data'                 => $notification['data'],
					'agent'                => $notification['agent'],
					'type'                 => $notification['type'],
					'topic'                => $notification['topic'],
					'blog_id'              => $notification['blog_id'],
					'push_notification_id' => $notification['push_notification_id'],
					'normal_notification'  => $notification['normal_notification'],
				);

				if ( empty( $notification['topic'] ) ) {
					$insert_query = "INSERT INTO {$this->push_queue_table} (`n_id`,`created`, `device`, `unique_hash`, `data`, `priority`, `user_id`, `type`, `agent`, `sent`) VALUES ";

					if ( ! empty( $user_devices ) ) {
						// Separate device token by app & users.
						$group_user_devices = $this->get_grouped_device_tokens( $user_devices );

						/**
						 * Here we queue notifications based on there users.
						 * One user can have ser devices ID.
						 */
						foreach ( $group_user_devices as $user_id => $devices ) {
							foreach ( $devices as $device ) {
								// Assign normal notification id in push notification.
								if ( isset( $notification_ids[ $user_id ] ) ) {
									$push_data['data']['notification_id'] = $notification_ids[ $user_id ];
								}

								$entries[] = array(
									$notification['push_notification_id'], // n_id.
									current_time( 'mysql', 1 ), // created.
									maybe_serialize( $device ), // device.
									$this->get_push_unique_hash( $notification, $device['push_type'] ), // unique_hash.
									maybe_serialize( $push_data ), // data.
									$notification['priority'], // priority.
									$user_id, // user_id.
									$notification['type'], // type.
									$notification['agent'], // agent.
									0, // sent.
								);
							}
						}
					}

					if ( ! empty( $devices_ids ) ) {
						// Separate device token by app & id.
						$group_devices_ids = $this->get_grouped_device_tokens_by_ids( $devices_ids );
						/**
						 * Here we queue notifications based on there users.
						 * One user can have ser devices ID.
						 */
						foreach ( $group_devices_ids as $id => $devices ) {
							foreach ( $devices as $device ) {
								$entries[] = array(
									$notification['push_notification_id'], // n_id.
									current_time( 'mysql', 1 ), // created.
									maybe_serialize( $device ), // device.
									$this->get_push_unique_hash( $notification, $device['push_type'] ), // unique_hash.
									maybe_serialize( $push_data ), // data.
									$notification['priority'], // priority.
									0, // user_id.
									$notification['type'], // type.
									$notification['agent'], // agent.
									0, // sent.
								);
							}
						}
					}
					$prepare_var = array();

					$query_row_placeholders = array();
					foreach ( $entries as $entry ) {
						foreach ( $entry as $e ) {
							$prepare_var[] = $e;
						}
						$query_row_placeholders[] = '(%d ,%s ,%s ,%s ,%s ,%d ,%d ,%s,%s, %d)';
					}
					$insert_query .= implode( ",\n ", $query_row_placeholders );

					if ( ! $wpdb->query( $wpdb->prepare( $insert_query, $prepare_var ) ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
						return new WP_Error( 'error', __( 'Error Sending Notifications : DB Error.', 'buddyboss-app' ) );
					}
				} else {

					$get_selected_push_service = bbapp_get_app_push_instance();

					if ( $get_selected_push_service ) {

						$add = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
							$this->push_queue_table,
							array(
								'created'     => current_time( 'mysql', 1 ),
								'device'      => $notification['topic'],
								'unique_hash' => $this->get_push_unique_hash( $notification, $get_selected_push_service->push_type() ),
								'data'        => maybe_serialize( $push_data ),
								'priority'    => $notification['priority'],
								'user_id'     => 0, // 0 means topic.
								'agent'       => $notification['agent'],
								'type'        => $notification['type'],
								'sent'        => 0,
							)
						);

						if ( ! $add ) {
							return new WP_Error( 'error', __( 'Error Sending Notifications : DB Error.', 'buddyboss-app' ) );
						}
					}
				}

				// make sure there are enough jobs tasked.
				$this->queue_require_jobs();

				return true;
			}

			return new WP_Error( 'error', __( 'Error Sending Notifications.', 'buddyboss-app' ) );
		}

		/**
		 * Create the normal notification for push being sent to user.
		 *
		 * @param array $notification Notification items.
		 *
		 * @return bool
		 */
		public function process_normal_notification( $notification ) {

			if ( isset( $notification['normal_notification'] ) && $notification['normal_notification'] ) {

				$default_push_data = array(
					'component_name'    => false,
					'component_action'  => false,
					'item_id'           => false,
					'secondary_item_id' => false,
					'user_ids'          => $notification['user_ids'],
				);

				$normal_push_data = ( isset( $notification['normal_notification_data'] ) ) ? $notification['normal_notification_data'] : array();

				if ( ! isset( $normal_push_data ) ) {
					$normal_push_data = array();
				}

				$normal_push_data = wp_parse_args( $normal_push_data, $default_push_data );

				if ( isset( $notification['push_notification_id'] ) ) {
					$default_push_data['item_id']           = $notification['push_notification_id']; // notification_id.
					$default_push_data['secondary_item_id'] = $notification['sent_as']; // user_id.
				}

				/**
				 * Validating Required Data.
				 */

				if (
					empty( $normal_push_data['component_action'] ) ||
					empty( $normal_push_data['component_name'] )
				) {
					return false;
				}

				if ( ! empty( $notification['topic'] ) ) {
					return false; // support no global for now.
				}

				return Notification::instance()->create_notification( $normal_push_data );
			}
			return false;
		}

		/**
		 * Returns a unique hash for a push notification. later help for grouping push.
		 *
		 * @param array  $notification Nitification items.
		 * @param string $push_type    Nitification type.
		 *
		 * @return string
		 */
		public function get_push_unique_hash( $notification, $push_type ) {
			return sha1(
				$notification['sent_as'] .
				maybe_serialize( $notification['data'] ) .
				$notification['secondary_text'] .
				$notification['primary_text'] .
				$notification['agent'] .
				$push_type
			);
		}

		/**
		 * Return the device tokens in exchange to user ids.
		 *
		 * @param array  $user_ids        Users id.
		 * @param string $min_app_version Minimum app version.
		 *
		 * @return array|bool
		 */
		protected function get_devices_for_users( $user_ids, $min_app_version = '1.0.0' ) {

			if ( empty( $user_ids ) ) {
				return false;
			}

			$fixed = array();
			foreach ( $user_ids as $uid ) {
				if ( (int) $uid > 0 ) {
					$fixed[] = (int) $uid;
				}
			}

			$user_ids = $fixed;

			global $wpdb;

			$get_selected_push_service = bbapp_get_app_push_instance();
			$push_type                 = $get_selected_push_service->push_type();

			$table = bbapp_get_network_table( 'bbapp_user_devices' );

			$sql = "SELECT * FROM {$table} WHERE 1=1";

			// Preparing the vars in serial way we used in prepare query.

			if ( ! empty( $user_ids ) ) {
				$user_id_in_placeholders = implode( ', ', array_fill( 0, count( $user_ids ), '%d' ) );
				$prepare_vars            = array();
				foreach ( $user_ids as $user_id ) {
					$prepare_vars[] = $user_id;
				}

				$sql .= $wpdb->prepare( " AND user_id IN ({$user_id_in_placeholders})", $prepare_vars ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}

			if ( ! empty( $push_type ) ) {
				$sql .= $wpdb->prepare( ' AND push_type = %s', $push_type );
			}

			$userdevices = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			if ( ! empty( $userdevices ) && ! is_wp_error( $userdevices ) ) {

				$filtered_devices = array();

				/**
				 * Filter the device token based on there App Version.
				 * These is needed to know target device app version support particular push notification.
				 */
				foreach ( $userdevices as $userdevice ) {
					if ( empty( $userdevice->app_ver ) ) {
						$userdevice->app_ver = '1.0.0';
					}
					// Token version should be greater or equal then $min_app_version.
					if ( version_compare( $userdevice->app_ver, $min_app_version ) >= 0 ) {
						$filtered_devices[] = $userdevice;
					}
				}

				return $filtered_devices;
			}

			return false;
		}

		/**
		 * Return the device tokens in exchange to non user ids.
		 *
		 * @param array  $device_ids        Users id.
		 * @param string $min_app_version Minimum app version.
		 *
		 * @return array|bool
		 */
		protected function get_devices_by_ids( $device_ids, $min_app_version = '1.0.0' ) {
			global $wpdb;

			$get_selected_push_service = bbapp_get_app_push_instance();
			$push_type                 = $get_selected_push_service->push_type();

			$table = bbapp_get_network_table( 'bbapp_user_devices' );

			$sql = "SELECT * FROM {$table} WHERE 1=1";

			// Preparing the vars in serial way we used in prepare query.
			if ( ! empty( $device_ids ) ) {
				$user_id_in_placeholders = implode( ', ', array_fill( 0, count( $device_ids ), '%s' ) );
				$prepare_vars            = array();
				foreach ( $device_ids as $device_id ) {
					$prepare_vars[] = $device_id;
				}

				$sql .= $wpdb->prepare( " AND id IN ({$user_id_in_placeholders})", $prepare_vars ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}

			if ( ! empty( $push_type ) ) {
				$sql .= $wpdb->prepare( ' AND push_type = %s', $push_type );
			}

			$userdevices = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			if ( ! empty( $userdevices ) && ! is_wp_error( $userdevices ) ) {

				$filtered_devices = array();

				/**
				 * Filter the device token based on there App Version.
				 * These is needed to know target device app version support particular push notification.
				 */
				foreach ( $userdevices as $userdevice ) {
					if ( empty( $userdevice->app_ver ) ) {
						$userdevice->app_ver = '1.0.0';
					}
					// Token version should be greater or equal then $min_app_version.
					if ( version_compare( $userdevice->app_ver, $min_app_version ) >= 0 ) {
						$filtered_devices[] = $userdevice;
					}
				}

				return $filtered_devices;
			}

			return false;
		}

		/**
		 * Check if there is any pending push to send is left.
		 *
		 * @return int
		 */
		public function has_pending_push() {
			global $wpdb;
			$count = $wpdb->get_var( "SELECT count(id) FROM {$this->push_queue_table} WHERE sent != 1" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			return (int) $count;
		}


		/**
		 * Group device tokens by user_id.
		 *
		 * @param array $devices User device information.
		 *
		 * @return mixed
		 */
		public function get_grouped_device_tokens( $devices ) {
			foreach ( $devices as $device ) {
				$group_devices[ $device->user_id ][] = array(
					'token'        => $device->device_token,
					'platform'     => $device->platform,
					'push_type'    => $device->push_type,
					'data'         => maybe_unserialize( $device->data ),
					'device_id'    => $device->device_id,
					'device_model' => $device->device_model,
				);
			}

			return $group_devices;
		}

		/**
		 * Group device tokens by id.
		 *
		 * @param array $devices User device information.
		 *
		 * @return mixed
		 */
		public function get_grouped_device_tokens_by_ids( $devices ) {
			foreach ( $devices as $device ) {
				$group_devices[ $device->id ][] = array(
					'token'        => $device->device_token,
					'platform'     => $device->platform,
					'push_type'    => $device->push_type,
					'data'         => maybe_unserialize( $device->data ),
					'device_id'    => $device->device_id,
					'device_model' => $device->device_model,
				);
			}

			return $group_devices;
		}


		/**
		 * Get the push batch.
		 * This get the batch based on unique group. & only one max push group will be returned.
		 *
		 * @param int  $max       max group to be returned.
		 *
		 * @param bool $group     default true. use this if you want to group push by there content.
		 * @param bool $mark_sent default true, this is used to mark returned notification as sent so they can't be used again.
		 *
		 * @return mixed
		 */
		public function get_push_batch( $max = 500, $group = true, $mark_sent = true ) {
			global $wpdb;

			if ( $group ) {

				$hash_with_most_push = $wpdb->get_var( "SELECT unique_hash FROM {$this->push_queue_table} WHERE sent!=1 GROUP BY unique_hash order by count(id) desc LIMIT 1" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				if ( $hash_with_most_push ) {
					$get_push = $wpdb->get_results( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

						$wpdb->prepare(
							"SELECT *FROM {$this->push_queue_table} WHERE sent!=1 && unique_hash=%s ORDER BY priority, id desc LIMIT %d", //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
							$hash_with_most_push,
							$max
						)
					); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

					if ( $mark_sent && ! empty( $get_push ) ) {
						$wpdb->query( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
							$wpdb->prepare(
								"UPDATE {$this->push_queue_table} SET sent=1 WHERE sent!=1 && unique_hash=%s ORDER BY priority, id desc LIMIT %d", //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
								$hash_with_most_push,
								$max
							)
						);
					}
				} else {
					return false;
				}
			} else {

				$get_push = $wpdb->get_results( $wpdb->prepare( "SELECT id, device, user_id, type, data FROM {$this->push_queue_table} WHERE sent!=1 ORDER BY priority, id desc LIMIT %d", $max ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				if ( $mark_sent && ! empty( $get_push ) ) {
					$wpdb->query( $wpdb->prepare( "UPDATE {$this->push_queue_table} SET sent=1 WHERE sent!=1 ORDER BY priority, id desc LIMIT %d", $max ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				}
			}

			// Convert to Array.
			foreach ( $get_push as $k => $v ) {
				$get_push[ $k ]           = (array) $v;
				$get_push[ $k ]['device'] = maybe_unserialize( $get_push[ $k ]['device'] );
				$get_push[ $k ]['data']   = maybe_unserialize( $get_push[ $k ]['data'] );
			}

			return $get_push;
		}


		/**
		 * Changes the status of notification.
		 *
		 * @param int  $push_notification_id Notification id.
		 * @param bool $status               Notification status.
		 *
		 * @return false|int
		 */
		public function change_notification_status( $push_notification_id, $status ) {
			return $this->update_notification(
				$push_notification_id,
				array(
					'status' => $status,
				)
			);
		}

		/**
		 * Add Notification Log.
		 *
		 * @param int    $push_notification_id Notification id.
		 * @param string $text                 Notification log.
		 *
		 * @return bool|false|int
		 */
		public function add_notification_log( $push_notification_id, $text ) {
			$notification = $this->get_notification( $push_notification_id );
			if ( ! $notification ) {
				return false;
			}
			if ( ! isset( $notification->data['logs'] ) ) {
				$notification->data['logs'] = array();
			}
			$notification->data['logs'][] = array(
				'log_text' => $text,
				'log_time' => current_time( 'mysql', 1 ),
			);

			return $this->update_notification(
				$push_notification_id,
				array(
					'data' => $notification->data,
				)
			);
		}

		/**
		 * Return the notification.
		 *
		 * @param int $push_notification_id Notification id.
		 *
		 * @return array|bool|null|object|void
		 */
		public function get_notification( $push_notification_id ) {
			global $wpdb;
			static $cache;

			// Return the cache.
			if ( is_array( $cache ) && isset( $cache[ $push_notification_id ] ) ) {
				return $cache[ $push_notification_id ];
			}

			$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );
			$get        = $wpdb->get_row( $wpdb->prepare( "SELECT *FROM {$table_name} WHERE id=%d", $push_notification_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			if ( empty( $get ) ) {
				return false;
			}

			$get->data = maybe_unserialize( $get->data );
			if ( ! is_array( $get->data ) ) {
				$get->data = array();
			}

			$cache[ $push_notification_id ] = $get;

			return $get;
		}

		/**
		 * Update Notification.
		 *
		 * @param int   $push_notification_id Notification id.
		 * @param array $data                 Notification items.
		 *
		 * @return false|int
		 */
		public function update_notification( $push_notification_id, $data ) {
			global $wpdb;

			$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

			if ( ManageApp::instance()->get_app_setting( 'app_maintenance_mode' ) ) {
				$data['data']['logs'][] = array(
					'log_text' => __( 'App is in maintenance mode. Push notification will not be sent.', 'buddyboss-app' ),
					'log_time' => current_time( 'mysql', 1 ),
				);
			}

			if ( isset( $data['data'] ) && is_array( $data['data'] ) ) {
				$data['data'] = maybe_serialize( $data['data'] );
			}

			return $wpdb->update( $table_name, $data, array( 'id' => $push_notification_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		}

		/**
		 * Delete push notification.
		 *
		 * @param int $push_notification_id Notificatoin id.
		 *
		 * @return bool|false|int
		 */
		public function delete_notification( $push_notification_id ) {
			global $wpdb;

			$notification = $this->get_notification( $push_notification_id );

			if ( empty( $notification ) ) {
				return false;
			}

			if ( 'processing' === $notification->status ) {
				return false;
			}

			$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

			return $wpdb->delete( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
				"{$table_name}",
				array( 'id' => $push_notification_id ),
				array( '%d' )
			);
		}

		/**
		 * Starts Push Delivery of Push Notification.
		 * This functions queues the push.
		 * This function uses the smart way to queue big amount of users. by batching them into small amounts and running
		 * back on background jobs.
		 *
		 * @param int $push_notification_id Notification id.
		 *
		 * @return bool
		 */
		public function do_push_delivery( $push_notification_id ) {
			global $wpdb;

			$push_notification_id = (int) $push_notification_id;

			$get = $this->get_notification( $push_notification_id );

			if ( empty( $get ) ) {
				return false;
			}

			$primary_text   = $get->primary_text;
			$secondary_text = $get->secondary_text;
			$target_type    = $get->target_type;
			$sent_as        = $get->sent_as;
			$data           = $get->data;
			$type           = $get->agent; // it's same later push notification type.
			$status         = $get->status;
			$date_updated   = $get->date_updated;
			$blog_id        = $get->blog_id;
			$is_schedule    = $get->is_schedule;
			$date_schedule  = $get->date_schedule;

			/**
			 * Only Pending & Processing Continue.
			 */
			if ( ! in_array( $status, array( 'processing', 'pending' ), true ) ) {
				return false;
			}

			/**
			 * If schedule date is greater then now.
			 */
			if ( $is_schedule && strtotime( $date_schedule ) > time() ) {
				return false;
			}

			$push_data                = ( isset( $data['push_data'] ) && is_array( $data['push_data'] ) ) ? $data['push_data'] : array();
			$normal_notification_data = ( isset( $data['normal_notification_data'] ) && is_array( $data['normal_notification_data'] ) ) ? $data['normal_notification_data'] : array();
			$push_topic               = ( isset( $data['push_topic'] ) && is_array( $data['push_topic'] ) ) ? $data['push_topic'] : '';

			$segment_id = ( isset( $data['segment_id'] ) ) ? $data['segment_id'] : false;
			if ( 'all_users' !== $target_type && empty( $segment_id ) ) {
				$this->add_notification_log( $push_notification_id, __( 'Sending failed because segment ID was not found.', 'buddyboss-app' ) );
				$this->change_notification_status( $push_notification_id, 'failed' );
				return false;
			}

			$notification_args = array(
				'primary_text'             => $primary_text,
				'secondary_text'           => $secondary_text,
				'user_ids'                 => array(),
				'sent_as'                  => $sent_as,
				'data'                     => $push_data,
				'type'                     => $type,
				'item_id'                  => $push_notification_id,
				'secondary_item_id'        => $sent_as,
				'topic'                    => $push_topic,
				'blog_id'                  => get_current_blog_id(),
				'push_notification_id'     => $push_notification_id,
				'normal_notification'      => ( isset( $data['normal_notification'] ) ) ? $data['normal_notification'] : false,
				'normal_notification_data' => $normal_notification_data,
			);
			$segments          = UserSegment::instance();

			if ( 'all_users' !== $target_type ) {
				$_segment_users = $segments->get_users( $segment_id, true );
				if ( empty( $_segment_users ) ) {
					$this->add_notification_log( $push_notification_id, __( 'No user found in user segment.', 'buddyboss-app' ) );
					$this->change_notification_status( $push_notification_id, 'failed' );

					return false;
				}
			}
			if ( ! isset( $data['queue_users_fetch_page'] ) ) {
				$data['queued_users']           = 0;
				$data['queue_users_fetch_page'] = 0;
			}

			if ( ! isset( $data['queue_users_fetch_limit'] ) ) {
				$data['queue_users_fetch_limit'] = $this->queue_users_per_manual_push_batch;
			}

			$data['queue_users_fetch_page'] ++;

			if ( 'all_users' !== $target_type ) {
				$fetch_users     = $segments->get_users( $segment_id, $data['queue_users_fetch_page'], $data['queue_users_fetch_limit'], true );
				$data['send_to'] = $segments->get_total_users( $segment_id, true );
			} else {
				$fetch_users     = $this->get_all_users( $data['queue_users_fetch_page'], $data['queue_users_fetch_limit'] );
				$data['send_to'] = $fetch_users['total'];
				if ( isset( $fetch_users['users'] ) && is_array( $fetch_users['users'] ) ) {
					$fetch_users = $fetch_users['users'];
				} else {
					$fetch_users = array();
				}
			}

			// Platform user setting manage manually notification.
			foreach ( $fetch_users as $key => $user_id ) {
				if ( function_exists( 'bp_is_labs_notification_preferences_support_enabled' ) && bp_is_labs_notification_preferences_support_enabled() ) {
					if ( 'no' === bp_get_user_meta( $user_id, 'notification_app_push', true ) || 'no' === bp_get_user_meta( $user_id, 'enable_notification_app', true ) ) {
						unset( $fetch_users[ $key ] );
					}
				}
			}

			$fetch_users = array_values( $fetch_users );

			/**
			 * Check if there are users to process or not.
			 */
			if ( empty( $fetch_users ) || $data['queued_users'] >= $data['send_to'] ) {
				if ( empty( $data['queued_users'] ) ) { // When queued user is also empty it's means segment didn't had users.
					$this->add_notification_log( $push_notification_id, __( 'No user found in user to send push.', 'buddyboss-app' ) );
					$this->change_notification_status( $push_notification_id, 'failed' );

					return false;
				} else { // else it's means current page don't have. but there already been queued on earlier jobs of this function.
					$count = $this->get_push_notification_queue_count( $push_notification_id );
					if ( empty( $count ) ) {
						$this->add_notification_log( $push_notification_id, __( 'All push notifications have been sent.', 'buddyboss-app' ) );
						$this->change_notification_status( $push_notification_id, 'sent' );
					} else {
						$jobs = Jobs::instance();
						// add the job.
						$jobs->add( 'process_manual_push', array( 'push_notification_id' => $push_notification_id ), 2 );
						$jobs->start();
						$this->queue_require_jobs();
					}

					return true;
				}
			}

			$notification_args['user_ids'] = $fetch_users;

			$send = $this->send( $notification_args );

			$data['queued_users'] += count( $fetch_users );

			// Update the notification data.
			$this->update_notification(
				$push_notification_id,
				array(
					'data' => $data,
				)
			);

			$jobs = Jobs::instance();
			// add the job.
			$jobs->add( 'process_manual_push', array( 'push_notification_id' => $push_notification_id ), 2 );
			$jobs->start();

			if ( is_wp_error( $send ) ) {
				return $send;
			}

			return true;
		}

		/**
		 * Make sure queue exists in jobs when there is queue in push list.
		 * Here we put randomly 1o jobs for each App ID. So each of them can get there required time.
		 */
		public function queue_require_jobs() {
			global $wpdb;

			$jobs         = Jobs::instance();
			$queued       = $jobs->has_process_items( 'push_batch' );
			$pending_push = $this->has_pending_push();

			if ( empty( $queued ) && ! empty( $pending_push ) ) {
				$jobs->add( 'push_batch' );
				$jobs->start();
			}
		}

		/**
		 * Method Executed when push_task is preformed on background jobs.
		 *
		 * @param object $task Push notification queue.
		 *
		 * @return bool
		 */
		public function do_push_batch( $task ) {
			// all push sending code should be here.
			maybe_unserialize( $task->data );

			$instance = bbapp_get_app_push_instance();

			if ( ! $instance ) {
				Logger::instance()->add( 'info_log', __( 'Skipping Push - as push is not configured.', 'buddyboss-app' ) );
				return false;
			}

			$do_group = $instance->do_group();

			/**
			 * Filter to use to change batch count.
			 *
			 * @param int $per_batch Per batch.
			 *
			 * @since 1.7.7
			 */
			$max_per_batch = (int) apply_filters( 'bbapp_max_per_batch_count', $instance->max_per_batch() );

			/**
			 * Filter to use to change batch loop.
			 *
			 * @param int $loop Loop.
			 *
			 * @since 2.2.10
			 */
			$loop = apply_filters( 'bbapp_push_batch_loop', 1 );

			for ( $i = 0; $i < $loop; $i++ ) {
				$push_batch = $this->get_push_batch( $max_per_batch, $do_group );

				$instance->service_send_push( $push_batch );
			}

			$this->queue_require_jobs();
		}

		/**
		 * This job execute the pending jobs task for manual push notification.
		 * In manual push we queue notification per 500 users on each batch.
		 * Why we do this is just for scaling purpose on low level WordPress Environment.
		 *
		 * @param object $task Push notification queue.
		 */
		public function do_process_manual_push( $task ) {

			// all push sending code should be here.
			$task_data = maybe_unserialize( $task->data );

			$push_notification_id = $task_data['push_notification_id'];

			if ( ! empty( $push_notification_id ) ) {
				$this->do_push_delivery( $push_notification_id );
			}

		}

		/**
		 * Return the view all push notification link.
		 *
		 * @return string|void
		 */
		public function get_view_all_link() {
			return admin_url( 'admin.php?page=bbapp-notification&setting=list' );
		}

		/**
		 * Return the new push notification link.
		 *
		 * @return string|void
		 */
		public function get_new_notification_link() {
			return admin_url( 'admin.php?page=bbapp-notification&setting=new' );
		}

		/**
		 * Return all users for push notification purpose.
		 * Note :- This function should only return with one order by for every time.
		 * Else the order of sending pushes will be broken and duplicate push attempt can caused.
		 *
		 * @param int $page  Current page number.
		 * @param int $limit Data per page.
		 *
		 * @return array
		 */
		public function get_all_users( $page = 1, $limit = 1 ) {
			global $wpdb;

			$devices_table = bbapp_get_network_table( 'bbapp_user_devices' );

			$page   = max( $page - 1, 0 );
			$offset = $limit * $page;

			$wp_user_capabilities = bbapp_get_global_db_prefix() . 'capabilities'; // this will force to collect user from correct blog.

			// Query only users who have device token registered.
			$sql = $wpdb->prepare( "SELECT DISTINCT u.ID as user_id FROM {$wpdb->users} as u, {$wpdb->usermeta} as um , {$devices_table} as devices WHERE u.ID = um.user_id AND u.ID = devices.user_id AND u.user_status = '0' AND (um.meta_key = %s) order by u.id ASC LIMIT %d OFFSET %d", $wp_user_capabilities, $limit, $offset ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			// Query only users who have device token registered.
			$sql = $wpdb->prepare( "SELECT DISTINCT u.ID as user_id FROM {$wpdb->users} as u, {$wpdb->usermeta} as um , {$devices_table} as devices WHERE u.ID = um.user_id AND u.ID = devices.user_id AND u.user_status = '0' AND (um.meta_key = %s)", $wp_user_capabilities ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			$sql = "SELECT count(ID) as count FROM {$wpdb->users} WHERE ID IN ({$sql})";

			$count = $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			$user_ids = array();

			if ( $results ) {
				foreach ( $results as $result ) {
					$user_ids[] = $result->user_id;
				}
			}

			return array(
				'users' => $user_ids,
				'total' => $count,
			);

		}

		/**
		 * Return the Push Settings.
		 *
		 * @return array
		 */
		public function get_settings() {

			// When Plugin is Enabled on NetWork Mode. The main SubSite Push Settings are Considered Everywhere.
			if ( bbapp()->is_network_activated() ) {
				switch_to_blog( 1 );
			}

			$bbapp_settings = ManageApp::instance()->get_app_settings();

			if ( bbapp()->is_network_activated() ) {
				restore_current_blog();
			}

			if ( ! isset( $bbapp_settings['push_settings'] ) || ! is_array( $bbapp_settings['push_settings'] ) ) {
				$bbapp_settings['push_settings'] = array();
			}

			return $bbapp_settings['push_settings'];
		}

		/**
		 * Update the Push Setting.
		 *
		 * @param array $settings Push settings.
		 *
		 * @return array
		 */
		public function update_settings( $settings ) {

			// When Plugin is Enabled on NetWork Mode. The main SubSite Push Settings are Considered Everywhere.
			if ( bbapp()->is_network_activated() ) {
				switch_to_blog( 1 );
			}

			// Merge Settings with parent.
			$bbapp_settings                  = ManageApp::instance()->get_app_settings();
			$bbapp_settings['push_settings'] = $settings;

			$return = ManageApp::instance()->update_app_settings( $bbapp_settings );

			if ( bbapp()->is_network_activated() ) {
				restore_current_blog();
			}

			return $return;
		}

		/**
		 * Get the push notification queue entry count by push notification id.
		 *
		 * @param int $push_id Push id.
		 *
		 * @return int
		 */
		public function get_push_notification_queue_count( $push_id ) {
			global $wpdb;

			return $wpdb->get_var( "SELECT count(*) FROM `{$this->push_queue_table}` where sent=0 AND n_id=" . $push_id ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		/**
		 * Get the push notification queue entry by push notification id.
		 *
		 * @param array  $args   Push queue items.
		 * @param string $output Push notification.
		 *
		 * @return array|object|null
		 */
		public function get_push_notification_queue( $args = array(), $output = OBJECT ) {
			global $wpdb;

			$default_args = array(
				'agent'   => 'manual',
				'user_id' => 0,
				'n_id'    => 0,
			);

			$args = wp_parse_args( $args, $default_args );

			$where_clause = array();

			if ( 'manual' !== $args['agent'] ) {
				$where_clause[] = $wpdb->prepare( 'agent=%s', $args['agent'] );
			}

			if ( ! empty( $args['user_id'] ) ) {
				$where_clause[] = $wpdb->prepare( 'user_id=%s', $args['user_id'] );
			}
			if ( ! empty( $args['n_id'] ) ) {
				$where_clause[] = $wpdb->prepare( 'n_id=%s', $args['n_id'] );
			}

			if ( ! empty( $where_clause ) ) {
				$where_clause = 'WHERE ' . implode( 'AND ', $where_clause );
			} else {
				$where_clause = '';
			}

			$limit_clause = '';

			if ( $args['per_page'] ) {
				$args['per_page']     = (int) $args['per_page'];
				$args['current_page'] = (int) $args['current_page'];

				$limit_clause  = " LIMIT {$args["per_page"]} ";
				$limit_clause .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
			}

			$sql = "SELECT  * FROM {$this->push_queue_table} {$where_clause}  {$limit_clause}";

			return $wpdb->get_results( $sql, $output ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		}

		/**
		 * Get the push notification queue entry by push notification id.
		 *
		 * @param array  $args   Push queue items.
		 * @param string $output Push notification.
		 *
		 * @return array|object|null
		 */
		public function get_distinct_push_notification_queue( $args = array(), $output = OBJECT ) {
			global $wpdb;

			$default_args = array(
				'agent'   => 'manual',
				'user_id' => 0,
				'n_id'    => 0,
			);

			$args = wp_parse_args( $args, $default_args );

			$where_clause = array();

			if ( 'manual' !== $args['agent'] ) {
				$where_clause[] = $wpdb->prepare( 'agent=%s', $args['agent'] );
			}

			if ( ! empty( $args['user_id'] ) ) {
				$where_clause[] = $wpdb->prepare( 'user_id=%s', $args['user_id'] );
			}
			if ( ! empty( $args['n_id'] ) ) {
				$where_clause[] = $wpdb->prepare( 'n_id=%s', $args['n_id'] );
			}

			if ( ! empty( $where_clause ) ) {
				$where_clause = 'WHERE ' . implode( 'AND', $where_clause );
			} else {
				$where_clause = '';
			}

			$limit_clause = '';

			if ( $args['per_page'] ) {
				$args['per_page']     = (int) $args['per_page'];
				$args['current_page'] = (int) $args['current_page'];

				$limit_clause  = " LIMIT {$args["per_page"]} ";
				$limit_clause .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
			}

			$sql = "SELECT DISTINCT user_id FROM {$this->push_queue_table} {$where_clause}  {$limit_clause}";
			return $wpdb->get_results( $sql, $output ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		}

		/**
		 * Get all push notification types.
		 *
		 * @return array
		 */
		public function get_all_push_types() {
			static $cache;

			if ( isset( $cache ) ) {
				return $cache;
			}

			$push_type                 = array(
				'custom' => array( 'admin_label' => __( 'Unspecified Notification', 'buddyboss-app' ) ),
				'manual' => array( 'admin_label' => __( 'Manual Push Notification', 'buddyboss-app' ) ),
			);
			$get_all_components        = apply_filters( 'bbapp_get_notification_component_names', array() ); // Todo pic from register type.
			$get_all_component_actions = apply_filters( 'bbapp_get_notification_component_actions', array() );
			foreach ( $get_all_components as $component_name ) {
				$actions = isset( $get_all_component_actions[ $component_name ] ) ? $get_all_component_actions[ $component_name ] : array();
				foreach ( $actions as $action_name => $action_data ) {
					$push_type[ "{$component_name}_{$action_name}" ] = $action_data;
				}
			}
			$cache = $push_type;

			return $cache;
		}

		/**
		 * Return Device string.
		 *
		 * @param string $device Device or Platform small latter name.
		 *
		 * @return string|void
		 */
		public function get_device_string( $device ) {
			if ( 'ios' === $device ) {
				return __( 'iOS', 'buddyboss-app' );
			}
			if ( 'android' === $device ) {
				return __( 'Android', 'buddyboss-app' );
			}

			return $device;
		}

		/**
		 * Nice Error for Error Code.
		 *
		 * @ref https://firebase.google.com/docs/cloud-messaging/http-server-ref
		 *
		 * @return array
		 */
		public function nice_error_codes() {
			return array(
				'0'  => _x( 'Unknown Error', 'Push Notification', 'buddyboss-app' ),
				'2'  => _x( 'Missing Registration', 'Push Notification', 'buddyboss-app' ),
				'3'  => _x( 'Invalid Registration', 'Push Notification', 'buddyboss-app' ),
				'4'  => _x( 'Not Registered', 'Push Notification', 'buddyboss-app' ),
				'5'  => _x( 'Invalid PackageName', 'Push Notification', 'buddyboss-app' ),
				'6'  => _x( 'Mismatched Session ID', 'Push Notification', 'buddyboss-app' ),
				'7'  => _x( 'Invalid Parameters', 'Push Notification', 'buddyboss-app' ),
				'8'  => _x( 'Message Too Big', 'Push Notification', 'buddyboss-app' ),
				'9'  => _x( 'Invalid DataKey', 'Push Notification', 'buddyboss-app' ),
				'10' => _x( 'Invalid Ttl', 'Push Notification', 'buddyboss-app' ),
				'11' => _x( 'Unavailable', 'Push Notification', 'buddyboss-app' ),
				'12' => _x( 'Internal Server Error', 'Push Notification', 'buddyboss-app' ),
				'13' => _x( 'Device Message Rate Exceeded', 'Push Notification', 'buddyboss-app' ),
				'14' => _x( 'Topics Message Rate Exceeded', 'Push Notification', 'buddyboss-app' ),
				'15' => _x( 'Invalid APNs Credential', 'Push Notification', 'buddyboss-app' ),
			);
		}

	}

endif;
