<?php
/**
 * BuddyBoss App Notifications.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

use BuddyBossApp\Api\Notification\V1\RestAPI;
use BuddyBossApp\Auth\Common;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Notification\Core\PostCommentNotification;
use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Contains Features Related to Notifications & Push Notification.
 * Class Notification
 */
class Notification {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	private static $instance;

	/**
	 * Push to normal notification.
	 *
	 * @var string $push_to_normal_notification
	 */
	private $push_to_normal_notification;

	/**
	 * User device table name.
	 *
	 * @var string $table_name
	 */
	protected $table_name = 'bbapp_user_devices';

	/**
	 * Class construct.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of class
	 *
	 * @return Notification
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load all notification hooks.
	 */
	protected function hooks() {

		$this->table_name = bbapp_get_network_table( $this->table_name );

		/**
		 * BuddyBossApp Push Notification Service.
		 */
		\BuddyBossApp\Notification\Services\Firebase::instance();

		$this->apply_bp_notifications_formats();

		/**
		 * Enable the support for app push notification preferences inside platform modern notification.
		 */
		add_filter( 'bb_app_notification_enabled', array( $this, 'bbapp_push_notification_enabled' ) );

		/**
		 * Trigger push when normal notification created.
		 * */
		// BuddyBoss Platform Notification.
		add_action( 'bp_notification_before_save', array( $this, 'bp_notification_before_save' ) );
		// BuddyBoss App fallback Notification.
		add_action(
			'bbapp_post_create_inbuilt_notification',
			array(
				$this,
				'bbapp_post_create_inbuilt_notification',
			),
			99,
			2
		);

		// Create Notification Count cache.
		add_action( 'bbapp_post_create_notification', array( $this, 'update_bbapp_notification_count' ), 99, 2 );
		add_action( 'bp_notification_after_save', array( $this, 'update_bp_notification_count' ), 99, 1 );
		add_action( 'bp_notification_before_update', array( $this, 'clear_bp_notification_count_cache' ), 99, 2 );

		// Mark read notification update unread count cache.
		add_action( 'bp_rest_after_notifications_mark_read', array( $this, 'update_bp_notification_count_after_mark_read' ), 99, 2 );
		add_action( 'bp_rest_after_notifications_mark_read', array( $this, 'bp_rest_notifications_update_item_callback' ), 99 );

		// Delete Notification Count cache.
		add_action( 'bp_notification_after_delete', array( $this, 'delete_notification_update_notification_count' ), 99, 3 );

		// This action will call after notification mark as read will successfully.
		// Fires after a notification is updated via the REST API.
		add_action( 'bp_rest_notifications_update_item', array( $this, 'bp_rest_notifications_update_item_callback' ), 10 );

		add_action( 'init', array( $this, 'load_init' ) );

		add_filter( 'bb_register_notification_preferences', array( $this, 'register_notification_preferences' ) );
		add_filter( 'bp_rest_account_settings_get_items', array( $this, 'bbapp_account_settings_get_items' ), 10, 3 );

		if ( bbapp_is_lite_app() && ! bbapp_is_grace_peroid() ) {
			$remove_app_notification_preferences = array(
				'bb_is_bb_new_mention_app_preference_type_render',
				'bb_is_bb_posts_new_comment_reply_app_preference_type_render',
				'bb_is_bb_account_password_app_preference_type_render',
				'bb_is_bb_activity_comment_app_preference_type_render',
				'bb_is_bb_activity_following_post_app_preference_type_render',
				'bb_is_bb_following_new_app_preference_type_render',
				'bb_is_bb_groups_details_updated_app_preference_type_render',
				'bb_is_bb_groups_promoted_app_preference_type_render',
				'bb_is_bb_groups_new_invite_app_preference_type_render',
				'bb_is_bb_groups_new_request_app_preference_type_render',
				'bb_is_bb_groups_request_accepted_app_preference_type_render',
				'bb_is_bb_groups_request_rejected_app_preference_type_render',
				'bb_is_bb_groups_new_message_app_preference_type_render',
				'bb_is_bb_groups_subscribed_activity_app_preference_type_render',
				'bb_is_bb_groups_subscribed_discussion_app_preference_type_render',
				'bb_is_bb_groups_new_zoom_app_preference_type_render',
				'bb_is_bb_forums_subscribed_discussion_app_preference_type_render',
				'bb_is_bb_forums_subscribed_reply_app_preference_type_render',
				'bb_is_bb_messages_new_app_preference_type_render',
				'bb_is_bb_connections_new_request_app_preference_type_render',
				'bb_is_bb_connections_request_accepted_app_preference_type_render',
			);

			foreach ( $remove_app_notification_preferences as $remove_app_notification_preference ) {
				add_filter( $remove_app_notification_preference, '__return_false' );
			}
			unset( $remove_app_notification_preferences );
		}
	}

	/**
	 * Load init.
	 *
	 * @since 1.7.90
	 */
	public function load_init() {
		PostCommentNotification::instance();
		SilentNotification::instance();
	}

	/**
	 * Apply all BuddyBoss Notifications formats registered by BuddyBoss App API.
	 */
	public function apply_bp_notifications_formats() {

		// Gather all notification and filter them.
		$accepted_args = 8;
		if ( class_exists( 'BP_Core_Notification_Abstract' ) ) {
			// when this class is introduced there is 9 params.
			$accepted_args = 9;
		}

		add_filter(
			'bp_notifications_get_notifications_for_user',
			array(
				$this,
				'bp_notifications_formats',
			),
			99999,
			$accepted_args
		);

		add_filter(
			'bp_get_the_notification_description',
			array(
				$this,
				'bp_get_the_notification_description',
			),
			99,
			2
		);

		add_filter(
			'bb_notifications_get_component_notification',
			array(
				$this,
				'bp_notifications_formats',
			),
			99999,
			$accepted_args
		);
	}

	/**
	 * Function to register push notification to normal notification so when normal
	 * notification is trigger push get send.
	 *
	 * @param string $component_name   Component name.
	 * @param string $component_action Action hook for component.
	 * @param string $push_type        Push type.
	 */
	public function register_push_to_normal_notification( $component_name, $component_action, $push_type = 'custom' ) {
		$this->push_to_normal_notification[ $component_name ][ $component_action ] = array(
			'push_type' => $push_type,
		);
	}

	/**
	 * Return the details of push to normal notification registration.
	 *
	 * @param string $component_name   Component name.
	 * @param string $component_action Action hook for component.
	 * @param int    $user_id          User id to check if particular notification is enabled.
	 *
	 * @return bool
	 */
	public function get_push_to_normal_notification( $component_name, $component_action, $user_id = 0 ) {

		if ( isset( $this->push_to_normal_notification[ $component_name ] ) && isset( $this->push_to_normal_notification[ $component_name ][ $component_action ] ) ) {
			return $this->push_to_normal_notification[ $component_name ][ $component_action ];
		}

		return false;
	}

	/**
	 * Send Push Notification on Creation of Normal Notification made from inbuilt notification.
	 *
	 * @param array $args Notification item.
	 * @param array $row  Notification database row.
	 */
	public function bbapp_post_create_inbuilt_notification( $args, $row ) {

		$can_push_to_normal = $this->get_push_to_normal_notification( $args['component_name'], $args['component_action'] );

		if ( ! $can_push_to_normal ) {
			return false;
		}
		$title = '';

		/**
		 * Filters notification output.
		 *
		 * @param int    $user_id           User ID.
		 * @param string $component_name    Notification component ID.
		 * @param string $component_action  Canonical notification action.
		 * @param int    $item_id           Notification item ID.
		 * @param int    $secondary_item_id Notification secondary item ID.
		 * @param int    $notification_id   Notification ID.
		 * @param string $format            Format of return. Either 'string' or 'object'.
		 * @param string $screen            Notification Screen type.
		 */
		$content = apply_filters( 'bbapp_get_notification_output', $args['user_id'], $args['component_name'], $args['component_action'], $args['item_id'], $args['secondary_item_id'], false, 'object', 'app_push' );

		if ( ! isset( $content[0] ) ) {
			return false;
		}

		$content = $content[0];

		$content = $this->get_override_notification_link( $content, $args['component_name'], $args['component_action'], $args['item_id'], $args['secondary_item_id'], 'object' );

		$link = $content->href;

		if ( isset( $content->title ) ) {
			$title = $content->title;
		}

		$push_args = array(
			'primary_text'      => $title,
			'secondary_text'    => $content->content,
			'user_ids'          => array( $args['user_id'] ),
			'sent_as'           => 1,
			'item_id'           => $row['id'],
			'secondary_item_id' => $row['secondary_item_id'],
			'type'              => $can_push_to_normal['push_type'],
			'data'              => array(
				'link'            => $link,
				'notification_id' => $row['id'],
			),
		);

		bbapp_send_push_notification( $push_args );
	}

	/**
	 * Function to add check if notification are enabled or not.
	 *
	 * @since 1.4.7
	 */
	public function bbapp_push_notification_enabled() {
		return ( function_exists( 'bp_is_active' ) && bp_is_active( 'notifications' ) && function_exists( 'bbapp_is_active' ) && bbapp_is_active( 'push_notification' ) );
	}

	/**
	 * Triggers when bp notification created.
	 *
	 * @param object $notification Notification item.
	 */
	public function bp_notification_before_save( $notification ) {
		if ( empty( $notification->id ) ) {
			add_action( 'bp_notification_after_save', array( $this, 'bp_notification_after_save' ), 99, 1 );
		}
	}

	/**
	 * Triggers when bp notification creation is done.
	 * Here we will create push notification is it's registered with bp notification.
	 *
	 * @param object $notification Notification item.
	 *
	 * @return bool
	 */
	public function bp_notification_after_save( $notification ) {
		$filter_users_by_subscription = true;
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			$can_push_to_normal = $this->get_push_to_normal_notification( $notification->component_name, $notification->component_action, $notification->user_id );

			if ( ! $can_push_to_normal ) {
				return false;
			}

			$push_type = $can_push_to_normal['push_type'];

		} else {

			if ( function_exists( 'bp_can_send_notification' ) ) {
				$component_name = $notification->component_name;

				// Platform is only action in below three components.
				if ( in_array(
					$component_name,
					array(
						'activity',
						'forums',
						'members',
						'core',
					),
					true
				) && 'bb_new_mention' === $notification->component_action ) {
					$component_name = 'members';
				}

				// Platform is only action in below three components.
				if ( 'bb_groups_new_message' === $notification->component_action ) {
					$component_name = 'groups';
				} elseif ( 'bb_messages_new' === $notification->component_action ) {
					$component_name = 'messages';
				}

				// Lite app disable platform notifications.
				if ( bbapp_is_lite_app() && ! bbapp_is_grace_peroid() ) {
					return false;
				}

				if ( ! bp_can_send_notification( $notification->user_id, $component_name, $notification->component_action, 'app' ) ) {
					return false;
				}

				$notification_setting = \BuddyBossApp\Admin\Notification::instance()->get_settings();
				$skip_users           = ! empty( $notification_setting['push.skip_active_members'] );

				if (
					function_exists( 'bb_can_send_push_notification' ) &&
					true !== bb_can_send_push_notification( $notification->user_id, array( 'skip_active_user' => $skip_users ) )
				) {
					return false;
				}

				$filter_users_by_subscription = false;
				$push_type                    = $component_name . '_' . $notification->component_action;
			}
		}

		$title = '';

		/**
		 * Filters notification output.
		 *
		 * @param int    $user_id           User ID.
		 * @param string $component_name    Notification component ID.
		 * @param string $component_action  Canonical notification action.
		 * @param int    $item_id           Notification item ID.
		 * @param int    $secondary_item_id Notification secondary item ID.
		 * @param int    $notification_id   Notification ID.
		 * @param string $format            Format of return. Either 'string' or 'object'.
		 * @param string $screen            Notification Screen type.
		 */
		$content = apply_filters( 'bbapp_get_notification_output', $notification->user_id, $notification->component_name, $notification->component_action, $notification->item_id, $notification->secondary_item_id, $notification->id, 'object', 'app_push' );

		if ( ! isset( $content[0] ) ) {
			return false;
		}

		$content = $content[0];

		$content = $this->get_override_notification_link( $content, $notification->component_name, $notification->component_action, $notification->item_id, $notification->secondary_item_id, 'object' );

		$link = $content->href;

		if ( isset( $content->title ) ) {
			$title = $content->title;
		}

		$push_args = array(
			'primary_text'                 => $title,
			'secondary_text'               => $content->content,
			'user_ids'                     => array( $notification->user_id ),
			'sent_as'                      => 1,
			'type'                         => $push_type,
			'filter_users_by_subscription' => $filter_users_by_subscription,
			'item_id'                      => $notification->id,
			'secondary_item_id'            => $notification->secondary_item_id,
			'data'                         => array(
				'link'             => $link,
				'notification_id'  => $notification->id,
				'component_name'   => $notification->component_name,
				'component_action' => $notification->component_action,
			),
		);

		$send = bbapp_send_push_notification( $push_args );
	}

	/**
	 * BP/Platfrom Notification Format Handler for web.
	 *
	 * @param string|array $return            Formated notification.
	 * @param int          $item_id           Notification item id.
	 * @param int          $secondary_item_id Notification secondary item id.
	 * @param int          $total_items       Total notification item number.
	 * @param string       $format            Notification format.
	 * @param string       $component_action  Action hook for component.
	 * @param string       $component_name    Component name.
	 * @param int          $notification_id   Notification id.
	 * @param string       $screen            Notification screen.
	 *
	 * @return array | void
	 */
	public function bp_notifications_formats( $return, $item_id, $secondary_item_id, $total_items, $format, $component_action, $component_name, $notification_id, $screen = 'web' ) {

		if ( empty( $format ) ) {
			$format = 'string';
		}

		/**
		 * Appends the custom formatted provided by BuddyBoss App notification abstract API.
		 *
		 * @param string $content           Formatted notification content.
		 * @param string $component_name    Component name.
		 * @param string $component_action  Component action.
		 * @param int    $item_id           Notification item id.
		 * @param int    $secondary_item_id Notification secondary item id.
		 * @param int    $notification_id   Notification id.
		 * @param int    $total_items       Total notification items.
		 * @param string $screen            Notification screen.
		 */
		$get_bbapp_format = apply_filters( 'bbapp_get_formatted_notification', false, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $screen );

		// Validate the return value & return if validated.
		if ( is_array( $get_bbapp_format ) && isset( $get_bbapp_format['text'] ) && isset( $get_bbapp_format['link'] ) ) {

			if ( 'string' === $format ) {

				if ( empty( $get_bbapp_format['link'] ) ) {
					$return = esc_html( $get_bbapp_format['text'] );
				} else {
					$return = '<a href="' . esc_url( $get_bbapp_format['link'] ) . '">' . esc_html( $get_bbapp_format['text'] ) . '</a>';
				}
			} else {

				$return = array(
					'text' => $get_bbapp_format['text'],
					'link' => $get_bbapp_format['link'],
				);

			}
		}

		/**
		 * Apply notification overrides if provided by BuddyBoss App notification abstract API.
		 */
		$_format = ( 'object' === $format ) ? 'array' : $format; // rename format to array instead of object.
		$return  = apply_filters( 'bbapp_get_formatted_notification_overrides', $return, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $_format, $screen );

		/**
		 * Capture needed param so they can be useful on bp_core_get_notifications_for_user filter later on.
		 */
		global $bbapp_notification_link_override_helper;

		$bbapp_notification_link_override_helper = array(
			'component_action'  => $component_action,
			'component_name'    => $component_name,
			'item_id'           => $item_id,
			'secondary_item_id' => $secondary_item_id,
		);

		return $return;
	}

	/**
	 * Change the override link for App End.
	 *
	 * @param string $description  Notification description.
	 * @param object $notification Notification items.
	 *
	 * @return bool
	 */
	public function bp_get_the_notification_description( $description, $notification ) {

		// Only override app url on rest api.

		$curr_url = add_query_arg( null, null );
		if ( strpos( $curr_url, 'wp-json' ) !== false ) {
			$description = $this->get_override_notification_link( $description, $notification->component_name, $notification->component_action, $notification->item_id, $notification->secondary_item_id, 'string' );
		}

		return $description;
	}

	/**
	 * Return the content with overrided app link.
	 *
	 * @param string|array|object $content           Notification content.
	 * @param string              $component_name    Component name.
	 * @param string              $component_action  Action hook for component.
	 * @param int                 $item_id           Notification item id.
	 * @param int                 $secondary_item_id Notification secondary item id.
	 * @param string              $format            Notification format.
	 *
	 * @return bool
	 */
	public function get_override_notification_link( $content, $component_name, $component_action, $item_id, $secondary_item_id, $format ) {
		/**
		 * Append custom app only links provided by BuddyBoss App Abstract Notification API.
		 */
		$link = false;

		if ( 'string' === $format ) {
			preg_match( '/href=["\'](.*)["\']/U', $content, $result_links );
			if ( isset( $result_links[1] ) && bbapp_is_valid_url( $result_links[1] ) ) {
				$link = $result_links[1];
			}
		} else {
			if ( is_array( $content ) ) {
				$link = $content['link'];
			} elseif ( is_object( $content ) ) {
				$link = $content->href;
			}
		}

		$override_link = apply_filters( 'bbapp_get_notification_app_link', $link, $component_name, $component_action, $item_id, $secondary_item_id );

		if ( $override_link !== $link ) {

			if ( 'string' === $format ) {
				if ( false === $override_link ) {
					$content = wp_strip_all_tags( $content );
				} else {
					$content = str_replace( $link, $override_link, $content );
				}
			} else {
				if ( is_array( $content ) ) {
					$content['link'] = $override_link;
				} elseif ( is_object( $content ) ) {
					$content->href = $override_link;
				}
			}
		}

		return $content;
	}

	/**
	 * Returns the Notification Table.
	 *
	 * @return string
	 */
	public function table_name_devices() {
		return $this->table_name;
	}

	/**
	 * Delete user device
	 *
	 * @param int $user_id user id.
	 *
	 * @since 1.4.7
	 */
	public function delete_user_devices_by_user_id( $user_id ) {
		global $wpdb;
		$wpdb->delete( $this->table_name, array( 'user_id' => $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Stores user device tokens on database.
	 *
	 * @param int    $user_id      User id.
	 * @param string $device_token Device token.
	 * @param bool   $auth_token   It's access token on v1 authentication and refresh token with v2 authentication.
	 * @param array  $device_data  Pass to device model and device id.
	 *
	 * @return bool|false|int
	 */
	public function register_device_for_user( $user_id, $device_token, $auth_token = false, $device_data = array() ) {
		global $wpdb, $bbapp_var;

		$device_token = trim( $device_token );

		/**
		 * If auth token is not provided use access token from header.
		 */
		if ( empty( $auth_token ) ) {
			if ( ! method_exists( '\BuddyBossApp\Auth\Common', 'get_access_token' ) ) {
				return false;
			}
			$bbapp_auth_common = Common::instance();
			$auth_token        = $bbapp_auth_common->get_access_token();
		}

		if ( empty( $auth_token ) || empty( $device_token ) ) {
			return false;
		}

		/**
		 * Get the auth jti value.
		 * jti a user unique key. on v2 same jti is used for refresh token & access token.
		 * For reference check authentication plugin JWT generation.
		 */
		$auth_jti = bbapp_get_jwt_jti( $auth_token );

		$auth_token_hash   = hash( 'sha256', $auth_jti );
		$device_token_hash = hash( 'sha256', $device_token );

		/**
		 * Delete any older entry which is not associated with current auth_token or user_id
		 */
		$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} WHERE  device_token_hash = %s AND (user_id != %d OR auth_token_hash != %s)", $device_token_hash, $user_id, $auth_token_hash ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$app_platform     = $bbapp_var['app_platform'];
		$app_base_version = $bbapp_var['app_code_version'];

		$device_id    = $app_platform;
		$device_model = $app_platform;
		if ( ! empty( $device_data ) ) {
			if ( ! empty( $device_data['device_id'] ) ) {
				$device_id = $device_data['device_id'];
			}
			if ( ! empty( $device_data['device_model'] ) ) {
				$device_model = $device_data['device_model'];
			}
		}
		/**
		 * App base version is required. to control core features.
		 */
		if ( empty( $app_base_version ) ) {
			return false;
		}

		/**
		 * Query for any previous device entry.
		 */
		$device_record = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE user_id = %s  AND auth_token_hash = %s ", $user_id, $auth_token_hash ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( empty( $device_record ) ) {
			$data = array();

			/**
			 * Helper hook to preload data on device token before it get into db.
			 *
			 * @param array  $data             Token data.
			 * @param string $device_token     Device token.
			 * @param int    $user_id          User id.
			 * @param string $app_platform     Device platform.
			 * @param string $app_base_version App base version.
			 */
			$data = apply_filters( 'bbapp_device_token_data', $data, $device_token, $user_id, $app_platform, $app_base_version );

			$data          = maybe_serialize( $data );
			$push_instance = bbapp_get_app_push_instance();
			$push_type     = '';

			if ( $push_instance ) {
				$push_type = $push_instance->push_type();
			}

			// 2. Add new row
			$insert = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
				$this->table_name,
				array(
					'device_token'      => $device_token,
					'user_id'           => $user_id,
					'device_token_hash' => $device_token_hash,
					'auth_token_hash'   => $auth_token_hash,
					'date_updated'      => current_time( 'mysql', 1 ),
					'date_registered'   => current_time( 'mysql', 1 ),
					'date_active'       => current_time( 'mysql', 1 ),
					'platform'          => $app_platform,
					'app_ver'           => $app_base_version,
					'push_type'         => $push_type,
					'device_id'         => $device_id,
					'device_model'      => $device_model,
					'data'              => $data,
				),
				array( '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s' )
			);

			return $insert;

		} else {

			$data = $device_record->data;

			$data = maybe_unserialize( $data );

			if ( ! is_array( $data ) ) {
				$data = array();
			}

			/**
			 * Filters token data.
			 *
			 * @param array  $data             Token data.
			 * @param string $device_token     Device token.
			 * @param int    $user_id          User id.
			 * @param string $app_platform     Device platform.
			 * @param string $app_base_version App base version.
			 */
			$data = apply_filters( 'bbapp_device_token_data', $data, $device_token, $user_id, $app_platform, $app_base_version );

			$data               = maybe_serialize( $data );
			$table_args         = array();
			$table_args_prepare = array();

			// Check the device token hash has exists in database.
			if ( isset( $device_record->device_token_hash ) && $device_record->device_token_hash !== $device_token_hash ) {
				$table_args['device_token_hash'] = $device_token_hash;
				$table_args_prepare[]            = '%s';
			}
			// Check the device token has exists in database.
			if ( isset( $device_record->device_token ) && $device_record->device_token !== $device_token ) {
				$table_args['device_token'] = $device_token;
				$table_args_prepare[]       = '%s';
			}
			// Check the App ver has exists in database.
			if ( isset( $device_record->app_ver ) && $device_record->app_ver !== $app_base_version ) {
				$table_args['app_ver'] = $app_base_version;
				$table_args_prepare[]  = '%s';
			}
			// Check the data has exists in database.
			if ( isset( $device_record->data ) && $device_record->data !== $data ) {
				$table_args['data']   = $data;
				$table_args_prepare[] = '%s';
			}

			/**
			 * Update the date active only once in hour or if any other data is changed.
			 * Keeping a delay will avoid not required database queries.
			 * Date last active doesn't need to be precise.
			 */

			$from = strtotime( $device_record->date_active ); // database.
			$diff = (int) abs( time() - $from );
			$hour = 3600; // seconds in hour.

			if ( ! empty( $table_args ) || $diff > $hour ) {
				$table_args['date_active'] = current_time( 'mysql', 1 );
				$table_args_prepare[]      = '%s';
			}

			/**
			 * ----
			 */

			if ( ! empty( $table_args ) ) {
				$table_args['date_updated'] = current_time( 'mysql', 1 );
				$table_args_prepare[]       = '%s';

				// 2. update old row
				return $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
					$this->table_name,
					array(
						'device_token_hash' => $device_token_hash,
						'device_token'      => $device_token,
						'date_updated'      => current_time( 'mysql', 1 ),
						'app_ver'           => $app_base_version,
						'device_id'         => $device_id,
						'device_model'      => $device_model,
						'data'              => $data,
					),
					array(
						'id' => $device_record->id,
					),
					$table_args_prepare,
					array( '%s' )
				);
			}
			return true;
		}
	}

	/**
	 * Marks unread notifications.
	 *
	 * @param int $user_id User id.
	 *
	 * @return false|int
	 */
	public function mark_notification_as_read( $user_id ) {
		global $wpdb;
		$table_name = bbapp_get_network_table( 'bbapp_notifications' );
		$update     = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
			$table_name,
			array(
				'unread' => 0,
			),
			array(
				'user_id' => $user_id,
				'blog_id' => get_current_blog_id(),
				'unread'  => 1,
			)
		);

		return $update;
	}

	/**
	 * Marks notification as a read.
	 *
	 * @param int $user_id         User id.
	 * @param int $notification_id Notification id.
	 *
	 * @since 1.8.90
	 * @return false|int
	 */
	public function mark_notification_as_read_by_id( $user_id, $notification_id ) {
		global $wpdb;

		if ( empty( $user_id ) || empty( $notification_id ) ) {
			return false;
		}

		$table_name = bbapp_get_network_table( 'bbapp_notifications' );
		$update     = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
			$table_name,
			array(
				'unread' => 0,
			),
			array(
				'user_id' => $user_id,
				'id'      => $notification_id,
				'blog_id' => get_current_blog_id(),
				'unread'  => 1,
			)
		);

		// Clear the unread notification count cache.
		wp_cache_delete( $user_id, 'bbapp_notifications_unread_count' );
		$this->get_notification_count( $user_id, true );

		return $update;
	}

	/**
	 * Get the user notification count value.
	 *
	 * @param int  $user_id      User id.
	 * @param bool $bypass_cache Cache data.
	 *
	 * @return int
	 */
	public function get_notification_count( $user_id, $bypass_cache = false ) {
		/**
		 * Added `function_exists` to support existing user otherwise it will return `0`.
		 */
		if ( $bypass_cache || function_exists( 'bp_notifications_get_unread_notification_count' ) ) {
			if ( class_exists( 'BP_Notifications_Notification' ) ) {
				$count = bbapp_notifications_get_unread_notification_count( $user_id );
				$get   = number_format_i18n( $count );
			} else {
				global $wpdb;
				$table_name = bbapp_get_network_table( 'bbapp_notifications' );
				$get        = $wpdb->get_var( $wpdb->prepare( "SELECT count(id) as count FROM {$table_name} WHERE `user_id` = %d AND blog_id=%d AND unread=%d", $user_id, $user_id, 1 ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				if ( ! is_numeric( $get ) ) {
					$get = 0;
				}
			}
			update_user_meta( $user_id, '_appb_notifications_count', $get );
		} else {
			$get = get_user_meta( $user_id, '_appb_notifications_count', true );
			if ( ! is_numeric( $get ) ) {
				$get = 0;
			}
		}

		return $get;
	}

	/**
	 * Update BuddyBoss App notification count.
	 *
	 * @param object $args Notification meta data.
	 * @param array  $row  Notification database row item.
	 *
	 * @return void
	 */
	public function update_bbapp_notification_count( $args, $row ) {
		if ( ! empty( $row['user_id'] ) ) {
			foreach ( $row['user_id'] as $user_id ) {
				$this->get_notification_count( $user_id, true );
			}
		}
	}

	/**
	 * Update BuddyPress notification count.
	 *
	 * @param object $notification Notification items.
	 *
	 * @return void
	 */
	public function update_bp_notification_count( $notification ) {
		if ( ! empty( $notification->user_id ) ) {
			wp_cache_delete( $notification->user_id, 'bbapp_notifications_unread_count' );

			$this->get_notification_count( $notification->user_id, true );
		}
	}

	/**
	 * Clear the unread notification count cache on notification update.
	 *
	 * @param array $update_args See BP_Notifications_Notification::update() for description.
	 * @param array $where_args  See BP_Notifications_Notification::update() for description.
	 *
	 * @since 2.3.60
	 * @return void
	 */
	public function clear_bp_notification_count_cache( $update_args, $where_args ) {
		// User ID is passed in where arugments.
		if ( ! empty( $where_args['user_id'] ) ) {
			wp_cache_delete( $where_args['user_id'], 'bbapp_notifications_unread_count' );
		} elseif ( ! empty( $where_args['id'] ) ) {
			$n = bp_notifications_get_notification( $where_args['id'] );
			wp_cache_delete( $n->user_id, 'bbapp_notifications_unread_count' );
		}
	}

	/**
	 * Update the notification count after mark as read.
	 *
	 * @param array           $notification_ids Notification ids.
	 * @param WP_REST_Request $request          Request object.
	 *
	 * @since 1.8.90
	 *
	 * @return void
	 */
	public function update_bp_notification_count_after_mark_read( $notification_ids, $request ) {
		if ( ! empty( $notification_ids ) ) {
			$user_id = get_current_user_id();

			wp_cache_delete( $user_id, 'bbapp_notifications_unread_count' );
			$this->get_notification_count( $user_id, true );
		}
	}

	/**
	 * Clear the unread notification count cache on notification delete.
	 *
	 * @param bool  $retval If record deleted or not.
	 * @param array $notifications Array of deleted notifications object.
	 * @param array $args   Associative array of columns/values, to determine
	 *                      which rows should be deleted. Of the format
	 *                      array( 'item_id' => 7, 'component_action' => 'members' ).
	 *
	 * @since 1.8.50
	 *
	 * @return void
	 */
	public function delete_notification_update_notification_count( $retval, $notifications, $args ) {
		if ( $retval && ! empty( $notifications ) ) {
			foreach ( $notifications as $notification ) {
				$this->update_bp_notification_count( $notification );
			}
		}
	}

	/**
	 * Get notifications.
	 *
	 * @param array $args Notification request data.
	 */
	public function get_notification( $args ) {

		$defaults = array(
			'per_page' => 10,
			'page'     => 1,
			'is_new'   => 1,
			'app_id'   => false,
		);

		$args = wp_parse_args( $args, $defaults );

		$results = array();

		/**
		 * The bp_has_notifications function will only exists when one of.
		 * BuddyBoss Platform or BuddyPress is enabled.
		 */
		if ( function_exists( 'bp_has_notifications' ) ) {

			if ( bp_has_notifications(
				array(
					'page'     => $args['page'],
					'per_page' => $args['per_page'],
					'is_new'   => $args['is_new'],
				)
			) ) {

				while ( bp_the_notifications() ) {

					bp_the_notification();

					$notification = buddypress()->notifications->query_loop->notification;

					$results[] = array(
						'id'                => bp_get_the_notification_id(),
						'item_id'           => $notification->item_id,
						'secondary_item_id' => $notification->secondary_item_id,
						'namespace'         => $notification->component_name,
						'action'            => $notification->component_action,
						'date_notified'     => $notification->date_notified,
					);

				}
			}
		}

		return $results;
	}

	/**
	 * Create bbapp normal notification using this function
	 * (don't get confuse with push notification.)
	 * You can add notification to single user or multiple users. use user_id, user_ids ( multiple users ).
	 *
	 * @param array $args Create notification reqested data.
	 *
	 * @return int|\WP_Error
	 */
	public function create_notification( $args ) {

		$defaults = array(
			'user_id'           => 0,
			'user_ids'          => array(),
			'item_id'           => 0,
			'secondary_item_id' => 0,
			'component_name'    => 'bbapp',
			'component_action'  => '',
			'blog_id'           => get_current_blog_id(),
			'version'           => 2,
		);

		$args = wp_parse_args( $args, $defaults );

		if ( empty( $args['component_action'] ) ) {
			return new \WP_Error( 'ACTION_MISSING', __( 'Action is missing.', 'buddyboss-app' ) );
		}

		if ( empty( $args['component_name'] ) ) {
			return new \WP_Error( 'NAME_MISSING', __( 'Action is missing.', 'buddyboss-app' ) );
		}

		global $wpdb;

		/**
		 * Check if user_ids are empty & user_id then consider user_id.
		 */
		if ( empty( $args['user_ids'] ) && is_int( $args['user_id'] ) ) {
			$args['user_ids'] = array(
				$args['user_id'],
			);
		}

		/**
		 * When user_ids found empty we don't have anything to do break the code.
		 */
		if ( empty( $args['user_ids'] ) ) {
			return false;
		}

		do_action( 'bbapp_pre_create_notification', $args );

		$row = array(
			'component_action'  => $args['component_action'],
			'component_name'    => $args['component_name'],
			'user_id'           => $args['user_ids'],
			'item_id'           => $args['item_id'],
			'secondary_item_id' => $args['secondary_item_id'],
			'date_notified'     => current_time( 'mysql', 1 ),
			'blog_id'           => (int) $args['blog_id'],
		);

		/**
		 * When BuddyPress or Platform not exists then use the inbuild bbapp table else platform & buddypress.
		 */
		if ( ! function_exists( 'bp_notifications_add_notification' ) ) {

			$entries = array();

			foreach ( $args['user_ids'] as $user_id ) {

				$entries[] = array(
					current_time( 'mysql', 1 ), // date_notified.
					$args['component_action'], // component_action.
					$args['component_name'], // component_name.
					$user_id, // user_id.
					$args['item_id'], // item_id.
					$args['secondary_item_id'], // secondary_item_id.
					intval( $args['blog_id'] ), // blog_id.
					2, // version.
				);

			}

			$table_name = bbapp_get_network_table( 'bbapp_notifications' );

			$insert_query = "INSERT INTO {$table_name} (`date_notified`, `component_action`, `component_name`, `user_id`, `item_id`, `secondary_item_id`, `blog_id`,`version`) VALUES ";

			$prepare_var            = array();
			$query_row_placeholders = array();

			foreach ( $entries as $entry ) {
				foreach ( $entry as $e ) {
					$prepare_var[] = $e;
				}
				$query_row_placeholders[] = '(%s ,%s ,%s ,%d ,%d ,%d , %d, %d)';
			}

			$insert_query .= implode( ",\n ", $query_row_placeholders );

			$insert = $wpdb->query( $wpdb->prepare( $insert_query, $prepare_var ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			if ( $insert ) {

				$row['id'] = $wpdb->insert_id;

				do_action( 'bbapp_post_create_inbuilt_notification', $args, $row );
				do_action( 'bbapp_post_create_notification', $args, $row );

				return $wpdb->insert_id;

			} else {

				return new \WP_Error( 'ERROR_CREATING', __( 'Unknown error while creating notification.', 'buddyboss-app' ) );

			}
		} else {

			/**
			 * When Platform enable.
			 */
			$entries = array();

			foreach ( $args['user_ids'] as $user_id ) {

				$entries[] = array(
					$user_id, // user_id.
					$args['item_id'], // item_id.
					$args['secondary_item_id'], // secondary_item_id.
					$args['component_name'], // component_name.
					$args['component_action'], // component_action.
					bp_core_current_time(), // date_notified.
					1, // is_new.
				);

			}

			$insert_id = $this->create_bp_bulk_notifications( $row, $entries );

			if ( is_wp_error( $insert_id ) || empty( $insert_id ) ) {
				return new \WP_Error( 'ERROR_CREATING', __( 'Unknown error while creating notification.', 'buddyboss-app' ) );
			}

			return $insert_id;
		}
	}

	/**
	 * Delete Notification From Table.
	 *
	 * @param array $args       Delete notification request data.
	 * @param bool  $only_bbapp Is it BB App.
	 *
	 * @return false|int
	 */
	public function delete_notification( $args, $only_bbapp = false ) {
		global $wpdb;

		$defaults = array(
			'id'                => false,
			'component_action'  => false,
			'user_id'           => false,
			'item_id'           => false,
			'secondary_item_id' => false,
			'component_name'    => false,
			'version'           => 2,
			'blog_id'           => get_current_blog_id(),
		);

		$args = wp_parse_args( $args, $defaults );

		$where        = array();
		$where_format = array();

		if ( $args['id'] ) {
			$where['id']    = $args['id'];
			$where_format[] = '%d';
		}

		if ( $args['component_action'] ) {
			$where['component_action'] = $args['component_action'];
			$where_format[]            = '%s';
		}

		if ( $args['user_id'] ) {
			$where['user_id'] = $args['user_id'];
			$where_format[]   = '%d';
		}

		if ( $args['item_id'] ) {
			$where['item_id'] = $args['item_id'];
			$where_format[]   = '%d';
		}

		if ( $args['secondary_item_id'] ) {
			$where['secondary_item_id'] = $args['secondary_item_id'];
			$where_format[]             = '%d';
		}

		if ( $args['component_name'] ) {
			$where['component_name'] = $args['component_name'];
			$where_format[]          = '%s';
		}

		if ( $args['blog_id'] && ! function_exists( 'bp_notifications_add_notification' ) ) {
			$where['blog_id'] = $args['blog_id'];
			$where_format[]   = '%d';
		}

		if ( $args['version'] && ! function_exists( 'bp_notifications_add_notification' ) ) {
			$where['version'] = $args['version'];
			$where_format[]   = '%d';
		}

		if ( function_exists( 'bp_notifications_add_notification' ) && ! $only_bbapp ) {
			global $bp;
			$wpdb->delete( $bp->notifications->table_name, $where, $where_format ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		// always delete from our table.
		$wpdb->delete( bbapp_get_network_table( 'bbapp_notifications' ), $where, $where_format ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

		return true;
	}

	/**
	 * Get Admin settting.
	 *
	 * @return array
	 */
	public function get_push_notification_admin_settings() {

		$admin_setting = array(
			'global'                   => 1,
			'manual_push_notification' => 1,
		);

		$sub_list = $this->get_push_notifications_subs_list();

		foreach ( $sub_list as $notification ) {
			foreach ( $notification['settings'] as $setting_key => $setting ) {
				$admin_setting[ $setting_key ] = ( isset( $setting['enabled'] ) ? $setting['enabled'] : '' );
			}
		}

		// force disable it when subscription is not available.
		foreach ( $sub_list as $notification ) {
			foreach ( $notification['settings'] as $setting_key => $setting ) {
				if ( isset( $setting['available'] ) && ! $setting['available'] ) {
					$admin_setting[ $setting_key ] = false;
				}
			}
		}

		return $admin_setting;
	}


	/**
	 * Transform the notification settings data for users with defaults overrides only.
	 * This function is used to determine for enable disable settings for users.
	 *
	 * @param array $data Get notification settings request parameters.
	 *
	 * @return array
	 */
	public function get_push_notification_settings_defaults( $data ) {

		if ( ! is_array( $data ) ) {
			$data = array();
		}

		$defaults = $this->get_push_notification_admin_settings();
		$args     = wp_parse_args( $data, $defaults );

		return $args;
	}

	/**
	 * Return the list of push notifications subs list.
	 *
	 * @return array
	 */
	public function get_push_notifications_subs_settings() {

		$subscriptions_setting_lists = array(
			array(
				'name'     => '',
				'label'    => '',
				'settings' => array(
					'global' => array(
						'label'       => __( 'All Notifications', 'buddyboss-app' ),
						'admin_label' => __( 'All Notifications', 'buddyboss-app' ),
						'enabled'     => true,
						'available'   => true,
						'is_topic'    => false,
					),
				),
			),
		);

		return apply_filters( 'bbapp_get_notification_subscriptions', $subscriptions_setting_lists );
	}

	/**
	 * Return the list of push notifications subs list.
	 *
	 * @return array
	 */
	public function get_push_notifications_subs_list() {
		$subscriptions_setting_lists = $this->get_push_notifications_subs_settings();

		return apply_filters( 'bbapp_filter_notification_subscriptions', $subscriptions_setting_lists );
	}

	/**
	 * Update the notification subscription list for user.
	 *
	 * @param int   $user_id  User id.
	 * @param array $sub_data User data.
	 *
	 * @return bool|int
	 */
	public function update_push_notification_subs( $user_id, $sub_data ) {
		if ( ! is_array( $sub_data ) ) {
			$sub_data = array();
		}

		return update_user_meta( $user_id, 'push_notification_settings', $sub_data );
	}

	/**
	 * Return the notification subscription list for user.
	 *
	 * @param int $user_id User id.
	 *
	 * @return bool|int
	 */
	public function get_push_notification_subs( $user_id ) {

		$sub_data = get_user_meta( $user_id, 'push_notification_settings', true );

		if ( ! is_array( $sub_data ) ) {
			$sub_data = array();
		}

		return $sub_data;
	}

	/**
	 * Return the notification subscription list for user.
	 *
	 * @param int  $user_id User id.
	 * @param bool $blog_id Has blog id or not.
	 *
	 * @return array
	 */
	public function get_push_notification_subs_for_user( $user_id, $blog_id = false ) {

		if ( ! $blog_id ) {
			$blog_id = get_current_blog_id();
		}

		$bbapp_app_id = ManageApp::instance()->get_app_id();

		$subscriptions_setting_lists = $this->get_push_notifications_subs_list();

		$subscriptions_setting_lists = apply_filters( 'bbapp_get_notification_subscriptions_for_user', $subscriptions_setting_lists, $user_id );

		$user_data = self::instance()->get_push_notification_subs( $user_id );

		// Add the topic_id.
		$changes = false;
		foreach ( $subscriptions_setting_lists as $setting_key => $subscriptions_setting_list ) {
			if ( isset( $subscriptions_setting_list['settings'] ) ) {
				foreach ( $subscriptions_setting_list['settings'] as $name => $value ) {
					if ( $value['is_topic'] ) {
						$value['topic'] = "{$bbapp_app_id}_{$name}_{$blog_id}";
					}
					$status = $subscriptions_setting_list['settings'][ $name ]['enabled'];
					if ( isset( $user_data[ $name ] ) ) {
						$status = $user_data[ $name ];
					} else {
						$changes            = true;
						$user_data[ $name ] = $status;
					}
					$value['enabled'] = (bool) $status;

					$subscriptions_setting_list['settings'][ $name ] = $value;
				}
				$subscriptions_setting_lists[ $setting_key ] = $subscriptions_setting_list;
			}
		}

		if ( $changes ) {
			self::instance()->update_push_notification_subs( $user_id, $user_data );
		}

		/**
		 * We keep all possible topics so we can disable them on app even when plugin remove them.
		 * This will be useful when some topic is subscribed by mobile and topic is removed on backend.
		 * In this case app will never know when to unsubscribe that topic, this feature will make app to unsubscribe.
		 * */

		$all_topics = get_option( 'bbapp_push_archive_topics' );

		if ( ! is_array( $all_topics ) ) {
			$all_topics = array();
		}

		$changes = false;

		// compare current topics with old.
		foreach ( $subscriptions_setting_lists as $setting_key => $subscriptions_setting_list ) {
			if ( isset( $subscriptions_setting_list['settings'] ) ) {
				foreach ( $subscriptions_setting_list['settings'] as $sub_name => $sub ) {
					if ( $sub['is_topic'] && ! isset( $all_topics[ $sub['topic'] ] ) ) {
						$all_topics[ $sub['topic'] ] = true;
						$changes                     = true;
					}
				}
			}
		}

		if ( $changes ) {
			update_option( 'bbapp_push_archive_topics', $all_topics );
		}

		// Add back not available topics into list.
		$not_exists_topics = $all_topics;
		foreach ( $subscriptions_setting_lists as $setting_key => $subscriptions_setting_list ) {
			if ( isset( $subscriptions_setting_list['settings'] ) ) {
				foreach ( $subscriptions_setting_list['settings'] as $sub_name => $sub ) {
					if ( isset( $sub['topic'] ) && isset( $not_exists_topics[ $sub['topic'] ] ) ) {
						unset( $not_exists_topics[ $sub['topic'] ] );
					}
				}
			}
		}

		// Deleted topic list.
		if ( isset( $subscriptions_setting_lists[0]['settings'] ) ) {
			foreach ( $not_exists_topics as $topic_name => $tmp ) {
				$subscriptions_setting_lists[0]['settings'][ 'deleted_' . $topic_name ] = array(
					'label'     => $topic_name,
					'enabled'   => false,
					'available' => false,
					'is_topic'  => true,
					'topic'     => $topic_name,
				);

			}
		}

		/**
		 * End Archive Topics Addition.
		 */
		return $subscriptions_setting_lists;
	}

	/**
	 * Filters out the user id. depending on there subscription to receive push notification.
	 *
	 * @param array  $user_ids  Users id.
	 * @param string $push_type Push type.
	 *
	 * @return array
	 */
	public function filter_user_ids_by_subscription( $user_ids, $push_type ) {
		global $wpdb;

		// If the push type is disabled from admin settings then return no users.
		if ( ! empty( $push_type ) ) {
			$admin_setting = $this->get_push_notification_admin_settings();
			if ( empty( $admin_setting[ $push_type ] ) ) {
				return array();
			}
		}

		$filtered_user_ids = array();

		/**
		 * Filter to use to Split them into chunks to put less stress on DB.
		 *
		 * @param int $per_batch Per batch chunk.
		 *
		 * @since 1.7.7
		 */
		$user_count   = (int) apply_filters( 'bbapp_user_subscription_count', 250 );
		$users_chunks = array_chunk( $user_ids, $user_count );

		foreach ( $users_chunks as $users ) {

			$users_comma = implode( ',', $users );

			$query_str = "SELECT user_id as user_id,meta_value as value FROM $wpdb->usermeta WHERE meta_key='push_notification_settings' AND user_id IN ( $users_comma )";

			$results = $wpdb->get_results( $query_str, OBJECT ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

			$users_meta = array();

			if ( ! empty( $results ) ) {
				foreach ( $results as $item ) {
					$users_meta[ $item->user_id ] = $item->value;
				}
			}

			foreach ( $user_ids as $user_id ) {

				if ( isset( $users_meta[ $user_id ] ) ) {
					$notifications_settings = maybe_unserialize( $users_meta[ $user_id ] );
				} else {
					$notifications_settings = array();
				}

				$notifications_settings = $this->get_push_notification_settings_defaults( $notifications_settings );

				// Check if we can send notification to user or not.
				if ( isset( $notifications_settings['global'] ) && $notifications_settings['global'] ) {
					$filtered_user_ids[] = $user_id;
				} elseif ( ! empty( $push_type ) && isset( $notifications_settings[ $push_type ] ) && ! empty( $notifications_settings[ $push_type ] ) ) {
					$filtered_user_ids[] = $user_id;
				} elseif ( empty( $push_type ) ) {
					$filtered_user_ids[] = $user_id;
				}
			}
		}

		return $filtered_user_ids;
	}

	/**
	 * Return the inbuilt notifications.
	 * This only fetch version 2 from DB.
	 *
	 * @param array $args         Parameters for getting inbuilt notification.
	 * @param bool  $is_all_fetch Get all data at a time.
	 *
	 * @return array|null|object
	 */
	public function get_inbuilt_notifications( $args, $is_all_fetch = false ) {
		global $wpdb;

		$defaults = array(
			'id'                => false,
			'user_id'           => get_current_user_id(),
			'item_id'           => false,
			'secondary_item_id' => false,
			'component_name'    => false,
			'component_action'  => false,
			'order_by'          => false,
			'sort_order'        => false,
			'is_new'            => false,
			'blog_id'           => get_current_blog_id(),
			'page'              => false,
			'per_page'          => false,
		);

		$args            = array_merge( $defaults, $args );
		$args['version'] = 2;

		/**
		 * Prepare the Where SQL.
		 */

		$where_sql   = array();
		$paged_sql   = '';
		$orderby_sql = '';

		if ( $args['id'] ) {
			$where_sql[] = $wpdb->prepare( 'id = %d', intval( $args['id'] ) );
		}

		if ( $args['user_id'] ) {
			$where_sql[] = $wpdb->prepare( 'user_id = %d', intval( $args['user_id'] ) );
		}

		if ( $args['item_id'] ) {
			$where_sql[] = $wpdb->prepare( 'item_id = %d', intval( $args['item_id'] ) );
		}

		if ( $args['secondary_item_id'] ) {
			$where_sql[] = $wpdb->prepare( 'secondary_item_id = %d', intval( $args['secondary_item_id'] ) );
		}

		if ( $args['component_name'] ) {
			$where_sql[] = $wpdb->prepare( 'component_name = %s', intval( $args['component_name'] ) );
		}

		if ( $args['component_action'] ) {
			$where_sql[] = $wpdb->prepare( 'component_action = %s', intval( $args['component_action'] ) );
		}

		if ( true === $is_all_fetch ) {
			if ( isset( $args['is_new'] ) ) { // is_new param getting 0 or 1 value.
				$where_sql[] = $wpdb->prepare( 'unread = %d', intval( $args['is_new'] ) );
			}
		} else {
			if ( $args['is_new'] ) {
				$where_sql[] = $wpdb->prepare( 'unread = %d', intval( $args['is_new'] ) );
			}
		}

		if ( $args['version'] ) {
			$where_sql[] = $wpdb->prepare( 'version = %d', intval( $args['version'] ) );
		}

		$registered_components = apply_filters( 'bp_notifications_get_registered_components', array() );

		if ( ! empty( $registered_components ) ) {
			foreach ( $registered_components as $k => $v ) {
				$registered_components[ $k ] = $wpdb->prepare( '%s', $v );
			}
			$registered_components = implode( ',', $registered_components );
			$where_sql[]           = "component_name IN ({$registered_components})";
		}

		if ( ! empty( $where_sql ) ) {
			$where_sql = ' WHERE ' . implode( ' AND ', $where_sql );
		} else {
			$where_sql = '';
		}

		/**
		 * Prepare the Paged
		 */
		if ( ! empty( $args['page'] ) && ! empty( $args['per_page'] ) ) {
			$page      = absint( $args['page'] );
			$per_page  = absint( $args['per_page'] );
			$offset    = $per_page * ( $page - 1 );
			$paged_sql = $wpdb->prepare( 'LIMIT %d, %d', $offset, $per_page );
		}

		/**
		 * Prepare Order By
		 */
		$conditions = array();

		// Order by.
		if ( ! empty( $args['order_by'] ) ) {
			$order_by               = implode( ', ', (array) $args['order_by'] );
			$conditions['order_by'] = "{$order_by}";
		}

		// Sort order direction.
		if ( ! empty( $args['sort_order'] ) && in_array( $args['sort_order'], array( 'ASC', 'DESC' ), true ) ) {
			$sort_order               = $args['sort_order'];
			$conditions['sort_order'] = "{$sort_order}";
		}

		// Custom ORDER BY.
		if ( ! empty( $conditions ) ) {
			$orderby_sql = 'ORDER BY ' . implode( ' ', $conditions );
		}

		$table_name = bbapp_get_network_table( 'bbapp_notifications' );

		$sql = "SELECT *FROM {$table_name} {$where_sql} {$orderby_sql} {$paged_sql}";

		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		if ( isset( $results ) && ! empty( $results ) ) {
			// Integer casting.
			foreach ( $results as $key => $result ) {
				$results[ $key ]->id                = (int) $results[ $key ]->id;
				$results[ $key ]->author_id         = (int) $results[ $key ]->author_id;
				$results[ $key ]->user_id           = (int) $results[ $key ]->user_id;
				$results[ $key ]->item_id           = (int) $results[ $key ]->item_id;
				$results[ $key ]->secondary_item_id = (int) $results[ $key ]->secondary_item_id;
				$results[ $key ]->blog_id           = (int) $results[ $key ]->blog_id;
				$results[ $key ]->unread            = (int) $results[ $key ]->unread;
			}
		}

		return $results;
	}

	/**
	 * This Function will update header after the notification is marked as read.
	 *
	 * @return void
	 */
	public function bp_rest_notifications_update_item_callback() {
		RestAPI::instance()->unread_notifications_count();
	}

	/**
	 * Create bp notification then clear object cache.
	 *
	 * @param array $row     Inserted row data.
	 * @param array $entries Notification data.
	 *
	 * @since 1.8.70
	 */
	public function create_bp_bulk_notifications( $row, $entries = array() ) {
		global $wpdb;
		$table_name             = buddypress()->notifications->table_name;
		$insert_query           = "INSERT INTO {$table_name} ( `user_id`, `item_id`, `secondary_item_id`, `component_name`, `component_action`, `date_notified`, `is_new`)  VALUES ";
		$prepare_var            = array();
		$query_row_placeholders = array();

		foreach ( $entries as $entry ) {
			foreach ( $entry as $e ) {
				$prepare_var[] = $e;
			}
			$query_row_placeholders[] = '(%d, %d ,%d ,%s ,%s ,%s, %d)';
		}

		$insert_query .= implode( ",\n ", $query_row_placeholders );
		$insert        = $wpdb->query( $wpdb->prepare( $insert_query, $prepare_var ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		if ( $insert ) {
			$args = array(
				'user_id'           => $row['user_id'],
				'item_id'           => $row['item_id'],
				'secondary_item_id' => $row['secondary_item_id'],
				'component_name'    => $row['component_name'],
				'component_action'  => $row['component_action'],
				'is_new'            => 1,
			);
			// Pull up a list of items matching the args.
			$notifications = \BP_Notifications_Notification::get( $args );
			foreach ( $notifications as $notification ) {
				bp_notifications_clear_all_for_user_cache( $notification->user_id );
				$this->get_notification_count( $notification->user_id, true );
				wp_cache_delete( $notification->id, 'bp_notifications' );
				wp_cache_delete( 'bp_notifications_check_access_' . $notification->user_id . '_' . $notification->id, 'bp_notifications' );
			}

			return $wpdb->insert_id;
		}

		return false;
	}

	/**
	 * Unset notifications based on license.
	 *
	 * @since 2.2.80
	 *
	 * @param array $notifications Notification array.
	 *
	 * @return array|mixed
	 */
	public function register_notification_preferences( $notifications ) {
		if ( ! bbapp_is_lite_live_app() ) {
			return $notifications;
		}

		unset( $notifications['mentions'] );
		unset( $notifications['members'] );
		unset( $notifications['activity'] );
		unset( $notifications['groups'] );
		unset( $notifications['forums'] );
		unset( $notifications['messages'] );
		unset( $notifications['friends'] );
		unset( $notifications['posts'] );

		return $notifications;
	}

	/**
	 * Account settings get items.
	 *
	 * @param object $navs     Navigation items.
	 * @param object $response Response object.
	 * @param object $request  Request object.
	 *
	 * @since 2.2.80
	 *
	 * @return void
	 */
	public function bbapp_account_settings_get_items( $navs, $response, $request ) {

		if ( bbapp_is_lite_live_app() ) {
			$data = $response->get_data();

			$allow_settings = array(
				'general',
				'notifications',
				'export',
				'delete-account',
			);

			foreach ( $data as $key => $value ) {
				if ( ! in_array( $value['slug'], $allow_settings, true ) ) {
					unset( $data[ $key ] );
				}

				if ( 'notifications' === $value['slug'] ) {
					foreach ( $value['children'] as $child_key => $child_value ) {
						if ( 'subscriptions' === $child_value['slug'] ) {
							unset( $data[ $key ]['children'][ $child_key ] );
						}
					}
				}
			}
			$data = array_values( $data );
			$response->set_data( $data );
		}
	}
}
