<?php
/**
 * Manual Recipients List.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

use BuddyBossApp\Admin\Notification\Compose;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Recipients list.
 * Class ManualRecipientsList
 *
 * @package BuddyBossApp\Notification
 */
class ManualRecipientsList extends \WP_List_Table {

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'singular' => __( 'Recipient', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'Recipients', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // does this table support ajax?
				'screen'   => 'push_recipients', // load push data in custom screen.
			)
		);

		add_action( 'admin_footer', array( &$this, 'admin_header' ) );

	}


	/**
	 * Style list table
	 */
	public function admin_header() {
		$page = ( isset( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

        if ( 'bbapp_pushnotifications' !== $page ) {
			return;
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'member'  => __( 'Member', 'buddyboss-app' ),
			'devices' => __( 'Devices', 'buddyboss-app' ),
		);
		if ( isset( $_REQUEST['push_id'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			/**
			 * Remove the devices column for the old push notification entries
			 */
			$single_push = Push::instance()->get_notification( bbapp_input_clean( wp_unslash( $_REQUEST['push_id'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
			if ( ! empty( $single_push ) && 'manual' === $single_push->agent ) {
				unset( $columns['devices'] );
			}
		}
		return $columns;
	}

	/**
	 * Members columns.
	 *
	 * @param array $item Column item.
	 */
	public function column_member( $item ) {
		if ( ! empty( $item['user_id'] ) ) {
			$user = get_userdata( $item['user_id'] );
			if ( false === $user ) {
				echo sprintf(
					"<div class='bbapp-member-col-entry'><strong>%s</strong></div>",
					esc_html( bbaap_get_user_display_name( $item['user_id'] ) )
				);
			} else {
				echo sprintf(
					"<div class='bbapp-member-col-entry'><a href='%s'><strong>%s</strong></a></div>",
					esc_url( admin_url( 'user-edit.php?user_id=' . $item['user_id'] ) ),
					esc_html( bbaap_get_user_display_name( $item['user_id'] ) )
				);
			}
		}
	}

	/**
	 * Get recipients for automatic push.
	 *
	 * @param array $item List table column item.
	 */
	public function column_devices( $item ) {
		$devices = $item['devices'];
		?>
		<ul class="bbapp-push-devices">
			<?php
			if ( ! empty( $devices ) ) {
				foreach ( $devices as $device ) {
					$nice_error     = '';
					$notice_class   = 'notice-error';
					$notice_tooltip = __( 'Delivery failed', 'buddyboss-app' );

					$status = 0;
					if ( ! empty( $device['devices_data'] ) ) {
						if ( isset( $device['devices_data']['error'] ) ) {
							$nice_error = $device['devices_data']['error']['message'];
						} elseif ( isset( $device['devices_data']['name'] ) ) {
							$status = 1;
						}
					} else {
						$status = $device['devices_status'];
						if ( 1 !== (int) $status ) {
							$nice_error = Push::instance()->nice_error_codes();
							$nice_error = isset( $nice_error[ $status ] ) ? $nice_error[ $status ] : $status[0];
						}
					}

					if ( 1 === (int) $status ) {
						$notice_class   = 'notice-success';
						$notice_tooltip = __( 'Delivery confirmed', 'buddyboss-app' );
					} else {
						$nice_error = "<br/><small>{$nice_error}</small>";
					}

					$tooltip_text = "<b>{$notice_tooltip}</b>{$nice_error}";
					?>
					<li class="<?php echo esc_attr( $notice_class ); ?>">
						<?php
						$html  = '<span style="vertical-align: super;" class="bbapp-device-badge">';
						$html .= ! empty( $device['device'] ) ? esc_html( Push::instance()->get_device_string( $device['device'] ) ) : esc_html( Push::instance()->get_device_string( $device['platform'] ) );
						$html .= '</span>';
						bbapp_tooltip( $tooltip_text, $html );
						?>
					</li>
					<?php
				}
			} else {
				$tooltip_text = sprintf( __( '<b>Delivery status</b><br/><small>Unknown</small>', 'buddyboss-app' ) );
				?>
				<li class="notice-bbapp-no-devices">
					<?php
					$html  = '<span style="vertical-align: super;" class="bbapp-device-badge">';
					$html .= esc_html__( 'N.A.', 'buddyboss-app' );
					$html .= '</span>';
					bbapp_tooltip( $tooltip_text, $html );
					?>
				</li>
				<?php
			}
			?>
		</ul>
		<?php
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {
		$push_id = isset( $_GET['push_id'] ) ? bbapp_input_clean( wp_unslash( $_GET['push_id'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$this->_column_headers = $this->get_column_info();

		$per_page     = $this->get_items_per_page( 'push_notifications_per_page', 10 );
		$current_page = $this->get_pagenum();

		$recipients_data = self::get_recipients_data( $push_id, $per_page, $current_page, true );

		$total_items = $recipients_data['total'];

		$this->items = $recipients_data['result'];

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);

	}

	/**
	 * Displays the table.
	 *
	 * @since 3.1.0
	 */
	public function display() {
		$singular = $this->_args['singular'];

		$this->screen->render_screen_reader_content( 'heading_list' );
		?>
		<table class="wp-list-table <?php echo esc_attr( implode( ' ', $this->get_table_classes() ) ); ?>">
			<thead>
			<tr>
				<?php $this->print_column_headers(); ?>
			</tr>
			</thead>

			<tbody id="the-list"
				<?php
				if ( $singular ) {
					echo esc_attr( " data-wp-lists='list:$singular'" );
				}
				?>
			>
			<?php $this->display_rows_or_placeholder(); ?>
			</tbody>

			<tfoot>
			<tr>
				<?php $this->print_column_headers( false ); ?>
			</tr>
			</tfoot>

		</table>
		<?php
		$this->display_tablenav( 'bottom' );
	}

	/**
	 * Displays the search box.
	 *
	 * @param string $text     The 'submit' button label.
	 * @param string $input_id ID attribute value for the search input field.
	 *
	 * @since 3.1.0
	 */
	public function search_box( $text, $input_id ) {
		if ( empty( $_REQUEST['s'] ) && ! $this->has_items() ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}

		$input_id = $input_id . '-search-input';

		if ( ! empty( $_REQUEST['page'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="page" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		if ( ! empty( $_REQUEST['action'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="action" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		if ( ! empty( $_REQUEST['push_id'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="push_id" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['push_id'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		if ( ! empty( $_REQUEST['orderby'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="orderby" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}

		if ( ! empty( $_REQUEST['order'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="order" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		if ( ! empty( $_REQUEST['post_mime_type'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="post_mime_type" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['post_mime_type'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		if ( ! empty( $_REQUEST['detached'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<input type="hidden" name="detached" value="' . esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['detached'] ) ) ) . '" />'; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
		?>
		<p class="search-box">
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_attr( $text ); ?>:</label>
			<input type="search" placeholder="<?php echo esc_html__( 'Search Recipients', 'buddyboss-app' ); ?>"
			id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>"/>
			<?php submit_button( $text, '', '', false, array( 'id' => 'search-submit' ) ); ?>
		</p>
		<?php
	}

	/**
	 * Get recipients data.
	 *
	 * @param int  $push_id      Push id.
	 * @param int  $per_page     Recipients per page.
	 * @param int  $current_page Current page number.
	 * @param bool $total_count  Total recipients number.
	 *
	 * @return array
	 */
	public static function get_recipients_data( $push_id, $per_page = 1, $current_page = 20, $total_count = false ) {

		$current_page = max( $current_page, 1 );

		$args = array(
			'item_id'      => $push_id,
			'current_page' => $current_page,
			'per_page'     => $per_page,
			'search'       => isset( $_REQUEST['s'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : '', //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
			'total'        => $total_count,
		);

		$push_history = Helpers::instance()->get_push_history( $args );

		$user_recipients = array();
		if ( ! empty( $push_history['users_loops'] ) ) {
			foreach ( $push_history['users_loops'] as $history_key => $history ) {

				$user_recipients[ $history_key ]['user_id'] = $history['user_id'];
				$user_recipients[ $history_key ]['push_id'] = $history['item_id'];
				$users_devices_loops                        = $push_history['users_devices_loops'];
				$devices_records                            = self::array_columns_search( $users_devices_loops, 'user_id', $history['user_id'] );
				$devices_records                            = self::array_columns_search( $devices_records, 'item_id', $history['item_id'] );
				foreach ( $devices_records as $key => $device_record ) {
					$user_recipients[ $history_key ]['devices'][ $key ] = array(
						'device'         => $device_record['device'],
						'platform'       => $device_record['platform'],
						'devices_status' => $device_record['status'],
						'devices_data'   => json_decode( $device_record['data'], true ),
					);
				}
			}
		}

		return array(
			'result' => $user_recipients,
			'total'  => $push_history['total_count'],
		);
	}

	/**
	 * Search column by arrays.
	 *
	 * @param array        $array       List table column items.
	 * @param array|string $column      List table column id.
	 * @param string       $search_item Search content.
	 *
	 * @return array
	 */
	public static function array_columns_search( $array, $column, $search_item ) {
		$filtered = array();

		foreach ( $array as $index => $columns ) {

			foreach ( $columns as $key => $value ) {
				if ( $key === $column && $value === $search_item ) {
					$filtered[] = $columns;
				}
			}
		}

		return $filtered;
	}

	/**
	 * No item found...
	 */
	public function no_items() {
		esc_html_e( 'No Recipients found.', 'buddyboss-app' );
	}
}
