<?php
/**
 * Manual notifcation list.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Renders list of push notifications sent on backend.
 * Class ManualNotificationList
 */
class ManualNotificationList extends \WP_List_Table {

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'screen'   => 'manual-notifications',
				'singular' => __( 'Push Notification', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'Push Notifications', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // does this table support ajax?
			)
		);

		add_action( 'admin_footer', array( &$this, 'admin_header' ) );

	}


	/**
	 * Style list table
	 */
	public function admin_header() {
		$page = ( isset( $_GET['page'] ) ) ? esc_attr( bbapp_input_clean( wp_unslash( $_GET['page'] ) ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( 'bbapp_pushnotifications' !== $page ) {
			return;
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'cb'         => '<input type="checkbox" />',
			'body'       => __( 'Message', 'buddyboss-app' ),
			'title'      => __( 'Title', 'buddyboss-app' ),
			'recipients' => __( 'Recipients', 'buddyboss-app' ),
			'sender'     => __( 'Sender', 'buddyboss-app' ),
			'created'    => __( 'Created', 'buddyboss-app' ),
			'status'     => __( 'Status', 'buddyboss-app' ),
		);

		return $columns;
	}


	/**
	 * Retrieve user apps data from the database
	 *
	 * @param int $per_page    Number of items per page.
	 * @param int $page_number Current page number.
	 *
	 * @return mixed
	 */
	public static function get_push_notifications( $per_page = 10, $page_number = 1 ) {

		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );
		$sql        = $wpdb->prepare( "SELECT * FROM {$table_name} WHERE blog_id = %d", get_current_blog_id() ); //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $_REQUEST['orderby'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended

			if ( 'sender' === bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
				$orderby = 'sent_as';
			} elseif ( 'created' === bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
				$orderby = 'date_updated';
			} else {
				$orderby = 'date_updated';
			}

			$sql .= ' ORDER BY ' . esc_sql( $orderby );
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' ASC'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		} else {
			$sql .= ' ORDER BY date_updated';
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' DESC'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}

		$sql .= " LIMIT $per_page";
		$sql .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;

		$result = $wpdb->get_results( $sql, 'ARRAY_A' ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return $result;
	}


	/**
	 * Delete a user app record.
	 *
	 * @param int $id user app ID.
	 *
	 * @return bool|false|int
	 */
	public static function delete_push_notification( $id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return Push::instance()->delete_notification( $id );
	}


	/**
	 * Returns the count of records in the database.
	 *
	 * @return null|string
	 */
	public static function record_count() {
		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE blog_id = %d", get_current_blog_id() ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}


	/** Text displayed when no user app data is available */
	public function no_items() {
		esc_html_e( 'No notifications available.', 'buddyboss-app' );
	}


	/**
	 * Render a column when no column specific method exist.
	 *
	 * @param array  $item        List table coumn item.
	 * @param string $column_name List table column name.
	 *
	 * @return mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'title':
				if ( empty( $item['primary_text'] ) ) {
					$item['primary_text'] = __( 'N/A', 'buddyboss-app' );
				}

				return '<strong>' . $item['primary_text'] . '</strong>';
			case 'sender':
				if ( empty( $item['sent_as'] ) ) {
					$item['sent_as'] = 1;
				}
				$user_id = $item['sent_as'];

				if ( ! empty( $user_id ) ) {
					printf(
						"<div class='bbapp-thumb-col-entry'><a href='%s'><strong>%s</strong></a></div>",
						esc_url( admin_url( 'user-edit.php?user_id=' . $user_id ) ),
						esc_html( bbaap_get_user_display_name( $user_id ) )
					);

				}

				return false;
			case 'created':
				$date = $item['date_created'];

				if ( '0000-00-00 00:00:00' === $date ) {
					$date = $item['date_updated'];
				}

				/* translators: %1$s: date format %2$s: time format */
				$date_time_format = sprintf( _x( '%1$s %2$s', 'date_at_time', 'buddyboss-app' ), get_option( 'date_format' ), get_option( 'time_format' ) );

				return get_date_from_gmt( $date, $date_time_format );
			case 'schedule':
				$is_schedule = $item['is_schedule'];
				if ( 1 === (int) $is_schedule ) {
					/* translators: %1$s: date format %2$s: time format */
					$date_time_format = sprintf( _x( '%1$s %2$s', 'date_at_time', 'buddyboss-app' ), get_option( 'date_format' ), get_option( 'time_format' ) );

					return sprintf( '%s (%s)', get_date_from_gmt( $item['date_schedule'], $date_time_format ), $item['data']['timezone'] );
				} else {
					return 0;
				}
			default:
				return 'N/A'; // Show the whole array for troubleshooting purposes.
		}
	}

	/**
	 * Sending manual push notification status.
	 *
	 * @param array $item List table column item.
	 */
	public function column_status( $item ) {

		$status      = $item['status'];
		$is_schedule = $item['is_schedule'];

		/* translators: %1$s: date format %2$s: time format */
		$date_time_format = sprintf( _x( '%1$s %2$s', 'date_at_time', 'buddyboss-app' ), get_option( 'date_format' ), get_option( 'time_format' ) );
		$status_data      = Helpers::instance()->get_push_status( $item['id'] );

		if ( '1' === $is_schedule && ! empty( $status_data['status'] ) ) {
			if ( 'sent_all' === $status_data['status'] || 'sent' === $status_data['status'] ) {
				$status = 'sent';
			} elseif ( 'failed' === $status && 'failed' === $status_data['status'] ) {
				$status = 'failed';
			}
		}

		if ( 'sent' === $status ) {
			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'All Notifications Sent', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['date_updated'], $date_time_format ) ) );

		} elseif ( 'pending' === $status ) {

			if ( 1 === (int) $is_schedule ) {
				echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'Scheduled', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['date_schedule'], $date_time_format ) ) );
			} else {
				echo sprintf( '<strong>%1$s</strong>', esc_html__( 'Pending', 'buddyboss-app' ) );
			}
		} elseif ( 'processing' === $status ) {
			echo sprintf( '<strong>%1$s |</strong> %2$d %3$s', esc_html__( 'Sending', 'buddyboss-app' ), esc_html( Helpers::instance()->get_device_left_count( $item['id'] ) ), esc_html__( 'Devices Left', 'buddyboss-app' ) );

		} elseif ( 'expired' === $status ) {
			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'Expired', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['date_expire'], $date_time_format ) ) );

		} elseif ( 'failed' === $status ) {
			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'Failed', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['date_updated'], $date_time_format ) ) );

		} else {
			$status = ucfirst( str_replace( '_', ' ', $status ) );
			echo esc_html( $status );
		}

	}

	/**
	 * Render the bulk edit checkbox
	 *
	 * @param array $item List table column item.
	 *
	 * @return string
	 */
	public function column_cb( $item ) {
		$disabled = false;
		if ( 'processing' === $item['status'] ) {
			$disabled = true;
		}

		return sprintf(
			'<input type="checkbox" name="bulk-delete[]" value="%s" ' . disabled( $disabled, true, false ) . '/>',
			$item['id']
		);
	}

	/**
	 * Render column recipients.
	 *
	 * @param arrray $item Column recipients.
	 */
	public function column_recipients( $item ) {
		$devices_count = Helpers::instance()->get_devices_count( $item['id'] );
		$send_to       = isset( $item['data']['send_to'] ) ? $item['data']['send_to'] : 0;
		$page          = ! empty( $_REQUEST['page'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		/* translators: %d number of total members for getting notification */
		$members_text = sprintf( _n( '%s Member', '%s Members', $send_to, 'buddyboss-app' ), number_format_i18n( $send_to ) );
		/* translators: %d total device count */
		$devices_text = sprintf( _n( '%s Device', '%s Devices', $devices_count, 'buddyboss-app' ), number_format_i18n( $devices_count ) );
		echo sprintf(
			'<a class="row-title" href="admin.php?page=%1$s&action=view_push&push_id=%2$s">%3$s</a><br /><span>%4$s</span>',
			esc_attr( $page ),
			absint( $item['id'] ),
			esc_html( $members_text ),
			esc_html( $devices_text )
		);
	}


	/**
	 * Method for name column
	 *
	 * @param array $item an array of DB data.
	 *
	 * @return string
	 */
	public function column_body( $item ) {

		$delete_nonce = wp_create_nonce( 'cc_delete_push_notification' );

		$title = nl2br( $item['secondary_text'] );
		$page  = ! empty( $_REQUEST['page'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$actions         = array();
		$actions['view'] = sprintf( '<a href="admin.php?page=%1$s&action=view_push&push_id=%2$s">%3$s</a>', esc_attr( $page ), absint( $item['id'] ), esc_html__( 'View', 'buddyboss-app' ) );
		// status other than processing can be deleted.
		if ( 'processing' === $item['status'] ) {
			$actions['delete'] = sprintf( '<a href="?page=%s&action=%s&push_notification=%s&_wpnonce=%s">%s</a>', esc_attr( $page ), 'delete', absint( $item['id'] ), $delete_nonce, esc_html__( 'Delete', 'buddyboss-app' ) );
		}

		return sprintf( '<a class="row-title" href="admin.php?page=%1$s&action=view_push&push_id=%2$s">%3$s</a>', esc_attr( $page ), absint( $item['id'] ), $title ) . $this->row_actions( $actions );
	}


	/**
	 * Returns an associative array containing the bulk action
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		$actions = array(
			'bulk-delete' => __( 'Delete', 'buddyboss-app' ),
		);

		return $actions;
	}


	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {

		$this->_column_headers = $this->get_column_info();

		/** Process bulk action */
		$this->process_bulk_action();

		$per_page     = $this->get_items_per_page( 'push_notifications_per_page', 10 );
		$current_page = $this->get_pagenum();
		$total_items  = self::record_count();

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);

		$this->items = self::get_push_notifications( $per_page, $current_page );

		foreach ( $this->items as $key => $item ) {
			$this->items[ $key ]['data']      = isset( $item['data'] ) ? maybe_unserialize( $item['data'] ) : array();
			$this->items[ $key ]['remaining'] = ( isset( $this->items[ $key ]['data']['send_to'] ) ) ? $this->items[ $key ]['data']['send_to'] : 0 - Helpers::instance()->get_devices_count( $item['id'] );
		}
	}

	/**
	 * Renders the Single Screen of Push Notification.
	 *
	 * @return void
	 * @file include/Admin/Notification/Notifications.php
	 */
	public function pushSingleScreen() {
		$push_id = ! empty( $_REQUEST['push_id'] ) ? (int) bbapp_input_clean( wp_unslash( $_REQUEST['push_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$push    = Push::instance()->get_notification( $push_id );
		if ( empty( $push_id ) || is_wp_error( $push ) ) {
			include bbapp()->plugin_dir . '/views/push-notifications/not-found.php';
		} else {

			include bbapp()->plugin_dir . '/views/push-notifications/single.php';
		}
	}

	/**
	 * Bulk actions.
	 */
	public function process_bulk_action() {

		// Detect when a bulk action is being triggered...
		if ( 'delete' === $this->current_action() ) {

			// In our file that handles the request, verify the nonce.
			$nonce = ! empty( $_REQUEST['_wpnonce'] ) ? esc_attr( bbapp_input_clean( wp_unslash( $_REQUEST['_wpnonce'] ) ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! wp_verify_nonce( $nonce, 'cc_delete_push_notification' ) ) {
				die( 'Go get a life script kiddies' );
			} elseif ( ! empty( $_GET['push_notification'] ) ) {
				self::delete_push_notification( absint( $_GET['push_notification'] ) );

			}
		}

		// If the delete bulk action is triggered.
		if ( ( isset( $_POST['action'] ) && 'bulk-delete' === $_POST['action'] )
			|| ( isset( $_POST['action2'] ) && 'bulk-delete' === $_POST['action2'] )
		) {

			$delete_ids = ! empty( $_POST['bulk-delete'] ) ? esc_sql( bbapp_input_clean( wp_unslash( $_POST['bulk-delete'] ) ) ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			// loop over the array of record IDs and delete them.
			foreach ( $delete_ids as $id ) {
				self::delete_push_notification( $id );
			}
		}
	}
}
