<?php
/**
 * Integration Abstract.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * This is abstract notification integration class.
 * Extending this class will provide all logical feature to register notification & push notification.
 *
 * Class IntegrationAbstract
 *
 * @package BuddyBossApp\Notification
 */
abstract class IntegrationAbstract {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	private static $instances = array();

	/**
	 * Component name.
	 *
	 * @var string $component_name
	 */
	private $component_name = 'bbapp';

	/**
	 * Class construct
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of this class
	 *
	 * @return IntegrationAbstract
	 */
	public static function instance() {
		$class = get_called_class();
		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->init_load();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Load integration abstract hooks.
	 */
	public function init_load() {
		add_action( 'init', array( $this, 'load' ), 99 );
		add_filter( 'bbapp_get_formatted_notification', array( $this, 'get_format_notification' ), 10, 8 );
		add_filter(
			'bbapp_get_formatted_notification_overrides',
			array(
				$this,
				'get_format_notification_override',
			),
			99,
			9
		);
		add_filter( 'bbapp_get_notification_app_link', array( $this, 'get_notification_app_links' ), 10, 5 );
	}

	/**
	 * This function is must needed to call.
	 *
	 * @param string $component_name Component name.
	 * @param bool   $only_api       Only allow this component api.
	 *
	 * @return void
	 */
	public function register_component( $component_name, $only_api = false ) {

		$this->component_name = $component_name;

		$add_filter = true;

		/**
		 * If it's only for API then we only allow this component api.
		 * what this will do is. it will hide all notification content related to bbapp component for web.
		 */
		if ( $only_api ) {
			$add_filter = false;
			$curr_url   = add_query_arg( null, null );
			// if current load is api. then add the filter.
			if ( false !== strpos( $curr_url, 'wp-json' ) ) {
				$add_filter = true;
			}
		}

		if ( $add_filter ) {

			// Register component on bp.
			add_filter(
				'bp_notifications_get_registered_components',
				function ( $component_names = array() ) use ( $component_name ) {
					// Force $component_names to be an array.
					if ( ! is_array( $component_names ) ) {
						$component_names = array();
					}

					// Check Component name already exists or not.
					if ( ! in_array( $component_name, $component_names, true ) ) {
						array_push( $component_names, $component_name );
					}

					return $component_names;
				}
			);

		}
	}

	/**
	 * Returns the component name.
	 *
	 * @return bool
	 */
	public function get_component_name() {
		return $this->component_name;
	}

	/**
	 * Abstract load mothod.
	 *
	 * @return mixed
	 */
	abstract public function load();

	/**
	 * Helper to register subscription type.
	 * The filter calling the global variable is in include/Notification/Actions.php.
	 *
	 * @param string $type        Register type.
	 * @param string $admin_label Admin label.
	 * @param string $user_label  User label.
	 * @param array  $extras      Extra params.
	 *
	 * @return mixed
	 */
	public function register_push_type( $type, $admin_label, $user_label, $extras = array() ) {
		global $bbapp_register_push_type;

		if ( ! is_array( $bbapp_register_push_type ) ) {
			$bbapp_register_push_type = array();
		}

		$extras_defaults = array(
			'enabled'    => true,
			'available'  => true,
			'is_topic'   => false,
			'push_group' => 'other',
		);

		$extras = wp_parse_args( $extras, $extras_defaults );
		if ( ! isset( $bbapp_register_push_type[ $type ] ) ) {
			$bbapp_register_push_type[ $type ] = array(
				'label'       => $user_label,
				'admin_label' => $admin_label,
				'enabled'     => $extras['enabled'],
				'available'   => $extras['available'],
				'is_topic'    => $extras['is_topic'], // if it’s not related to backend & app has to handle.
			);

			$this->add_push_type_in_push_group( $extras['push_group'], array( $type ) );
		}

		return $type;
	}

	/**
	 * Helper to register subscription type.
	 * The filter calling the global variable is in include/Notification/Actions.php.
	 *
	 * @param string $name        Subscription name.
	 * @param string $label       Subscription label.
	 * @param string $admin_label Admin subscription label.
	 * @param array  $extras      Subscription meta.
	 *
	 * @return mixed returns the name of the subscription.
	 *
	 * @see        include/Notification/Actions.php
	 * @deprecated 1.3.3
	 */
	public function register_subscription_type( $name, $label, $admin_label, $extras = array() ) {
		global $bbapp_register_push_groups;
		_deprecated_function( __FUNCTION__, '1.3.7', 'register_push_type()' );

		foreach ( $bbapp_register_push_groups as $push_group => $bbapp_register_push_group ) {
			if ( in_array( $name, $bbapp_register_push_groups[ $push_group ]['settings'], true ) ) {
				$extras['push_group'] = $push_group;
			}
		}

		$name = $this->register_push_type( $name, $admin_label, $label, $extras );

		return $name;
	}

	/**
	 * Helper to register subscription group.
	 * The filter calling the global variable is in include/Notification/Actions.php.
	 *
	 * @param string $name     Subscription name.
	 * @param string $label    Subscription label.
	 * @param array  $settings Subscription grop settings.
	 *
	 * @return mixed returns the name of the subscription.
	 * @see include/Notification/Actions.php
	 * @deprecated
	 */
	public function register_subscription_group( $name, $label, $settings = array() ) {
		_deprecated_function( __FUNCTION__, '1.3.7', 'register_push_group()' );

		$this->register_push_group( $name, $label );
		$this->add_push_type_in_push_group( $name, $settings );

		return $name;
	}

	/**
	 * Add push type in push group.
	 *
	 * @param string $name       Push type group name.
	 * @param array  $push_types Push types.
	 *
	 * @since 1.4.1
	 *
	 * @return mixed
	 */
	private function add_push_type_in_push_group( $name, $push_types ) {
		global $bbapp_register_push_groups;

		if ( ! is_array( $bbapp_register_push_groups ) ) {
			$bbapp_register_push_groups = array();
		}

		if ( 'other' === $name && ! isset( $bbapp_register_push_groups['other'] ) ) {
			$bbapp_register_push_groups['other'] = array( 'label' => __( 'Other', 'buddyboss-app' ) );
		}

		if ( isset( $bbapp_register_push_groups[ $name ] ) ) {

			if ( empty( $bbapp_register_push_groups[ $name ]['settings'] ) ) {
				$bbapp_register_push_groups[ $name ]['settings'] = array();
			}
			// Todo: key update [settings to push_types].
			$bbapp_register_push_groups[ $name ]['settings'] = array_unique(
				array_merge( $bbapp_register_push_groups[ $name ]['settings'], $push_types )
			);
		}

		return $name;
	}

	/**
	 * Helper to register subscription group.
	 * The filter calling the global variable is in include/Notification/Actions.php.
	 *
	 * @param string $name  Group name.
	 * @param string $label Group label.
	 *
	 * @return mixed returns the name of the subscription.
	 * @see include/Notification/Actions.php
	 */
	public function register_push_group( $name, $label ) {
		global $bbapp_register_push_groups;

		if ( ! is_array( $bbapp_register_push_groups ) ) {
			$bbapp_register_push_groups = array();
		}

		if ( ! isset( $bbapp_register_push_groups[ $name ] ) ) {
			$bbapp_register_push_groups[ $name ] = array(
				'label' => $label,
			);
		}

		return $name;
	}

	/**
	 * Sends the push notification to users.
	 * Also can create normal notification.
	 * Push Notification can be ignored based on subscription.
	 *
	 * @param array  $notification {
	 *                              'primary_text'         (required) Primary Text
	 *                              'secondary_text'       (required) Secondary Text
	 *                              'user_ids'             (required) Users ID to send Push Notifications.
	 *                              'device_ids'           (required) Users device IDs to send Push Notifications.
	 *                              'sent_as'              (optional) default 1,
	 *                              'data'                 (optional) default array(),
	 *                              'agent'                (optional) default 'event',
	 *                              'topic'                (optional) default '',
	 *                              'priority'             (optional) default 9,
	 *                              'blog_id'              (optional) default get_current_blog_id(),
	 *                              'subscription_type'    (optional) default 'manual_push_notification' replaced by push_type, deprecated 1.3.3
	 *                              'push_notification_id' (optional) // Notification Push ID
	 *                              'normal_notification'  (optional) // true or false
	 *                              'normal_notification_data'  (optional) // Param should be same as send_normal() function.
	 *                              }.
	 * @param string $type push type (optional - recommended).
	 *
	 * @return \WP_Error | boolean
	 */
	public function send_push( $notification, $type = '' ) {
		if ( isset( $notification['normal_notification_data'] ) ) {
			if ( ! empty( $notification['normal_notification_data']['component_name'] ) ) {
				$notification['normal_notification_data']['component_name'] = $this->component_name;
			}
			if ( ! empty( $component_action ) ) {
				$notification['normal_notification_data']['component_action'] = $component_action;
			}
		}

		$notification['type'] = $type;
		return bbapp_send_push_notification( $notification );
	}


	/**
	 * Sends a normal notification on backend.
	 *
	 * @param array $data {
	 *              'user_id' // default 0,
	 *              'user_ids' // default array(),
	 *              'item_id' // default 0,
	 *              'secondary_item_id' // default  0,
	 *              'component_name' // default 'bbapp',
	 *              'component_action' // default '',
	 *              'blog_id' // default get_current_blog_id(),
	 *              'version' // default 2
	 *              }.
	 *
	 * @return bool
	 */
	public function send_normal( $data ) {
		if ( ! isset( $data['component_name'] ) ) {
			$data['component_name'] = $this->component_name;
		}

		return Notification::instance()->create_notification( $data );
	}

	/**
	 * Remove notification.
	 *
	 * @param array $args {
	 *                   'component_action'
	 *                   'user_id'
	 *                   'item_id'
	 *                   'secondary_item_id'
	 *                   'component_name'
	 *                   'blog_id'
	 *                   } delete by.
	 * @param bool  $only_bbapp Remove notification only from the app.
	 *
	 * @return false|int
	 */
	public function delete_normal( $args, $only_bbapp = false ) {
		return Notification::instance()->delete_notification( $args, $only_bbapp );
	}

	/**
	 * You can override notifications links for app only.
	 * This links will be override on API & Push Notifications related to given component.
	 * This function should be override.
	 *
	 * @param string $link              Notification link.
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action hook.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 *
	 * @return string
	 */
	public function notification_app_links( $link, $component_name, $component_action, $item_id, $secondary_item_id ) {
		return $link;
	}

	/**
	 * This get attach to the action bbapp_get_notification_app_link & calls notification_app_links safely.
	 *
	 * @param string $link              Notification link.
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action hook.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 *
	 * @return string
	 */
	public function get_notification_app_links( $link, $component_name, $component_action, $item_id, $secondary_item_id ) {

		$get_link = $this->notification_app_links( $link, $component_name, $component_action, $item_id, $secondary_item_id );

		if ( isset( $get_link ) ) {
			if ( ! bbapp_is_valid_url( $get_link ) ) {
				$get_link = false;
			}
			$link = $get_link;
		}

		return $link;
	}

	/**
	 * Register format the notifications format for API, Web & Push
	 *
	 * For override only by class.
	 *
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action hook.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 * @param int    $notification_id   Notification id.
	 * @param int    $total_items       Total item number.
	 * @param string $screen            web | app_push | web_push.
	 *
	 * @return bool {
	 *  "title" => "" // Optional
	 *  "link" => "" // Must be URL eg. http://example.com/example/example
	 *  "text" => "" // Link Text eg. Someone replied to your topic.
	 * }
	 */
	public function format_notification( $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $screen ) {
		return false;
	}

	/**
	 * Attached to the bbapp notification render filter.
	 *
	 * @param string $content           Notification content.
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action hook.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 * @param int    $notification_id   Notification id.
	 * @param int    $total_items       Total item number.
	 * @param string $screen            web | app_push | web_push.
	 *
	 * @return mixed
	 */
	public function get_format_notification( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $screen = 'web' ) {

		$return = $this->format_notification( $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $screen = 'web' );

		if ( isset( $return['text'] ) && isset( $return['link'] ) ) {
			if ( ! isset( $return['title'] ) ) {
				$return['title'] = '';
			}

			return $return;
		}

		return $content;
	}


	/***
	 * Override the existing registered notifications format for API, Web & Push.
	 *
	 * @param string         $content           Notification content.
	 * @param string         $component_name    Component name.
	 * @param string         $component_action  Component action hook.
	 * @param int            $item_id           Item id.
	 * @param int            $secondary_item_id Secondary item id.
	 * @param int            $notification_id   Notification id.
	 * @param int            $total_items       Total item number.
	 * @param array | string $format            Only used in web version.
	 * @param string         $screen            web | app_push | web_push.
	 *
	 * @return mixed array {
	 *  "title" => "" // Optional
	 *  "link" => "" // Must be URL eg. http://example.com/example/example
	 *  "text" => "" // Link Text eg. Someone replied to your topic.
	 * }
	 */
	public function format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen ) {
		return $content;
	}

	/**
	 * Attached to the bbapp notification render filter.
	 *
	 * @param string         $content           Notification content.
	 * @param string         $component_name    Component name.
	 * @param string         $component_action  Component action hook.
	 * @param int            $item_id           Item id.
	 * @param int            $secondary_item_id Secondary item id.
	 * @param int            $notification_id   Notification id.
	 * @param int            $total_items       Total item number.
	 * @param array | string $format            Only used in web version.
	 * @param string         $screen            web | app_push | web_push.
	 *
	 * @return mixed
	 */
	public function get_format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen = 'web' ) {
		if ( 'array' !== $format ) {
			return $content;
		}

		if ( 'array' === $format && ! isset( $content['title'] ) ) {
			$content['title'] = '';
		}

		$return = $this->format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen );

		if ( 'array' === $format && isset( $return['text'] ) && isset( $return['link'] ) ) {
			return $return;
		}

		return $content;
	}

	/**
	 * Register Normal Notification (BuddyBoss Platform Legacy Notification) to Push Notification.
	 * So when this is register when ever a normal notification is created the push will be sent to user also.
	 * Note - This function only works with platform legacy notification system. in modern platform notification it's
	 * automatic supported with all notification registered with platform notification API.
	 *
	 * @param string $component_name   Component name.
	 * @param string $component_action Component action hook.
	 * @param string $push_type        Name of the push subscription. you can create using
	 *                                    $this->register_push_type().
	 */
	public function register_push_to_normal( $component_name, $component_action, $push_type = 'custom' ) {
		Notification::instance()->register_push_to_normal_notification( $component_name, $component_action, $push_type );
	}
}
