<?php
/**
 * Holds Notification helper functions.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

/**
 * Notification helper class.
 */
class Helpers {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Notification table name.
	 *
	 * @var string $table_name
	 */
	private $table_name = '';

	/**
	 * Construct method.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Helpers
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load additional hooks such as : admin_init
	 * NOTE : Method is similar to concept of init
	 *
	 * @return void
	 */
	public function load() {
		global $wpdb;
		$this->table_name = "{$wpdb->prefix}bbapp_push_notifications_history";
	}

	/**
	 * Get push history.
	 *
	 * @param array  $args          Set of arguments.
	 * @param string $distinct_by   Distinct by field.
	 * @param string $extra_columns Set of extra columns.
	 *
	 * @return array
	 */
	public function get_push_history( $args, $distinct_by = 'user_id', $extra_columns = 'item_id' ) {
		global $wpdb;
		$default_args = array(
			'user_id'      => false,
			'item_id'      => false,
			'per_page'     => 10,
			'current_page' => 1,
			'orderby'      => 'id',
			'order'        => 'desc',
			'status'       => 'all',
			'search'       => false,
			'total'        => false,
			'type'         => false,
			'not_type'     => false,
			'platform'     => array( 'ios', 'android' ),
		);

		$args = array_merge( $default_args, $args );

		$where_clause = array();

		if ( ! empty( $args['user_id'] ) ) {
			$where_clause[] = $wpdb->prepare( 'user_id=%s', $args['user_id'] );
		}
		if ( ! empty( $args['item_id'] ) ) {
			$where_clause[] = $wpdb->prepare( 'item_id=%s', $args['item_id'] );
		}

		if ( 'all' !== $args['status'] ) {
			$where_clause[] = $wpdb->prepare( 'status=%s', $args['status'] );
		}

		if ( ! empty( $args['platform'] ) && is_array( $args['platform'] ) ) {
			$where_clause[] = wp_unslash( $wpdb->prepare( ' platform IN (%s)', implode( "','", $args['platform'] ) ) );
		}

		if ( ! empty( $args['type'] ) ) {
			if ( is_array( $args['type'] ) ) {
				$where_clause[] = wp_unslash( $wpdb->prepare( 'type IN (%s)', implode( "','", $args['type'] ) ) );
			} else {
				$where_clause[] = $wpdb->prepare( 'type=%s', $args['type'] );
			}
		}
		if ( ! empty( $args['not_type'] ) ) {
			if ( is_array( $args['not_type'] ) ) {
				$where_clause[] = wp_unslash( $wpdb->prepare( 'type NOT IN (%s)', implode( "','", $args['not_type'] ) ) );
			} else {
				$where_clause[] = $wpdb->prepare( 'type!=%s', $args['not_type'] );
			}
		}

		$searched_users_ids = $this->get_searched_users_ids( $args['search'] );

		if ( ! empty( $searched_users_ids ) && is_array( $searched_users_ids ) ) {
			$where_clause[] = ' user_id IN (' . implode( ',', $searched_users_ids ) . ')';
		}

		// Multisite support.
		$where_clause[] = $wpdb->prepare( 'blog_id=%s', get_current_blog_id() );

		$where_clause_query = '';
		if ( ! empty( $where_clause ) ) {
			$where_clause_query = 'WHERE ' . implode( ' AND ', $where_clause );
		}

		$limit_clause = '';

		if ( $args['per_page'] ) {
			$args['per_page']     = (int) $args['per_page'];
			$args['current_page'] = (int) $args['current_page'];

			$limit_clause  = " LIMIT {$args["per_page"]} ";
			$limit_clause .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
		}

		$order_by_clause = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order_by_clause .= ' ORDER BY ' . esc_sql( $args['orderby'] );
			$order_by_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$extra_columns      = is_array( $extra_columns ) ? implode( ' , ', $extra_columns ) : $extra_columns;
		$total_member_count = 0;

		if ( true === $args['total'] ) {
			$total_member_sql = "SELECT SQL_CALC_FOUND_ROWS DISTINCT({$distinct_by}) , {$extra_columns} FROM {$this->table_name} {$where_clause_query}";
			$wpdb->get_var( $total_member_sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			$total_member_count = $wpdb->num_rows;
		}

		$list_of_member_sql     = "SELECT DISTINCT({$distinct_by}) , {$extra_columns} FROM {$this->table_name} {$where_clause_query} {$order_by_clause} {$limit_clause} ";
		$list_of_member_results = $wpdb->get_results( $list_of_member_sql, ARRAY_A ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$user_ids               = array_column( $list_of_member_results, 'user_id' );

		if ( ! empty( $user_ids ) && is_array( $user_ids ) ) {
			$where_clause[] = ' user_id IN (' . implode( ',', $user_ids ) . ')';

		}

		if ( ! empty( $where_clause ) ) {
			$where_clause_query = 'WHERE ' . implode( ' AND ', $where_clause );
		}

		$sql = "SELECT * FROM {$this->table_name} $where_clause_query {$order_by_clause}";

		return array(
			'total_count'         => $total_member_count,
			'users_loops'         => $list_of_member_results,
			'users_devices_loops' => $wpdb->get_results( $sql, ARRAY_A ), //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		);
	}

	/**
	 * Get searched users ids.
	 *
	 * @param string $search_string Search term.
	 *
	 * @return array|false
	 */
	public function get_searched_users_ids( $search_string ) {
		if ( empty( $search_string ) ) {
			return false;
		}
		$users = new \WP_User_Query(
			array(
				'search'         => '*' . esc_attr( $search_string ) . '*',
				'fields'         => 'ID',
				'search_columns' => array(
					'user_login',
					'user_nicename',
					'user_email',
				),
			)
		);

		return $users->get_results();
	}

	/**
	 * Get all push history.
	 *
	 * @param int $push_id Push notification id.
	 *
	 * @return null|array|object
	 */
	public function get_push_history_by_push_id( $push_id ) {

		global $wpdb;

		return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `{$this->table_name}` where item_id=%s", $push_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Get devices count
	 *
	 * @param int $push_id Push notification id.
	 *
	 * @return null|string
	 */
	public function get_devices_count( $push_id ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM `{$this->table_name}` where item_id=%s", $push_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Push notification status.
	 *
	 * @param int $push_id Push notification id.
	 *
	 * @return array
	 */
	public function get_push_status( $push_id ) {
		$push_history = $this->get_push_history_by_push_id( $push_id );
		$data         = array(
			'sent'  => 0,
			'error' => 0,
			'total' => 0,
		);
		foreach ( $push_history as $history ) {

			// New firebase response support.
			if ( ! empty( $history->data ) ) {
				$history_data = json_decode( $history->data, true );
				if ( isset( $history_data['name'] ) ) {
					$history->status = 1;
				}
			}

			if ( 1 === (int) $history->status ) {
				$data['sent'] ++;
			} else {
				$data['error'] ++;
			}
		}
		$data['total'] = $data['sent'] + $data['error'];

		$push_status = array(
			'data'   => $data,
		);

		if ( ( ! empty( $data['total'] ) && ! empty( $data['sent'] ) ) && $data['total'] === $data['sent'] ) {
			$push_status['status'] = 'sent_all';
		} elseif ( $data['total'] === $data['error'] ) {
			$push_status['status'] = 'failed';
		} else {
			$push_status['status'] = 'sent';
		}

		return $push_status;
	}

	/**
	 * Push notification status.
	 *
	 * @param int $push_id Push notification id.
	 *
	 * @return array
	 */
	public function get_push_single_status( $push_id ) {
		global $wpdb;

		$status             = array();
		$status['total']    = $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM `{$this->table_name}` WHERE item_id=%s ", $push_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$status['sent']     = $status['total']; // It will show the count of notifications of total devices.
		$status['delivery'] = $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM `{$this->table_name}` WHERE item_id=%s AND status=1", $push_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! $status['delivery'] ) {
			$status['delivery'] = $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM `{$this->table_name}` WHERE item_id=%s AND data LIKE '%name%'", $push_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.LikeWildcardsInQuery
		}

		return $status;
	}

	/**
	 * Get the remained device count.
	 *
	 * @param int $push_id Push notification id.
	 *
	 * @return null|int|string
	 */
	public function get_device_left_count( $push_id ) {
		$total_push    = Push::instance()->get_push_notification_queue_count( $push_id );
		$total_process = $this->get_devices_count( $push_id );

		return $total_push - $total_process;
	}
}
