<?php
/**
 * Holds functionality related post comments push notifications.
 *
 * @package BuddyBossApp\Notification\Core
 */

namespace BuddyBossApp\Notification\Core;

use BuddyBossApp\Admin\Settings;
use BuddyBossApp\Notification\IntegrationAbstract;

/**
 * Post Comment reply push notification support.
 * Class PostCommentNotification
 *
 * @package BuddyBossApp\Notification\Core
 */
class PostCommentNotification extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Load methods.
	 */
	public function load() {
		if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {
			$this->hooks();
		}
		if ( 'push_notifications' !== Settings::instance()->get_current_sub_tab() || ! defined( 'BP_PLATFORM_VERSION' ) ) {
			$this->push_types();
			$this->link_notifications();
		}
	}

	/**
	 * Load hooks and filter.
	 *
	 * @since 1.7.90
	 */
	public function hooks() {
		add_action( 'wp_insert_comment', array( $this, 'send_post_comment_notification' ), 10, 2 );
		add_action( 'transition_comment_status', array( $this, 'bbapp_post_comment_on_status_change' ), 10, 3 );
	}

	/**
	 * Register Subscriptions Types
	 */
	public function push_types() {
		// Comment.
		$this->register_push_group( 'comments', __( 'Comments', 'buddyboss-app' ) );
		/* translators: %s: User display name. */
		$this->register_push_type( 'bbapp_new_comment_reply', __( 'A user receives a reply to their post comment', 'buddyboss-app' ), __( 'When someone replies to your comment', 'buddyboss-app' ), array( 'push_group' => 'comments' ) );
	}

	/**
	 * Link Normal Notification to Push.
	 */
	public function link_notifications() {
		// post comment.
		$this->register_push_to_normal( 'core', 'bb_posts_new_comment_reply', 'bbapp_new_comment_reply' );
	}

	/**
	 * Attached to the bbapp notification render filter.
	 *
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action hook.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 * @param int    $notification_id   Notification id.
	 * @param int    $total_items       Total item number.
	 * @param string $screen            web | app_push | web_push.
	 *
	 * @return array|void
	 */
	public function format_notification( $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $screen ) {
		/**
		 * This will help to update notification content for web.
		 * You need to return data in following structure
		 *  array(
		 *      'text' => Notification text/html,
		 *      'link' => link of content,
		 *  )
		 */

		if ( 'bbapp_new_comment_reply' === $component_action ) {
			return array(
				'text' => sprintf(
				/* translators: 1: Reply text. */
					__( 'Replied to your comment: "%1$s"', 'buddyboss-app' ),
					get_comment_text( $item_id ),
				),
				'link' => get_comment_link( $item_id ),
			);
		}
	}

	/**
	 * Send post comment notification.
	 *
	 * @param int         $comment_id The comment ID.
	 * @param \WP_Comment $comment    Comment object.
	 *
	 * @since 1.7.90
	 * @return false|void
	 */
	public function send_post_comment_notification( $comment_id, $comment ) {

		// Don't send notification if the comment hasn't been approved.
		if ( empty( $comment->comment_approved ) ) {
			return false;
		}

		// Get the post.
		$post = get_post( $comment->comment_post_ID );
		if (
			! is_a( $post, 'WP_Post' ) ||
			( isset( $post->post_type ) && 'post' !== $post->post_type ) // Allow only for the WP default post only.
		) {
			return false;
		}

		if ( 0 !== $comment->comment_parent ) {

			$parent_comment = get_comment( $comment->comment_parent );

			// Get the user id by the comment author email.
			$comment_author_id = email_exists( $comment->comment_author_email );

			if ( ! empty( $comment_author_id ) && (int) $comment_author_id === (int) $parent_comment->user_id ) {
				return false;
			}

			// Collect push data from comment meta data.
			$comment_author_id        = ! empty( $comment_author_id ) ? $comment_author_id : $comment->user_id;
			$comment_content          = $comment->comment_content;
			$excerpt                  = wp_strip_all_tags( $comment_content );
			$comment_author_name      = ! empty( $comment_author_id ) ? bbaap_get_user_display_name( $comment_author_id ) : $comment->comment_author;
			$comment_link             = get_comment_link( $comment_id );
			$parent_comment_author_id = (int) $parent_comment->user_id;
			$device_ids               = array();
			$user_ids                 = array();

			if ( $parent_comment_author_id ) {
				$user_ids[] = $parent_comment_author_id;
			} else {
				if ( class_exists( '\BuddyBossApp\Auth\UserDevice' ) ) {
					$device_uuid = get_comment_meta( $parent_comment->comment_ID, 'bb-app-device-id' );
					$device_ids  = \BuddyBossApp\Auth\UserDevice::instance()->get_device_ids_by_uuid( $device_uuid );
				}
			}

			if ( ! empty( $excerpt ) ) {
				$text = sprintf(
				/* translators: Excerpt. */
					__( 'Replied to your comment: "%s"', 'buddyboss-app' ),
					$excerpt
				);
			} else {
				$text = __( 'Replied to your comment', 'buddyboss-app' );
			}

			$push_args = array(
				'primary_text'             => $comment_author_name,
				'secondary_text'           => $text,
				'data'                     => array(),
				'push_data'                => array(
					'link' => $comment_link,
				),
				'subscription_type'        => 'bbapp_new_comment_reply',
				'normal_notification'      => true,
				'normal_notification_data' => array(
					'component_name'    => 'core',
					'component_action'  => 'bbapp_new_comment_reply',
					'item_id'           => $comment_id,
					'secondary_item_id' => $comment_author_id,
				),
			);

			if ( ! empty( $user_ids ) ) {
				$push_args['user_ids'] = $user_ids;
			}
			if ( ! empty( $user_ids ) ) {
				$push_args['device_ids'] = $device_ids;
			}

			$this->send_push(
				$push_args,
				'bbapp_new_comment_reply'
			);

			update_comment_meta( $comment_id, 'bbapp_comment_push_after_approved', 1 );
		}
	}

	/**
	 * Check post comment status on transition_comment_status hook and send the new comment reply push if not sent already.
	 *
	 * @param string      $new_status New comment status.
	 * @param string      $old_status Previous comment status.
	 * @param \WP_Comment $comment    Comment data.
	 *
	 * @since 1.7.90
	 */
	public function bbapp_post_comment_on_status_change( $new_status, $old_status, $comment ) {
		$push_already_sent = get_comment_meta( $comment->comment_ID, 'bbapp_comment_push_after_approved', true );
		if (
			empty( $push_already_sent ) &&
			'approved' === $new_status &&
			in_array( $old_status, array( 'unapproved', 'spam' ), true )
		) {
			$this->send_post_comment_notification( $comment->comment_ID, $comment );
		}
	}

}
