<?php
/**
 * Automatic notification list.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Renders list of push notifications sent on backend.
 * Class AutomaticNotificationList
 */
class AutomaticNotificationList extends \WP_List_Table {

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'screen'        => 'automatic-notifications',
				'singular'      => __( 'Push Notification', 'buddyboss-app' ), // singular name of the listed records.
				'plural'        => __( 'Push Notifications', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'          => false, // does this table support ajax?
				'is_search_box' => true, // Custom param for showing search box.
			)
		);

		add_action( 'admin_footer', array( &$this, 'admin_header' ) );

	}

	/**
	 * Style list table
	 */
	public function admin_header() {
		$page = ( isset( $_GET['page'] ) ) ? esc_attr( bbapp_input_clean( wp_unslash( $_GET['page'] ) ) ) : false; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( 'bbapp_pushnotifications' !== $page ) {
			return;
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'type'       => __( 'Type', 'buddyboss-app' ),
			'recipients' => __( 'Recipients', 'buddyboss-app' ),
			'devices'    => __( 'Devices', 'buddyboss-app' ),
			'status'     => __( 'Status', 'buddyboss-app' ),
		);

		return $columns;
	}

	/**
	 * Method for name column
	 *
	 * @param array $item an array of DB data.
	 */
	public function column_type( $item ) {
		global $bbapp_register_push_type;
		$type_labels = $this->get_notification_type_label( $item['type'] );
		if ( ! empty( $type_labels ) ) {
			echo esc_html( $type_labels );
		} elseif ( isset( $bbapp_register_push_type[ $item['type'] ] ) ) {
			echo esc_html( $bbapp_register_push_type[ $item['type'] ]['admin_label'] ); // Custom type.
		} else {
			echo esc_html__( 'Unspecified Notification', 'buddyboss-app' );
		}
	}

	/**
	 * Get recipients for automatic push.
	 *
	 * @param array $item Notification item.
	 */
	public function column_recipients( $item ) {
		$user_id = $item['user_id'];
		if ( ! empty( $user_id ) ) {
			printf(
				"<div class='bbapp-thumb-col-entry'><a href='%s'><strong>%s</strong></a></div>",
				esc_url( admin_url( 'user-edit.php?user_id=' . $user_id ) ),
				esc_html( bbaap_get_user_display_name( $user_id ) )
			);
		} elseif ( 0 === (int) $user_id ) {
			printf(
				"<div class='bbapp-thumb-col-entry'><strong>%s</strong></div>",
				esc_html__( 'Guest user', 'buddyboss-app' )
			);
		}
	}

	/**
	 * Get recipients for automatic push.
	 *
	 * @param array $item Notification item.
	 */
	public function column_devices( $item ) {
		$devices = $item['devices'];
		if ( ! empty( $devices ) ) {
			?>
			<ul class="bbapp-push-devices">
				<?php
				foreach ( $devices as $device ) {
					$nice_error     = '';
					$notice_class   = 'notice-error';
					$notice_tooltip = __( 'Delivery failed', 'buddyboss-app' );
					$status         = 0;
					if ( ! empty( $device['devices_data'] ) ) {
						if ( isset( $device['devices_data']['error'] ) ) {
							$nice_error = $device['devices_data']['error']['message'];
						} elseif ( isset( $device['devices_data']['name'] ) ) {
							$status = 1;
						}
					} else {
						$status = $device['devices_status'];
						if ( 1 !== (int) $status ) {
							$nice_error = Push::instance()->nice_error_codes();
							$nice_error = isset( $nice_error[ $status ] ) ? $nice_error[ $status ] : $status[0];
						}
					}

					if ( 1 === (int) $status ) {
						$notice_class   = 'notice-success';
						$notice_tooltip = __( 'Delivery confirmed', 'buddyboss-app' );
					} else {
						$nice_error = "<br/><small>{$nice_error}</small>";
					}

					$tooltip_text = "<b>{$notice_tooltip}</b>{$nice_error}";
					?>
					<li class="<?php echo esc_attr( $notice_class ); ?>">
						<?php
						$html = '<span style="vertical-align: super;" class="bbapp-device-badge">';
						$html .= ! empty( $device['device'] ) ? esc_html( Push::instance()->get_device_string( $device['device'] ) ) : esc_html( Push::instance()->get_device_string( $device['platform'] ) );
						$html .= '</span>';
						bbapp_tooltip( $tooltip_text, $html );
						?>
					</li>
				<?php } ?>
			</ul>
			<?php
		}
	}

	/**
	 * Column status.
	 *
	 * @param array $item Notification item.
	 */
	public function column_status( $item ) {

		$status           = '';
		$date_time_format = sprintf( '%1$s %2$s', esc_html( get_option( 'date_format' ) ), esc_html( get_option( 'time_format' ) ) );
		$status_data      = Helpers::instance()->get_push_status( $item['item_id'] );
		if ( ! empty( $status_data['status'] ) ) {
			if ( 'sent_all' === $status_data['status'] || 'sent' === $status_data['status'] ) {
				$status = 'sent';
			} elseif ( 'failed' === $status_data['status'] ) {
				$status = 'failed';
			}
		}

		if ( 'sent' === $status ) {

			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'All Notifications Sent', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['processed_at'], $date_time_format ) ) );

		} elseif ( 'processing' === $status ) {

			echo sprintf( '<strong>%1$d |</strong> %2$d %3$d <br /><span>%4$s</span>', esc_html__( 'Sending', 'buddyboss-app' ), esc_html( Helpers::instance()->get_device_left_count( $item['id'] ) ), esc_html__( 'Devices Left', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['processed_at'], $date_time_format ) ) );

		} elseif ( 'expired' === $status ) {

			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'Expired', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['processed_at'], $date_time_format ) ) );

		} elseif ( 'failed' === $status ) {

			echo sprintf( '<strong>%1$s</strong> <br /><span>%2$s</span>', esc_html__( 'Failed', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['processed_at'], $date_time_format ) ) );

		} else {
			$status = ucfirst( str_replace( '_', ' ', $status ) );
			echo esc_html( $status );
		}

	}

	/**
	 * Render a column when no column specific method exist.
	 *
	 * @param array  $item        Column item.
	 * @param string $column_name Column name.
	 *
	 * @return mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			default:
				return 'N/A'; // Show the whole array for troubleshooting purposes.
		}
	}

	/**
	 * Retrieve user apps data from the database
	 *
	 * @param int $per_page    Per page.
	 * @param int $page_number Page number.
	 *
	 * @return mixed
	 */
	public static function get_push_notifications( $per_page = 10, $page_number = 1 ) {
		$args = array(
			'current_page' => $page_number,
			'per_page'     => $per_page,
			'search'       => isset( $_REQUEST['s'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			'total'        => true,
			'not_type'     => array( 'manual_push_notification', 'silent_push' ),
			'orderby'      => 'processed_at',
		);

		$push_history         = Helpers::instance()->get_push_history(
			$args,
			'type',
			array(
				'user_id',
				'item_id',
				'processed_at',
				'data',
			)
		);
		$automatic_push_users = array();
		if ( ! empty( $push_history['users_loops'] ) ) {
			foreach ( $push_history['users_loops'] as $history_key => $history ) {
				$automatic_push_users[ $history_key ]['user_id']      = $history['user_id'];
				$automatic_push_users[ $history_key ]['item_id']      = $history['item_id'];
				$automatic_push_users[ $history_key ]['type']         = $history['type'];
				$automatic_push_users[ $history_key ]['processed_at'] = $history['processed_at'];
				$automatic_push_users_devices                         = array();
				$users_devices_loops                                  = $push_history['users_devices_loops'];
				$devices_records                                      = ManualRecipientsList::array_columns_search( $users_devices_loops, 'user_id', $history['user_id'] );
				$devices_records                                      = ManualRecipientsList::array_columns_search( $devices_records, 'item_id', $history['item_id'] );
				$devices_records                                      = ManualRecipientsList::array_columns_search( $devices_records, 'processed_at', $history['processed_at'] );
				$devices_records                                      = ManualRecipientsList::array_columns_search( $devices_records, 'data', $history['data'] );
				foreach ( $devices_records as $key => $device_record ) {
					if ( $history['type'] === $device_record['type'] ) {
						$automatic_push_users_devices[ $key ] = array();
						if ( ! in_array( $device_record['device'], $automatic_push_users_devices[ $key ], true ) ) {
							$automatic_push_users_devices[ $key ] = array(
								'device'         => $device_record['device'],
								'platform'       => $device_record['platform'],
								'devices_status' => $device_record['status'],
								'type'           => $device_record['type'],
								'user_id'        => $device_record['user_id'],
								'devices_data'   => ! empty( $device_record['data'] ) ? json_decode( $device_record['data'], true ) : '',
							);
						}
					}
				}
				$automatic_push_users[ $history_key ]['devices'] = $automatic_push_users_devices;
			}
		}

		return array(
			'result' => $automatic_push_users,
			'total'  => $push_history['total_count'],
		);
	}

	/** Text displayed when no user app data is available */
	public function no_items() {
		esc_html_e( 'No notifications available.', 'buddyboss-app' );
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {

		$this->_column_headers = $this->get_column_info();

		$per_page     = $this->get_items_per_page( 'push_notifications_per_page', 10 );
		$current_page = $this->get_pagenum();

		$automatic_push = self::get_push_notifications( $per_page, $current_page );
		$this->items    = $automatic_push['result'];
		$total_items    = $automatic_push['total'];

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => (int) $per_page, // WE have to determine how many items to show on a page.
			)
		);

	}

	/**
	 * Displays the table structure.
	 *
	 * @param string $which which position of table.
	 *
	 * @since 1.4.1
	 */
	public function display_tablenav( $which ) {
		?>
		<div class="tablenav <?php echo esc_attr( $which ); ?>">

			<div class="alignleft actions">
				<?php $this->bulk_actions(); ?>
			</div>
			<?php
			$this->extra_tablenav( $which );
			$this->pagination( $which );
			?>
			<br class="clear"/>
		</div>
		<?php
	}

	/**
	 * Displays the search box.
	 *
	 * @param string $text     The 'submit' button label.
	 * @param string $input_id ID attribute value for the search input field.
	 *
	 * @since 3.1.0
	 */
	public function search_box( $text, $input_id ) {
		if ( empty( $_REQUEST['s'] ) && ! $this->has_items() ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		$input_id = $input_id . '-search-input';

		if ( ! empty( $_REQUEST['sub-setting'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo sprintf( '<input type="hidden" name="sub-setting" value="%s"/>', esc_attr( wp_unslash( $_REQUEST['sub-setting'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}
		if ( ! empty( $_REQUEST['orderby'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo sprintf( '<input type="hidden" name="orderby" value="%s"/>', esc_attr( wp_unslash( $_REQUEST['orderby'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}
		if ( ! empty( $_REQUEST['order'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo sprintf( '<input type="hidden" name="order" value="%s"/>', esc_attr( wp_unslash( $_REQUEST['order'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		?>
		<p class="search-box">
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_html( $text ); ?>:</label>
			<input type="search" placeholder="<?php echo esc_html__( 'Search Recipients', 'buddyboss-app' ); ?>" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php esc_attr( _admin_search_query() ); ?>"/>
			<?php submit_button( $text, '', '', false, array( 'id' => 'search-submit' ) ); ?>
		</p>
		<?php
	}

	/**
	 * This function to prepare all legacy notification labels.
	 *
	 * @since 1.4.7.1
	 * @return array
	 */
	public function prepare_legacy_notification_labels() {
		$bbapp_legacy_admin_labels = array(
			'manual_push_notification' => __( 'Manual notifications sent by the admin', 'buddyboss-app' ),
		);

		if ( function_exists( 'bp_is_active' ) ) {

			// Activity.
			if ( bp_is_active( 'activity' ) ) {

				$legacy_admin_labels       = array(
					'bp_activity_new_at_mention' => __( 'A member is mentioned in another member’s update', 'buddyboss-app' ),
					'bp_activity_new_reply'      => __( 'A member receives a reply to an update or comment they’ve posted', 'buddyboss-app' ),
				);
				$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );
			}

			// Groups.
			if ( bp_is_active( 'groups' ) ) {
				$legacy_admin_labels = array(
					'bp_groups_new_invite'             => __( 'A member is invited to join a group', 'buddyboss-app' ),
					'bp_groups_admin_promotion'        => __( 'A member is promoted to a group organizer', 'buddyboss-app' ),
					'bp_groups_admin_promotion_mod'    => __( 'A member is promoted to a moderator', 'buddyboss-app' ),
					'bp_groups_new_membership_request' => __( 'A member receives a request to join a private group they organize', 'buddyboss-app' ),
					'bp_groups_request_completed'      => __( 'A member\'s request to join a group has been approved', 'buddyboss-app' ),
					'bp_groups_request_denied'         => __( 'A member\'s request to join a group has been denied', 'buddyboss-app' ),
					'bp_groups_send_new_message'       => __( 'A member receives a new group message', 'buddyboss-app' ),
					'bb_groups_details_updated'        => __( 'A group\'s details are updated', 'buddyboss-app' ),
					'bb_groups_request_accepted'       => __( 'A member is accepted into a group', 'buddyboss-app' ),
					'bb_groups_request_rejected'       => __( 'A member is rejected from joining a group', 'buddyboss-app' ),
				);

				$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );

				if ( function_exists( 'bb_platform_pro' ) ) {
					$legacy_admin_labels = array(
						'bp_zoom_meeting_scheduled' => __( 'A Zoom meeting has been scheduled in one of your groups', 'buddyboss-app' ),
						'bp_zoom_webinar_scheduled' => __( 'A Zoom webinar has been scheduled in one of your groups', 'buddyboss-app' ),
					);

					$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );
				}
			}

			// Friends / Connections.
			if ( bp_is_active( 'friends' ) ) {
				$legacy_admin_labels = array(
					'bp_friends_friendship_request'  => __( 'A member receives a new connection request', 'buddyboss-app' ),
					'bp_friends_friendship_accepted' => __( 'A member’s connection request is accepted', 'buddyboss-app' ),
				);

				$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );
			}

			// Messages.
			if ( bp_is_active( 'messages' ) ) {
				$legacy_admin_labels = array(
					'bp_messages_new_message' => __( 'A member receives a new message', 'buddyboss-app' ),
				);

				$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );
			}
		}

		// Forum.
		if ( function_exists( 'bbpress' ) ) {
			$legacy_admin_labels = array(
				'bbp_new_reply'      => __( 'A member receives a new reply to a discussion they\'ve created', 'buddyboss-app' ),
				'bbp_new_at_mention' => __( 'A discussion is created in a forum a member is subscribed to', 'buddyboss-app' ),
			);

			$bbapp_legacy_admin_labels = array_merge( $bbapp_legacy_admin_labels, $legacy_admin_labels );
		}

		return apply_filters( 'prepare_legacy_notification_labels', $bbapp_legacy_admin_labels );
	}

	/**
	 * This function to prepare all modern notification labels.
	 *
	 * @since [BBAPPVERION]
	 * @return array
	 */
	public function bp_prepare_modern_notification_labels() {
		$modern_notification = array();
		if ( function_exists( 'bb_register_notification_preferences' ) ) {
			$notifications       = bb_register_notification_preferences();
			$all_notifications   = array();
			$modern_notification = array();

			if ( ! empty( $notifications ) ) {
				$notifications = array_column( $notifications, 'fields', null );

				foreach ( $notifications as $key => $val ) {
					$all_notifications = array_merge( $all_notifications, $val );
				}

				$all_notifications_admin_labels = array_column( $all_notifications, 'admin_label', 'key' );
				$all_notifications              = array_column( $all_notifications, 'notifications', 'key' );

				foreach ( $all_notifications as $notification_type => $registered_notification ) {
					if ( ! empty( $registered_notification[0] ) && ! empty( $registered_notification[0]['component'] ) && $registered_notification[0]['component_action'] ) {
						$component                        = $registered_notification[0]['component'];
						$component_action                 = $registered_notification[0]['component_action'];
						$push_key                         = $component . '_' . $component_action;
						$modern_notification[ $push_key ] = $all_notifications_admin_labels[ $notification_type ];
					}
				}
			}
		}

		return $modern_notification;
	}

	/**
	 * This function get the push type by label.
	 *
	 * @param string $push_type push notification type.
	 *
	 * @since 1.4.7.1
	 * @return mixed|string
	 */
	public function get_notification_type_label( $push_type ) {
		$modern_notification_labels = $this->bp_prepare_modern_notification_labels();
		if ( isset( $modern_notification_labels[ $push_type ] ) ) {
			return $modern_notification_labels[ $push_type ];
		}
		$legacy_notification_labels = $this->prepare_legacy_notification_labels();
		if ( isset( $legacy_notification_labels[ $push_type ] ) ) {
			return $legacy_notification_labels[ $push_type ];
		}

		return '';
	}

}



