<?php
/**
 * Notification action.
 *
 * @package BuddyBossApp\Notification
 */

namespace BuddyBossApp\Notification;

use BuddyBossApp\Admin\Notification as AdminNotifications;
use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Contain actions & hooks related to notifications.
 * Class Actions
 */
class Actions {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of class
	 *
	 * @return Actions
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 */
	protected function hooks() {
		add_action( 'delete_user', array( $this, 'on_delete_user' ) );
		add_action( 'wpmu_delete_user', array( $this, 'on_delete_user' ) );
		add_action( 'bbapp_every_5min', array( $this, 'schedule_push_trigger' ) );
		add_action( 'bbapp_every_5min', array( $this, 'bb_push_notification_recalculate_segment' ) );
		add_filter( 'bbapp_get_notification_output', array( $this, 'get_notification_output' ), 99, 8 );
		add_filter( 'bbapp_get_notification_subscriptions', array( $this, 'add_notification_subscriptions' ), 99, 1 );
		add_filter( 'bbapp_filter_notification_subscriptions', array( $this, 'filter_get_notification_subscriptions' ) );
		add_filter( 'bbapp_rest_notifications_get_items_query_args', array( $this, 'filter_lite_app_notification_components' ), 10, 2 );
		add_filter( 'bp_rest_notifications_get_items_query_args', array( $this, 'filter_lite_app_notification_components' ), 10, 2 );
	}

	/**
	 * Return the registered subscription types to filter.
	 *
	 * @param array $subscriptions_lists get default setting of notification.
	 *
	 * @return array
	 * @see IntegrationAbstract
	 * @see Notification
	 */
	public function add_notification_subscriptions( array $subscriptions_lists ) {
		global $bbapp_register_push_type, $bbapp_register_push_groups, $bbapp_notification_setting_list;

		if ( ! is_array( $bbapp_register_push_type ) ) {
			$bbapp_register_push_type = array();
		}
		if ( ! is_array( $bbapp_register_push_groups ) ) {
			$bbapp_register_push_groups = array();
		}
		if ( ! is_array( $bbapp_notification_setting_list ) ) {
			$bbapp_notification_setting_list = array();
		}
		foreach ( $bbapp_register_push_groups as $bbapp_register_subscription_group_key => $bbapp_register_push_group ) {
			$bbapp_notification_setting          = array();
			$bbapp_notification_setting['name']  = $bbapp_register_subscription_group_key;
			$bbapp_notification_setting['label'] = $bbapp_register_push_group['label'];
			if ( isset( $bbapp_register_push_group['settings'] ) ) {
				foreach ( $bbapp_register_push_group['settings'] as $key => $bbapp_register_subscription_group_setting ) {
					$bbapp_notification_setting['settings'][ $bbapp_register_subscription_group_setting ] = $bbapp_register_push_type[ $bbapp_register_subscription_group_setting ];
				}
			}
			$bbapp_notification_setting_list[ $bbapp_register_subscription_group_key ] = $bbapp_notification_setting;
		}

		return array_merge( $subscriptions_lists, $bbapp_notification_setting_list );
	}

	/**
	 * Return the Filtered Notification list by Notification settings
	 *
	 * @param array $subscriptions_lists Notification subscriptions lists.
	 *
	 * @return array
	 */
	public function filter_get_notification_subscriptions( array $subscriptions_lists ) {
		$settings = AdminNotifications::instance()->get_settings();
		if ( empty( $settings ) ) {
			return $subscriptions_lists;
		}

		$filterd_subscriptions_lists = array();
		foreach ( $subscriptions_lists as $subscriptions_list ) {
			if ( in_array( $subscriptions_list['name'], array( '', 'announcements' ), true ) ) {
				$filterd_subscriptions_lists[] = $subscriptions_list;
				continue;
			}
			if ( ! empty( $subscriptions_list['settings'] ) ) {
				foreach ( $subscriptions_list['settings'] as $setting_key => $subscriptions_setting ) {
					if ( isset( $settings[ $setting_key ] ) && empty( $settings[ $setting_key ] ) ) {
						unset( $subscriptions_list['settings'][ $setting_key ] );
					}
				}
				if ( ! empty( $subscriptions_list['settings'] ) ) {
					$filterd_subscriptions_lists[] = $subscriptions_list;
				}
			}
		}

		return $filterd_subscriptions_lists;
	}

	/**
	 * Remove notification components for lite app.
	 *
	 * @param array $args Notification args.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return array
	 *
	 * @since 2.2.80
	 */
	public function filter_lite_app_notification_components( $args, $request ) {
		$components_to_remove = array(
			'groups',
			'activity',
			'friends',
			'xprofile',
			'blogs',
			'forums',
			'members',
			'messages',
		);

		if ( bbapp_is_lite_live_app() && ! empty( $args['component_name'] ) && is_array( $args['component_name'] ) ) {
			$args['component_name'] = array_diff( $args['component_name'], $components_to_remove );
		}

		return $args;
	}

	/**
	 * Return the notification output.
	 * Helps to get the output of single notification.
	 * Used at Notification Screen & When Sending Push From Notification.
	 *
	 * @param int        $user_id           Notification user id.
	 * @param string     $component_name    Notification component name.
	 * @param string     $component_action  Notification component action.
	 * @param string|int $item_id           Notification items id.
	 * @param string|int $secondary_item_id Notification secondory item id.
	 * @param string|int $notification_id   Notification id.
	 *
	 * @param string     $format            Format of notification. (object, array).
	 * @param string     $screen            Which screen we need to show. (web,push).
	 *
	 * @return mixed|void
	 */
	public function get_notification_output( $user_id, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $format = 'object', $screen = 'web' ) {

		if ( function_exists( 'buddypress' ) ) {
			$bp = buddypress();
		}

		/**
		 * Switch current user on this set of block to get correct information.
		 * BP render all notifications based on current logged in user instead of user_id.
		 * & This function is also called on place where logged in user might not be user_id of notification.
		 * So then for getting a correct information we have to manipulate user info temporary within
		 * this function.
		 */
		if ( isset( $bp ) ) {
			$previous_displayed_user = clone $bp->displayed_user;
			if ( $bp->displayed_user && isset( $bp->displayed_user->id ) && $bp->displayed_user->id !== $user_id ) {
				$displayed_user               = get_user_by( 'id', $user_id );
				$bp->displayed_user->id       = $displayed_user->ID;
				$bp->displayed_user->userdata = bp_core_get_core_userdata( $user_id );
				$bp->displayed_user->fullname = isset( $bp->displayed_user->userdata->display_name ) ? $bp->displayed_user->userdata->display_name : '';
				$bp->displayed_user->domain   = bp_core_get_user_domain( $user_id );
			}
			$previous_loggedin_user = clone $bp->loggedin_user;
			if ( $bp->loggedin_user && isset( $bp->loggedin_user->id ) && $bp->loggedin_user->id !== $user_id ) {
				$loggedin_user               = get_user_by( 'id', $user_id );
				$bp->loggedin_user->id       = $loggedin_user->ID;
				$bp->loggedin_user->userdata = bp_core_get_core_userdata( $user_id );
				$bp->loggedin_user->fullname = isset( $bp->displayed_user->userdata->display_name ) ? $bp->displayed_user->userdata->display_name : '';
				$bp->loggedin_user->domain   = bp_core_get_user_domain( $user_id );
			}
		}

		// We prefer that extended profile component-related notifications use
		// the component_name of 'xprofile'. However, the extended profile child
		// object in the $bp object is keyed as 'profile', which is where we need
		// to look for the registered notification callback.
		if ( 'xprofile' === $component_name ) {
			$component_name = 'profile';
		}

		// Callback function exists.
		if ( function_exists( 'buddypress' ) && isset( $bp->{$component_name}->notification_callback ) && is_callable( $bp->{$component_name}->notification_callback ) ) {

			// Function should return an object.
			if ( 'object' === $format ) {

				// Retrieve the content of the notification using the callback.
				$content = call_user_func( $bp->{$component_name}->notification_callback, $component_action, $item_id, $secondary_item_id, 1, 'array', $notification_id, $screen );

				$content = apply_filters(
					'bb_notifications_get_component_notification',
					$content,
					$item_id,
					$secondary_item_id,
					1,
					'array',
					$component_action,
					$component_name,
					$notification_id,
					$screen
				);

				// Create the object to be returned.
				$notification_object = new \stdClass();

				// Minimal backpat with non-compatible notification
				// callback functions.
				if ( is_string( $content ) ) {
					$notification_object->content = $content;
					$notification_object->href    = bp_loggedin_user_domain();
				} else {
					$notification_object->title   = $content['title'];
					$notification_object->content = $content['text'];
					$notification_object->href    = $content['link'];
				}

				$renderable[] = $notification_object;

				// Return an array of content strings.
			} else {
				$content = call_user_func( $bp->{$component_name}->notification_callback, $component_action, $item_id, $secondary_item_id, 1, 'string', $notification_id, $screen );

				$content = apply_filters(
					'bb_notifications_get_component_notification',
					$content,
					$item_id,
					$secondary_item_id,
					1,
					'string',
					$component_action,
					$component_name,
					$notification_id,
					$screen
				);

				$renderable[] = $content;
			}

			// @deprecated format_notification_function - 1.5
		} elseif ( function_exists( 'buddypress' ) && isset( $bp->{$component_name}->format_notification_function ) && function_exists( $bp->{$component_name}->format_notification_function ) ) {
			$renderable[] = call_user_func( $bp->{$component_name}->notification_callback, $component_action, $item_id, $secondary_item_id, 1 );

			// Allow non BuddyPress components to hook in.
		} else {

			// The array to reference with apply_filters_ref_array().
			$ref_array = array(
				$component_action,
				$item_id,
				$secondary_item_id,
				1,
				$format,
				$component_action, // Duplicated so plugins can check the canonical action name.
				$component_name,
				$notification_id,
				$screen,
			);

			// Function should return an object.
			if ( 'object' === $format ) {

				/**
				 * Filters the notification content for notifications created by plugins.
				 * If your plugin extends the {@link BP_Component} class, you should use the
				 * 'notification_callback' parameter in your extended
				 * {@link BP_Component::setup_globals()} method instead.
				 *
				 * @param string $content Component action. Deprecated. Do not do checks against this! Use
				 *                                      the 6th parameter instead - $component_action_name.
				 * @param int $item_id Notification item ID.
				 * @param int $secondary_item_id Notification secondary item ID.
				 * @param int $action_item_count Number of notifications with the same action.
				 * @param string $format Format of return. Either 'string' or 'object'.
				 * @param string $component_action_name Canonical notification action.
				 * @param string $component_name Notification component ID.
				 * @param int $id Notification ID.
				 *
				 * @return string|array If $format is 'string', return a string of the notification content.
				 *                      If $format is 'object', return an array formatted like:
				 *                      array( 'text' => 'CONTENT', 'link' => 'LINK' )
				 * @since 1.9.0
				 * @since 2.6.0 Added $component_action_name, $component_name, $id as parameters.
				 */

				$content = apply_filters_ref_array( 'bp_notifications_get_notifications_for_user', $ref_array );

				// Create the object to be returned.
				$notification_object = new \stdClass();

				// Minimal backpat with non-compatible notification
				// callback functions.
				if ( is_string( $content ) ) {
					$notification_object->content = $content;
					if ( class_exists( 'bp_loggedin_user_domain' ) ) {
						$notification_object->href = bp_loggedin_user_domain();
					} else {
						$notification_object->href = '';
					}
				} else {
					$notification_object->title   = $content['title'];
					$notification_object->content = $content['text'];
					$notification_object->href    = $content['link'];
				}

				$renderable[] = $notification_object;

				// Return an array of content strings.
			} else {

				/** This filters is documented in bp-notifications/bp-notifications-functions.php */
				$renderable[] = apply_filters_ref_array( 'bp_notifications_get_notifications_for_user', $ref_array );
			}
		}

		// If renderable is empty array, set to false.
		if ( empty( $renderable ) ) {
			$renderable = false;
		}

		$return = apply_filters( 'bp_core_get_notifications_for_user', $renderable, $user_id, $format );

		/**
		 * Switch back to previous user once code is executed.
		 */
		if ( isset( $bp ) ) {
			if ( isset( $previous_displayed_user ) && isset( $previous_displayed_user->id ) && $previous_displayed_user->id !== $bp->displayed_user->id ) {
				$bp->displayed_user = $previous_displayed_user;
			}

			if ( isset( $previous_loggedin_user ) && isset( $previous_loggedin_user->id ) && $previous_loggedin_user->id !== $bp->loggedin_user->id ) {
				$bp->loggedin_user = $previous_loggedin_user;
			}
		}

		/**
		 * Filters the final array of notifications to be displayed for a user.
		 *
		 * @param array|bool $renderable Array of notifications to render or false if no notifications.
		 * @param int $user_id ID of the user whose notifications are being displayed.
		 * @param string $format Display format requested for the notifications.
		 *
		 * @since 1.6.0
		 */
		return $return;
	}

	/**
	 * Delete all notification related to that user.
	 *
	 * @param int $user_id User id.
	 */
	public function on_delete_user( $user_id ) {
		if ( function_exists( 'bbapp_notifications' ) ) {

			bbapp_notifications()->delete_notification(
				array(
					'user_id' => $user_id,
					'blog_id' => false,
				),
				true
			);

			// Delete devices token.
			bbapp_notifications()->delete_user_devices_by_user_id( $user_id );

		}
	}

	/**
	 * This function clears all expires schedules.
	 */
	public function clear_expire_schedule_push() {

		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

		$gmnow = gmdate( 'Y-m-d H:i:s' );

		$get_results = $wpdb->get_results( $wpdb->prepare( "SELECT *FROM {$table_name} WHERE date_expire < %s AND is_schedule=1 AND status='pending'", $gmnow ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		foreach ( $get_results as $push ) {

			$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$table_name,
				array(
					'status' => 'expired',
				),
				array(
					'id' => $push->id,
				)
			);

		}

	}

	/**
	 * Do the Schedule trigger of Push Notifications.
	 *
	 * @return bool
	 */
	public function schedule_push_trigger() {
		$lock_key = 'bbapp_push_schedule_task_lock';

		/**
		 * Filters scheduled items process.
		 *
		 * @param int $task Number of items.
		 */
		$item_exec = apply_filters( 'bbapp_schedule_task_items_process', 2 );

		if ( '1' === get_transient( $lock_key ) ) {
			\BuddyBossApp\Tools\Logger::instance()->add( 'push_notification', __( 'Schedule Push Task Found Locked, Skipped Current Task.', 'buddyboss-app' ) );

			return false;
		}

		set_transient( $lock_key, '1', MINUTE_IN_SECONDS * 10 );

		$exec_push_per_batch = 2;

		/**
		 * Process all expired push first.
		 */
		$this->clear_expire_schedule_push();

		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

		$gmnow = gmdate( 'Y-m-d H:i:s' );

		// get push to execute.
		$results = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table_name} WHERE date_schedule < %s AND is_schedule=1 AND status='pending' LIMIT %d", $gmnow, $exec_push_per_batch ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$i = 0;

		foreach ( $results as $result ) {

			if ( $i > $item_exec ) {
				break;
			}

			// Start the delivery of push notification.
			Push::instance()->do_push_delivery( $result->id );

			$i ++;
		}

		// once task finish release the lock.
		delete_transient( $lock_key );
	}

	/**
	 * Recalculate user based on segment for filters member type.
	 *
	 * @since 1.7.6
	 */
	public function bb_push_notification_recalculate_segment() {
		$exec_recalculate_segment_per_batch = 2;

		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

		$gmnow = gmdate( 'Y-m-d H:i:s', strtotime( '+30 minutes' ) );

		/**
		 * Filter to pass for user segment time.
		 *
		 * @param string $gmnow Date-time.
		 *
		 * @since 1.7.6
		 */
		$gmnow = apply_filters( 'bbapp_notification_recalculate_user_segment_datetime', $gmnow );

		// get push to execute.
		$results = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table_name} WHERE date_schedule < %s AND is_schedule=1 AND status='pending' and target_type='filtered_users' and data NOT REGEXP '.*\"is_sync\";i:1.*' LIMIT %d", $gmnow, $exec_recalculate_segment_per_batch ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $results ) ) {
			foreach ( $results as $result ) {

				if ( isset( $result->data ) && ! empty( $result->data ) ) {
					$data = maybe_unserialize( $result->data );
					// Check if segment array is set and it's not empty.
					if ( isset( $data['segment_id'] ) && ! empty( $data['segment_id'] ) && isset( $data['segment'] ) && ! empty( $data['segment'] ) ) {
						$bbapp_user_segment = \BuddyBossApp\UserSegment::instance()->bbapp_create_user_segment( $result->id, $data['segment_id'], $data['segment'] );
						if ( is_array( $bbapp_user_segment ) && ! empty( $bbapp_user_segment['completed'] ) ) {
							$data['is_sync'] = 1;
							$data['send_to'] = \BuddyBossApp\UserSegment::instance()->get_total_users( $data['segment_id'], true );
							// Update the notification data.
							\BuddyBossApp\Notification\Push::instance()->update_notification(
								$result->id,
								array(
									'data' => $data,
								)
							);
						}
					}
				}
			}
		}
	}
}
