<?php
/**
 * Network class.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Server;

/**
 * Network class.
 */
class Network extends WP_REST_Controller {
	/**
	 * API namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/v1';
	/**
	 * Rest base.
	 *
	 * @var string
	 */
	protected $rest_base = 'network';
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 * Singleton instance.
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new Network();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}
	/**
	 * Define all actions and filters here.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter( 'bbapp_feature_availability_multisite_network', array( $this, 'feature_availability' ), 1, 1 );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/sites/',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

	}

	/**
	 * Changes the feature status based on WordPress requirements.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function feature_availability( $feature ) {
		$enabled                       = ( is_multisite() && bbapp()->is_network_activated() );
		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Retrieve network site info.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @apiPrivate
	 * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @api           {GET} /wp-json/buddyboss-app/core/v1/network/sites Networks sites of user
	 * @apiName       GetNetworkSitesOfUser
	 * @apiGroup      NetworkSettings
	 * @apiVersion    1.0.0
	 * @apiPermission LoggedInUser
	 */
	public function get_items( $request ) {
		// Array $bbapp_var.
		global $bbapp_var;

		if ( ! bbapp()->is_network_activated() ) {
			return new \WP_Error( 'rest_not_available', __( 'Information is not available.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$args_default = array(
			'user_id' => get_current_user_id(),
		);

		/**
		 * If current user is super admin, so super admin can see other user sites list based on user_id.
		 * If current user is not super admin so that user can see only his/her sites.
		 */
		if ( ! empty( $request['user_id'] ) ) {
			$sites = array();
			if ( ( get_current_user_id() === $request['user_id'] ) || true === is_super_admin( get_current_user_id() ) ) {
				$args_default['user_id'] = $request['user_id'];
				$sites                   = get_blogs_of_user( $args_default['user_id'] );
			}
		} else {
			$sites = $this->get_network_sites();
		}

		$sites = apply_filters( 'bbapp_rest_pre_sites', $sites, $request );

		$response = array();
		if ( ! empty( $sites ) ) {
			$instance = Branding::instance();

			foreach ( $sites as $site ) {

				$data                         = array();
				$data['blog_id']              = $site->userblog_id;
				$data['blogname']             = $site->blogname;
				$data['domain']               = $site->domain;
				$data['path']                 = $site->path;
				$data['siteurl']              = $site->siteurl;
				$data['app_home_screen_logo'] = null;
				$data['app_splash']           = null;

				// SSL fix on siteurl.
				if ( is_ssl() ) {
					$data['siteurl'] = str_replace( 'http://', 'https://', $data['siteurl'] );
				}

				switch_to_blog( $site->userblog_id );

				$branding_opt = $instance->get_options();

				$filename = 'home_screen_logo.png';
				if ( isset( $branding_opt['uploads_hash'] ) && isset( $branding_opt['uploads_hash'][ $filename ] ) ) {
					$data['app_home_screen_logo'] = $instance->get_branding_file( $filename, $branding_opt['uploads_hash'][ $filename ] );
				}

				$filename = 'home_screen_logo_v1.png';
				if ( isset( $branding_opt['uploads_hash'][ $filename ] ) ) {
					$data['app_home_screen_logo_v1'] = $instance->get_branding_file( $filename, $branding_opt['uploads_hash'][ $filename ] );
				}

				$filename = 'splash_screen_img.png';
				if ( isset( $branding_opt['uploads_hash'] ) && isset( $branding_opt['uploads_hash'][ $filename ] ) ) {
					$data['app_splash'] = $instance->get_branding_file( $filename, $branding_opt['uploads_hash'][ $filename ] );
				}

				// Make null when empty.
				if ( empty( $data['app_home_screen_logo'] ) ) {
					$data['app_home_screen_logo'] = null;
				}

				if ( empty( $data['app_splash'] ) ) {
					$data['app_splash'] = null;
				}

				restore_current_blog();

				$response[] = $data;

			}
		}
		$response = apply_filters( 'bbapp_rest_sites_response', $response, $request );

		return rest_ensure_response( $response );
	}
	/**
	 * Get all the sites in network.
	 */
	public function get_network_sites() {
		$sites = array();
		$args  = array(
			'number'                 => '',
			'update_site_meta_cache' => false,
		);

		$_sites = get_sites( $args );

		foreach ( $_sites as $site ) {
			$sites[ $site->id ] = (object) array(
				'userblog_id' => $site->id,
				'blogname'    => $site->blogname,
				'domain'      => $site->domain,
				'path'        => $site->path,
				'site_id'     => $site->network_id,
				'siteurl'     => $site->siteurl,
				'archived'    => $site->archived,
				'mature'      => $site->mature,
				'spam'        => $site->spam,
				'deleted'     => $site->deleted,
			);
		}

		/**
		 * Filters the list of sites a user belongs to.
		 *
		 * @param object[] $sites   An array of site objects belonging to the user.
		 * @param int      $user_id User ID.
		 *
		 * @since MU (3.0.0)
		 */
		return apply_filters( 'bbapp_get_network_sites', $sites );
	}

	/**
	 * Check if a given request has access to notification items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! empty( $request['user_id'] ) && ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bbapp_rest_authorization_required',
				__( 'Sorry, you are not allowed to see notifications.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_rest_network_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Collection params.
	 *
	 * @return mixed|void
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to network sites addressed to a specific user.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_network_collection_params', $params );
	}

}
