<?php
/**
 * Holds block parsing functionality.
 *
 * @package BuddyBossApp\NativeAppPage
 */

namespace BuddyBossApp\NativeAppPage;

use BuddyBossApp\ClientCommon;
use IvoPetkov;
use BuddyBossApp\Styling;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Contain App Page Gutenberg related things.
 */
class ParseBlocks {

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Class constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return ParseBlocks
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load hooks and methods.
	 */
	public function load() {
	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
	}

	/**
	 * Parse and modify Gutenberg blocks for the AppPages.
	 *
	 * @param array $blocks       Gutenberg Blocks.
	 * @param array $parent_style Parent style array.
	 *
	 * @return array array Modify Gutenberg Blocks.
	 */
	public static function parse_gutenberg_blocks( $blocks, $parent_style = array() ) {
		$new_app_page_data = array();
		$common            = ClientCommon::instance();
		foreach ( $blocks as $key => $block_data ) {
			if ( ! empty( $block_data['blockName'] ) ) {

				$activity_component = false;
				if ( class_exists( 'BuddyPress' ) && bp_is_active( 'activity' ) ) {
					$activity_component = true;
				}

				$groups_component = false;
				if ( class_exists( 'BuddyPress' ) && bp_is_active( 'groups' ) ) {
					$groups_component = true;
				}

				$members_component = false;
				if ( class_exists( 'BuddyPress' ) && bp_is_active( 'members' ) ) {
					$members_component = true;
				}

				$notification_component = false;
				if ( ( class_exists( 'BuddyPress' ) && bp_is_active( 'notifications' ) ) || bbapp_is_active( 'push_notification' ) ) {
					$notification_component = true;
				}

				if ( 'bbapp/groups' === $block_data['blockName'] && false === $groups_component ) {
					continue;
				} elseif ( 'bbapp/members' === $block_data['blockName'] && false === $members_component ) {
					continue;
				} elseif ( 'bbapp/activity' === $block_data['blockName'] && false === $activity_component ) {
					continue;
				} elseif ( 'bbapp/notifications' === $block_data['blockName'] && false === $notification_component ) {
					continue;
				} elseif ( ( 'bbapp/forums' === $block_data['blockName'] || 'bbapp/topics' === $block_data['blockName'] ) && ! function_exists( 'bbpress' ) ) {
					continue;
				} elseif ( ( 'bbapp/courses' === $block_data['blockName'] ) && ( ! $common->is_active_logic( array( 'bbapp_learner_api' ) ) ) ) {
					continue;
				} elseif ( ( 'bbapp/mp-courses' === $block_data['blockName'] ) && ( ! bbapp_is_memberpress_lms_enabled() ) ) {
					continue;
				} elseif ( ( 'bbapp/tutor-courses' === $block_data['blockName'] ) && ( ! bbapp_is_tutor_lms_plugins_active() ) ) {
					continue;
				} elseif ( ( 'bbapp/h5p' === $block_data['blockName'] ) && ! class_exists( 'H5P_Plugin' ) ) {
					continue;
				}

				if (
					bbapp_is_lite_live_app() &&
					in_array(
						$block_data['blockName'],
						array(
							'bbapp/activity',
							'bbapp/forums',
							'bbapp/groups',
							'bbapp/members',
							'bbapp/topics',
						),
						true
					)
				) {
					continue;
				}

				$render = render_block( $block_data );
				$render = trim( $render );

				if ( empty( $render ) ) {
					continue;
				}
				switch ( $block_data['blockName'] ) {
					case 'core/image':
					case 'core/video':
					case 'core/audio':
						$app_page_data = self::parse_media_block( $block_data, $render );
						break;
					case 'core/gallery':
						$app_page_data = self::parse_gallery_block( $block_data, $render );
						break;
					case 'core/paragraph':
						$app_page_data = self::parse_paragraph_block( $block_data, $render, $parent_style );
						break;
					case 'core/heading':
						$app_page_data = self::parse_heading_block( $block_data, $render, $parent_style );
						break;
					case 'core/quote':
						$app_page_data = self::parse_quote_block( $block_data, $render, $parent_style );
						break;
					case 'core/table':
						$app_page_data = self::parse_table_block( $block_data, $render, $parent_style );
						break;
					case 'core/list':
						$app_page_data = self::parse_common_block( $block_data, $render, $parent_style );
						break;
					case 'core/html':
						$app_page_data = self::parse_html_block( $block_data, $render );
						break;
					case 'core/cover':
						$app_page_data = self::parse_cover_block( $block_data, $render );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'core/media-text':
						$app_page_data = self::parse_media_text_block( $block_data, $render, $parent_style );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'core/buttons':
						$app_page_data = self::parse_buttons_block( $block_data, $render, $parent_style );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'core/button':
						$app_page_data = self::parse_button_block( $block_data, $render );
						break;
					case 'core/columns':
						$app_page_data = self::parse_columns_block( $block_data, $render, $parent_style );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'core/embed':
						$app_page_data = self::parse_embed_block( $block_data, $render );
						break;
					case 'bbapp/courses':
						$app_page_data = self::parse_courses_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/mp-courses':
						$app_page_data = self::parse_mp_courses_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/tutor-courses':
						$app_page_data = self::parse_tutor_courses_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/topics':
						$app_page_data = self::parse_topics_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/forums':
						$app_page_data = self::parse_forums_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/groups':
						$app_page_data = self::parse_groups_block( $block_data, $render, $parent_style );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'bbapp/members':
						$app_page_data = self::parse_members_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/activity':
						$app_page_data = self::parse_activity_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/notifications':
						$app_page_data = self::parse_notifications_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/qlinks':
						$app_page_data = self::parse_quick_links_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/h5p':
						$app_page_data = self::parse_h5p_block( $block_data, $render, $parent_style );
						break;
					case 'core/code':
						$app_page_data = self::parse_code_block( $block_data, $render );
						break;
					case 'core/pullquote':
						$app_page_data = self::parse_pullquote_block( $block_data, $render );
						break;
					case 'core/preformatted':
						$app_page_data = self::parse_preformatted_block( $block_data, $render, $parent_style );
						break;
					case 'core/verse':
						$app_page_data = self::parse_verse_block( $block_data, $render );
						break;
					case 'core/file':
						$app_page_data = self::parse_file_block( $block_data, $render, $parent_style );
						break;
					case 'core/separator':
						$app_page_data = self::parse_separator_block( $block_data, $render );
						break;
					case 'core/spacer':
						$app_page_data = self::parse_spacer_block( $block_data, $render );
						break;
					case 'core/group':
						$app_page_data = self::parse_group_block( $block_data, $render, $parent_style );
						$app_page_data = self::instance()->remove_style_from_render( $app_page_data );
						break;
					case 'bbapp/posts':
						$app_page_data = self::parse_posts_block( $block_data, $render, $parent_style );
						break;
					case 'bbapp/directory':
						$app_page_data = self::parse_directory_block( $block_data, $render, $parent_style );
						break;
					default:
						$app_page_data = self::parse_custom_block( $block_data, $render );
						$app_page_data = apply_filters( 'bbapp_custom_block_data', $app_page_data, $block_data, $render );
				}

				$app_page_data = apply_filters( 'bbapp_render_app_page_data', $app_page_data, $block_data );
				array_push( $new_app_page_data, $app_page_data );
			}
		}

		return $new_app_page_data;
	}

	/**
	 * Remove style from render.
	 *
	 * @param array $app_page_data App data.
	 *
	 * @since 1.6.7
	 *
	 * @return mixed
	 */
	public function remove_style_from_render( $app_page_data ) {
		// Remove style sttribute from render and data.
		if ( ! empty( $app_page_data['render'] ) ) {
			$app_page_data['render'] = preg_replace( '/(<[^>]+) style=".*?"/i', '$1', $app_page_data['render'] );
		}
		if ( ! empty( $app_page_data['content'] ) ) {
			$app_page_data['content'] = preg_replace( '/(<[^>]+) style=".*?"/i', '$1', $app_page_data['content'] );
		}

		return $app_page_data;
	}

	/**
	 * Parse Default block.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_custom_block( $block_data, $render ) {
		$attribute = array();
		$style     = array();

		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;

		$content                  = wp_strip_all_tags( $render );
		$app_page_data['content'] = $content;

		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Media blocks like Image, Video and Audio.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_media_block( $block_data, $render ) {
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;

		$content = '';
		if ( preg_match( '@src="([^"]+)"@', $render, $match ) ) {
			preg_match( '@src="([^"]+)"@', $render, $match );
			$src     = array_pop( $match );
			$content = $src;
		}

		if ( preg_match_all( '~<a(.*?)href="([^"]+)"(.*?)>~', $render, $matches ) ) {
			preg_match_all( '~<a(.*?)href="([^"]+)"(.*?)>~', $render, $matches );
			if ( isset( $matches[2][0] ) ) {
				$attribute['linkSrc'] = $matches[2][0];
			}
		}

		if ( preg_match( '@alt="([^"]+)"@', $block_data['innerHTML'], $match ) ) {
			preg_match( '@alt="([^"]+)"@', $block_data['innerHTML'], $match );
			$alt              = array_pop( $match );
			$attribute['alt'] = $alt;
		}

		if ( preg_match( '@preload="([^"]+)"@', $block_data['innerHTML'], $match ) ) {
			preg_match( '@preload="([^"]+)"@', $block_data['innerHTML'], $match );
			$preload              = array_pop( $match );
			$attribute['preload'] = $preload;
		}

		$style           = array();
		$original_render = $render;
		$render          = preg_replace(
			array( '/figure/', '/figcaption/' ),
			array( 'div', 'span' ),
			$render
		);

		$attribute['src']         = trim( $content );
		$app_page_data['content'] = $content;

		if ( 'core/image' === $block_data['blockName'] ) {
			// Get Default Image Width and Height.
			if ( ! empty( $content ) ) {
				$style = self::bbapp_get_image_size( $content, $block_data, $style );
			}

			if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $original_render, $content_matches ) ) {
				preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $original_render, $content_matches );
				$attribute['caption'] = isset( $content_matches[2][0] ) ? $content_matches[2][0] : $content_matches[0][0];
				$style                = self::get_caption_color( $attribute['caption'], $style );
			}
			$style['align'] = 'left';
		} else {
			$attribute['autoplay'] = false;
			$attribute['loop']     = false;
			$attribute['muted']    = false;
		}

		if ( preg_match( '/autoplay/', $render ) ) {
			$attribute['autoplay'] = true;
		}

		if ( preg_match( '/loop/', $render ) ) {
			$attribute['loop'] = true;
		}

		if ( preg_match( '/muted/', $render ) ) {
			$attribute['muted'] = true;
		}

		if ( preg_match( '/controls/', $render ) ) {
			$attribute['controls'] = true;
		}

		if ( preg_match( '/playsinline/', $render ) ) {
			$attribute['playsinline'] = true;
		}

		if ( preg_match( '/target=/', $render ) ) {
			$attribute['targetBlank'] = true;
		}

		if ( 'core/audio' === $block_data['blockName'] ) {
			unset( $attribute['controls'] );
			unset( $attribute['muted'] );
		}

		if ( isset( $block_data['attrs']['className'] ) ) {
			if ( preg_match( '/rounded/', $block_data['attrs']['className'] ) ) {
				$style['image_style'] = 'rounded';
			}
		}

		if ( preg_match_all( '~<video(.*?)poster="([^"]+)"(.*?)>~', $render, $matches ) ) {
			preg_match_all( '~<video(.*?)poster="([^"]+)"(.*?)>~', $render, $matches );
			if ( isset( $matches[2][0] ) ) {
				$attribute['poster'] = $matches[2][0];
			}
		}

		if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $original_render, $content_matches ) ) {
			preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $original_render, $content_matches );
			$attribute['caption'] = isset( $content_matches[2][0] ) ? $content_matches[2][0] : $content_matches[0][0];
			$style                = self::get_caption_color( $attribute['caption'], $style );
		}

		$style                  = self::get_block_style( $block_data, $style );
		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$attribute              = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Gallery Block
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_gallery_block( $block_data, $render ) {
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;
		$gallery_item            = array();

		// WordPress 5.9 version provide inner block as image block.
		if ( ! empty( $block_data['innerBlocks'] ) ) {
			foreach ( $block_data['innerBlocks'] as $inner_blocks ) {
				$inner_render = render_block( $inner_blocks );
				$inner_render = trim( $inner_render );
				if ( ! empty( $inner_render ) ) {
					$image_item        = self::parse_media_block( $inner_blocks, $inner_render );
					$gallery_attribute = array(
						'url'             => $image_item['data']['src'],
						'sizes'           => array(
							'width'  => ! empty( $image_item['style']['width'] ) ? $image_item['style']['width'] : 0,
							'height' => ! empty( $image_item['style']['height'] ) ? $image_item['style']['height'] : 0,
						),
						'linkDestination' => ! empty( $image_item['data']['linkDestination'] ) ? $image_item['data']['linkDestination'] : 'none',
						'linkSrc'         => ! empty( $image_item['data']['linkSrc'] ) ? $image_item['data']['linkSrc'] : 'none',
						'targetBlank'     => ! empty( $image_item['data']['targetBlank'] ) ? $image_item['data']['targetBlank'] : 'none',
						'color'           => ! empty( $image_item['style']['color'] ) ? $image_item['style']['color'] : '',
						'border'          => ! empty( $image_item['style']['border'] ) ? $image_item['style']['border'] : '',
						'image_style'     => ! empty( $image_item['style']['image_style'] ) ? $image_item['style']['image_style'] : '',
					);

					$gallery_data['item_caption'] = ( ! empty( $image_item['data'] ) && ! empty( $image_item['data']['caption'] ) ) ? $image_item['data']['caption'] : '';

					if ( empty( $gallery_attribute ) && empty( $gallery_data ) ) {
						continue;
					}

					$gallery_item[] = array(
						'content'   => $image_item['render'],
						'attribute' => (object) $gallery_attribute,
						'data'      => (object) $gallery_data,
					);
				}
			}
		} else {
			// Old WordPress 5.8 version support added.
			if ( preg_match_all( '/<figure(.*?)>(.*?)<\/figure>/s', $render, $gallery_content, PREG_SET_ORDER ) ) {
				foreach ( $gallery_content as $key => $gallery_content_item ) {
					$gallery_attribute = array();

					$inner_blocks = isset( $block_data['attrs']['ids'][ $key ] ) ? $block_data['attrs']['ids'][ $key ] : '';

					if ( ! empty( $inner_blocks ) ) {
						$attachment_id = $inner_blocks;
						$gallery_url   = wp_get_attachment_url( $attachment_id );
						if ( ! empty( $gallery_url ) ) {
							$link_destination = 'none';
							$link_src         = 'none';
							if ( isset( $block_data['attrs']['linkTo'] ) ) {
								switch ( $block_data['attrs']['linkTo'] ) {
									case 'media':
									case 'file':
										$link_destination = $block_data['attrs']['linkTo'];
										$link_src         = $gallery_url;
										break;
									case 'attachment':
									case 'post':
										$link_destination = $block_data['attrs']['linkTo'];
										$link_src         = get_attachment_link( $attachment_id );
										break;
								}
							}
							$image_meta        = wp_get_attachment_metadata( $attachment_id );
							$gallery_attribute = array(
								'url'             => wp_get_attachment_url( $inner_blocks ),
								'sizes'           => array(
									'width'  => ! empty( $image_meta['width'] ) ? $image_meta['width'] : 0,
									'height' => ! empty( $image_meta['height'] ) ? $image_meta['height'] : 0,
								),
								'linkDestination' => $link_destination,
								'linkSrc'         => $link_src,
							);
						}
					}
					$gallery_data = array();
					if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $gallery_content_item[0], $content_matches ) ) {
						$gallery_caption_position     = array_search( 'blocks-gallery-caption', $content_matches[1], true );
						$gallery_data['item_caption'] = $content_matches[2][ $gallery_caption_position ];
						$style                        = self::get_caption_color( $gallery_data['item_caption'] );
					}
					$gallery_content = preg_replace(
						array( '/figure/', '/figcaption/' ),
						array( 'div', 'span' ),
						$gallery_content_item[0]
					);

					if ( empty( $gallery_attribute ) && empty( $gallery_data ) ) {
						continue;
					}

					$gallery_item[] = array(
						'content'   => $gallery_content,
						'attribute' => (object) $gallery_attribute,
						'data'      => (object) $gallery_data,
					);
				}
			}
		}

		$render                        = preg_replace(
			array( '/figure/', '/figcaption/' ),
			array( 'div', 'span' ),
			$render
		);
		$app_page_data['content']      = $render;
		$app_page_data['gallery_item'] = $gallery_item;

		$style['cropped'] = false;
		if ( preg_match( '/is-cropped/', $render ) ) {
			$style['cropped'] = true;
		}

		$style['columns'] = 3;
		if ( isset( $block_data['attrs']['columns'] ) ) {
			if ( preg_match( '/(?P<name>\w+)-(?P<digit>\d+)/', $render, $matches ) ) {
				preg_match( '/(?P<name>\w+)-(?P<digit>\d+)/', $render, $matches );
				$style['columns'] = absint( $matches['digit'] );
			}
		}

		// Gallery and Gallery item caption.
		$attribute = array();
		if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $block_data['innerHTML'], $content_matches ) ) {
			$matches = preg_grep( '/blocks-gallery-caption/', $content_matches[1] );
			if ( $matches ) { // WordPress 5.8 or older.
				foreach ( $matches as $caption_position => $matche ) {
					$attribute['gallery_caption'] = $content_matches[2][ $caption_position ];
				}
			} elseif ( ! empty( $content_matches[2][0] ) ) { // WordPress 5.9 or new.
				$attribute['gallery_caption'] = $content_matches[2][0];
			} else {
				$attribute['gallery_caption'] = '';
			}
			$style = self::get_caption_color( $attribute['gallery_caption'], $style );
		}

		$style['align']         = 'left';
		$style                  = self::get_block_style( $block_data, $style );
		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$app_page_data['data']  = (object) $attribute;
		$app_page_data['style'] = (object) $style;

		return $app_page_data;
	}

	/**
	 * Parse Paragraph Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_paragraph_block( $block_data, $render, $parent_style = array() ) {
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;

		preg_match( '@src="([^"]+)"@', $render, $match );

		$render         = preg_replace( '/<p>(.+)<\/p>/Uuis', '$1', $render );
		$split_data     = preg_split( '/(<img[^>]+\>)/i', $render, - 1, PREG_SPLIT_DELIM_CAPTURE );
		$split_data     = array_filter( $split_data );
		$style          = array();
		$new_split_data = array();

		if ( ! empty( $split_data ) ) {
			foreach ( $split_data as $data_key => $data_text ) {
				$new_split_data[ $data_key ]['type'] = 'text';
				$new_split_data[ $data_key ]['data'] = do_shortcode( $data_text );

				if ( preg_match( '@src="([^"]+)"@', $data_text, $match ) ) {
					$style = array();
					preg_match( '@src="([^"]+)"@', $data_text, $match );
					$src = array_pop( $match );
					preg_match( '/width:(.*)px;/', $data_text, $match );
					$src_inline_width = array_pop( $match );

					$new_split_data[ $data_key ]['type'] = 'image';
					$new_split_data[ $data_key ]['src']  = $src;

					if ( ! empty( $src_inline_width ) ) {
						$style['width'] = (int) trim( $src_inline_width );
					} elseif ( ! empty( $src ) ) {
							$style = self::bbapp_get_image_size( $src, $block_data, $style );
					}

					$new_split_data[ $data_key ]['style'] = $style;
				}
			}
		}

		$app_page_data['content'] = (array) $new_split_data;
		$style['textAlign']       = ! empty( $block_data['attrs']['align'] ) ? $block_data['attrs']['align'] : 'left';
		$style['color']           = null;
		$style['dropCap']         = false;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$app_page_data['data']    = array();
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse table Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 * @throws object \Exception is the base class for.
	 */
	public static function parse_table_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$style['textColor']       = '';
		$style['layout']          = 'regular';
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$background_color         = isset( $style['backgroundColor'] ) ? $style['backgroundColor'] : '';
		$text_color               = isset( $style['color'] ) ? $style['color'] : '';
		$app_page_data['content'] = $render;

		$dom = new IvoPetkov\HTML5DOMDocument();
		$dom->loadHTML( $render );
		$styles = array(
			'background-color' => $background_color,
			'text-color'       => $text_color,
		);
		$table  = array();
		$count  = 0;

		foreach ( $dom->querySelectorAll( 'tr' ) as $r => $row ) {
			if ( isset( $style['layout'] ) && 'stripes' === $style['layout'] ) {
				( $count % 2 ) ? $styles['background-color'] = $background_color : $styles['background-color'] = '#f0f0f0';
			}

			// Table header & footer set selected/default color.
			if ( 'thead' === $row->parentNode->tagName || 'tfoot' === $row->parentNode->tagName ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$styles['background-color'] = $background_color;
			} else {
				$count ++;
			}

			foreach ( $row->querySelectorAll( 'th' ) as $c => $column_data ) {
				$styles['is_head'] = true;
				$td_inner_html     = $column_data->innerHTML; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				// Prepare Custom HTML Block.
				$table = self::create_table_rows( $table, $r, $c, $td_inner_html, $styles, $column_data->getAttribute( 'data-align' ) );
			}

			foreach ( $row->querySelectorAll( 'td' ) as $c => $column_data ) {
				$styles['is_head'] = false;
				$td_inner_html     = $column_data->innerHTML; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				// Prepare Custom HTML Block.
				$table = self::create_table_rows( $table, $r, $c, $td_inner_html, $styles, $column_data->getAttribute( 'data-align' ) );
			}
		}

		if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $render, $content_matches ) ) {
			preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $render, $content_matches );
			$table['caption'] = isset( $content_matches[2][0] ) ? $content_matches[2][0] : $content_matches[0][0];
			$style            = self::get_caption_color( $table['caption'], $style );
		}

		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$app_page_data['data']  = $table;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Table row data collect.
	 *
	 * @param array  $table         Table array.
	 * @param string $r             Table row key.
	 * @param string $c             Table column key.
	 * @param string $td_inner_html Inner html.
	 * @param array  $styles        Style array.
	 * @param string $align         Column align attribute.
	 *
	 * @return mixed
	 */
	public static function create_table_rows( $table, $r, $c, $td_inner_html, $styles, $align ) {
		// Prepare Custom HTML Block.
		$columns['type']   = 'core/paragraph';
		$columns['render'] = $td_inner_html;

		preg_match( '@src="([^"]+)"@', $td_inner_html, $match );
		$td_inner_html  = preg_replace( '/<p>(.+)<\/p>/Uuis', '$1', $td_inner_html );
		$split_data     = preg_split( '/(<img[^>]+\>)/i', $td_inner_html, - 1, PREG_SPLIT_DELIM_CAPTURE );
		$new_split_data = array();

		if ( is_array( $split_data ) ) {
			foreach ( $split_data as $data_key => $data_text ) {
				$new_split_data[ $data_key ]['type'] = 'text';
				$new_split_data[ $data_key ]['data'] = $data_text;

				if ( preg_match( '@src="([^"]+)"@', $data_text, $match ) ) {
					$style = array();
					preg_match( '@src="([^"]+)"@', $data_text, $match );
					$src = array_pop( $match );

					$new_split_data[ $data_key ]['type'] = 'image';
					$new_split_data[ $data_key ]['src']  = $src;

					if ( ! empty( $src ) ) {
						$style = self::bbapp_get_image_size( $src, $td_inner_html, $style );
					}

					$new_split_data[ $data_key ]['style'] = $style;
				}
				if ( preg_match( '@href="([^"]+)"@', $data_text, $match ) ) {
					$style = array();
					preg_match( '@href="([^"]+)"@', $data_text, $match );
					$src = array_pop( $match );

					$new_split_data[ $data_key ]['type'] = 'a';
					$new_split_data[ $data_key ]['src']  = $src;

					$new_split_data[ $data_key ]['style'] = $style;
				}
			}
		}

		$columns['content']                   = $new_split_data;
		$columns['data']                      = array( 'html' => $td_inner_html );
		$columns['style']                     = array( 'align' => ! empty( $align ) ? $align : 'left' );
		$table['rows'][ $r ]['style']         = $styles;
		$table['rows'][ $r ]['columns'][ $c ] = $columns;

		return $table;
	}

	/**
	 * Parse cover Block.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 * @throws object \Exception exception handler.
	 */
	public static function parse_cover_block( $block_data, $render ) {
		$content_url = ( ! empty( $block_data['attrs'] ) && ! empty( $block_data['attrs']['url'] ) ) ? $block_data['attrs']['url'] : '';

		/**
		 * The`as3cf_filter_post_local_to_s3`  filters added for filtering S3 URLs in custom content.
		 *
		 * @param string $content_url Content URL.
		 */
		$content_url = apply_filters( 'as3cf_filter_post_local_to_s3', $content_url );

		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['content'] = $content_url;

		$attribute = array();
		$style     = array();

		$style['isRepeated']   = false;
		$style['dimRatio']     = 100;
		$style['hasParallax']  = false;
		$style['contentAlign'] = 'center';

		$dom = new IvoPetkov\HTML5DOMDocument();
		$dom->loadHTML( $render );
		$render_output           = isset( $dom->querySelector( 'p.wp-block-cover-text' )->outerHTML ) ? $dom->querySelector( 'p.wp-block-cover-text' )->outerHTML : '';
		$attribute               = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['render'] = ( ! empty( $render_output ) ) ? $render_output : $render;
		$app_page_data['data']   = $attribute;
		$style                   = self::get_block_style( $block_data, $style );

		if ( ! empty( $block_data['innerBlocks'] ) ) {
			$app_page_data['coverData'] = bbapp_parse_inner_block( $block_data['innerBlocks'], $style );
		}

		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$app_page_data['style'] = $style;

		return self::get_backward_compatibility_cover_block( $app_page_data, $block_data );
	}

	/**
	 * Parse Media-Text Block.
	 *
	 * @param array  $block_data   Gutenberg data attributes.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return mixed
	 */
	public static function parse_media_text_block( $block_data, $render, $parent_style ) {

		$app_page_data['type']      = $block_data['blockName'];
		$app_page_data['render']    = $render;
		$style                      = array();
		$app_page_data['content']   = $render;
		$style['width']             = null;
		$style['height']            = null;
		$attribute['media_text']    = bbapp_parse_inner_block( $block_data['innerBlocks'] );
		$attribute['mediaSizeSlug'] = 'full';
		$attribute                  = self::get_block_attributes( $block_data, $attribute );
		$style['mediaPosition']     = 'left';
		$style['verticalAlignment'] = 'center';
		$style['isStackedOnMobile'] = false;

		if ( ! isset( $block_data['attrs']['isStackedOnMobile'] ) ) {
			$style['isStackedOnMobile'] = true; // Is attr is not coming then it's return true.
		}

		if ( preg_match_all( '~<a(.*?)href="([^"]+)"(.*?)>~', $render, $matches ) ) {
			if ( isset( $matches[2][0] ) ) {
				$attribute['linkSrc'] = $matches[2][0];
			}
		}

		$style                  = self::get_block_style( $block_data, $style, array() );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Button Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 * @throws object \Exception Exception data.
	 */
	public static function parse_buttons_block( $block_data, $render, $parent_style ) {
		$app_page_data['type'] = $block_data['blockName'];
		$dom                   = new IvoPetkov\HTML5DOMDocument();
		$dom->loadHTML( $block_data['innerHTML'] );
		$render_output           = isset( $dom->querySelector( 'div.wp-block-button' )->outerHTML ) ? $dom->querySelector( 'div.wp-block-button' )->outerHTML : '';
		$render_output           = preg_replace( array( '/div/' ), array( 'p' ), $render_output );
		$app_page_data['render'] = $render_output;
		if ( isset( $block_data['innerBlocks'] ) ) {
			foreach ( $block_data['innerBlocks'] as $key => $block_inner_blocks ) {
				$render                              = render_block( $block_inner_blocks );
				$render                              = trim( $render );
				$app_page_data['buttonData'][ $key ] = self::parse_button_block( $block_inner_blocks, $render );
			}
		}
		$style                  = array();
		$style['align']         = 'left';
		$style                  = self::get_block_style( $block_data, $style, $parent_style );
		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Button Block.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 * @throws object \Exception Exception data.
	 */
	public static function parse_button_block( $block_data, $render ) {
		$app_page_data['type']    = $block_data['blockName'];
		$content                  = ( ! empty( $render ) ) ? preg_replace( '@<(script|style)[^>]*?>.*?</\\1>@si', '', $render ) : '';
		$content                  = ( ! empty( $content ) ) ? preg_replace( '/\n/', '', wp_strip_all_tags( $content ) ) : '';
		$app_page_data['content'] = $content;
		$attribute['button_text'] = $content;

		$dom = new IvoPetkov\HTML5DOMDocument();
		$dom->loadHTML( $block_data['innerHTML'] );
		$render_output           = ( ! empty( $dom ) && ! empty( $dom->querySelector( 'div.wp-block-button' ) ) ) ? $dom->querySelector( 'div.wp-block-button' )->outerHTML : '';
		$render_output           = preg_replace( array( '/div/' ), array( 'p' ), $render_output );
		$app_page_data['render'] = ( ! empty( $render_output ) ) ? $render_output : $render;
		$default_theme           = wp_get_theme();
		$default_bg_color        = '';

		if ( 'twentynineteen' === $default_theme->get_stylesheet() ) {
			$default_bg_color = '#0073aa';
		}

		$style['button_style']    = 'rounded';
		$style['backgroundColor'] = $default_bg_color;
		$style['color']           = '#ffffff';

		if ( preg_match_all( '~<a(.*?)href="([^"]+)"(.*?)>~', $render, $matches ) ) {
			preg_match_all( '~<a(.*?)href="([^"]+)"(.*?)>~', $render, $matches );
			$attribute['flagUrl'] = false;
			if ( isset( $matches[2][0] ) ) {
				$attribute['url'] = $matches[2][0];
				$url              = wp_parse_url( $matches[2][0] );

				if ( 'https' === $url['scheme'] ) {
					$attribute['flagUrl'] = true;
				}
			}
		}

		if ( preg_match( '/target=/', $render ) ) {
			$attribute['targetBlank'] = true;
		}

		$style                  = self::get_block_style( $block_data, $style );
		$style                  = self::get_backward_compatibility_block_style( $block_data, $style );
		$attribute              = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Columns Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_columns_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$attribute                = array();
		$app_page_data['content'] = $render;

		if ( isset( $block_data['innerBlocks'] ) && ! empty( $block_data['innerBlocks'] ) ) {
			foreach ( $block_data['innerBlocks'] as $column_key => $column_blocks ) {
				$column_render = render_block( $column_blocks );
				if ( ! empty( trim( $column_render ) ) && isset( $column_blocks['innerBlocks'] ) && ! empty( $column_blocks['innerBlocks'] ) ) {
					$sub_parent_style     = self::get_block_style( $column_blocks, array() );
					$inner_block_response = bbapp_parse_inner_block( $column_blocks['innerBlocks'], $sub_parent_style );
					if ( isset( $inner_block_response ) && ! empty( $inner_block_response ) ) {
						$attribute[ $column_key ] = $inner_block_response;
					}
				}
			}
		}

		$style['columns']       = ! empty( $attribute ) ? count( $attribute ) : 0;
		$app_page_data['data']  = ! empty( $render ) ? array_values( $attribute ) : array();
		$app_page_data['style'] = ! empty( $render ) ? self::get_block_style( $block_data, $style ) : array();

		return $app_page_data;
	}

	/**
	 * Parse Heading Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Block render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_heading_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$content                  = wp_strip_all_tags( $render );
		$app_page_data['content'] = $content;
		$style['align']           = 'left';
		if ( ! isset( $block_data['attrs']['level'] ) && ! isset( $block_data['attrs']['fontSize'] ) ) {
			$block_data['attrs']['level'] = '2';
		}
		$style                  = self::get_block_style( $block_data, $style, $parent_style );
		$app_page_data['data']  = array();
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Quote Block.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Block render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_quote_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$content                  = wp_strip_all_tags( $render );
		$app_page_data['content'] = $content;

		if ( preg_match_all( '~<p(.*?)>(.*?)</p>~', $render, $content_matches ) ) {
			preg_match_all( '~<p(.*?)>(.*?)</p>~', $render, $content_matches );
			$app_page_data['content'] = wp_strip_all_tags( $content_matches[0][0] );
		}

		if ( preg_match_all( '~<cite>(.*?)</cite>~', $render, $citation_matches ) ) {
			preg_match_all( '~<cite>(.*?)</cite>~', $render, $citation_matches );
			$attribute['citation'] = $citation_matches[1][0];
		}

		$attribute['fontSize'] = 'normal';
		$style                 = array();
		$style['layout']       = 'default';
		$style                 = self::get_block_style( $block_data, $style, $parent_style );
		if ( isset( $block_data['innerBlocks'] ) ) {
			$app_page_data['quote_blocks'] = bbapp_parse_inner_block( $block_data['innerBlocks'], $style );
		}
		$attribute              = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse embed Block like YouTube and Vimeo.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Block render html.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_embed_block( $block_data, $render ) {
		$app_page_data['type'] = $block_data['blockName'];

		$content = wp_strip_all_tags( $render );
		if ( preg_match( "'<div class=\"wp-block-embed__wrapper\">(.*?)</div>'si", $render, $content_matches ) ) {
			preg_match( "'<div class=\"wp-block-embed__wrapper\">(.*?)</div>'si", $render, $content_matches );
			$content = wp_strip_all_tags( $content_matches[1] );
		}

		$content_url = $content;
		$content     = self::get_cached_oembed( $content_url );

		$content = ! empty( $content ) ? preg_replace( '/(<iframe[^>]+) width="(.*?)"/i', '$1 width="100%"', $content ) : $content; // Set width as 100% for iframe.
		$content = ! empty( $content ) ? preg_replace( '/(<iframe[^>]+) (.*?)src="\/\/(.*?)/i', '$1 src="https://$2', $content ) : $content; // Add https to iframe src.
		$content = ! empty( $content ) ? preg_replace( '/(<img[^>]+) width="(.*?)"/i', '$1 width="100%"', $content ) : $content; // Set width as 100% for img tag.
		$content = ! empty( $content ) ? preg_replace( '/(<div[^>]+) (.*?)style="width:(.*?);(.*?)/i', '$1 style="width:100%;$2', $content ) : $content; // Add width ad 100% to div style.
		$content = ! empty( $content ) ? preg_replace( '/(<script[^>]+) (.*?)src="\/\/(.*?)/i', '$1 src="https://$2', $content ) : $content; // Add https to script src.
		$content = ! empty( $content ) ? preg_replace( '/(<iframe[^>]+) style="(.*?)(height:[^;]*;)(.*?)"/i', '$1 style="$2$4"', $content ) : $content; // Remove height attribute from iframe.
		$content = ! empty( $content ) ? preg_replace( '/(<div[^>]+) (.*?)max-height:(.*?);(.*?)/i', '$1 $2$4', $content ) : $content; // Remove max-height from div tag style.

		// If smugmug block replace iframe url with content url.
		if ( ! empty( $block_data['attrs']['providerNameSlug'] ) && 'smugmug' === $block_data['attrs']['providerNameSlug'] ) {
			$content = ! empty( $content ) ? preg_replace( '/src="([^"]+)"/', 'src="' . $content_url . '"', $content ) : $content; // Replace iframe url with content url.
		}

		// If issuu block not then remove the blank div.
		if ( ! empty( $block_data['attrs']['providerNameSlug'] ) && 'issuu' !== $block_data['attrs']['providerNameSlug'] ) {
			$content = ! empty( $content ) ? preg_replace( '/<div[^>]+><\/div>/i', '', $content ) : $content; // Remove empty div tag.
		}

		$app_page_data['render']  = ! empty( $content ) ? $content : '';
		$app_page_data['content'] = ! empty( $content ) ? $content : '';
		$style                    = array();
		$attribute['provider']    = '';
		$attribute['type']        = '';
		$attribute['align']       = 'center';
		// Fetch the embed link.
		if ( preg_match_all( '/<div(.*?)class="wp-block-embed__wrapper"(.*?)>(.*?)<\/div>/s', $render, $content_matches ) ) {
			$attribute['src'] = wp_strip_all_tags( $content_matches[0][0] );
		}

		if ( preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $render, $content_matches ) ) {
			preg_match_all( '~<figcaption(.*?)>(.*?)</figcaption>~', $render, $content_matches );
			$attribute['caption'] = isset( $content_matches[2][0] ) ? $content_matches[2][0] : $content_matches[0][0];
			$style                = self::get_caption_color( $attribute['caption'], $style );
		}

		$attribute              = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = self::get_block_style( $block_data, $style );

		return $app_page_data;
	}

	/**
	 * Get cached oembed content with performance optimizations.
	 *
	 * @param string $content_url The URL to fetch oembed content for.
	 *
	 * @since 2.10.0
	 * @return string|false The oembed content or false on failure.
	 */
	public static function get_cached_oembed( $content_url ) {
		if ( empty( $content_url ) ) {
			return false;
		}

		$cache_key      = 'bbapp_oembed_' . md5( $content_url );
		$cached_content = wp_cache_get( $cache_key, 'bbapp_oembed' );

		if ( false !== $cached_content ) {
			return $cached_content;
		}

		add_filter( 'http_request_args', array( __CLASS__, 'modify_oembed_http_args' ), 10, 2 );
		$content = wp_oembed_get( $content_url );
		remove_filter( 'http_request_args', array( __CLASS__, 'modify_oembed_http_args' ), 10 );

		/**
		 * Determines the duration in seconds for which cached data remains valid.
		 *
		 * This variable is used to specify the time period after which the cached
		 * data should be considered stale and either refreshed or removed.
		 * It plays a critical role in cache management, improving performance
		 * while ensuring data consistency.
		 *
		 * A lower value means more frequent cache updates, ensuring fresher data
		 * but potentially increasing server load. Higher values reduce server load
		 * but may lead to outdated data being served.
		 *
		 * @since 2.10.0
		 */
		$cache_expiration = apply_filters( 'bbapp_oembed_cache_expiration', 3600 );

		if ( empty( $content ) ) {
			$fallback_content = self::get_oembed_fallback( $content_url );
			wp_cache_set( $cache_key, $fallback_content, 'bbapp_oembed', $cache_expiration );

			return $fallback_content;
		}

		wp_cache_set( $cache_key, $content, 'bbapp_oembed', $cache_expiration );

		return $content;
	}

	/**
	 * Modify HTTP request arguments for oembed requests to add timeout.
	 *
	 * @param array  $args HTTP request arguments.
	 * @param string $url  Request URL.
	 *
	 * @since 2.10.0
	 * @return array Modified HTTP request arguments.
	 */
	public static function modify_oembed_http_args( $args, $url ) {
		if ( strpos( $url, 'oembed' ) !== false ||
		     strpos( $url, 'youtube' ) !== false ||
		     strpos( $url, 'vimeo' ) !== false ||
		     strpos( $url, 'twitter' ) !== false ||
		     strpos( $url, 'instagram' ) !== false ) {
			$args['timeout']    = 3;
			$args['blocking']   = true;
			$args['user-agent'] = 'BuddyBoss App/1.0 (+' . home_url() . ')';
		}

		return $args;
	}

	/**
	 * Provide fallback content when oembed fails.
	 *
	 * @param string $content_url The original URL.
	 *
	 * @since 2.10.0
	 * @return string Fallback HTML content.
	 */
	public static function get_oembed_fallback( $content_url ) {
		$domain        = wp_parse_url( $content_url, PHP_URL_HOST );
		$fallback_html = sprintf(
			'<div class="bbapp-oembed-fallback"><a href="%1$s" target="_blank" rel="noopener">%2$s</a><p>%3$s</p></div>',
			esc_url( $content_url ),
			esc_html( $content_url ),
			sprintf( esc_html__( 'View content on %s', 'buddyboss-app' ), esc_html( $domain ) )
		);

		return $fallback_html;
	}

	/**
	 * Parse Common Block like HTML and List which render same.
	 *
	 * @param array  $block_data   Block data.
	 * @param string $render       Render html.
	 * @param array  $parent_style Parent style array.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_common_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style                    = self::get_block_style( $block_data, array(), $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse HTML Block like HTML.
	 *
	 * @param array  $block_data Block data.
	 * @param string $render     Render html.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_html_block( $block_data, $render ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$app_page_data['data']    = array();
		$app_page_data['style']   = array();

		return $app_page_data;
	}

	/**
	 * Parse Courses Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_courses_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$attribute['title']       = __( 'Courses', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';

		$type = 'fetch';
		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$course_ids = array();

			if ( ! empty( $block_data['attrs']['courses'] ) ) {
				$course_ids = $block_data['attrs']['courses'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $course_ids,
				'request_params' => array(
					'orderby'  => 'include',
					'include'  => $course_ids,
					'per_page' => count( $course_ids ),
				),
			);
		} else {
			$per_page    = ( ! empty( $block_data['attrs']['per_page'] ) ) ? absint( $block_data['attrs']['per_page'] ) : 5;
			$per_page    = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'orderby'   => ( ! empty( $block_data['attrs']['orderby'] ) ) ? $block_data['attrs']['orderby'] : 'id',
					'order'     => ( ! empty( $block_data['attrs']['order'] ) ) ? $block_data['attrs']['order'] : 'desc',
					'page'      => 1,
					'per_page'  => $per_page,
					'mycourses' => ( ! empty( $block_data['attrs']['filter'] ) && 'mine' === $block_data['attrs']['filter'] ) ? 1 : 0,
				),
			);

			if ( ( bbapp_is_learndash_enabled() && method_exists( '\LearnDash_Settings_Section', 'get_section_setting' ) ) ) {

				if ( 'yes' === \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_category' ) && ! empty( $block_data['attrs']['ld_cats'] ) && is_array( $block_data['attrs']['ld_cats'] ) ) {
					$data_source['request_params']['categories'] = implode( ',', $block_data['attrs']['ld_cats'] );
				}

				if ( 'yes' === \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_tag' ) && ! empty( $block_data['attrs']['ld_tags'] ) && is_array( $block_data['attrs']['ld_tags'] ) ) {
					$data_source['request_params']['tags'] = implode( ',', $block_data['attrs']['ld_tags'] );
				}

				if ( ! empty( $data_source['request_params']['categories'] ) && ! empty( $data_source['request_params']['tags'] ) ) {
					$data_source['request_params']['tax_relation'] = 'OR';  // category and tag both result should be show using OR relation.
				}
			}
		}

		$data_source['route']     = '/buddyboss-app/learndash/v1/courses';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse MemberPress LMS Courses Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @since 2.2.80
	 * @return array $app_page_data
	 */
	public static function parse_mp_courses_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$attribute['title']       = __( 'Courses', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';
		$type                     = 'fetch';

		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$course_ids = array();

			if ( ! empty( $block_data['attrs']['courses'] ) ) {
				$course_ids = $block_data['attrs']['courses'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $course_ids,
				'request_params' => array(
					'include' => $course_ids,
				),
			);
		} else {
			$per_page    = ( ! empty( $block_data['attrs']['per_page'] ) ) ? absint( $block_data['attrs']['per_page'] ) : 5;
			$per_page    = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'order'      => ( ! empty( $block_data['attrs']['orderby'] ) ) ? $block_data['attrs']['orderby'] : 'id', // MP LMS courses order by. Default is id.
					'order_dir'  => ( ! empty( $block_data['attrs']['order'] ) ) ? $block_data['attrs']['order'] : 'desc', // MP LMS courses order direction. Default is desc.
					'page'       => 1,
					'per_page'   => $per_page,
					'my_courses' => ( ! empty( $block_data['attrs']['filter'] ) && 'mine' === $block_data['attrs']['filter'] ) ? true : false,
				),
			);

			if ( bbapp_is_memberpress_lms_enabled() ) {

				if ( ! empty( $block_data['attrs']['mp_cats'] ) && is_array( $block_data['attrs']['mp_cats'] ) ) {
					$data_source['request_params']['category'] = implode( ',', $block_data['attrs']['mp_cats'] );
				}

				if ( ! empty( $block_data['attrs']['mp_tags'] ) && is_array( $block_data['attrs']['mp_tags'] ) ) {
					$data_source['request_params']['tag'] = implode( ',', $block_data['attrs']['mp_tags'] );
				}

				if ( ! empty( $data_source['request_params']['category'] ) && ! empty( $data_source['request_params']['tag'] ) ) {
					$data_source['request_params']['tax_relation'] = 'OR';  // category and tag both result should be show using OR relation.
				}
			}
		}

		$data_source['route']     = '/mp/v1/courses';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Tutor Courses Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @since 2.2.80
	 * @return array $app_page_data
	 */
	public static function parse_tutor_courses_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$attribute['title']       = __( 'Courses', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';
		$type                     = 'fetch';

		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$course_ids = array();

			if ( ! empty( $block_data['attrs']['courses'] ) ) {
				$course_ids = $block_data['attrs']['courses'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $course_ids,
				'request_params' => array(
					'orderby' => 'include',
					'include' => $course_ids,
				),
			);
		} else {
			$per_page    = ( ! empty( $block_data['attrs']['per_page'] ) ) ? absint( $block_data['attrs']['per_page'] ) : 5;
			$per_page    = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'orderby'   => ( ! empty( $block_data['attrs']['orderby'] ) ) ? $block_data['attrs']['orderby'] : 'id',
					'order'     => ( ! empty( $block_data['attrs']['order'] ) ) ? $block_data['attrs']['order'] : 'desc',
					'page'      => 1,
					'per_page'  => $per_page,
					'mycourses' => ( ! empty( $block_data['attrs']['filter'] ) && 'mine' === $block_data['attrs']['filter'] ) ? 1 : 0,
				),
			);

			if ( bbapp_is_tutor_lms_plugins_active() ) {

				if ( ! empty( $block_data['attrs']['tutor_cats'] ) && is_array( $block_data['attrs']['tutor_cats'] ) ) {
					$data_source['request_params']['categories'] = implode( ',', $block_data['attrs']['tutor_cats'] );
				}

				if ( ! empty( $block_data['attrs']['tutor_tags'] ) && is_array( $block_data['attrs']['tutor_tags'] ) ) {
					$data_source['request_params']['tags'] = implode( ',', $block_data['attrs']['tutor_tags'] );
				}

				if ( ! empty( $data_source['request_params']['categories'] ) && ! empty( $data_source['request_params']['tags'] ) ) {
					$data_source['request_params']['tax_relation'] = 'OR';  // category and tag both result should be show using OR relation.
				}
			}
		}

		$data_source['route']     = '/buddyboss-app/tutor/v1/courses';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Topics Block
	 *
	 * @param array  $block_data   Block data attributes.
	 * @param string $render       Block render content.
	 * @param array  $parent_style Array of parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_topics_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$attribute['title']       = __( 'Discussions', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';

		$type = 'fetch';
		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$discussion_ids = array();

			if ( ! empty( $block_data['attrs']['discussions'] ) ) {
				$discussion_ids = $block_data['attrs']['discussions'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $discussion_ids,
				'request_params' => array(
					'orderby'  => 'include',
					'include'  => $discussion_ids,
					'per_page' => count( $discussion_ids ),
				),
			);
		} else {

			$order_by = 'popular';
			$order    = 'desc';
			$page     = 1;
			$per_page = 5;

			// Order by.
			if ( ! empty( $block_data['attrs']['orderby'] ) ) {
				$order_by = $block_data['attrs']['orderby'];
			}

			// Order.
			if ( ! empty( $block_data['attrs']['order'] ) ) {
				$order = $block_data['attrs']['order'];
			}

			// Per Page.
			if ( ! empty( $block_data['attrs']['per_page'] ) ) {
				$per_page = absint( $block_data['attrs']['per_page'] );
				$per_page = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			}

			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'orderby'  => $order_by,
					'order'    => $order,
					'page'     => $page,
					'per_page' => $per_page,
				),
			);
		}

		$data_source['route']                    = '/buddyboss/v1/topics';
		$data_source['request_params']['_embed'] = true;
		$attribute['data_source']                = $data_source;
		$style                                   = self::get_block_style( $block_data, array(), $parent_style );
		$attribute                               = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']                   = $attribute;
		$app_page_data['style']                  = $style;

		return $app_page_data;
	}

	/**
	 * Parse Forums Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_forums_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Forums', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';

		$type = 'fetch';
		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$forum_ids = array();

			if ( ! empty( $block_data['attrs']['forums'] ) ) {
				$forum_ids = $block_data['attrs']['forums'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $forum_ids,
				'request_params' => array(
					'orderby'  => 'include',
					'include'  => $forum_ids,
					'per_page' => count( $forum_ids ),
				),
			);
		} else {
			$order_by     = 'date';
			$order        = 'desc';
			$page         = 1;
			$per_page     = 5;
			$show_myforum = 0;

			// Order by.
			if ( ! empty( $block_data['attrs']['orderby'] ) ) {
				$order_by = $block_data['attrs']['orderby'];
			}

			// Order.
			if ( ! empty( $block_data['attrs']['order'] ) ) {
				$order = $block_data['attrs']['order'];
			}

			// Per Page.
			if ( ! empty( $block_data['attrs']['per_page'] ) ) {
				$per_page = absint( $block_data['attrs']['per_page'] );
				$per_page = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			}

			// My Forum.
			if ( ! empty( $block_data['attrs']['show_myforum'] ) ) {
				$show_myforum = get_current_user_id();
			}

			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'orderby'  => $order_by,
					'order'    => $order,
					'page'     => $page,
					'per_page' => $per_page,
					'author'   => $show_myforum,
				),
			);
		}

		$data_source['route']                    = '/buddyboss/v1/forums';
		$data_source['request_params']['_embed'] = true;
		$attribute['data_source']                = $data_source;
		$style                                   = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                               = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']                   = $attribute;
		$app_page_data['style']                  = $style;

		return $app_page_data;
	}

	/**
	 * Parse Groups Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_groups_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Groups', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';

		$type = 'fetch';
		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$group_ids = array();

			if ( ! empty( $block_data['attrs']['groups'] ) ) {
				$group_ids = $block_data['attrs']['groups'];
			}

			$data_source = array(
				'type'           => 'ids',
				'ids'            => $group_ids,
				'request_params' => array(
					'orderby'  => 'include',
					'include'  => $group_ids,
					'per_page' => count( $group_ids ),
				),
			);
		} else {
			$gtype    = 'newest';
			$order_by = 'date_created';
			$order    = 'desc';
			$page     = 1;
			$per_page = 5;

			// Order by.
			if ( ! empty( $block_data['attrs']['orderby'] ) ) {
				$gtype = $block_data['attrs']['orderby'];

				if ( 'newest' === $gtype ) {
					$order_by = 'date_created';
				} elseif ( 'active' === $gtype ) {
					$order_by = 'last_activity';
				} elseif ( 'popular' === $gtype ) {
					$order_by = 'total_member_count';
				} elseif ( 'alphabetical' === $gtype ) {
					$order_by = 'name';
				} elseif ( 'random' === $gtype ) {
					$order_by = 'random';
				} else {
					$order_by = 'last_activity';
				}
			}

			// Order.
			if ( ! empty( $block_data['attrs']['order'] ) ) {
				$order = $block_data['attrs']['order'];
			}

			// Per Page.
			if ( ! empty( $block_data['attrs']['per_page'] ) ) {
				$per_page = absint( $block_data['attrs']['per_page'] );
				$per_page = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			}

			$data_source = array(
				'type'           => 'fetch',
				'request_params' => array(
					'type'     => $gtype,
					'orderby'  => $order_by,
					'order'    => $order,
					'page'     => $page,
					'per_page' => $per_page,
				),
			);

			// My Groups.
			if ( ! empty( $block_data['attrs']['show_mygroup'] ) ) {
				$data_source['request_params']['user_id'] = get_current_user_id();
			}

			// if user login then we have to pass show hidden.
			if ( is_user_logged_in() ) {
				$data_source['request_params']['show_hidden'] = true;
			}

			// Only Forums group.
			if ( ! empty( $block_data['attrs']['only_forum'] ) ) {
				$data_source['request_params']['only_forum'] = '1';
			}

			// Hide Private groups.
			if ( ! empty( $block_data['attrs']['hide_private'] ) ) {
				$data_source['request_params']['status'] = 'public';
			}
		}

		$data_source['route']     = '/buddyboss/v1/groups';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Members Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_members_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Members', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';
		$filter_type              = 'fetch';

		if ( ! empty( $block_data['attrs']['filter_type'] ) ) {
			$filter_type = $block_data['attrs']['filter_type'];
		}

		if ( 'ids' === $filter_type ) {
			$member_ids = array();

			if ( ! empty( $block_data['attrs']['members'] ) ) {
				$member_ids = $block_data['attrs']['members'];
			}

			$data_source['request_params'] = array(
				'include'  => $member_ids,
				'per_page' => count( $member_ids ),
			);
		} else {
			$type     = 'newest';
			$per_page = 5;

			// Type.
			if ( ! empty( $block_data['attrs']['type'] ) ) {
				$type = $block_data['attrs']['type'];
			}
			// Per Page.
			if ( ! empty( $block_data['attrs']['per_page'] ) ) {
				$per_page = absint( $block_data['attrs']['per_page'] );
				$per_page = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			}

			$data_source['request_params'] = array(
				'type'     => $type,
				'per_page' => $per_page,
			);
		}

		$data_source['route']     = '/buddyboss/v1/members';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Activity Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_activity_block( $block_data, $render, $parent_style ) {

		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Activity', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';
		$page                     = 1;
		$per_page                 = 5;

		// Per Page.
		if ( ! empty( $block_data['attrs']['per_page'] ) ) {
			$per_page = absint( $block_data['attrs']['per_page'] );
			$per_page = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
		}

		$data_source = array(
			'type'           => 'fetch',
			'request_params' => array(
				'page'     => $page,
				'per_page' => $per_page,
				'_embed'   => true
			),
		);

		// My Activity.
		if ( ! empty( $block_data['attrs']['show_my_activity'] ) && is_user_logged_in() ) {
			$data_source['request_params']['user_id'] = get_current_user_id();
		}

		// Only group activity.
		if ( ! empty( $block_data['attrs']['only_group_activity'] ) ) {
			$data_source['request_params']['scope']     = 'groups';
			$data_source['request_params']['component'] = 'groups';
			$data_source['request_params']['type']      = 'activity_update';
		}

		$data_source['route']     = '/buddyboss/v1/activity';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Notifications Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_notifications_block( $block_data, $render, $parent_style ) {
		$attribute                         = array();
		$style                             = array();
		$app_page_data['type']             = $block_data['blockName'];

		$app_page_data['render']           = $render;
		$app_page_data['content']          = $render;
		$style['backgroundColor']          = '';
		$styling                           = Styling::instance()->get_styling_colors();
		$style['headerTextColor']          = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']                = __( 'Notifications', 'buddyboss-app' );
		$attribute['icon_url']             = '';
		$attribute['tint_color']           = '#908DA1';

		$per_page    = ( ! empty( $block_data['attrs']['per_page'] ) ) ? absint( $block_data['attrs']['per_page'] ) : 5;
		$per_page    = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
		$data_source = array(
			'request_params' => array(
				'orderby'    => ( ! empty( $block_data['attrs']['orderby'] ) ) ? $block_data['attrs']['orderby'] : 'date_notified',
				'sort_order' => ( ! empty( $block_data['attrs']['order'] ) ) ? $block_data['attrs']['order'] : 'DESC', // Please use uppercase DESC or ASC.
				'page'       => 1,
				'per_page'   => $per_page,
			),
		);

		$data_source['route']     = '/buddyboss-app/v1/notifications';
		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']             = $attribute;
		$app_page_data['style']            = $style;

		return $app_page_data;
	}

	/**
	 * Parse Ab Quick Links Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_quick_links_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Quick Links', 'buddyboss-app' );
		$quick_links_array        = array();
		$quick_links              = array();

		if ( ! empty( $block_data['attrs']['quickLinkObj'] ) ) {
			$quick_links_array = $block_data['attrs']['quickLinkObj'];
		}

		$locked_quick_link = bbapp_lite_app_license_locked_quick_link();

		foreach ( $quick_links_array as $key => $quick_link ) {
			if ( bbapp_is_lite_live_app() &&
				! empty( $quick_link['id'] ) &&
				! empty( $locked_quick_link[ $quick_link['type'] ] ) &&
				in_array( $quick_link['id'], $locked_quick_link[ $quick_link['type'] ], true )
			) {
				continue;
			}

			$quick_links[ $key ]['type']          = isset( $quick_link['type'] ) ? $quick_link['type'] : 'links';
			$quick_links[ $key ]['data']['title'] = ! empty( $quick_link['title'] ) ? $quick_link['title'] : '';
			$quick_links[ $key ]['data']['id']    = ! empty( $quick_link['id'] ) ? $quick_link['id'] : '';
			$quick_links[ $key ]['icon']          = \BuddyBossApp\Common\IconPicker::instance()->get_icon_for_api( $quick_link, '' );
			$quick_links[ $key ]['data']['url']   = isset( $quick_link['link'] ) ? esc_url( $quick_link['link'] ) : '';

			if ( 'link' === $quick_link['type'] ) {
				$quick_links[ $key ]['data']['url'] = ! empty( $quick_link['id'] ) ? esc_url( self::quick_link_default_links( $quick_link['id'] ) ) : '';
			} elseif ( 'core_app_screen' === $quick_link['type'] ) {
				$quick_links[ $key ]['data']['url'] = ! empty( $quick_link['id'] ) ? esc_url( home_url( 'bbapp/screen/' . $quick_link['id'] ) ) : '';
			} elseif ( 'app_page' === $quick_link['type'] ) {
				$quick_links[ $key ]['data']['url'] = ! empty( $quick_link['id'] ) ? esc_url( home_url( 'bbapp/page/' . $quick_link['id'] ) ) : '';
			}

			$quick_links[ $key ]['deeplink_data']['url'] = esc_url( $quick_links[ $key ]['data']['url'] ); // @todo: to be removed app is no longer using it.
		}

		$attribute['quick_links'] = array_values( $quick_links );
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Get the quick link user profile links.
	 *
	 * @param string $link_slug Platform link slug.
	 *
	 * @return string
	 */
	public static function quick_link_default_links( $link_slug ) {

		if ( function_exists( 'bp_is_active' ) && ! empty( bp_loggedin_user_domain() ) ) {

			if ( bp_is_active( 'xprofile' ) && 'profile' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_profile_slug() );
			}

			if ( bp_is_active( 'settings' ) && bp_get_settings_slug() === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_settings_slug() );
			}

			if ( bp_is_active( 'activity' ) && 'activities' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_activity_slug() );
			}

			if ( bp_is_active( 'notifications' ) && 'notifications' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_notifications_slug() );
			}

			if ( bp_is_active( 'messages' ) && 'messages' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_messages_slug() );
			}

			if ( bp_is_active( 'friends' ) && 'friends' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_friends_slug() );
			}

			if ( bp_is_active( 'groups' ) && 'groups' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_groups_slug() );
			}

			if ( bp_is_active( 'forums' ) && 'forums' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bbp_get_root_slug() );
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_media_support_enabled' ) && bp_is_profile_media_support_enabled() && 'photos' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_media_slug() );
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_document_support_enabled' ) && bp_is_profile_document_support_enabled() && 'documents' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_document_slug() );
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_video_support_enabled' ) && bp_is_profile_video_support_enabled() && 'videos' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_video_slug() );
			}

			if ( bp_is_active( 'invites' ) && function_exists( 'bp_allow_user_to_send_invites' ) && true === bp_allow_user_to_send_invites() && 'invites' === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . bp_get_invites_slug() );
			}
		}

		return apply_filters( 'bbapp_quick_link_deeplink_links', '', $link_slug );
	}

	/**
	 * Parse H5P Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_h5p_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;

		preg_match( '@src="([^"]+)"@', $render, $match );

		$render         = preg_replace( '/<p>(.+)<\/p>/Uuis', '$1', $render );
		$split_data     = preg_split( '/(<iframe[^>]+\>)/i', $render, - 1, PREG_SPLIT_DELIM_CAPTURE );
		$split_data     = array_filter( $split_data );
		$style          = array();
		$new_split_data = array();

		if ( ! empty( $split_data ) && is_array( $split_data ) ) {
			foreach ( $split_data as $data_text ) {
				if ( preg_match_all( '/<(?:iframe|script) [^>]*src *= *["\']?([^"\']*)/i', $render, $matches ) ) {
					$style = array();
					$src   = array_pop( $matches );

					preg_match( '@width="([^"]+)"@', $data_text, $match );
					$width = array_pop( $match );

					preg_match( '@height="([^"]+)"@', $data_text, $match );
					$height                           = array_pop( $match );
					$new_split_data['type']           = 'iframe';
					$new_split_data['src']            = isset( $src[0] ) ? $src[0] : null;
					$new_split_data['resizer_js_src'] = isset( $src[1] ) ? $src[1] : esc_url_raw( content_url( 'plugins/h5p/h5p-php-library/js/h5p-resizer.js' ) );
					$style['width']                   = $width;
					$style['height']                  = $height;
					$new_split_data['style']          = $style;
				}
			}
		}

		// H5P Title.
		$attribute['title']        = '';
		$attribute['content_type'] = isset( $block_data['attrs']['content_type'] ) ? $block_data['attrs']['content_type'] : '';
		$attribute['h5p_id']       = isset( $block_data['attrs']['h5p_id'] ) ? $block_data['attrs']['h5p_id'] : 0;
		$app_page_data['content']  = (array) $new_split_data;
		$attribute                 = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']     = $attribute;
		$app_page_data['style']    = $style;

		return $app_page_data;
	}

	/**
	 * Parse Code block content.
	 *
	 * @param array  $block_data Array of block data.
	 * @param string $render     Content to be rendered on the app side.
	 *
	 * @return array
	 */
	public static function parse_code_block( $block_data, $render ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = wp_strip_all_tags( $render );
		$style                    = self::get_block_style( $block_data );
		$attribute                = self::get_block_attributes( $block_data );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Pull Quote block content.
	 *
	 * @param array  $block_data Array of block data.
	 * @param string $render     App render content.
	 *
	 * @return array
	 */
	public static function parse_pullquote_block( $block_data, $render ) {
		$style                   = array();
		$attribute               = array();
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;

		// get blockquote content.
		preg_match_all( '#<blockquote.*?>([\s\S]*?)</blockquote>#', $render, $block_content_matches );

		if ( isset( $block_content_matches[1] ) ) {
			$block_content = $block_content_matches[1][0];
			preg_match_all( "/(<([\w]+)[^>]*>)(.*?)(<\/\\2>)/", $block_content, $matches, PREG_SET_ORDER );
			if ( isset( $matches[0][0] ) ) {
				$app_page_data['content'] = $matches[0][0];
			}
			if ( isset( $matches[1][3] ) ) {
				$attribute['citation'] = $matches[1][3];
			}
		}

		$style['customMainColor'] = '#555555';
		$style['layout']          = 'default';
		$style                    = self::get_block_style( $block_data, $style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Preformatted block content.
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array
	 */
	public static function parse_preformatted_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = wp_strip_all_tags( $render );
		$style                    = self::get_block_style( $block_data, array(), $parent_style );
		$attribute                = self::get_block_attributes( $block_data );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Verse block content.
	 *
	 * @param array  $block_data Array of block data.
	 * @param string $render     App render content.
	 *
	 * @return array
	 */
	public static function parse_verse_block( $block_data, $render ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = strip_tags( $render, '<br>' );
		$style                    = self::get_block_style( $block_data );
		$attribute                = self::get_block_attributes( $block_data );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse File block content.
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array
	 */
	public static function parse_file_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$attribute['show_button'] = true;
		$attribute                = self::get_block_attributes( $block_data, $attribute );

		if ( preg_match_all( '#<a.*?>([\s\S]*?)</a>#', $render, $block_content_matches ) ) {
			preg_match_all( '#<a.*?>([\s\S]*?)</a>#', $render, $block_content_matches );
			if ( isset( $block_content_matches[1][0] ) ) {
				$app_page_data['content'] = $block_content_matches[1][0];
			}
			if ( true === $attribute['show_button'] && isset( $block_content_matches[1][1] ) ) {
				$attribute['button_text'] = $block_content_matches[1][1];
			}
		}

		if ( preg_match( '/target=/', $render ) ) {
			$attribute['targetBlank'] = true;
		}

		$style                  = self::get_block_style( $block_data, array(), $parent_style );
		$app_page_data['data']  = $attribute;
		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Separator block content.
	 *
	 * @param array  $block_data Array of block data.
	 * @param string $render     App render content.
	 *
	 * @return array
	 */
	public static function parse_separator_block( $block_data, $render ) {
		$app_page_data['type']    = $block_data['blockName'];
		$style['layout']          = 'default';
		$style['backgroundColor'] = '#000000';
		$style                    = self::get_block_style( $block_data, $style );
		$app_page_data['render']  = $render;
		$app_page_data['content'] = null;
		$app_page_data['data']    = array();
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Spacer block content.
	 *
	 * @param array  $block_data Array of block data.
	 * @param string $render     App render content.
	 *
	 * @return array
	 */
	public static function parse_spacer_block( $block_data, $render ) {
		$app_page_data['type'] = $block_data['blockName'];
		$attribute             = array();

		if ( ! isset( $block_data['attrs']['height'] ) ) {
			$attribute['height'] = '100px';
		} elseif ( empty( $block_data['attrs']['height'] ) ) {
			$attribute['height'] = '';
		} else {
			// Supporting the older versions (5.8) of WordPress.
			$block_height        = $block_data['attrs']['height'];
			$height              = str_replace( ' ', '', strtolower( $block_data['attrs']['height'] ) );
			$unit                = preg_replace( '/\d+/', '', $height );
			$unit                = str_replace( '.', '', $unit );
			$attribute['height'] = empty( $unit ) ? $block_height . 'px' : $block_height;
		}

		$app_page_data['render']  = $render;
		$app_page_data['content'] = null;
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = array();

		return $app_page_data;
	}

	/**
	 * Parse Spacer block content.
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array
	 */
	public static function parse_group_block( $block_data, $render, $parent_style ) {
		$app_page_data['type']   = $block_data['blockName'];
		$app_page_data['render'] = $render;
		$attribute               = array();
		$style                   = array();
		$style                   = self::get_block_style( $block_data, $style, $parent_style );
		$style                   = self::get_backward_compatibility_block_style( $block_data, $style );
		$attribute               = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']   = $attribute;

		if ( isset( $block_data['innerBlocks'] ) ) {
			$app_page_data['group_blocks'] = bbapp_parse_inner_block( $block_data['innerBlocks'], $style );
		}

		if ( ! empty( $app_page_data['group_blocks'] ) ) {
			foreach ( $app_page_data['group_blocks'] as $key => $block_content ) {
				if ( ! empty( $block_content['render'] ) ) {
					$app_page_data['group_blocks'][ $key ]['render'] = preg_replace( '/(<[^>]+) style=".*?"/i', '$1', $block_content['render'] );
				}
				if ( ! empty( $block_content['content'][0]['data'] ) ) {
					$app_page_data['group_blocks'][ $key ]['content'][0]['data'] = preg_replace( '/(<[^>]+) style=".*?"/i', '$1', $block_content['content'][0]['data'] );
				}
			}
		}
		// If allignment is not set.
		if ( ! isset( $style['align'] ) ) {
			$style['align'] = 'space-between';
		}

		$app_page_data['style'] = $style;

		return $app_page_data;
	}

	/**
	 * Parse Posts Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @return array $app_page_data
	 */
	public static function parse_posts_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;
		$style['backgroundColor'] = '';
		$styling                  = Styling::instance()->get_styling_colors();
		$style['headerTextColor'] = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
		$attribute['title']       = __( 'Blog', 'buddyboss-app' );
		$attribute['icon_url']    = '';
		$attribute['tint_color']  = '#908DA1';
		$type                     = 'fetch';

		if ( ! empty( $block_data['attrs']['type'] ) ) {
			$type = $block_data['attrs']['type'];
		}

		if ( 'ids' === $type ) {
			$post_ids = array();

			if ( ! empty( $block_data['attrs']['posts'] ) ) {
				$post_ids = $block_data['attrs']['posts'];
			}

			$request_params = array(
				'include'  => $post_ids,
				'per_page' => count( $post_ids ),
			);
		} else {
			$page         = 1;
			$order_by     = ! empty( $block_data['attrs']['orderby'] ) ? $block_data['attrs']['orderby'] : 'date';
			$order        = ! empty( $block_data['attrs']['order'] ) ? $block_data['attrs']['order'] : 'desc';
			$cat_ids      = ( ! empty( $block_data['attrs']['cats'] ) && is_array( $block_data['attrs']['cats'] ) ) ? implode( ',', $block_data['attrs']['cats'] ) : '';
			$tag_ids      = ( ! empty( $block_data['attrs']['tags'] ) && is_array( $block_data['attrs']['tags'] ) ) ? implode( ',', $block_data['attrs']['tags'] ) : '';
			$author_ids   = ( ! empty( $block_data['attrs']['users'] ) && is_array( $block_data['attrs']['users'] ) ) ? implode( ',', $block_data['attrs']['users'] ) : '';
			$date         = ! empty( $block_data['attrs']['date'] ) ? $block_data['attrs']['date'] : 'all-dates';
			$per_page     = ! empty( $block_data['attrs']['per_page'] ) ? $block_data['attrs']['per_page'] : 5;
			$per_page     = ( 1 <= $per_page ) && ( $per_page <= 20 ) ? $per_page : 5;
			$sticky_posts = '';

			if ( isset( $block_data['attrs']['sticky_posts'] ) && 'true' === $block_data['attrs']['sticky_posts'] ) {
				$sticky_posts = true;
			}

			if ( isset( $block_data['attrs']['sticky_posts'] ) && 'false' === $block_data['attrs']['sticky_posts'] ) {
				$sticky_posts = false;
			}

			$categories_list = array();
			if ( ! empty( $cat_ids ) ) {

				$categories_terms = get_categories(
					array(
						'include' => $cat_ids,
						'fields'  => 'id=>name',
					)
				);

				if ( ! empty( $categories_terms ) ) {
					$categories_list = $categories_terms;
				}
			}

			$request_params = array(
				'page'     => $page,
				'orderby'  => $order_by,
				'order'    => $order,
				'date'     => $date,
				'per_page' => $per_page,
			);

			if ( ! empty( $cat_ids ) ) {
				$request_params['categories'] = $cat_ids;
			}

			if ( ! empty( $tag_ids ) ) {
				$request_params['tags'] = $tag_ids;
			}

			// The category and tag both result should be show using OR relation.
			if ( ! empty( $tag_ids ) && ! empty( $cat_ids ) ) {
				$request_params['tax_relation'] = 'OR';
			}

			if ( '' !== $sticky_posts ) {
				$request_params['sticky'] = $sticky_posts;
			}

			if ( ! empty( $categories_list ) ) {
				$request_params['categories_list'] = $categories_list;
			}

			if ( ! empty( $author_ids ) ) {
				$request_params['author'] = $author_ids;
			}
		}

		$data_source = array(
			'route'          => '/wp/v2/posts',
			'request_params' => $request_params,
		);

		$attribute['data_source'] = $data_source;
		$style                    = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']    = $attribute;
		$app_page_data['style']   = $style;

		return $app_page_data;
	}

	/**
	 * Parse Directory Block
	 *
	 * @param array  $block_data   Array of block data.
	 * @param string $render       App render content.
	 * @param array  $parent_style Parent style.
	 *
	 * @since 2.3.40
	 * @return array $app_page_data
	 */
	public static function parse_directory_block( $block_data, $render, $parent_style ) {
		$attribute                = array();
		$style                    = array();
		$app_page_data['type']    = $block_data['blockName'];
		$app_page_data['render']  = $render;
		$app_page_data['content'] = $render;

		$post_type = ! empty( $block_data['attrs']['postType'] ) ? $block_data['attrs']['postType'] : 'post';
		$order_by  = ! empty( $block_data['attrs']['orderBy'] ) ? $block_data['attrs']['orderBy'] : 'ID';
		$order     = ! empty( $block_data['attrs']['order'] ) ? $block_data['attrs']['order'] : 'ASC';
		$per_page  = ! empty( $block_data['attrs']['perPage'] ) ? $block_data['attrs']['perPage'] : 10;

		$directory_settings = ! empty( $block_data['attrs']['directorySettings'] ) ? $block_data['attrs']['directorySettings'] : array();
		$show_search        = ! empty( $directory_settings['showSearch'] ) ? $directory_settings['showSearch'] : false;
		$show_filter        = ! empty( $directory_settings['showFilter'] ) ? $directory_settings['showFilter'] : false;
		$show_avatar        = ! empty( $directory_settings['showAvatar'] ) ? $directory_settings['showAvatar'] : false;
		$show_comment       = ! empty( $directory_settings['showComment'] ) ? $directory_settings['showComment'] : false;
		$show_bookmark      = ! empty( $directory_settings['showBookmark'] ) ? $directory_settings['showBookmark'] : false;
		$thumbnail          = ! empty( $directory_settings['showFeaturedImage'] ) ? $directory_settings['showFeaturedImage'] : false;

		$directory_filters = ! empty( $block_data['attrs']['directoryFilters'] ) ? $block_data['attrs']['directoryFilters'] : array();

		$settings = array(
			'show_search' => $show_search,
			'show_filter' => $show_filter,
		);

		$request_params = array(
			'post_type' => $post_type,
			'order_by'  => $order_by,
			'order'     => $order,
			'per_page'  => $per_page,
		);

		$directory_meta = ! empty( $block_data['attrs']['directoryMeta'] ) ? $block_data['attrs']['directoryMeta'] : array();

		foreach ( $directory_meta as $index => $meta ) {
			$settings[ 'show_meta_' . ( $index + 1 ) ] = ! empty( $meta['isShow'] ) ? $meta['isShow'] : false;

			if ( ! empty( $meta['isShow'] ) ) {
				$request_params[ 'meta_' . ( $index + 1 ) ] = ! empty( $meta['metaValue'] ) ? $meta['metaValue'] : '';
			}
		}

		$request_params['thumbnail']     = ! empty( $thumbnail ) ? $thumbnail : '';
		$request_params['show_avatar']   = ! empty( $show_avatar ) && $show_avatar;
		$request_params['show_comment']  = ! empty( $show_comment ) && $show_comment;
		$request_params['show_bookmark'] = ! empty( $show_bookmark ) && $show_bookmark;

		if ( ! empty( $show_filter ) && ! empty( $directory_filters ) ) {
			$settings['filter_options'] = array_values(
				array_filter(
					$directory_filters,
					function ( $filter ) {
						return ! empty( $filter['checked'] );
					}
				)
			);
		}

		$data_source = array(
			'route'          => '/buddyboss-app/v1/directory-block',
			'request_params' => $request_params,
		);

		$attribute['data_source']        = $data_source;
		$attribute['directory_settings'] = $settings;
		$style                           = self::get_block_style( $block_data, $style, $parent_style );
		$attribute                       = self::get_block_attributes( $block_data, $attribute );
		$app_page_data['data']           = $attribute;
		$app_page_data['style']          = $style;

		return $app_page_data;
	}

	/**
	 * Color which used in Gutenberg blocks.
	 *
	 * @return array
	 */
	public static function gutenberg_color_array() {

		$gutenberg_colors = array(
			'very-light-gray'                      => '#eeeeee',
			'very-dark-gray'                       => '#313131',
			'primary'                              => '#0073a8',
			'secondary'                            => '#005075',
			'dark-gray'                            => '#111111',
			'light-gray'                           => '#767676',
			'black'                                => '#000000',
			'cyan-bluish-gray'                     => '#abb8c3',
			'white'                                => '#ffffff',
			'pale-pink'                            => '#f78da7',
			'vivid-red'                            => '#cf2e2e',
			'luminous-vivid-orange'                => '#ff6900',
			'luminous-vivid-amber'                 => '#fcb900',
			'light-green-cyan'                     => '#7bdcb5',
			'vivid-green-cyan'                     => '#00d084',
			'pale-cyan-blue'                       => '#8ed1fc',
			'vivid-cyan-blue'                      => '#0693e3',
			'vivid-purple'                         => '#9b51e0',
			'subtle-light-gray'                    => '#f3f4f5',
			'subtle-pale-green'                    => '#e9fbe5',
			'subtle-pale-blue'                     => '#e7f5fe',
			'subtle-pale-pink'                     => '#fcf0ef',
			'vivid-cyan-blue-to-vivid-purple'      => 'linear-gradient(135deg,rgba(6,147,227,1) 0%,rgb(155,81,224) 100%)',
			'light-green-cyan-to-vivid-green-cyan' => 'linear-gradient(135deg,rgb(122,220,180) 0%,rgb(0,208,130) 100%)',
			'luminous-vivid-amber-to-luminous-vivid-orange' => 'linear-gradient(135deg,rgba(252,185,0,1) 0%,rgba(255,105,0,1) 100%)',
			'luminous-vivid-orange-to-vivid-red'   => 'linear-gradient(135deg,rgba(255,105,0,1) 0%,rgb(207,46,46) 100%)',
			'very-light-gray-to-cyan-bluish-gray'  => 'linear-gradient(135deg,rgb(238,238,238) 0%,rgb(169,184,195) 100%)',
			'cool-to-warm-spectrum'                => 'linear-gradient(135deg,rgb(74,234,220) 0%,rgb(151,120,209) 20%,rgb(207,42,186) 40%,rgb(238,44,130) 60%,rgb(251,105,98) 80%,rgb(254,248,76) 100%)',
			'blush-light-purple'                   => 'linear-gradient(135deg,rgb(255,206,236) 0%,rgb(152,150,240) 100%)',
			'blush-bordeaux'                       => 'linear-gradient(135deg,rgb(254,205,165) 0%,rgb(254,45,45) 50%,rgb(107,0,62) 100%)',
			'luminous-dusk'                        => 'linear-gradient(135deg,rgb(255,203,112) 0%,rgb(199,81,192) 50%,rgb(65,88,208) 100%)',
			'pale-ocean'                           => 'linear-gradient(135deg,rgb(255,245,203) 0%,rgb(182,227,212) 50%,rgb(51,167,181) 100%)',
			'electric-grass'                       => 'linear-gradient(135deg,rgb(202,248,128) 0%,rgb(113,206,126) 100%)',
			'midnight'                             => 'linear-gradient(135deg,rgb(2,3,129) 0%,rgb(40,116,252) 100%)',
		);

		// Get from WordPress Core theme json wp-includes/theme.json.
		$style_settings = \WP_Theme_JSON_Resolver::get_merged_data()->get_settings();
		if ( ! empty( $style_settings ) ) {
			if ( ! empty( $style_settings['color']['palette']['default'] ) ) {
				$styles = $style_settings['color']['palette']['default'];
				foreach ( $styles as $style ) {
					$gutenberg_colors[ $style['slug'] ] = $style['color'];
				}
			}
			if ( ! empty( $style_settings['color']['gradients']['default'] ) ) {
				$styles = $style_settings['color']['gradients']['default'];
				foreach ( $styles as $style ) {
					$gutenberg_colors[ $style['slug'] ] = $style['gradient'];
				}
			}
		}

		return $gutenberg_colors;
	}

	/**
	 * Get caption colors.
	 *
	 * @param string $caption_html Caption html.
	 * @param array  $style        Styles.
	 *
	 * @since 1.6.7
	 *
	 * @return array|mixed
	 */
	public static function get_caption_color( $caption_html, $style = array() ) {
		if ( preg_match_all( '~<(.*?) class="(.*?)">(.*?)</(.*?)>~', $caption_html, $content_matches ) ) {
			if ( ! empty( $content_matches[2][0] ) ) {
				$gutenberg_color = self::gutenberg_color_array();
				foreach ( $gutenberg_color as $key => $value ) {
					$preg_match = '/' . $key . '/i';
					if ( preg_match( $preg_match, $content_matches[2][0] ) ) {
						$style['caption_color'] = $value;
					}
				}
			}
		}
		if ( preg_match_all( '~<mark style="(.*?)"~', $caption_html, $content_matches ) ) {
			if ( ! empty( $content_matches[1][0] ) ) {
				$matches_explode = explode( ':', $content_matches[1][0] );
				if ( ! empty( $matches_explode[1] ) ) {
					$style['caption_bgcolor'] = $matches_explode[1];
				}
			}
		}

		return $style;
	}

	/**
	 * Function prepare the block style.
	 *
	 * @param array $block_data   Gutenberg data attributes.
	 * @param array $style        Style array.
	 * @param array $parent_style Parent style array.
	 *
	 * @since 1.6.7
	 * @return array|mixed
	 */
	public static function get_block_style( $block_data, $style = array(), $parent_style = array() ) {
		$style['parent_style'] = (object) $parent_style;

		if ( isset( $block_data['attrs'] ) ) {

			if ( ! empty( $block_data['attrs']['className'] ) ) {
				$default_theme              = wp_get_theme();
				$default_bg_color           = '';
				$default_text_outline_color = '';

				if ( 'twentynineteen' === $default_theme->get_stylesheet() ) {
					$default_bg_color           = '#0073aa';
					$default_text_outline_color = '#0073aa';
				}

				switch ( $block_data['attrs']['className'] ) {
					case 'is-style-outline':
						$style['button_style']    = 'outline';
						$style['color']           = '#000000';
						$style['backgroundColor'] = '#FFFFFF';
						if ( ! empty( $default_text_outline_color ) ) {
							$style['color'] = $default_text_outline_color;
						}
						break;
					case 'is-style-squared':
						$style['button_style'] = 'squared';
						if ( ! empty( $default_bg_color ) ) {
							$style['backgroundColor'] = $default_bg_color;
						}
						$style['color'] = '#ffffff';
						break;
					case 'is-style-fill':
						$style['button_style']    = 'fill';
						$style['backgroundColor'] = '';
						$style['color']           = '';
						break;
					case 'is-style-stripes':
						$style['layout'] = 'stripes';
						break;
					case 'is-style-solid-color':
						$style['layout'] = 'solid-color';
						break;
					case 'is-style-wide':
						$style['layout'] = 'wide';
						break;
					case 'is-style-dots':
						$style['layout'] = 'dots';
						break;
					case 'is-style-rounded':
						$style['border']['radius'] = '9999px';
						break;
					case 'is-style-plain':
						$style['layout'] = 'plain';
						break;
				}
			}

			if ( ! empty( $block_data['attrs']['textAlign'] ) ) {
				$style['textAlign'] = $block_data['attrs']['textAlign'];
			}

			if ( ! empty( $block_data['attrs']['align'] ) ) {
				$style['align'] = $block_data['attrs']['align'];
			}

			if (
				! empty( $block_data['attrs']['layout']['justifyContent'] ) &&
				in_array(
					$block_data['attrs']['layout']['justifyContent'],
					array(
						'left',
						'right',
						'center',
					),
					true
				)
			) {
				$style['align'] = $block_data['attrs']['layout']['justifyContent'];
			}

			if (
				! empty( $block_data['attrs']['contentJustification'] ) &&
				in_array(
					$block_data['attrs']['contentJustification'],
					array(
						'left',
						'right',
						'center',
					),
					true
				)
			) {
				$style['align'] = $block_data['attrs']['contentJustification'];
			}

			if ( ! empty( $block_data['attrs']['layout']['type'] ) ) {
				$style['type'] = $block_data['attrs']['layout']['type'];
			}

			if ( ! empty( $block_data['attrs']['layout']['orientation'] ) ) {
				$style['orientation'] = $block_data['attrs']['layout']['orientation'];
			}

			if ( ! empty( $block_data['attrs']['layout']['flexWrap'] ) ) {
				$style['flexWrap'] = $block_data['attrs']['layout']['flexWrap'];
			}

			if ( ! empty( $block_data['attrs']['level'] ) ) {
				switch ( $block_data['attrs']['level'] ) {
					case '1':
						$style['header'] = 'h1';
						break;
					case '2':
						$style['header'] = 'h2';
						break;
					case '3':
						$style['header'] = 'h3';
						break;
					case '4':
						$style['header'] = 'h4';
						break;
					case '5':
						$style['header'] = 'h5';
						break;
					case '6':
						$style['header'] = 'h6';
						break;
					default:
						$style['header'] = 'h2';
				}
			}
			if ( ! empty( $style['header'] ) ) {
				$block_data['attrs']['fontSize'] = $style['header'];
			}

			if ( ! empty( $block_data['attrs']['fontSize'] ) ) {
				switch ( $block_data['attrs']['fontSize'] ) {
					case 'h1':
					case 'large':
						$style['fontSize'] = '36px';
						break;
					case 'h2':
						$style['fontSize'] = '27px';
						break;
					case 'h3':
						$style['fontSize'] = '21px';
						break;
					case 'h4':
						$style['fontSize'] = '18px';
						break;
					case 'h5':
						$style['fontSize'] = '14px';
						break;
					case 'h6':
						$style['fontSize'] = '12px';
						break;
					case 'small':
						$style['fontSize'] = '13px';
						break;
					case 'normal':
						$style['fontSize'] = '16px';
						break;
					case 'medium':
						$style['fontSize'] = '20px';
						break;
					case 'x-large':
						$style['fontSize'] = '42px';
						break;
					case 'high':
					case 'huge':
						$style['fontSize'] = '48px';
						break;
					default:
						$style['fontSize'] = '16px';
				}
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['fontSize'] ) ) {
				$style['fontSize'] = $block_data['attrs']['style']['typography']['fontSize'];
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['fontStyle'] ) ) {
				$style['fontStyle'] = $block_data['attrs']['style']['typography']['fontStyle'];
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['fontWeight'] ) ) {
				$style['fontWeight'] = $block_data['attrs']['style']['typography']['fontWeight'];
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['textTransform'] ) ) {
				$style['textTransform'] = $block_data['attrs']['style']['typography']['textTransform'];
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['textDecoration'] ) ) {
				$style['textDecoration'] = $block_data['attrs']['style']['typography']['textDecoration'];
			}

			if ( ! empty( $block_data['attrs']['style']['typography']['letterSpacing'] ) ) {
				$style['letterSpacing'] = $block_data['attrs']['style']['typography']['letterSpacing'];
			}

			if ( ! empty( $block_data['attrs']['customFontSize'] ) ) {
				$style['fontSize'] = $block_data['attrs']['customFontSize'];
			}

			$color_name_array = self::gutenberg_color_array();

			if ( ! empty( $block_data['attrs']['textColor'] ) ) {
				$style['color'] = array_key_exists( $block_data['attrs']['textColor'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['textColor'] ] : $block_data['attrs']['textColor'];
			}
			if ( ! empty( $block_data['attrs']['style']['color']['text'] ) ) {
				$style['color'] = $block_data['attrs']['style']['color']['text'];
			}

			if ( ! empty( $block_data['attrs']['customBackgroundColor'] ) ) {
				$style['backgroundColor'] = array_key_exists( $block_data['attrs']['customBackgroundColor'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['customBackgroundColor'] ] : $block_data['attrs']['customBackgroundColor'];
			}
			if ( ! empty( $block_data['attrs']['backgroundColor'] ) ) {
				$style['backgroundColor'] = array_key_exists( $block_data['attrs']['backgroundColor'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['backgroundColor'] ] : $block_data['attrs']['backgroundColor'];
			}
			if ( ! empty( $block_data['attrs']['style']['color']['background'] ) ) {
				$style['backgroundColor'] = $block_data['attrs']['style']['color']['background'];
			}

			if ( ! empty( $block_data['attrs']['gradient'] ) ) {
				$style['gradient'] = array_key_exists( $block_data['attrs']['gradient'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['gradient'] ] : $block_data['attrs']['gradient'];
			}
			if ( ! empty( $block_data['attrs']['style']['color']['gradient'] ) ) {
				$style['gradient'] = $block_data['attrs']['style']['color']['gradient'];
			}

			if ( ! empty( $block_data['attrs']['customMainColor'] ) ) {
				$style['customMainColor'] = $block_data['attrs']['customMainColor'];
			}

			if ( ! empty( $block_data['attrs']['overlayColor'] ) ) {
				$style['overlayColor'] = array_key_exists( $block_data['attrs']['overlayColor'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['overlayColor'] ] : $block_data['attrs']['overlayColor'];
			}

			if ( ! empty( $block_data['attrs']['borderColor'] ) ) {
				$style['borderColor'] = array_key_exists( $block_data['attrs']['borderColor'], $color_name_array ) ? $color_name_array[ $block_data['attrs']['borderColor'] ] : $block_data['attrs']['borderColor'];
			}

			if ( ! empty( $block_data['attrs']['customGradient'] ) ) {
				$style['customGradient'] = $block_data['attrs']['customGradient'];
			}

			if ( ! empty( $block_data['attrs']['customOverlayColor'] ) ) {
				$style['customOverlayColor'] = $block_data['attrs']['customOverlayColor'];
			}

			if ( ! empty( $block_data['attrs']['style'] ) && ! empty( $block_data['attrs']['style']['color'] ) && ! empty( $block_data['attrs']['style']['color']['duotone'] ) ) {
				$style['color']['duotone']['shadows']    = ! empty( $block_data['attrs']['style']['color']['duotone'][0] ) ? $block_data['attrs']['style']['color']['duotone'][0] : '';
				$style['color']['duotone']['highlights'] = ! empty( $block_data['attrs']['style']['color']['duotone'][1] ) ? $block_data['attrs']['style']['color']['duotone'][1] : '';
			}

			if ( ! empty( $block_data['attrs']['style']['border'] ) && is_array( $block_data['attrs']['style']['border']['radius'] ) ) {
				$radius                    = $block_data['attrs']['style']['border']['radius'];
				$style['border']['radius'] = array(
					'topLeft'     => $radius['topLeft'],
					'topRight'    => $radius['topRight'],
					'bottomLeft'  => $radius['bottomLeft'],
					'bottomRight' => $radius['bottomRight'],
				);
			}

			if ( ! empty( $block_data['attrs']['style']['border'] ) && ! is_array( $block_data['attrs']['style']['border']['radius'] ) ) {
				$style['border']['radius'] = $block_data['attrs']['style']['border']['radius'];
			}

			if ( ! empty( $block_data['attrs'] ) && ! empty( $block_data['attrs']['style'] ) && ! empty( $block_data['attrs']['style']['border'] ) && isset( $block_data['attrs']['style']['border']['width'] ) && ! is_array( $block_data['attrs']['style']['border']['width'] ) ) {
				$style['border']['width'] = $block_data['attrs']['style']['border']['width'];
			}

			if ( ! empty( $block_data['attrs'] ) && ! empty( $block_data['attrs']['style'] ) && ! empty( $block_data['attrs']['style']['border'] ) && isset( $block_data['attrs']['style']['border']['style'] ) && ! is_array( $block_data['attrs']['style']['border']['style'] ) ) {
				$style['border']['style'] = $block_data['attrs']['style']['border']['style'];
			}

			if ( ! empty( $block_data['attrs']['width'] ) ) {
				$style['width'] = $block_data['attrs']['width'];
			}

			if ( ! empty( $block_data['attrs']['height'] ) ) {
				$style['height'] = $block_data['attrs']['height'];
			}

			if ( ! empty( $block_data['attrs']['sizeSlug'] ) ) {
				$style['sizeSlug'] = $block_data['attrs']['sizeSlug'];
			}

			if ( ! empty( $block_data['attrs']['linkTarget'] ) ) {
				$style['linkTarget'] = $block_data['attrs']['linkTarget'];
			}

			if ( ! empty( $block_data['attrs']['linkTo'] ) ) {
				$style['linkTo'] = $block_data['attrs']['linkTo'];
			}

			if ( ! empty( $block_data['attrs']['dropCap'] ) ) {
				$style['dropCap'] = $block_data['attrs']['dropCap'];
			}

			if ( 'core/table' === $block_data['blockName'] ) {
				$style['hasFixedLayout'] = isset( $block_data['attrs']['hasFixedLayout'] );
			}

			if ( ! empty( $block_data['attrs']['mediaPosition'] ) ) {
				$style['mediaPosition'] = $block_data['attrs']['mediaPosition'];
			}

			if ( ! empty( $block_data['attrs']['imageFill'] ) ) {
				$style['imageFill'] = $block_data['attrs']['imageFill'];
			}

			if ( ! empty( $block_data['attrs']['mediaType'] ) ) {
				$style['mediaType'] = $block_data['attrs']['mediaType'];
			}
			if ( ! empty( $block_data['attrs']['mediaWidth'] ) ) {
				$style['mediaWidth'] = $block_data['attrs']['mediaWidth'];
			}
			if ( ! empty( $block_data['attrs']['mediaLink'] ) ) {
				$style['mediaLink'] = $block_data['attrs']['mediaLink'];
			}

			if ( ! empty( $block_data['attrs']['verticalAlignment'] ) ) {
				$style['verticalAlignment'] = $block_data['attrs']['verticalAlignment'];
			}

			if ( ! empty( $block_data['attrs']['focalPoint'] ) ) {
				$style['focalPoint']['x'] = isset( $block_data['attrs']['focalPoint']['x'] ) ? ( $block_data['attrs']['focalPoint']['x'] * 100 ) . '%' : '100%';
				$style['focalPoint']['y'] = isset( $block_data['attrs']['focalPoint']['y'] ) ? ( $block_data['attrs']['focalPoint']['y'] * 100 ) . '%' : '100';
			}

			if ( ! empty( $block_data['attrs']['isRepeated'] ) ) {
				$style['isRepeated'] = $block_data['attrs']['isRepeated'];
			}

			if ( isset( $block_data['attrs']['dimRatio'] ) ) {
				$style['dimRatio'] = $block_data['attrs']['dimRatio'];
			}

			if ( ! empty( $block_data['attrs']['hasParallax'] ) ) {
				$style['hasParallax'] = $block_data['attrs']['hasParallax'];
			}

			if ( ! empty( $block_data['attrs']['minHeight'] ) ) {
				$style['minHeight'] = $block_data['attrs']['minHeight'];
			}

			if ( ! empty( $block_data['attrs']['contentAlign'] ) ) {
				$style['contentAlign'] = $block_data['attrs']['contentAlign'];
			}

			if ( ! empty( $block_data['attrs']['borderRadius'] ) ) {
				$style['borderRadius'] = $block_data['attrs']['borderRadius'];
			}

			if ( ! empty( $block_data['attrs']['columns'] ) ) {
				$style['columns'] = absint( $block_data['attrs']['columns'] );
			}

			if ( ! empty( $block_data['attrs']['mediaId'] ) ) {
				$attachment_image_url = wp_get_attachment_image_url( $block_data['attrs']['mediaId'] );

				if ( $attachment_image_url ) {
					$style = self::bbapp_get_image_size( $attachment_image_url, $block_data, $style );
				}
			}

			if ( ! empty( $block_data['attrs']['headerTextColor'] ) ) {
				$style['headerTextColor'] = $block_data['attrs']['headerTextColor'];
			}
		}

		return $style;
	}

	/**
	 * Function prepare the block attribute.
	 *
	 * @param array $block_data Gutenberg data attributes.
	 * @param array $attribute  Attribute array.
	 *
	 * @since 1.6.7
	 * @return array|mixed
	 */
	public static function get_block_attributes( $block_data, $attribute = array() ) {

		if ( ! empty( $block_data['attrs'] ) ) {

			if ( ! empty( $block_data['attrs']['linkDestination'] ) ) {
				$attribute['linkDestination'] = $block_data['attrs']['linkDestination'];
			}

			if ( ! empty( $block_data['attrs']['className'] ) && 'is-style-large' === $block_data['attrs']['className'] ) {
				$attribute['fontSize'] = 'large';
			}

			if ( ! empty( $block_data['attrs']['url'] ) ) {
				/**
				 * The`as3cf_filter_post_local_to_s3`  filters added for filtering S3 URLs in custom content.
				 *
				 * @param string $url Attribute URL.
				 */
				$attribute['url'] = apply_filters( 'as3cf_filter_post_local_to_s3', $block_data['attrs']['url'] );
			}

			if ( ! empty( $block_data['attrs']['mediaSizeSlug'] ) ) {
				$attribute['mediaSizeSlug'] = $block_data['attrs']['mediaSizeSlug'];
			}

			if ( ! empty( $block_data['attrs']['mediaId'] ) ) {
				$attribute['url'] = wp_get_attachment_image_url( $block_data['attrs']['mediaId'], $attribute['mediaSizeSlug'] );
			}

			if ( ! empty( $block_data['attrs']['providerNameSlug'] ) ) {
				$attribute['provider'] = $block_data['attrs']['providerNameSlug'];
			}

			if ( ! empty( $block_data['attrs']['type'] ) ) {
				$attribute['type'] = $block_data['attrs']['type'];
			}

			if ( ! empty( $block_data['attrs']['align'] ) ) {
				$attribute['align'] = $block_data['attrs']['align'];
			}

			if ( ! empty( $block_data['attrs']['title'] ) ) {
				$attribute['title'] = $block_data['attrs']['title'];
			}

			if ( ! empty( $block_data['attrs']['icon_url'] ) ) {
				$attribute['icon_url'] = $block_data['attrs']['icon_url'];
			}

			if ( ! empty( $block_data['attrs']['tint_color'] ) ) {
				$attribute['tint_color'] = $block_data['attrs']['tint_color'];
			}

			if ( ! empty( $block_data['attrs']['noTeaser'] ) ) {
				$attribute['noTeaser'] = $block_data['attrs']['noTeaser'];
			}

			if ( ! empty( $block_data['attrs']['customText'] ) ) {
				$attribute['customText'] = $block_data['attrs']['customText'];
			}

			if ( isset( $block_data['attrs']['showDownloadButton'] ) ) {
				$attribute['show_button'] = $block_data['attrs']['showDownloadButton'];
			}

			if ( ! empty( $block_data['attrs']['href'] ) ) {
				$attribute['file_url'] = $block_data['attrs']['href'];
			}
		}

		return $attribute;
	}

	/**
	 * Function prepare the block style for old customer.
	 *
	 * @param array $block_data Gutenberg data attributes.
	 * @param array $style      Style array.
	 *
	 * @since 1.6.7
	 * @return array|mixed
	 */
	public static function get_backward_compatibility_block_style( $block_data, $style ) {
		if ( ! empty( $block_data ) ) {
			if ( ! empty( $block_data['attrs']['align'] ) ) {
				$style['alignSelf'] = $block_data['attrs']['align'];
			}
			$color_name_array = self::gutenberg_color_array();
			if ( ! empty( $block_data['attrs']['textColor'] ) ) {
				$style['textColor'] = $color_name_array[ $block_data['attrs']['textColor'] ];
			}
			if ( ! empty( $block_data['attrs']['style']['color']['text'] ) ) {
				$style['textColor'] = $block_data['attrs']['style']['color']['text'];
			}

			// Add radius if exists.
			if ( isset( $block_data['attrs']['style']['border']['radius'] ) ) {
				$style['radius'] = $block_data['attrs']['style']['border']['radius'];
			}
		}

		return $style;
	}

	/**
	 * Old cover block support added.
	 *
	 * @param array $app_page_data Block data for render in app.
	 * @param array $block_data    Gutenberg block data.
	 *
	 * @return mixed
	 */
	public static function get_backward_compatibility_cover_block( $app_page_data, $block_data ) {
		if ( isset( $block_data['innerBlocks'] ) && ! empty( $block_data['innerBlocks'] ) ) { // core/cover block.
			$content = wp_strip_all_tags( $block_data['innerBlocks'][0]['innerHTML'] );
		} else { // core/cover block.
			$content = wp_strip_all_tags( $block_data['innerHTML'] );
		}

		$app_page_data['data']['coverText'] = $content;

		if ( isset( $block_data['innerBlocks'][0] ) && ! empty( $content ) ) {
			$render                         = render_block( $block_data['innerBlocks'][0] );
			$render                         = trim( $render );
			$app_page_data['coverTextData'] = self::parse_paragraph_block( $block_data['innerBlocks'][0], $render );
		}

		return $app_page_data;
	}

	/**
	 * Get cached image sizes.
	 *
	 * @param string       $url        Url.
	 * @param array|string $block_data Block data.
	 * @param array        $style      Style.
	 *
	 * @since 1.7.2
	 * @return array|mixed
	 */
	public static function bbapp_get_image_size( $url, $block_data, $style = array() ) {
		// Get the image width and height fron block data attribute if available.

		$style['width']  = ( ! empty( $block_data['attrs']['width'] ) ) ? $block_data['attrs']['width'] : 0;
		$style['height'] = ( ! empty( $block_data['attrs']['height'] ) ) ? $block_data['attrs']['height'] : 0;

		// Get the image width and height from the dom element available.
		if ( empty( $style['width'] ) || empty( $style['height'] ) ) {
			$dom = new \DOMDocument();

			if ( ! empty( $block_data['innerHTML'] ) ) {
				$dom->loadXML( $block_data['innerHTML'] ); // Load the XML.
			} else {
				$dom->loadHTML( $block_data ); // Load the HTML.
			}

			$elements = $dom->getElementsByTagName( 'img' );// Get all the img elements.
			$width    = ( ! empty( $elements[0] ) && ! empty( $elements[0]->getAttribute( 'width' ) ) ) ? $elements[0]->getAttribute( 'width' ) : 0;
			$height   = ( ! empty( $elements[0] ) && ! empty( $elements[0]->getAttribute( 'height' ) ) ) ? $elements[0]->getAttribute( 'height' ) : 0;

			if ( ! empty( $width ) && ! empty( $height ) ) {
				$style['width']  = $width;
				$style['height'] = $height;
			}
		}

		$size_slug = '';

		// Get image width and height form attachment id.
		if ( empty( $style['width'] ) || empty( $style['height'] ) ) {
			if ( ! empty( $block_data['attrs']['id'] ) ) {
				$attachment_id = $block_data['attrs']['id'];
				$size_slug     = ! empty( $block_data['attrs']['sizeSlug'] ) ? $block_data['attrs']['sizeSlug'] : '';
			} elseif ( ! empty( $block_data['attrs']['mediaId'] ) ) { // media-text block.
				$attachment_id = $block_data['attrs']['mediaId'];
				$size_slug     = ! empty( $block_data['attrs']['mediaSizeSlug'] ) ? $block_data['attrs']['mediaSizeSlug'] : '';
			} elseif ( ! empty( $block_data['innerHTML'] ) ) {
				$dom = new \DOMDocument();
				$dom->loadXML( $block_data['innerHTML'] ); // Load the XML.
				$elements      = $dom->getElementsByTagName( 'img' );// Get all the img elements.
				$img_class     = ( ! empty( $elements[0] ) && ! empty( $elements[0]->getAttribute( 'class' ) ) ) ? explode( 'wp-image-', $elements[0]->getAttribute( 'class' ) ) : '';
				$attachment_id = 0;

				if ( ! empty( $img_class ) && ! empty( $img_class[1] ) ) {
					preg_match( '!\D+!', $img_class[1], $match_string );

					$size_string   = ! empty( $match_string[0] ) ? explode( 'size-', $match_string[0] ) : '';
					$size_slug     = ( ! empty( $size_string ) && ! empty( $size_string[1] ) ) ? (string) $size_string[1] : '';
					$attachment_id = (int) $img_class[1];
				}
			}

			if ( ! empty( $attachment_id ) ) {
				$wp_attachment_metadata = maybe_unserialize( get_post_meta( $attachment_id, '_wp_attachment_metadata', true ) );
				$style['width']         = ( ! empty( $wp_attachment_metadata['width'] ) ) ? $wp_attachment_metadata['width'] : 0;
				$style['height']        = ( ! empty( $wp_attachment_metadata['height'] ) ) ? $wp_attachment_metadata['height'] : 0;

				if ( isset( $wp_attachment_metadata['sizes'][ $size_slug ] ) ) {
					$style['width']  = ! empty( $wp_attachment_metadata['sizes'][ $size_slug ]['width'] ) ? $wp_attachment_metadata['sizes'][ $size_slug ]['width'] : $style['width'];
					$style['height'] = ! empty( $wp_attachment_metadata['sizes'][ $size_slug ]['height'] ) ? $wp_attachment_metadata['sizes'][ $size_slug ]['height'] : $style['height'];
				}
			}
		}

		// If above methods don't get the image width and height then we need to use `getimagesize` function and cache the image width and height.
		if ( empty( $style['width'] ) || empty( $style['height'] ) ) {
			global $post;

			$bbapp_image_cache = get_post_meta( $post->ID, '_bbapp_image_cache', true );
			$bbapp_image_cache = ! empty( $bbapp_image_cache ) ? $bbapp_image_cache : array();

			if ( ! empty( $bbapp_image_cache[ md5( $url ) ] ) ) {
				$sizes = $bbapp_image_cache[ md5( $url ) ];
			} else {
				$imagesize                        = getimagesize( $url );
				$sizes                            = array(
					'width'  => $imagesize[0],
					'height' => $imagesize[1],
				);
				$bbapp_image_cache[ md5( $url ) ] = $sizes;
				update_post_meta( $post->ID, '_bbapp_image_cache', $bbapp_image_cache );
			}
			$style['width']  = ! empty( $sizes['width'] ) ? $sizes['width'] : 0;
			$style['height'] = ! empty( $sizes['height'] ) ? $sizes['height'] : 0;
		}

		return $style;
	}
}
