<?php
/**
 * Holds app pages gutenberg blocks related functionality.
 *
 * @package BuddyBossApp\NativeAppPage
 */

namespace BuddyBossApp\NativeAppPage;

// Exit if accessed directly.
use WP_Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Contain App Page Gutenberg related things.
 */
class Gutenberg {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Gutenberg constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Gutenberg
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		$this->hooks();
	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
		add_filter( 'allowed_block_types_all', array( $this, 'bbapp_allowed_block_types' ), 9999, 2 );
		add_filter( 'bbapp_ld_rest_prepare_course', array( $this, 'native_gutenberg_api_response' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_lesson', array( $this, 'native_gutenberg_api_response' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_topic', array( $this, 'native_gutenberg_api_response' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_quiz', array( $this, 'native_gutenberg_api_response' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_quiz_question', array( $this, 'native_gutenberg_api_response' ), 10, 3 );

		/**
		 * This filter is used to Set Question data like question_id to Question Object.
		 *
		 * Here we set Data like question_id to the Question Object so that
		 * we can use that in another method called `native_gutenberg_api_response` to get Native content.
		 */
		add_filter( 'bbapp_ld_prepare_quiz_question_answer', array( $this, 'prepare_question_data' ), 10, 2 );

		// Blog post and BBPress Support.
		add_action( 'rest_api_init', array( $this, 'register_content_native_field' ), 99 );

		// Add BuddyBossApp State if Post use App Editor.
		add_filter( 'display_post_states', array( $this, 'add_bbapp_post_state' ), 10, 2 );

		// Add/Remove App Editor edit link in dashboard.
		add_filter( 'post_row_actions', array( $this, 'filter_post_row_actions' ), 11, 2 );

		add_filter( 'bbapp_custom_block_data', array( $this, 'old_wp_bbapp_custom_block_data' ), 10, 3 );
		add_filter( 'bbapp_app_page_app_blocks', array( $this, 'old_wp_bbapp_app_page_app_blocks' ), 10, 3 );
		add_action( 'post_updated', array( $this, 'clear_image_cache' ), 10, 3 );
	}

	/**
	 * Fires after post updated.
	 *
	 * @param int     $post_id Post id.
	 * @param WP_Post $post_after Post object.
	 * @param WP_Post $post_before Post object.
	 *
	 * @since 1.7.2
	 * @return void
	 */
	public function clear_image_cache( $post_id, $post_after, $post_before ) {
		delete_post_meta( $post_id, '_bbapp_image_cache' );
	}

	/**
	 * Return the list of app blocks.
	 *
	 * @param WP_Post $post Wp post.
	 *
	 * @return string[]
	 */
	public function app_blocks( $post ) {
		$allowed_block_types = array(
			'core/paragraph',
			'core/heading',
			'core/quote',
			'core/list',
			'core/list-item',
			'core/table',
			'core/gallery',
			'core/audio',
			'core/video',
			'core/image',
			'core/cover',
			'core/html',
			'core/media-text',
			'core/buttons',
			'core/button',
			'core/columns',
			'core/column',
			'core-embed/vimeo',
			'core-embed/youtube',
			'core/code',
			'core/pullquote',
			'core/preformatted',
			'core/verse',
			'core/file',
			'core/group',
			'core/separator',
			'core/spacer',
			'core/embed',
		);
		$default_app_page    = bbapp_get_default_app_page();
		$post_id             = ( $post instanceof \WP_Block_Editor_Context ) ? $post->post->ID : $post->ID;

		// only show those block if app page. not show bbapp widget on other cpt.
		if ( $post_id !== $default_app_page ) {
			$allowed_block_types[] = 'bbapp/my-progress';
			$allowed_block_types[] = 'bbapp/posts';
			$allowed_block_types[] = 'bbapp/directory';

			if ( function_exists( 'learndash_get_option' ) ) {
				$allowed_block_types[] = 'bbapp/courses';
			}

			if ( bbapp_is_memberpress_lms_enabled() ) {
				$allowed_block_types[] = 'bbapp/mp-courses';
			}

			if ( bbapp_is_tutor_lms_plugins_active() ) {
				$allowed_block_types[] = 'bbapp/tutor-courses';
			}

			if ( class_exists( 'bbPress' ) ) {
				$allowed_block_types[] = 'bbapp/topics';
				$allowed_block_types[] = 'bbapp/forums';
			}

			if ( function_exists( 'bp_is_active' ) ) {
				if ( bp_is_active( 'groups' ) ) {
					$allowed_block_types[] = 'bbapp/groups';
				}

				if ( bp_is_active( 'members' ) ) {
					$allowed_block_types[] = 'bbapp/members';
				}

				if ( bp_is_active( 'activity' ) ) {
					$allowed_block_types[] = 'bbapp/activity';
				}
			}

			/** Members widget added that's why remove the user widget. */
			$allowed_block_types[] = 'bbapp/notifications';
			$allowed_block_types[] = 'bbapp/qlinks';
		}

		if ( class_exists( 'H5P_Plugin' ) ) {
			$allowed_block_types[] = 'bbapp/h5p';
		}

		/**
		 * Filters app page blocks.
		 *
		 * @param array $allowed_block_types Allowed block types.
		 * @param int   $post_id             Post id.
		 * @param int   $default_app_page    Default app page id.
		 */
		return apply_filters( 'bbapp_app_page_app_blocks', $allowed_block_types, $post_id, $default_app_page );
	}

	/**
	 * Allowed block types.
	 *
	 * @param array   $allowed_block_types Allowed block types.
	 * @param WP_Post $post                WP post.
	 *
	 * @return array
	 */
	public function bbapp_allowed_block_types( $allowed_block_types, $post ) {
		global $pagenow;

		if ( empty( $post ) || 'widgets.php' === $pagenow ) {
			return $allowed_block_types;
		}

		$post_type = ( ( $post instanceof \WP_Block_Editor_Context ) && ! empty( $post->post->post_type ) ) ? $post->post->post_type : ( ! empty( $post->post_type ) ? $post->post_type : '' );

		// Allowed all block types except `app_page` cpt.
		if ( 'app_page' !== $post_type ) {
			return $allowed_block_types;
		}

		$allowed_block_types = $this->app_blocks( $post );

		/**
		 * App Boss allowed block types.
		 *
		 * @param array $allowed_block_types Allowed block types.
		 */
		return apply_filters( 'bbapp_allowed_block_types', $allowed_block_types );
	}

	/**
	 * Register bbpress response fields.
	 *
	 * @since 1.0,0
	 * @return void
	 */
	public function register_content_native_field() {
		$object_types = array(
			'bp_forums',
			'topics',
			'post', // Blog post support.
		);

		foreach ( $object_types as $object_type ) {
			register_rest_field(
				$object_type,
				'content_native',
				array(
					'get_callback'    => array( $this, 'get_content_native' ),
					'update_callback' => null,
					'schema'          => null,
				)
			);
		}
	}

	/**
	 * Parse Gutenberg content for Forums an Topic.
	 *
	 * @param object          $object     Native object.
	 * @param string          $field_name Field name.
	 * @param WP_REST_Request $request    Rest request.
	 *
	 * @return array
	 */
	public function get_content_native( $object, $field_name, $request ) {
		$post_id = isset( $object['id'] ) ? absint( $object['id'] ) : 0;
		// Set Native App page Content.
		$app_page_editor_content = get_post_meta( $post_id, '_app_page_editor_content', true );

		if ( empty( $app_page_editor_content ) ) {
			$native_post = get_post( $post_id );

			// Editor content to native content support added.
			if ( $this->only_has_supported_blocks( $native_post->post_content, $native_post ) ) {
				$app_page_editor_content = $native_post->post_content;
			}
		}

		return ! empty( $app_page_editor_content ) ? bbapp_parse_gutenberg_content( $app_page_editor_content ) : array();
	}

	/**
	 * Set Question data needed like question_id.
	 *
	 * Here we set Data like question_id to the Question Object so that
	 * we can use that in another method called `native_gutenberg_api_response` to get Native content.
	 *
	 * @param array  $question Question data.
	 * @param object $que_post Question object.
	 *
	 * @return mixed
	 */
	public function prepare_question_data( $question, $que_post ) {
		$question_post_id = null;

		if ( function_exists( 'learndash_get_question_post_by_pro_id' ) ) {
			$pro_id           = $que_post->getId();
			$question_post_id = learndash_get_question_post_by_pro_id( $pro_id );
		}

		if ( is_array( $question ) ) { // support new learndash api.
			$question['is_hint_enable'] = $que_post->isTipEnabled();
			$question['question_id']    = isset( $question_post_id ) ? $question_post_id : $que_post->getQuestionPostId();
		} else { // support old learndash learner api.
			$question->is_hint_enable = $que_post->isTipEnabled();
			$question->question_id    = isset( $question_post_id ) ? $question_post_id : $que_post->getQuestionPostId();
		}

		return $question;
	}

	/**
	 * Check weather content has only supported blocks if it founds any unsupported block it will return false.
	 *
	 * @param string  $content Block content.
	 * @param WP_Post $post    WP post.
	 *
	 * @return bool
	 */
	public function only_has_supported_blocks( $content, $post ) {
		$allowd_blocks = $this->app_blocks( $post );
		$return        = true;
		$parsed_blocks = parse_blocks( $content );

		foreach ( $parsed_blocks as $key => $block_data ) {
			if ( isset( $block_data['blockName'] ) && ! in_array( $block_data['blockName'], $allowd_blocks, true ) ) {
				$return = false;
				break;
			}
		}

		return $return;
	}

	/**
	 * Add Native Gutenberg Content into the Response.
	 *
	 * @param WP_REST_Response $response The response object.
	 * @param WP_Post          $post     Post object.
	 * @param WP_REST_Request  $request  Request object.
	 *
	 * @return WP_REST_Response $response The response object.
	 */
	public function native_gutenberg_api_response( $response, $post, $request ) {
		$post_id = ! empty( $post->question_id ) ? $post->question_id : $post->ID;

		// Set Native App page Content.
		$app_page_editor_content = get_post_meta( $post_id, '_app_page_editor_content', true );
		$app_page_editor_content = ! empty( $app_page_editor_content ) ? $app_page_editor_content : '';

		/**
		 * Create excerpt from bbapp editor content
		 * We will checking if any paragraph added in bbapp editor. Using first paragraph text as excerpt
		 */
		if ( isset( $response->data['excerpt'] ) ) {
			$response->data['excerpt_native'] = array();
			$app_page_editor_excerpt          = bbapp_parse_gutenberg_excerpt( $app_page_editor_content );

			if ( ! empty( $app_page_editor_excerpt ) ) {
				$response->data['excerpt_native']['rendered'] = $app_page_editor_excerpt;
			} else {
				// Editor content in matched bbapp supported block to add native content.
				if ( $this->only_has_supported_blocks( $post->post_content, $post ) ) {
					$response->data['excerpt_native']['rendered'] = bbapp_parse_gutenberg_excerpt( $post->post_content );
				}
			}
		}

		$post_type = get_post_type( $post_id );

		/**
		 * Just don't add content_native param on lists endpoints.
		 * We can know this by checking the callback functions.
		 * Mostly endpoint uses get_items for list contents. so this is the trick.
		 *
		 * Currently we filter data by context only for course so below condition added only for course.
		 */
		if ( 'sfwd-courses' === $post_type && isset( $request->get_attributes()['callback'][1] ) && 'get_items' === $request->get_attributes()['callback'][1] ) {
			return $response;
		}

		if( defined( 'UNCANNY_REPORTING_VERSION' ) && false !== strpos( $post->post_content, 'tincanny' ) ) {
			return $response;
		}

		$response_key = 'content_native';

		if ( 'sfwd-question' === $post_type ) {
			$response_key = 'title_native';

			// Set Native App page Content.
			$use_app_page_editor_hint = get_post_meta( $post_id, 'use_app_page_editor_tipMsg', true );
			$use_app_page_editor_hint = empty( $use_app_page_editor_hint ) ? 'yes' : $use_app_page_editor_hint;
			$app_page_editor_hint     = get_post_meta( $post_id, '_app_page_editor_tipMsg', true );
			$app_page_editor_hint     = ! empty( $app_page_editor_hint ) ? $app_page_editor_hint : '';
			$is_hint_enable           = $post->is_hint_enable;

			if ( 'yes' === $use_app_page_editor_hint && ! empty( $app_page_editor_hint ) && ! empty( $is_hint_enable ) ) {
				$response->data['hint_native'] = '<p>' . $app_page_editor_hint . '</p>';
			} else {
				$response->data['hint_native'] = '';
			}
		}

		if ( ! empty( $app_page_editor_content ) ) {
			$response->data[ $response_key ] = bbapp_parse_gutenberg_content( $app_page_editor_content );
		} elseif ( isset( $post->post_content ) ) {
			// Editor content in matched bbapp supported block to add native content.
			if ( $this->only_has_supported_blocks( $post->post_content, $post ) ) {
				$response->data[ $response_key ] = bbapp_parse_gutenberg_content( $post->post_content );
			} else {
				$response->data[ $response_key ] = array();
			}
		}

		return $response;
	}

	/**
	 * Check whether post use App Editor or not.
	 *
	 * @param int $post_id Post id.
	 *
	 * @return bool
	 */
	public static function is_use_bbapp_editor( $post_id ) {
		$app_page_editor_content = get_post_meta( $post_id, '_app_page_editor_content', true );
		$app_page_editor_content = ! empty( $app_page_editor_content ) ? $app_page_editor_content : '';

		if ( empty( $app_page_editor_content ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Add App Editor state.
	 *
	 * Adds a new "App Editor" post state to the post table.
	 *
	 * Fired by `display_post_states` filter.
	 *
	 * @param array    $post_states An array of post display states.
	 * @param \WP_Post $post        The current post object.
	 *
	 * @since  3.0.0
	 * @access public
	 *
	 * @return array A filtered array of post display states.
	 */
	public function add_bbapp_post_state( $post_states, $post ) {
		if ( self::is_use_bbapp_editor( $post->ID ) ) {
			$post_states['bbapp_editor'] = __( 'App Editor', 'buddyboss-app' );
		}

		return $post_states;
	}

	/**
	 * Add/Remove App Editor edit link in dashboard.
	 *
	 * Add or remove an edit link to the post action links on the post list table.
	 *
	 * Fired by `post_row_actions` filters.
	 *
	 * @access public
	 *
	 * @param array   $actions An array of row action links.
	 * @param WP_Post $post    The current post object.
	 *
	 * @return array An updated array of row action links.
	 */
	public function filter_post_row_actions( $actions, $post ) {
		$url = add_query_arg(
			array(
				'post'         => $post->ID,
				'action'       => 'edit',
				'bbapp_editor' => true,
			),
			admin_url( 'post.php' )
		);

		if ( self::is_use_bbapp_editor( $post->ID ) ) {
			$actions['edit_with_bbapp'] = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url ), __( 'Use App Editor', 'buddyboss-app' ) );
		}

		return $actions;
	}

	/**
	 * WordPress old version block support with new code.
	 *
	 * @param array $app_page_data App page data.
	 * @param array $block_data    Block data.
	 * @param array $render        Render data.
	 *
	 * @return array|mixed
	 */
	public function old_wp_bbapp_custom_block_data( $app_page_data, $block_data, $render ) {

		switch ( $block_data['blockName'] ) {
			case 'core-embed/twitter':
			case 'core-embed/youtube':
			case 'core-embed/wordpress':
			case 'core-embed/soundcloud':
			case 'core-embed/spotify':
			case 'core-embed/flickr':
			case 'core-embed/vimeo':
			case 'core-embed/animoto':
			case 'core-embed/cloudup':
			case 'core-embed/crowdsignal':
			case 'core-embed/dailymotion':
			case 'core-embed/imgur':
			case 'core-embed/issuu':
			case 'core-embed/kickstarter':
			case 'core-embed/meetup-com':
			case 'core-embed/mixcloud':
			case 'core-embed/reddit':
			case 'core-embed/reverbnation':
			case 'core-embed/screencast':
			case 'core-embed/scribd':
			case 'core-embed/slideshare':
			case 'core-embed/smugmug':
			case 'core-embed/speaker-deck':
			case 'core-embed/tiktok':
			case 'core-embed/ted':
			case 'core-embed/tumblr':
			case 'core-embed/videopress':
			case 'core-embed/wordpress-tv':
			case 'core-embed/amazon-kindle':
				$block_data['blockName'] = 'core/embed';
				$app_page_data           = ParseBlocks::parse_embed_block( $block_data, $render );
				break;
		}

		return $app_page_data;
	}

	/**
	 * WordPress 5.5.3 or old version block support.
	 *
	 * @param array $allowed_block_types Allowed block types.
	 * @param int   $post_id             Post id.
	 * @param int   $default_app_page    Default app page.
	 *
	 * @return array|mixed
	 */
	public function old_wp_bbapp_app_page_app_blocks( $allowed_block_types, $post_id, $default_app_page ) {
		global $wp_version;

		if ( version_compare( $wp_version, '5.5.3', '<=' ) ) {
			$allowed_old_block_types = array(
				'core-embed/twitter',
				'core-embed/youtube',
				'core-embed/wordpress',
				'core-embed/soundcloud',
				'core-embed/spotify',
				'core-embed/flickr',
				'core-embed/vimeo',
				'core-embed/animoto',
				'core-embed/cloudup',
				'core-embed/crowdsignal',
				'core-embed/dailymotion',
				'core-embed/imgur',
				'core-embed/issuu',
				'core-embed/kickstarter',
				'core-embed/meetup-com',
				'core-embed/mixcloud',
				'core-embed/reddit',
				'core-embed/reverbnation',
				'core-embed/screencast',
				'core-embed/scribd',
				'core-embed/slideshare',
				'core-embed/smugmug',
				'core-embed/speaker-deck',
				'core-embed/tiktok',
				'core-embed/ted',
				'core-embed/tumblr',
				'core-embed/videopress',
				'core-embed/wordpress-tv',
				'core-embed/amazon-kindle',
			);
			$allowed_block_types     = array_merge( $allowed_block_types, $allowed_old_block_types );
		}

		return $allowed_block_types;
	}
}
