<?php
/**
 * Holds native app page block functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use BuddyBossApp\Admin\SetupAdmin;
use BuddyBossApp\Integrations\H5P\Main;
use BuddyBossApp\NativeAppPage\Deprecated_Content;
use BuddyBossApp\NativeAppPage\DirectoryBlock;
use BuddyBossApp\NativeAppPage\Gutenberg;
use WP_Post;
use WP_Post_Type;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Contain code for App Page Custom Post Type.
 */
class NativeAppPage {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * NativeAppPage constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return NativeAppPage
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 */
	public function load() {
		// Load Gutenberg Class.
		Gutenberg::instance();
		Deprecated_Content::instance();
		DirectoryBlock::instance();
		Main::instance(); // H5P plugin support.
		$this->hooks();

	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'register_app_page_cpt' ), 0 );
		add_action( 'wp_head', array( $this, 'fallback_support' ), 999 );
		add_action( 'template_redirect', array( $this, 'redirect_single_app_page' ) );
		add_action( 'admin_init', array( $this, 'is_activate_gutenberg_editor' ) );
		add_filter( 'rest_prepare_app_page', array( $this, 'modify_app_page_rest_response' ), 10, 3 );
		add_filter( 'post_row_actions', array( $this, 'remove_classic_editor_link' ), 99999, 2 );
		add_action( 'add_meta_boxes', array( $this, 'add_app_page_editor_metaboxes' ) );
		add_action( 'save_post', array( $this, 'save_app_page_gutenberg_editor_checkbox' ), 10, 1 );
		add_action( 'pre_get_posts', array( $this, 'exclude_default_app_page' ), 10 );
		add_filter( 'views_edit-app_page', array( $this, 'bbapp_exclude_default_page_count' ) );

		add_filter(
			'use_block_editor_for_post_type',
			array(
				$this,
				'app_page_force_gutenberg_editor',
			),
			9999999,
			2
		);
		add_filter(
			'use_block_editor_for_post',
			array(
				$this,
				'app_page_force_gutenberg_editor_post',
			),
			9999999,
			2
		);

		add_action( 'admin_head', array( $this, 'app_page_head' ), 10 );
		add_action( 'admin_print_footer_scripts', array( $this, 'app_page_footer' ), 99999 );
		add_action( 'enqueue_block_editor_assets', array( $this, 'widget_footer' ), 999999 );
		add_action( 'enqueue_block_editor_assets', array( $this, 'gutenberg_variables' ), 999999 );
		add_filter( 'block_categories_all', array( $this, 'block_categories' ), 0, 2 );
		add_filter( 'bbapp_app_menus_allowed_post_types', array( $this, 'add_app_pages_in_app_menus' ), 10, 1 );

		// Remove patterns from app pages.
		add_filter( 'block_editor_settings_all', array( $this, 'remove_block_patterns_support' ) );

		// Classic Editor plugin.
		if ( class_exists( 'Classic_Editor' ) && bbapp_is_learndash_enabled() ) {
			add_action( 'init', array( $this, 'classic_editor_init' ) );
		}

		add_filter( 'rest_user_query', array( $this, 'bbapp_search_user_by_name_filter' ), 10, 2 );

		if ( bbapp_is_learndash_enabled() ) {
			add_action( 'wp', array( $this, 'bbapp_filter_unsupported_blocks' ) );
		}

		if ( isset( $_GET['mobile-view-content'] ) ) {
			add_filter( 'wp_link_pages_link', array( $this, 'bbapp_wp_link_pages_link' ), 10, 1 );
		}

		add_filter( 'nav_menu_meta_box_object', array( $this, 'bbapp_remove_app_pages_appearance_menu' ), 99, 1 );
	}

	/**
	 * Function to filter unsupported blocks.
	 *
	 * @since 1.4.5
	 */
	public function bbapp_filter_unsupported_blocks() {
		global $post;

		if ( ! isset( $post ) ) {
			return;
		}

		$blocks           = array_filter( wp_list_pluck( parse_blocks( $post->post_content ), 'blockName' ) );
		$supported_blocks = self::instance()->bbapp_supported_block_list();
		$diff_blocks      = array_diff( $blocks, $supported_blocks );
		$ld_post_types    = array(
			'sfwd-lessons',
			'sfwd-courses',
			'sfwd-topic',
			'sfwd-quiz',
			'sfwd-question',
			'sfwd-certificates',
			'groups',
			'sfwd-assignment',
		);

		if ( in_array( get_post_type( $post ), $ld_post_types, true ) && ! empty( $diff_blocks ) ) {
			add_filter(
				'ld_template_args_learndash_lesson_video',
				array(
					$this,
					'bbapp_ld_template_args_learndash_lesson_video',
				),
				10,
				3
			);
		}
	}

	/**
	 * Init Classic editor
	 */
	public function classic_editor_init() {
		add_filter(
			'ld_template_args_learndash_lesson_video',
			array(
				$this,
				'bbapp_ld_template_args_learndash_lesson_video',
			),
			10,
			3
		);
	}

	/**
	 * Classed enable and hide the video content when mobile view content enable.
	 *
	 * @param array  $args     Arguments.
	 * @param string $filepath File path.
	 * @param bool   $echo     If needs to print.
	 *
	 * @return mixed
	 */
	public function bbapp_ld_template_args_learndash_lesson_video( $args, $filepath, $echo ) {
		if ( ! empty( $_GET['mobile-view-content'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$args['video_content'] = '';
		}

		return $args;
	}

	/**
	 * Prepare the fallback webview preview, by hiding everything other then content.
	 */
	public function fallback_support() {
		global $post;

		if ( isset( $_GET['mobile-view-content'] ) && isset( $post ) && ! empty( $post ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			?>
			<style>
				html {
					margin: 0px !important;
				}

				.bbapp_hide_all {
					display: none !important;
				}

				body {
					display: none;
				}

				/** caused by admin bar */
				body:before {
					display: none !important;
				}

				.sticky-header .site-content {
					padding-top: 0 !important;
				}

				body.single-post .entry-content {
					padding-top: 40px;
				}

				<?php
				$theme = wp_get_theme(); // gets the current theme.

				if ( 'BuddyBoss Theme' === $theme->name || 'BuddyBoss Theme' === $theme->parent_theme ) :
					?>
                .lms-topic-sidebar-wrapper,
                #learndash-course-header,
                .bb-learndash-banner,
                .bb-single-course-sidebar,
                .learndash-wrapper .ld-course-status,
                .learndash-wrapper .ld-progress,
                .learndash-wrapper .learndash-shortcode-wrap,
                .learndash-wrapper .ld-course-status.ld-course-status-enrolled,
                .learndash-wrapper .bb-learndash-content-wrap > div.ld-item-list,
                .bb-about-instructor,
                .learndash-wrapper .learndash_content_wrap .ld-content-actions,
                .learndash-wrapper .ld-tabs .ld-tabs-navigation,
                .comments-area,
                .wpProQuiz_content {
                    display: none !important;
                }

				#learndash-page-content,
				.learndash_content_wrap {
					padding-top: 10px;
				}
				<?php else : ?>
				.learndash-wrapper .ld-course-status,
                .learndash-wrapper .ld-progress,
                .learndash-wrapper .learndash-shortcode-wrap,
				.learndash-wrapper .ld-course-status.ld-course-status-enrolled,
				.learndash_post_sfwd-courses > .learndash-wrapper > div.ld-item-list,
				.learndash-wrapper .ld-tabs .ld-tabs-navigation,
				.learndash-wrapper .ld-content-actions,
				.learndash-wrapper > .ld-lesson-status,
				.learndash-wrapper > .ld-topic-status,
				.learndash-wrapper .ld-breadcrumbs,
                .comments-area,
				.wpProQuiz_content {
					display: none !important;
				}
				<?php endif; ?>
			</style>
			<script>
                jQuery( document ).ready( function() {
                    let $ele = jQuery( '.bbapp_main_content_area' );
                    let $ele_link = jQuery( '.page-links:first' );

                    jQuery( 'body' ).find( '*' ).addClass( 'bbapp_hide_all' );
                    $ele.removeClass( 'bbapp_hide_all' );
                    $ele_link.removeClass( 'bbapp_hide_all' );

                    $ele.find( '.bbapp_hide_all' ).each( function() {
                        jQuery( this ).removeClass( 'bbapp_hide_all' );
                    } );

                    $ele_link.find( '.bbapp_hide_all' ).each( function(index, value) {
                        jQuery( this ).removeClass( 'bbapp_hide_all' );
                    } );

                    if ( 0 !== $ele.parent().length ) {
                        while ( $ele.parent().prop( 'nodeName' ) !== jQuery( 'html' ).prop( 'nodeName' ) ) {
                            jQuery( $ele.parent() ).removeClass( 'bbapp_hide_all' ).addClass( 'bbapp_main_content_parent' );
                            $ele = $ele.parent();
                        }
                    }

                    jQuery( 'body' ).show().removeClass( 'bbapp_main_content_parent' );
                } );
            </script>
			<?php
			add_filter( 'the_content', array( $this, 'wrap_fallback_div' ), 999 );
		}
	}

	/**
	 * Wrap post div for fallback support.
	 *
	 * @param string $content Content.
	 *
	 * @return string
	 */
	public function wrap_fallback_div( $content ) {
		return "<div class='bbapp_main_content_area'>" . $content . '</div>';
	}

	/**
	 * User links.
	 *
	 * @return array|false|mixed
	 */
	public function quick_link_user_links() {
		$quick_link_user_links_items = array();

		if ( function_exists( 'bp_is_active' ) ) {
			if ( bp_is_active( 'xprofile' ) ) {
				$item_xprofile                 = array(
					'ID'    => 'profile',
					'title' => __( 'Profile', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'user',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_xprofile;
			}

			if ( bp_is_active( 'settings' ) ) {
				$item_settings                 = array(
					'ID'    => bp_get_settings_slug(),
					'title' => __( 'Account', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'cog',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_settings;
			}

			if ( bp_is_active( 'activity' ) ) {
				$item_activity                 = array(
					'ID'    => 'activities',
					'title' => __( 'Timeline', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'activity',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_activity;
			}

			if ( bp_is_active( 'notifications' ) ) {
				$item_notification             = array(
					'ID'    => 'notifications',
					'title' => __( 'Notifications', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'bell',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_notification;
			}

			if ( bp_is_active( 'messages' ) ) {
				$item_messages                 = array(
					'ID'    => 'messages',
					'title' => __( 'Messages', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'comments',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_messages;
			}

			if ( bp_is_active( 'friends' ) ) {
				$item_friends                  = array(
					'ID'    => 'friends',
					'title' => __( 'Connections', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'user-plus',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_friends;
			}

			if ( bp_is_active( 'groups' ) ) {
				$item_groups                   = array(
					'ID'    => 'groups',
					'title' => __( 'Groups', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'users',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_groups;
			}

			if ( bp_is_active( 'forums' ) ) {
				$item_forums                   = array(
					'ID'    => 'forums',
					'title' => __( 'Forums', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'comments-square',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_forums;
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_media_support_enabled' ) && bp_is_profile_media_support_enabled() ) {
				$item_media                    = array(
					'ID'    => 'photos',
					'title' => __( 'Photos', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'image',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_media;
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_document_support_enabled' ) && bp_is_profile_document_support_enabled() ) {
				$item_documents                = array(
					'ID'    => 'documents',
					'title' => __( 'Documents', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'folder-alt',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_documents;
			}

			if ( bp_is_active( 'media' ) && function_exists( 'bp_is_profile_video_support_enabled' ) && bp_is_profile_video_support_enabled() ) {
				$item_videos                   = array(
					'ID'    => 'videos',
					'title' => __( 'Videos', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'video',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_videos;
			}

			if ( bp_is_active( 'invites' ) && function_exists( 'bp_allow_user_to_send_invites' ) && true === bp_allow_user_to_send_invites() ) {
				$item_invites                  = array(
					'ID'    => 'invites',
					'title' => __( 'Email Invites', 'buddyboss-app' ),
					'icon'  => array(
						'id'   => 'at',
						'type' => 'buddyboss',
					),
				);
				$quick_link_user_links_items[] = $item_invites;
			}
		}

		/**
		 * Filters quick links user links.
		 *
		 * @param array $quick_link_user_links_items Quick link items.
		 */
		$quick_link_user_links_items = apply_filters( 'bbapp_quick_link_user_links_items', $quick_link_user_links_items );

		$quick_link_user_links_items = json_decode( wp_json_encode( $quick_link_user_links_items ), false ); // make array to object.

		return $quick_link_user_links_items;
	}

	/**
	 * Function to get the post months and years.
	 *
	 * @since 1.4.4
	 *
	 * @return array
	 */
	public function bbapp_get_post_months() {
		global $wpdb, $wp_locale;

		$months           = $wpdb->get_results( "SELECT DISTINCT YEAR( post_date ) AS year, MONTH( post_date ) AS month FROM $wpdb->posts WHERE post_type = 'post' AND post_status = 'publish' ORDER BY post_date DESC" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$organized_months = array();
		$count            = 0;

		if ( ! empty( $months ) ) {
			foreach ( $months as $arc_row ) {
				if ( 0 === (int) $arc_row->year ) {
					continue;
				}

				$month                               = zeroise( $arc_row->month, 2 );
				$year                                = $arc_row->year;
				$organized_months[ $count ]['key']   = zeroise( $arc_row->month, 2 ) . '-' . $year;
				$organized_months[ $count ]['label'] = $wp_locale->get_month( $month ) . ' ' . $year;

				$count ++;
			}
		}

		return json_decode( wp_json_encode( $organized_months ), false ); // make array to object.
	}

	/**
	 * Add App Page into the App Menu.
	 *
	 * @param array $allowed_post_types Allowed post types.
	 *
	 * @return array $allowed_post_types
	 */
	public function add_app_pages_in_app_menus( $allowed_post_types ) {
		$allowed_post_types[] = 'app_page';

		return $allowed_post_types;
	}

	/**
	 * Forcefully Enable Gutenberg for the APP page type.
	 *
	 * @param bool   $use_block_editor If needs to use block editor.
	 * @param string $post_type        Post type.
	 *
	 * @return bool
	 */
	public function app_page_force_gutenberg_editor( $use_block_editor, $post_type ) {
		if ( 'app_page' === $post_type ) {
			return true;
		}

		return $use_block_editor;
	}

	/**
	 * Forcefully Enable Gutenberg for the APP page type.
	 *
	 * @param bool    $use_block_editor If needs to use block editor.
	 * @param WP_Post $post             Wp post.
	 *
	 * @return bool
	 */
	public function app_page_force_gutenberg_editor_post( $use_block_editor, $post ) {
		if ( 'app_page' === $post->post_type ) {
			return true;
		}

		return $use_block_editor;
	}

	/**
	 * Register CPT APP Page.
	 */
	public function register_app_page_cpt() {
		$labels = array(
			'name'                  => _x( 'App Pages', 'App Pages name', 'buddyboss-app' ),
			'singular_name'         => _x( 'App Page', 'App Pages Singular name', 'buddyboss-app' ),
			'menu_name'             => __( 'App Pages', 'buddyboss-app' ),
			'name_admin_bar'        => __( 'App Pages', 'buddyboss-app' ),
			'archives'              => __( 'App Page Archives', 'buddyboss-app' ),
			'attributes'            => __( 'App Page Attributes', 'buddyboss-app' ),
			'parent_item_colon'     => __( 'Parent App Page:', 'buddyboss-app' ),
			'all_items'             => __( 'All App Pages', 'buddyboss-app' ),
			'add_new_item'          => __( 'Add New App Page', 'buddyboss-app' ),
			'add_new'               => __( 'Add New', 'buddyboss-app' ),
			'new_item'              => __( 'New Item', 'buddyboss-app' ),
			'edit_item'             => __( 'Edit App Page', 'buddyboss-app' ),
			'update_item'           => __( 'Update App Page', 'buddyboss-app' ),
			'view_item'             => __( 'View App Page', 'buddyboss-app' ),
			'view_items'            => __( 'View App Pages', 'buddyboss-app' ),
			'search_items'          => __( 'Search App Pages', 'buddyboss-app' ),
			'not_found'             => __( 'Not found', 'buddyboss-app' ),
			'not_found_in_trash'    => __( 'Not found in Trash', 'buddyboss-app' ),
			'featured_image'        => __( 'Featured Image', 'buddyboss-app' ),
			'set_featured_image'    => __( 'Set featured image', 'buddyboss-app' ),
			'remove_featured_image' => __( 'Remove featured image', 'buddyboss-app' ),
			'use_featured_image'    => __( 'Use as featured image', 'buddyboss-app' ),
			'uploaded_to_this_item' => __( 'Uploaded to this item', 'buddyboss-app' ),
			'items_list'            => __( 'App Pages list', 'buddyboss-app' ),
			'items_list_navigation' => __( 'App Pages list navigation', 'buddyboss-app' ),
			'filter_items_list'     => __( 'Filter App Pages list', 'buddyboss-app' ),
		);

		$args = array(
			'label'                 => __( 'App Pages', 'buddyboss-app' ),
			'description'           => __( 'App Pages Description', 'buddyboss-app' ),
			'labels'                => $labels,
			'supports'              => array( 'title', 'editor' ),
			'public'                => false,
			'show_ui'               => current_user_can( 'manage_options' ),
			'show_in_menu'          => false,
			'menu_position'         => 64,
			'menu_icon'             => 'dashicons-smartphone',
			'show_in_admin_bar'     => false,
			'show_in_nav_menus'     => true,
			'can_export'            => false,
			'has_archive'           => false,
			'exclude_from_search'   => true,
			'capability_type'       => 'page',
			'show_in_rest'          => true,
			'rest_base'             => 'app_pages',
			'rest_controller_class' => 'WP_REST_Posts_Controller',
		);

		register_post_type( 'app_page', $args );
	}

	/**
	 * App page cpt make as `publicly_queryable` for deeplinking but we don't expose that post-type on web so added redirection to Home page.
	 */
	public function redirect_single_app_page() {
		if ( is_singular( 'app_page' ) ) {
			wp_safe_redirect( get_home_url() );
			exit();
		}
	}

	/**
	 * Check whether Gutenberg editor activated for allowed Post types or not.
	 *
	 * Show error message respective.
	 */
	public function is_activate_gutenberg_editor() {
		$wp_version = get_bloginfo( 'version' );

		// Bail out if not admin page.
		if ( ! is_admin() ) {
			return;
		}

		$post_id = ! empty( $_GET['post'] ) ? bbapp_input_clean( wp_unslash( absint( $_GET['post'] ) ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// If current page is post.php and post isset than query for its post type.
		if ( 'post.php' === SetupAdmin::instance()->get_page_now() && ! empty( $post_id ) || 'post-new.php' === SetupAdmin::instance()->get_page_now() ) {

			// Do something with $post_id. For example, you can get the full post object:.
			$post = get_post( $post_id );

			if ( ! empty( $post_id ) ) {
				$post_type = $post->post_type;
			} else {
				$post_type = isset( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : 'post'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			}

			// Allowed only for App page CPT.
			if ( 'app_page' !== $post_type ) {
				return;
			}

			// Update app license data.
			ManageApp::instance()->verify_license_data();

			// Check if WordPress version less than 5.0.
			if ( version_compare( $wp_version, '5.0', '<' ) ) {
				// Remove Classic editor for the CPT App page.
				remove_post_type_support( $post_type, 'editor' );
				$this->wp_required_editor_notice();
			}

			if ( version_compare( $wp_version, '5.0', '>=' ) && function_exists( 'use_block_editor_for_post_type' ) ) {
				if ( false === use_block_editor_for_post_type( 'app_page' ) ) {
					// Remove Classic editor for the CPT App page.
					remove_post_type_support( $post_type, 'editor' );
					$this->gutenberg_required_editor_notice();
				}
			}
		}
	}

	/**
	 * Show error message if current editor is not Gutenberg Editor.
	 */
	public function gutenberg_required_editor_notice() {
		$class = 'notice notice-error';

		/**
		 * Filters gutenberg required message.
		 *
		 * @param string $message Mesage.
		 */
		$message = sprintf( apply_filters( 'bbapp_gutenberg_editor_required_err_message', __( 'Please enable Gutenberg in WordPress for your mobile app to work properly.', 'buddyboss-app' ) ) );

		printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), wp_kses_post( $message ) );
	}

	/**
	 * Show error message if current editor is not Gutenberg Editor.
	 */
	public function wp_required_editor_notice() {
		$class = 'notice notice-error';

		/**
		 * Filters editor required message.
		 *
		 * @param string $message Message.
		 */
		$message = sprintf( apply_filters( 'bbapp_page_wp_required_err_message', __( 'Please update to WordPress 5.0 or above for your mobile app to work properly.', 'buddyboss-app' ) ) );

		printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), wp_kses_post( $message ) );
	}

	/**
	 * Modify API response.
	 *
	 * @param WP_REST_Response $data    WP REST API response of a post.
	 * @param WP_Post          $post    The post being returned.
	 * @param WP_REST_Request  $request WP REST API request.
	 *
	 * @return WP_REST_Response Response containing the permalink_template.
	 */
	public function modify_app_page_rest_response( $data, $post, $request ) {
		if ( is_admin() ) {
			return $data;
		}

		// Return with actual data if < WordPress 5.0.
		if ( ! function_exists( 'parse_blocks' ) ) {
			return $data;
		}

		$app_page_content      = $post->post_content;
		$new_app_page_data     = bbapp_parse_gutenberg_content( $app_page_content );
		$data->data['content'] = $new_app_page_data;

		return $data;
	}

	/**
	 * Remove Classic Editor for the App Page CPT.
	 *
	 * @param array   $actions Action array.
	 * @param WP_Post $post    Object.
	 *
	 * @return array $actions
	 */
	public function remove_classic_editor_link( $actions, $post ) {
		// Remove Classic editor link.
		if ( 'app_page' === $post->post_type ) {
			unset( $actions['classic'] );
		}

		return $actions;
	}

	/**
	 * Add App Page Gutenberg Editor Meta boxes.
	 */
	public function add_app_page_editor_metaboxes() {
		$post_types = get_post_types();

		foreach ( $post_types as $post_type ) {
			add_meta_box(
				'app_page_gutenberg_editor', // $id
				__( 'App Editor', 'buddyboss-app' ), // $title
				array( $this, 'show_app_page_gutenberg_editor' ), // $callback
				$post_type,
				'normal', // $context
				'high' // $priority
			);
		}
	}

	/**
	 * Callback function for the App Page Gutenberg Editor on Button click.
	 *
	 * @param WP_Post $post Wp post.
	 */
	public function show_app_page_gutenberg_editor( $post ) {

		// Bail out if CPT is APPPage.
		if ( 'app_page' === $post->post_type ) {
			return;
		}

		$this->add_app_boss_editor_button( 'content' );
	}

	/**
	 * Save App page Gutenberg Editor Checkbox value.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_app_page_gutenberg_editor_checkbox( $post_id ) {
		$get_post_type = get_post_type( $post_id );

		// Bail out if CPT is APPPage.
		if ( 'app_page' === $get_post_type || 'revision' === $get_post_type ) {
			return;
		}

		// nonce verification from WordPress edit post.
		if ( isset( $_POST['_wpnonce'] ) && empty( $_POST['_wpnonce'] ) && false === wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'update-post_' . $post_id ) ) {
			return;
		}

		// Classic Editor Content.
		if ( isset( $_POST['app-page-editor-content'] ) ) {
			update_post_meta( $post_id, '_app_page_editor_content', sanitize_post_field( 'post_content', wp_unslash( $_POST['app-page-editor-content'] ), $post_id, 'db' ) ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}
	}

	/**
	 * Exclude Default App Page From post type listing.
	 *
	 * @param WP_Query $query Wp query.
	 *
	 * @return WP_Query
	 */
	public function exclude_default_app_page( $query ) {
		if ( is_admin() && 'app_page' === $query->get( 'post_type' ) ) {
			$bbapp_default_apppage = get_option( 'bbapp_default_apppage', false );

			if ( ! empty( $bbapp_default_apppage ) ) {
				$post__not_in = $query->get( 'post__not_in' );
				$post__not_in = array_merge( array( $bbapp_default_apppage ), $post__not_in );

				$query->set( 'post__not_in', $post__not_in );
			}
		}

		return $query;
	}

	/**
	 * Add Style on WP Admin Head.
	 *
	 * Apply style for the AppPage Editor only.
	 */
	public function app_page_head() {
		$screen = get_current_screen();

		if ( empty( $screen->post_type ) || 'edit' === $screen->base ) {
			return;
		}

		$allowed_post_types = $this->get_editor_allowed_cpt();

		if ( ! in_array( $screen->post_type, $allowed_post_types, true ) ) {
			return;
		}

		$post_gutenberg_url = '#';

		if ( empty( $native_editor_id ) ) {
			$native_editor_id = 'content';
		}

		$default_apppage = bbapp_get_default_app_page();

		if ( ! empty( $default_apppage ) ) {
			$post_gutenberg_url = site_url() . '/wp-admin/post.php?post=' . $default_apppage . '&action=edit&default_app_page=true&editor_id=' . $native_editor_id;
		}

		$safe_get            = filter_input_array( INPUT_GET, FILTER_DEFAULT );
		$is_default_app_page = isset( $safe_get['default_app_page'] ) ? $safe_get['default_app_page'] : '';

		// Include AppPage Header Code.
		include bbapp()->plugin_dir . 'views/app-page/header.php';
	}

	/**
	 * Add Script on WP Admin Footer.
	 */
	public function app_page_footer() {
		$screen = get_current_screen();

		if ( empty( $screen->post_type ) ) {
			return;
		}

		$safe_get            = filter_input_array( INPUT_GET, FILTER_DEFAULT );
		$is_default_app_page = isset( $safe_get['default_app_page'] ) ? $safe_get['default_app_page'] : '';
		$editor_id           = isset( $safe_get['editor_id'] ) ? $safe_get['editor_id'] : 'content';
		$post_id             = isset( $safe_get['post'] ) ? absint( $safe_get['post'] ) : 0;
		$post_type           = get_post_type( $post_id );

		/**
		 * Filters offline editor for custom post types.
		 *
		 * @param array $cpt List of post types.
		 */
		$offline_blocks_support_post_type = apply_filters(
			'bbapp_editor_offline_blocks_allowed_cpt',
			array(
				'sfwd-courses',
				'sfwd-lessons',
				'sfwd-topic',
			)
		);

		$disallow_blocks_for_other_cpts        = $this->disallow_blocks_other_cpts();
		$disallow_blocks_for_other_cpts        = wp_json_encode( $disallow_blocks_for_other_cpts );
		$allow_blocks_for_editor_to_app_editor = $this->bbapp_supported_block_list();
		$allow_blocks_for_editor_to_app_editor = wp_json_encode( $allow_blocks_for_editor_to_app_editor );

		// Include AppPage Footer Code.
		include bbapp()->plugin_dir . 'views/app-page/footer.php';
	}

	/**
	 * Function to ge the list of plugin supported blocks.
	 *
	 * @since 1.4.5
	 *
	 * @return mixed|void
	 */
	public function bbapp_supported_block_list() {
		/**
		 * Filters allowed block for app editor.
		 *
		 * @param array $blocks Allowed blocks.
		 */
		$allow_blocks_for_editor_to_app_editor = apply_filters(
			'allow_blocks_for_editor_to_app_editor',
			array(
				'core/paragraph',
				'core/heading',
				'core/quote',
				'core/list',
				'core/table',
				'core/gallery',
				'core/audio',
				'core/video',
				'core/image',
				'core/cover',
				'core/html',
				'core/media-text',
				'core/buttons',
				'core/button',
				'core/columns',
				'core/column',
				'core/embed',
				'core-embed/vimeo',
				'core-embed/youtube',
				'core/code',
				'core/pullquote',
				'core/preformatted',
				'core/verse',
				'core/file',
				'core/group',
				'core/separator',
				'core/spacer',
				'bbapp/h5p',
			)
		);

		return $allow_blocks_for_editor_to_app_editor;
	}

	/**
	 * Add Button next to `Add Media` in Classic Editor.
	 *
	 * @param string $native_editor_id Native editor id.
	 */
	public function add_app_boss_editor_button( $native_editor_id ) {
		global $post;

		// Only add App Editor for the Content Block.
		if ( 'content' !== $native_editor_id ) {
			return;
		}

		$post_id   = isset( $post->ID ) ? $post->ID : 0;
		$post_type = get_post_type( $post_id );

		if ( ! in_array( $post_type, $this->get_editor_allowed_cpt(), true ) ) {
			return;
		}

		if ( empty( $native_editor_id ) ) {
			$native_editor_id = 'content';
		}

		$app_page_editor = get_post_meta( $post_id, '_app_page_editor_' . $native_editor_id, true );
		$app_page_editor = ! empty( $app_page_editor ) ? $app_page_editor : '';
		$app_page_editor = apply_filters( 'ap_gutenberg_app_page_editor_content', $app_page_editor, $native_editor_id );

		// Include App Editor Button Code.
		include bbapp()->plugin_dir . 'views/app-page/editor-button.php';
	}

	/**
	 * Create Default APP Page.
	 */
	public function create_default_apppage() {
		$app_home_page = bbapp_get_page_by_title( 'Home', OBJECT, 'app_page' );

		if ( ! $app_home_page ) {
			$current_user = wp_get_current_user();

			$post_content  = '';
			$post_content .= '<!-- wp:bbapp/notifications -->';
			$post_content .= '<div class="wp-block-bbapp-notifications"></div>';
			$post_content .= '<!-- /wp:bbapp/notifications -->';

			$page = array(
				'post_title'   => __( 'Home', 'buddyboss-app' ),
				'post_status'  => 'publish',
				'post_author'  => $current_user->ID,
				'post_type'    => 'app_page',
				'post_content' => $post_content,
			);

			// insert the post into the database.
			wp_insert_post( $page );
		}
	}

	/**
	 * Register custom block category.
	 *
	 * @since 1.0.0
	 *
	 * @param array       $block_categories Array of current block categories.
	 * @param object|bool $post             WP_Post instance of post being edited.
	 *
	 * @return array $block_categories.
	 */
	public function block_categories( $block_categories = array(), $post = false ) {
		$block_categories[] = array(
			'slug'  => 'bbapp-widgets-blocks',
			'title' => esc_html__( 'BuddyBoss App Blocks', 'buddyboss-app' ),
			'icon'  => false,
		);

		// Always return $default_block_categories.
		return $block_categories;
	}

	/**
	 * Filters the list of available list table views.
	 *
	 * @param string[] $views An array of available list table views.
	 *
	 * @return string[]
	 */
	public function bbapp_exclude_default_page_count( $views ) {
		global $wpdb;

		$bbapp_default_apppage = bbapp_get_default_app_page();

		if ( is_admin() && ! empty( $bbapp_default_apppage ) ) {
			$total   = $wpdb->get_var( "SELECT COUNT(*) FROM $wpdb->posts WHERE (post_status = 'publish' OR post_status = 'draft' OR post_status = 'pending') AND post_type = 'app_page' " ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$publish = $wpdb->get_var( "SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'publish' AND post_type = 'app_page'" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			/**
			 * Update all and publish value return any value
			 */
			$views['all']     = preg_replace( '/\(.+\)/U', '(' . ( $total - 1 ) . ')', $views['all'] );
			$views['publish'] = preg_replace( '/\(.+\)/U', '(' . ( $publish - 1 ) . ')', $views['publish'] );
		}

		return $views;
	}

	/**
	 * Remove patterns from app pages.
	 *
	 * @param array $settings Settings array.
	 *
	 * @return mixed
	 */
	public function remove_block_patterns_support( $settings ) {
		global $current_screen;

		if ( 'app_page' === $current_screen->post_type ) {
			$settings['__experimentalBlockPatterns'] = array();
		}

		return $settings;
	}

	/**
	 * Block list for app pages.
	 *
	 * @since 1.4.3
	 *
	 * @return array
	 */
	public function disallow_blocks_other_cpts() {
		/**
		 * Filter blocks for custom post types.
		 *
		 * @param array $blocks List of blocks.
		 */
		return apply_filters(
			'bbapp_disallowed_block_for_other_cpt',
			array(
				'bbapp/paragraph',
				'bbapp/audio',
				'bbapp/heading',
				'bbapp/image',
				'bbapp/list',
				'bbapp/video',
				'bbapp/html',
				'bbapp/quote',
				'bbapp/media-text',
				'bbapp/gallery',
				'bbapp/table',
				'bbapp/cover',
				'bbapp/button',
				'bbapp/columns',
				'bbapp/column',
				'bbapp/embed',
				'bbapp-embed/vimeo',
				'bbapp-embed/youtube',
				'bbapp/my-progress',
				'bbapp/courses',
				'bbapp/mp-courses',
				'bbapp/tutor-courses',
				'bbapp/topics',
				'bbapp/forums',
				'bbapp/groups',
				'bbapp/members',
				'bbapp/notifications',
				'bbapp/qlinks',
				'bbapp/activity',
				'bbapp/posts',
				'bbapp/directory',
			)
		);
	}

	/**
	 * Footer scripts for widget.
	 *
	 * @since 1.4.3
	 *
	 * @return void
	 */
	public function widget_footer() {
		global $pagenow;

		if ( 'widgets.php' !== $pagenow && 'customize.php' !== $pagenow ) {
			return;
		}

		$disallow_blocks_for_other_cpts = $this->disallow_blocks_other_cpts();

		// Prepare the JavaScript object data
		$script_data = wp_json_encode( $disallow_blocks_for_other_cpts );

		// JavaScript logic
		$js = "
        jQuery(window).on('load', function () {
            if (typeof wp.blocks === 'undefined') {
                return;
            }

            const blocks = wp.blocks.getBlockTypes();
            const disallowedBlockTypes = JSON.parse(" . wp_json_encode( $script_data ) . ");

            blocks.forEach(function (block) {
                if (jQuery.inArray(block.name, disallowedBlockTypes) > -1) {
                    console.log(block);
                    wp.blocks.unregisterBlockType(block.name);
                }
            });
        });
    ";

		// Enqueue the script using wp_add_inline_script
		wp_add_inline_script( 'wp-blocks', $js );
	}

	/**
	 * Add gutenberg related variables for the blocks.
	 *
	 * @since 2.1.70
	 * @return void
	 */
	public function gutenberg_variables() {
		$months = $this->bbapp_get_post_months();

		$user_links = $this->quick_link_user_links();
		$core_menus = bbapp_get_core_appmenu();
		$default_page = bbapp_get_default_app_page();

		// Prepare the JavaScript object data
		$data = array(
			'years_months' => $months,
			'user_links'   => $user_links,
			'core_menus'   => $core_menus,
			'default_page' => $default_page,
		);

		// Use wp_add_inline_script to add JS safely
		$js = "
        if (typeof window.gutenberg_post_block === 'undefined') {
            window.gutenberg_post_block = [];
        }
        window.gutenberg_post_block['years_months'] = " . wp_json_encode( $data['years_months'] ) . ";
        
        if (typeof window.gutenberg_quick_link_blocks === 'undefined') {
            window.gutenberg_quick_link_blocks = [];
        }
        window.gutenberg_quick_link_blocks['user_links'] = " . wp_json_encode( $data['user_links'] ) . ";
        window.gutenberg_quick_link_blocks['core_menus'] = " . wp_json_encode( $data['core_menus'] ) . ";
        window.gutenberg_quick_link_blocks['default_page'] = " . wp_json_encode( $data['default_page'] ) . ";
    ";

		wp_add_inline_script( 'wp-blocks', $js );
	}

	/**
	 * Search Blog post auther search by display name only.
	 *
	 * @param array           $prepared_args Prepared arguments.
	 * @param WP_REST_Request $request       Rest request.
	 *
	 * @return mixed
	 */
	public function bbapp_search_user_by_name_filter( $prepared_args, $request ) {
		if ( true === (bool) $request->get_param( 'is_author_search' ) ) {
			$prepared_args['search_columns'] = array( 'display_name' );
		}

		return $prepared_args;
	}

	/**
	 * Get editor allowed custom post type.
	 *
	 * @since 1.6.0
	 * @return mixed|void
	 */
	public function get_editor_allowed_cpt() {
		$allowed_post_types = apply_filters(
			'bbapp_editor_allowed_cpt',
			array(
				'app_page',
				'post', // Blog post support.
				'sfwd-courses',
				'sfwd-lessons',
				'sfwd-topic',
				'sfwd-quiz',
			)
		);

		return $allowed_post_types;
	}

	/**
	 * Page break block support.
	 *
	 * @param string $link Link.
	 *
	 * @since 2.0.70
	 * @return array|string|string[]|null
	 */
	public function bbapp_wp_link_pages_link( $link ) {
		return preg_replace( '/(href=["\'])(.*?)["\']/', '$1$2?mobile-view-content=1"', $link );
	}

	/**
	 * Remove 'App Pages' from nav menu list, if current screen is appearance nav-menu
	 *
	 * @param WP_Post_Type|false $post The post object for nav menu item.
	 *
	 * @since 2.1.60
	 *
	 * @return object|bool
	 */
	public function bbapp_remove_app_pages_appearance_menu( $post ) {
		if ( function_exists( 'get_current_screen' ) ) {
			$screen = get_current_screen();

			if ( 'app_page' === $post->name && isset( $screen->id ) && 'nav-menus' === $screen->id ) {
				return false;
			}
		}

		return $post;
	}
}
