<?php

namespace BuddyBossApp\Migration;

use \WP_Error as WP_Error;

class RestApi {
	protected $namespace = 'appboss/core/v1';
	private static $instance;

	public function __construct() {
	}

	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register WordPress endpoint routes.
	 */
	public function register_routes() {

		register_rest_route(
			$this->namespace,
			'/menu',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'menu' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'allow_more' => array(
							'type'              => 'boolean',
							'default'           => true,
							'enum'              => array( true, false ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/menu/(?P<app_id>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'menu' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'allow_more' => array(
							'type'              => 'boolean',
							'default'           => true,
							'enum'              => array( true, false ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/device_token',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'update_device_token' ),
					'args'                => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/app_details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'app_details' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/settings',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( \BuddyBossApp\AppSettings::instance(), 'rest_response' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/settings/(?P<app_id>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( \BuddyBossApp\AppSettings::instance(), 'rest_response' ),
				),
			)
		);


		register_rest_route(
			$this->namespace,
			'/profile/details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_profile_details' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/search/details',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_search_details' ),
				),
			)
		);

		/**
		 * This endpoint is only for Health check
		 */
		register_rest_route(
			$this->namespace,
			'/cache_test',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'permission_callback' => '__return_true',
					'callback'            => array( $this, 'get_cache_test' ),
				),
			)
		);

		/**
		 * This endpoint is feedback endpoint.
		 */
		register_rest_route(
			$this->namespace,
			'/feedback',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'send_feedback' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'message' => array(
							'required'          => true,
							'type'              => 'string',
							'validate_callback' => 'rest_validate_request_arg',
						),
						'fields'  => array(
							'description' => __( 'Feedback fields.', 'buddyboss-app' ),
							'default'     => array(),
						),
						'type'    => array(
							'description'       => __( 'Feedback Type.', 'buddyboss-app' ),
							'type'              => 'string',
							'default'           => 'bug',
							'enum'              => array( 'bug', 'feedback' ),
							'validate_callback' => 'rest_validate_request_arg',
						),
					),
				),
			)
		);
	}

	/**
	 * @param $request
	 *
	 * @return mixed
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/core/v1/menu/:app_id App Menu
	 * @apiName CoreMenuSettings
	 * @apiGroup CoreSettings
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Core menu settings
	 * @apiParam {String} app_id App Id
	 * @apiUse apidocForCoreMenuSettingsV1
	 */
	public function menu( $request ) {
		global $wp_rest_server;
		$app_id = $request->get_param( 'app_id' );

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/core/v1/menu/{$app_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$request_curl->set_file_params( $request->get_file_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return WP_Error
     * @apiPrivate
	 * @api {PATCH} /wp-json/appboss/core/v1/device_token Update device token
	 * @apiName UpdateDeviceToken
	 * @apiGroup CoreSettings
	 * @apiVersion 1.0.0
	 * @apiPermission LoggedInUser
	 * @apiDescription Update device token
	 * @apiParam {String} [app_id=false] App Id
	 * @apiParam {String} [unique_id=false] Used interchangeably with app_id, one is required
	 */
	public function update_device_token( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/core/v1/device_token' );
		$request_curl->set_body_params( $request->get_params() );
		$request_curl->set_file_params( $request->get_file_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	/**
	 * @param $request
	 *
	 * @return array
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/core/v1/profile/details Profile tabs details
	 * @apiName GetAProfileDetails
	 * @apiGroup CoreSettings
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Get a profile details
	 * @apiError {Object}  Response object
	 */
	public function get_profile_details( $request ) {

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/core/v1/profile/details' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return array
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/core/v1/search/details Search tabs details
	 * @apiName GetASearchDetails
	 * @apiGroup CoreSettings
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription Get a search details
	 * @apiError {Object}  Response object
	 */
	public function get_search_details( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/core/v1/search/details' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $request
	 *
	 * @return mixed|\WP_REST_Response
     * @apiPrivate
	 * @api {GET} /wp-json/appboss/core/v1/app_details App details
	 * @apiName GetAppDetails
	 * @apiGroup CoreSettings
	 * @apiVersion 1.0.0
	 * @apiPermission Public
	 * @apiDescription This endpoint returns the signature information from bbapp server.
	 * @apiUse apidocForGetAppDetailsV1
	 */
	public function app_details( $request ) {
		global $wp_rest_server;
		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/core/v1/app_details' );
		$request_curl->set_query_params( $request->get_params() );
		$request_curl->set_file_params( $request->get_file_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );

	}

	public function get_cache_test( $request ) {
		global $wp_rest_server;
		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/core/v1/cache_test' );
		$request_curl->set_query_params( $request->get_params() );
		$request_curl->set_file_params( $request->get_file_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Feedback api callback.
	 *
	 * @param $request
	 *
	 * @return array|bool|WP_Error
     * @apiPrivate
	 * @api           {POST} /wp-json/appboss/core/v1/feedback
	 * @apiGroup      CoreSettings
	 * @apiVersion    1.0.0
	 * @apiPermission LoggedInUser
	 */
	public function send_feedback( $request ) {
		global $wp_rest_server;
		$request_curl = new \WP_REST_Request( 'POST', '/buddyboss-app/core/v1/feedback' );
		$request_curl->set_body_params( $request->get_params() );
		$request_curl->set_file_params( $request->get_file_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()["_embed"] ) ) {
			$query_params["_embed"] = $request->get_query_params()["_embed"];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
