<?php

namespace BuddyBossApp\Migration\Notification;

use \BuddyBossApp\Notification\Notification;

/**
 * BuddyBossApp Notification.
 * This Notification uses the same response as the latest platform API.
 *
 * Class RestAPIv2
 *
 * @package BuddyBossApp\Notification
 */
class RestAPIv2 extends \WP_REST_Controller {

	public static $instance;

	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'appboss/core/v2';
		$this->rest_base = 'notifications';
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register the component routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'A unique numeric ID for the notification.', 'buddypress' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve notifications.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 */
	public function get_items( $request ) {
		$request['sort_order'] = strtoupper( $request['sort_order'] );
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/core/v2/notifications' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to notification items.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() || ( get_current_user_id() !== $request['user_id'] && ! $this->can_see() ) ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you are not allowed to see notifications.', 'buddypress' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_items` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response
	 */
	public function get_item( $request ) {
		global $wp_rest_server;
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/core/v2/notifications/{$notification_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to get information about a specific notification.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you are not allowed to see the notification.', 'buddypress' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$notification = $this->get_notification_object( $request );

		if ( true === $retval && ( empty( $notification ) || is_null( $notification->item_id ) ) ) {
			$retval = new \WP_Error(
				'bbapp_rest_notification_invalid_id',
				__( 'Invalid notification ID.', 'buddypress' ),
				array(
					'status' => 404,
				)
			);
		}

		if ( true === $retval && ! $this->can_see( $notification->id ) ) {
			$retval = new \WP_Error(
				'rest_not_logged_in',
				__( 'Sorry, you cannot view this notification.', 'buddypress' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the notifications `get_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_get_item_permissions_check', $retval, $request );
	}

	/**
	 * Update a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function update_item( $request ) {
		global $wp_rest_server;
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/core/v2/notifications/{$notification_id}" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to update a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_Error|bool
	 */
	public function update_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `update_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_update_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function delete_item( $request ) {
		global $wp_rest_server;
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];

		$request_curl = new \WP_REST_Request( 'DELETE', "/buddyboss-app/core/v2/notifications/{$notification_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check if a given request has access to delete a notification.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|\WP_Error
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = $this->get_item_permissions_check( $request );

		/**
		 * Filter the notifications `delete_item` permissions check.
		 *
		 * @param bool|\WP_Error   $retval  Returned value.
		 * @param \WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bbapp_rest_notifications_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Can this user see the notification?
	 *
	 * @param int $notification_id Notification ID.
	 *
	 * @return bool
	 */
	protected function can_see( $notification_id = 0 ) {

		$notification = $this->get_notification_object( $notification_id );

		// Check notification access.
		if ( ! empty( $notification ) && $notification->user_id == get_current_user_id() ) {
			return true;
		}

		// Moderators as well.
		if ( current_user_can( 'bp_moderate' ) || current_user_can( 'manage_options' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Get notification object.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @return \Notification|string A notification object.
	 */
	public function get_notification_object( $request ) {
		$notification_id = is_numeric( $request ) ? $request : (int) $request['id'];

		/**
		 * From BP OR Platform
		 */
		if ( function_exists( 'bp_notifications_get_notification' ) ) {

			$notification = bp_notifications_get_notification( $notification_id );

			if ( empty( $notification->id ) ) {
				return '';
			}
		} else {
			/**
			 * From Builtin BuddyBoss App.
			 */

			// Actually, query it.
			$notifications = Notification::instance()->get_inbuilt_notifications( array( 'id' => $notification_id ) );

			/**
			 * Add is_new property. for making it compatible.
			 */
			foreach ( $notifications as $k => $v ) {
				$notifications[ $k ]->is_new = $v->unread;
			}

			if ( ! isset( $notifications[0] ) ) {
				return '';
			}

			$notification = $notifications[0];

		}

		return $notification;
	}

	/**
	 * Select the item schema arguments needed for the EDITABLE method.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 */
	public function get_endpoint_args_for_item_schema( $method = \WP_REST_Server::CREATABLE ) {
		$args = \WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'get_item';

		if ( \WP_REST_Server::EDITABLE === $method ) {
			$key = 'update_item';

			// Only switching the is_new property can be achieved.
			$args                      = array_intersect_key( $args, array( 'is_new' => true ) );
			$args['is_new']['default'] = 0;
		} elseif ( \WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 */
		return apply_filters( "bp_rest_notifications_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the notification schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bp_notifications',
			'type'       => 'object',
			'properties' => array(
				'id'                => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'A unique numeric ID for the notification.', 'buddypress' ),
					'readonly'    => true,
					'type'        => 'integer',
				),
				'user_id'           => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the user the notification is addressed to.', 'buddypress' ),
					'type'        => 'integer',
					'default'     => get_current_user_id(),
				),
				'item_id'           => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the item associated with the notification.', 'buddypress' ),
					'type'        => 'integer',
				),
				'secondary_item_id' => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The ID of the secondary item associated with the notification.', 'buddypress' ),
					'type'        => 'integer',
				),
				'component'         => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The name of the BuddyPress component the notification relates to.', 'buddypress' ),
					'type'        => 'string',
				),
				'action'            => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'The name of the component\'s action the notification is about.', 'buddypress' ),
					'type'        => 'string',
				),
				'date'              => array(
					'description' => __( 'The date the notification was created, in the site\'s timezone.', 'buddypress' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit' ),
				),
				'is_new'            => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Whether it\'s a new notification or not.', 'buddypress' ),
					'type'        => 'integer',
					'default'     => 1,
				),
				'description'       => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Whether it\'s a new notification or not.', 'buddypress' ),
					'type'        => 'integer',
					'default'     => 1,
					'properties'  => array(
						'rendered' => array(
							'description' => __( 'HTML description for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
							'readonly'    => true,
						),
					),
				),
				'link_url'          => array(
					'context'     => array( 'view', 'edit' ),
					'description' => __( 'Link URL for the notification.', 'buddypress' ),
					'type'        => 'string',
				),
			),
		);

		/**
		 * Filters the notifications schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bp_rest_notification_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Get the query params for the notifications collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		// Remove the search argument.
		unset( $params['search'] );

		$params['order_by'] = array(
			'description'       => __( 'Name of the field to order according to.', 'buddypress' ),
			'default'           => 'id',
			'type'              => 'string',
			'enum'              => array(
				'id',
				'date_notified',
				'item_id',
				'secondary_item_id',
				'component_name',
				'component_action',
			),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['sort_order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddypress' ),
			'default'           => 'ASC',
			'type'              => 'string',
			'enum'              => array( 'ASC', 'DESC' ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_name'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component', 'buddypress' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['component_action'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific component\'s action name.', 'buddypress' ),
			'default'           => '',
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to notifications addressed to a specific user.', 'buddypress' ),
			'default'           => get_current_user_id(),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific item ID.', 'buddypress' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['secondary_item_id'] = array(
			'description'       => __( 'Limit result set to notifications associated with a specific secondary item ID.', 'buddypress' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['is_new'] = array(
			'description'       => __( 'Limit result set to items from specific states.', 'buddypress' ),
			'default'           => true,
			'type'              => 'boolean',
			'sanitize_callback' => 'rest_sanitize_boolean',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bbapp_rest_notifications_collection_params', $params );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
