<?php
/**
 * Holds third party membership functions.
 *
 * @package BuddyBossApp\Memberships
 */

namespace BuddyBossApp\Memberships;

use WP_Error;

/**
 * Membership utility class.
 */
trait MembershipsUtils {

	/**
	 * Check if user has memberpress membership.
	 *
	 * @param int $id Memberpress membership id.
	 *
	 * @return WP_Error|bool
	 */
	public static function memberpress_has_access( $id ) {
		$has_access = false;

		if ( class_exists( 'MeprUtils' ) ) {
			$user                         = \MeprUtils::get_currentuserinfo();
			$active_product_subscriptions = $user->active_product_subscriptions();

			if ( in_array( $id, $active_product_subscriptions, true ) ) {
				$has_access = true;
			}
		} else {
			return new WP_Error( 'not_available', __( 'MemberPress is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $has_access;
	}

	/**
	 * This will return ids of Memberpress(memberpressproduct) which current(logged-in) user have access to.
	 *
	 * @return WP_Error|array Consists Ids
	 */
	public static function memberpress_access_list() {
		$access_list = array();

		if ( class_exists( 'MeprUtils' ) ) {
			$user                         = \MeprUtils::get_currentuserinfo();
			$active_product_subscriptions = $user->active_product_subscriptions();
			$access_list                  = $active_product_subscriptions;
		} else {
			return new WP_Error( 'not_available', __( 'MemberPress is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $access_list;
	}

	/**
	 * Check if user has access to WC Membership.
	 *
	 * @param int $plan_id Membership id.
	 *
	 * @return WP_Error|bool
	 */
	public static function woo_membership_has_access( $plan_id ) {
		$has_access = false;

		// NOTE : Using wc_memberships_get_user_active_memberships() NOT wc_memberships_get_user_memberships().
		if ( class_exists( 'WC_Memberships_User_Memberships' ) && function_exists( 'wc_memberships_get_user_active_memberships' ) ) {
			$user_memberships = wc_memberships_get_user_active_memberships();
			$access_list      = array();

			// Note : Can't use wp_list_pluck($user_memberships, 'plan_id') since string casting was NOT possible.
			foreach ( $user_memberships as $membership ) {
				if ( isset( $membership->plan_id ) ) {
					$access_list[] = $membership->plan_id;
				}
			}

			if ( in_array( $plan_id, $access_list, true ) ) {
				$has_access = true;
			}
		} else {
			return new WP_Error( 'not_available', __( 'WooCommerce Memberships is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $has_access;
	}

	/**
	 * If user has access to PM Pro membership.
	 *
	 * @param int $level_id Membership level id.
	 *
	 * @return WP_Error|bool
	 */
	public static function pm_pro_membership_has_access( $level_id ) {
		$has_access = false;

		// Only if pmpro-memberships(constant: PMPRO_VERSION) plugin is active.
		if ( defined( 'PMPRO_VERSION' ) ) {
			$user_id          = get_current_user_id();
			$user_memberships = pmpro_getMembershipLevelsForUser( $user_id );
			$access_list      = array();

			// Note : Can't use wp_list_pluck($user_memberships, 'ID') since string casting was NOT possible.
			foreach ( $user_memberships as $index => $membership ) {
				if ( isset( $membership->ID ) ) {
					$access_list[] = $membership->ID;
				}
			}

			if ( in_array( $level_id, $access_list, true ) ) {
				$has_access = true;
			}
		} else {
			return new WP_Error( 'not_available', __( 'Paid Memberships Pro is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $has_access;
	}

	/**
	 * This will return ids of WooCommerce(woo-memberships) which current(logged-in) user have access to
	 *
	 * @return WP_Error|array Consists Ids
	 */
	public static function woo_membership_access_list() {
		$access_list = array();

		// NOTE : Using wc_memberships_get_user_active_memberships() NOT wc_memberships_get_user_memberships().
		if ( class_exists( 'WC_Memberships_User_Memberships' ) && function_exists( 'wc_memberships_get_user_active_memberships' ) ) {
			$user_memberships = wc_memberships_get_user_active_memberships();
			$access_list      = array();

			// Note : Can't use wp_list_pluck($user_memberships, 'plan_id') since string casting was NOT possible.
			foreach ( $user_memberships as $membership ) {
				if ( isset( $membership->plan_id ) ) {
					$access_list[] = (string) $membership->plan_id;
				}
			}
		} else {
			return new WP_Error( 'not_available', __( 'WooCommerce Memberships is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $access_list;
	}


	/**
	 * This will return ids of PmProMembership(pm-pro-membership) which current(logged-in) user have access to
	 *
	 * @return WP_Error|array Consists Ids
	 */
	public static function pm_pro_membership_access_list() {
		$access_list = array();

		// Only if pmpro-memberships(constant: PMPRO_VERSION) plugin is active.
		if ( defined( 'PMPRO_VERSION' ) ) {
			$user_id          = get_current_user_id();
			$user_memberships = pmpro_getMembershipLevelsForUser( $user_id );

			// Note : Can't use wp_list_pluck($user_memberships, 'ID') since string casting was NOT possible.
			foreach ( $user_memberships as $membership ) {
				if ( isset( $membership->ID ) ) {
					$access_list[] = (string) $membership->ID;
				}
			}
		} else {
			return new WP_Error( 'not_available', __( 'Paid Memberships Pro is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $access_list;
	}


	/**
	 * Check if user has course access.
	 *
	 * @param int $course_id Course id.
	 *
	 * @return WP_Error|bool
	 */
	public static function learndash_course_has_access( $course_id ) {
		$has_access = false;

		if ( function_exists( 'sfwd_lms_has_access' ) ) {
			if ( sfwd_lms_has_access( $course_id ) ) {
				$has_access = true;
			}
		} else {
			return new WP_Error( 'not_available', __( 'LearnDash is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $has_access;

	}

	/**
	 * This will return ids of LearnDash(learndash-course) which current(logged-in) user have access to
	 *
	 * @return WP_Error|array Consists Ids
	 */
	public static function learndash_course_access_list() {
		$access_list = array();

		if ( function_exists( 'learndash_user_get_enrolled_courses' ) ) {
			$current_user_id = get_current_user_id();
			$access_list     = learndash_user_get_enrolled_courses( $current_user_id, array(), true );
		} else {
			return new WP_Error( 'not_available', __( 'LearnDash is inactive or integration is not available', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		return $access_list;
	}
}
