<?php

namespace BuddyBossApp\Migration;

use WP_Error;
use WP_REST_Controller;
use WP_REST_SERVER;

// API for memberpress.
class Memberships extends WP_REST_Controller {

	protected $namespace = 'appboss/v2';

	private static $instance;
	private $route_slug = 'memberships';


	/**
	 * Memberships constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance()
	}

	/**
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function load() {
		add_action( 'rest_api_init', array( $this, 'registerRoutes' ), 99 );
	}

	/**
	 *
	 */
	public function registerRoutes() {
		register_rest_route(
			$this->namespace,
			$this->route_slug . '/(?P<integration_type>\S+)' . '/(?P<id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'itemDetails' ),
					'permission_callback' => array( $this, 'permissionCallback' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type' ),
							'enum'        => array( 'memberpress', 'woo-membership', 'pm-pro-membership' ),
						),
						'id'               => array(
							'required'    => true,
							'type'        => 'number',
							'description' => __( 'Unique Identifier' ),
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			$this->route_slug . '/(?P<integration_type>\S+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'accessDetails' ),
					'permission_callback' => array( $this, 'permissionCallback' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type' ),
							'enum'        => array( 'memberpress', 'woo-membership', 'pm-pro-membership' ),
						),
					),
				),
			)
		);

	}

	/**
	 * @return mixed
	 */
	public function permissionCallback() {
		return is_user_logged_in();
	}


	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_REST_Response\WP_Error
	 * @api            {GET} /wp-json/appboss/v2/memberships/:integration_type/:id Membership details
	 * @apiName        MembershipItemInfo
	 * @apiGroup       Memberships
	 * @apiVersion     2.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get information for particular membership item
	 * @apiUse         apidocForMembershipItemInfo
	 */
	public function itemDetails( $request ) {
		$integration_type = $request->get_param( 'integration_type' );
		$id               = $request->get_param( 'id' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/v2/memberships/{$integration_type}/{$id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}


	/**
	 * @param $request
	 * @apiPrivate
	 * @return WP_REST_Response\WP_Error
	 * @api            {GET} /wp-json/appboss/v2/memberships/:integration_type Membership access details
	 * @apiName        MembershipAccessDetails
	 * @apiGroup       Memberships
	 * @apiVersion     2.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get access information for particular membership
	 * @apiUse         apidocForMembershipAccessDetails
	 */
	public function accessDetails( $request ) {
		$integration_type = $request->get_param( 'integration_type' );

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/v2/memberships/{$integration_type}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Dispatch the request item.
	 *
	 * @param $request
	 *
	 * @return mixed
	 */
	protected function dispatch( $request ) {

		$query_params = $request->get_params();

		if ( isset( $request->get_query_params()['_embed'] ) ) {
			$query_params['_embed'] = $request->get_query_params()['_embed'];
		}

		$request->set_query_params( $query_params );

		$server   = rest_get_server();
		$response = $server->dispatch( $request );

		return $response;
	}
}
