<?php
/**
 * Holds third party memberships API.
 *
 * @package BuddyBossApp\Memberships
 */

namespace BuddyBossApp\Memberships;

use WP_Error;
use WP_REST_Controller;
use WP_REST_SERVER;
use WP_REST_Request;
use WP_REST_Response;

/**
 * API for third party memberhips.
 */
class BBAPPMemberships extends WP_REST_Controller {

	/**
	 * Rest namespace (Must be protected).
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss-app/v2';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Rest route slug.
	 *
	 * @var string $route_slug
	 */
	private $route_slug = 'memberships';


	/**
	 * Memberships constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of this class.
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register rest routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->route_slug . '/(?P<integration_type>\S+)/(?P<id>\d+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'item_details' ),
					'permission_callback' => array( $this, 'permissionCallback' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type', 'buddyboss-app' ),
							'enum'        => array( 'memberpress', 'woo-membership', 'pm-pro-membership' ),
						),
						'id'               => array(
							'required'    => true,
							'type'        => 'number',
							'description' => __( 'Unique Identifier', 'buddyboss-app' ),
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			$this->route_slug . '/(?P<integration_type>\S+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'access_details' ),
					'permission_callback' => array( $this, 'permissionCallback' ),
					'args'                => array(
						'integration_type' => array(
							'required'    => true,
							'type'        => 'string',
							'description' => __( 'Integration Type', 'buddyboss-app' ),
							'enum'        => array( 'memberpress', 'woo-membership', 'pm-pro-membership' ),
						),
					),
				),
			)
		);
	}

	/**
	 * Rest api permission callback.
	 *
	 * @return mixed
	 */
	public function permissionCallback() {
		return is_user_logged_in();
	}

	/**
	 * Membership item details.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return WP_REST_Response|\WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/v2/memberships/:integration_type/:id Membership details
	 * @apiName        MembershipItemInfo
	 * @apiGroup       Memberships
	 * @apiVersion     2.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get information for particular membership item
	 * @apiUse         apidocForMembershipItemInfo
	 */
	public function item_details( $request ) {
		$integration_type       = $request->get_param( 'integration_type' );
		$id                     = $request->get_param( 'id' );
		$response['has_access'] = false;

		switch ( $integration_type ) {
			case 'memberpress':
				$response['has_access'] = MembershipsUtils::memberpress_has_access( $id );

				if ( is_wp_error( $response['has_access'] ) ) {
					return $response['has_access'];
				}
				break;
			case 'woo-membership':
				$response['has_access'] = MembershipsUtils::woo_membership_has_access( $id );

				if ( is_wp_error( $response['has_access'] ) ) {
					return $response['has_access'];
				}
				break;
			case 'pm-pro-membership':
				$response['has_access'] = MembershipsUtils::pm_pro_membership_has_access( $id );

				if ( is_wp_error( $response['has_access'] ) ) {
					return $response['has_access'];
				}
				break;
			default:
				return new WP_Error( 'not_found', 'Integration is not found' );
		}

		return rest_ensure_response( $response );
	}


	/**
	 * Membership access details.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @return WP_REST_Response|\WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/v2/memberships/:integration_type Membership access details
	 * @apiName        MembershipAccessDetails
	 * @apiGroup       Memberships
	 * @apiVersion     2.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get access information for particular membership
	 * @apiUse         apidocForMembershipAccessDetails
	 */
	public function access_details( $request ) {
		$integration_type        = $request->get_param( 'integration_type' );
		$response['access_list'] = false;

		switch ( $integration_type ) {
			case 'memberpress':
				$response['access_list'] = MembershipsUtils::memberpress_access_list();

				if ( is_wp_error( $response['access_list'] ) ) {
					return $response['access_list'];
				}
				break;
			case 'woo-membership':
				$response['access_list'] = MembershipsUtils::woo_membership_access_list();

				if ( is_wp_error( $response['access_list'] ) ) {
					return $response['access_list'];
				}
				break;
			case 'pm-pro-membership':
				$response['access_list'] = MembershipsUtils::pm_pro_membership_access_list();

				if ( is_wp_error( $response['access_list'] ) ) {
					return $response['access_list'];
				}
				break;
			case 'learndash-course':
				$response['access_list'] = MembershipsUtils::learndash_course_access_list();

				if ( is_wp_error( $response['access_list'] ) ) {
					return $response['access_list'];
				}
				break;
			default:
				return new WP_Error( 'not_found', 'Integration is not found' );
		}

		return rest_ensure_response( $response );
	}
}
