<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Quiz;

use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;
use \WP_Post;
use \WP_Query;

class QuizUserRest extends QuizRest {

	protected static $instance;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/user_quizdata', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'user_quizdata' ),
					'permission_callback' => array( $this, 'get_user_quizdata_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}

	/**
	 * Check if a given request has access to quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_user_quizdata_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `user_quizdata` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_user_quizdata_permissions_check', $retval, $request );
	}


	/**
	 * Get Quiz user data.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/quiz/user_quizdata Get Quiz user data.
	 * @apiName        GetLDQuizUserData
	 * @apiGroup       Quiz
	 * @apiDescription Get Quiz user data.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 */
	public function user_quizdata( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/quiz/user_quizdata' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * @param $args
	 *
	 * @return array
	 */
	private function get_user_quizdata( $args ) {
		$offset  = ( $args['paged'] - 1 ) * $args['per_page'];

		$user_id = get_current_user_id();
		$quizzes = get_user_meta( $user_id, '_sfwd-quizzes', true );

		$quiz_ids              = wp_list_pluck( $quizzes, 'quiz' );
		$quiz_total_query_args = array(
			'post_type' => 'sfwd-quiz',
			'fields'    => 'ids',
			'orderby'   => 'title', //$atts['quiz_orderby'],
			'order'     => 'ASC', //$atts['quiz_order'],
			'nopaging'  => true,
			'post__in'  => $quiz_ids,
		);
		$quiz_query            = new WP_Query( $quiz_total_query_args );
		$quizzes_tmp           = array();
		if ( ! empty( $quizzes ) && ! empty( $quiz_query->posts ) ) {
			foreach ( $quiz_query->posts as $post_idx => $quiz_id ) {
				foreach ( $quizzes as $quiz_idx => $quiz_attempt ) {
					if ( $quiz_attempt['quiz'] == $quiz_id ) {
						$quiz_key                 = $quiz_attempt['time'] . '-' . $quiz_attempt['quiz'];
						$quizzes_tmp[ $quiz_key ] = $quiz_attempt;
						unset( $quizzes[ $quiz_idx ] );
					}
				}
			}
		}
		$quizzes = $quizzes_tmp;

		if ( isset( $request['order'] ) && $request['order'] == 'ASC' ) {
			ksort( $quizzes );
		} else {
			krsort( $quizzes );
		}

		$total_posts = count( $quizzes );
		$quizzes     = array_slice( $quizzes, $offset, $args['per_page'] );

		foreach ( $quizzes as $quizdata ) {
			$quiz_settings = learndash_get_setting( $quizdata['quiz'] );
			$quiz_post     = get_post( $quizdata['quiz'] );
			$date          = $quizdata['time'] + get_option( 'gmt_offset' ) * 3600;
			$date          = date( "Y-m-d H:i:s", $date );

			$quiz_statistic = array(
				'quiz'            => $quiz_post,
				'result'          => $quizdata['percentage'],
				'score'           => $quizdata['score'],
				'count'           => ( isset( $quizdata['question_show_count'] ) && ! empty( $quizdata['question_show_count'] ) ) ? $quizdata['question_show_count'] : $quizdata['count'],
				'awarded_points'  => $quizdata['points'],
				'possible_points' => $quizdata['total_points'],
				'Date'            => mysql_to_rfc3339( $date ),
			);

			if ( true === $quizdata['has_graded'] && ! empty( $quizdata['graded'] ) ) {
				foreach ( $quizdata['graded'] as $quiz_question_id => $graded ) {
					$graded_post = get_post( $graded['post_id'] );
					if ( $graded_post instanceof WP_Post ) {
						$quiz_statistic['graded'][ $quiz_question_id ] = array(
							'title'          => get_the_title( $graded['post_id'] ),
							'status'         => $graded['status'],
							'awarded_points' => $graded['points_awarded'],
						);
					}
				}
			}

			$certificate_post_id = intval( $quiz_settings['certificate'] );
			if ( ! empty( $certificate_post_id ) ) {
				$certificate_post = get_post( $certificate_post_id );
				$certificate      = learndash_certificate_details( $quizdata['quiz'], $user_id );

				if ( ( isset( $certificate['certificateLink'] ) ) && ( ! empty( $certificate['certificateLink'] ) ) ) {
					$quiz_statistic['certificate'] = array(
						'title'    => get_the_title( $quizdata['quiz'] ),
						'filename' => sanitize_file_name( $quiz_post->post_title ) . "-" . sanitize_file_name( $certificate_post->post_title ) . ".pdf",
						'link'     => $certificate['certificateLink'],
					);
				}
			}

			$result[] = $quiz_statistic;
		}

		return array( 'posts' => $result, 'total_posts' => $total_posts );
	}
}