<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Quiz;

use BuddyBossApp\Migration\LearnDash\V1\Core\LDRestController;
use BuddyBossApp\Migration\LearnDash\V1\Course\CoursesError;
use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;
use \WP_Error;
use \WP_Query;
use \LearnDash_Settings_Section;
use \LDLMS_Factory_Post;
use \WpProQuiz_Controller_Quiz;
use \WpProQuiz_Model_FormMapper;
use \WpProQuiz_Model_Quiz;
use \WpProQuiz_Model_QuizMapper;
use \WpProQuiz_Model_Form;

class QuizRest extends LDRestController {

	protected static $instance;

	protected $post_type = 'sfwd-quiz';

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Quiz items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_items_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the Quiz `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quizzes_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_item_permissions_check( $request ) {

		$retval = true;

		/**
		 * Filter the quiz `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base, array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)', array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}


	/**
	 * Retrieve Quizzes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/quiz Get Quizzes
	 * @apiName        GetLDQuizzes
	 * @apiGroup       Quiz
	 * @apiDescription Retrieve Quizzes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure result set includes specific IDs.
	 * @apiParam {String} [after]  Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to excludes those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', '/buddyboss-app/learndash/v1/quiz' );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}


	/**
	 * Retrieve Quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {GET} /wp-json/appboss/learndash/v1/quiz/:id Get Topic
	 * @apiName        GetLDQuiz
	 * @apiGroup       Quiz
	 * @apiDescription Retrieve single Quiz
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function get_item( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'GET', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}" );
		$request_curl->set_query_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {

		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 * @since 0.1.0
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'quiz',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full url of image .' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['mode'] = array(
			'description' => __( 'Mode of the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['lesson'] = array(
			'description' => __( 'The Lesson id for the object' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'The Topic id for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['has_course_access'] = array(
			'description' => __( 'Whether or not user have the object access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed'  ),
		);

		$schema['properties']['has_content_access'] = array(
			'description' => __( 'Whether or not user have the object content access.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['completed'] = array(
			'description' => __( 'The object is completed by current user or not.' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['can_take_again'] = array(
			'description' => __( 'Whether or not user can take the object again.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['form']     = array(
			'description' => __( 'Pre request form for object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);
		$schema['properties']['settings'] = array(
			'description' => __( 'Settings for object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

}