<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Quiz;

use \WP_REST_Server;
use \WP_REST_Request;
use \WP_REST_Response;

class QuizQuestionsRest extends QuizRest {

	protected static $instance;
	protected $helper;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		$this->helper    = QuizQuestionsHelper::instance();
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 0.1.0
	 */
	public function register_routes() {

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/start', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'start' ),
					'permission_callback' => array( $this, 'get_start_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/next', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'next' ),
					'permission_callback' => array( $this, 'get_next_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/prev', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'prev' ),
					'permission_callback' => array( $this, 'get_prev_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/check', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'check' ),
					'permission_callback' => array( $this, 'get_check_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );

		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/upload', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'upload' ),
					'permission_callback' => array( $this, 'get_upload_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );


		register_rest_route( $this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/save', array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save' ),
					'permission_callback' => array( $this, 'get_save_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			) );
	}

	/**
	 * Check if a given request has access to start quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_start_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `start` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_start_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to next quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_next_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `next` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_next_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to prev quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_prev_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `prev` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_prev_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to check quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_check_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `check` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_check_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to upload quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_upload_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `upload` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_upload_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to save quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 * @since 0.1.0
	 */
	public function get_save_permissions_check( $request ) {

		$retval = true;
		if ( ! is_user_logged_in() ) {
			$retval = QuizError::instance()->userNotLoggedIn();
		}

		/**
		 * Filter the quiz `save` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_ld_quiz_save_permissions_check', $retval, $request );
	}


	/**
	 * Start Quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/start Start Quiz
	 * @apiName        StartLDQuiz
	 * @apiGroup       Quiz
	 * @apiDescription Start Quiz
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function start( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/start" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Next Quiz Question.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/next Next Quiz Question
	 * @apiName        LDQuizQuestionNext
	 * @apiGroup       Quiz
	 * @apiDescription Next Quiz Question
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function next( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/next" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Previous Quiz Question.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/prev Previous Quiz Question
	 * @apiName        LDQuizQuestionPrevious
	 * @apiGroup       Quiz
	 * @apiDescription Previous Quiz Question
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function prev( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/prev" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Check Quiz Question is correct or not.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/check Check Quiz Question is correct or not.
	 * @apiName        LDQuizQuestionCheck
	 * @apiGroup       Quiz
	 * @apiDescription Check Quiz Question is correct or not.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {array} [answer] Answer array for quiz questions.
	 */
	public function check( $request ) {

		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/check" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Upload Quiz Question answers file.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/upload Upload Quiz Question answers file.
	 * @apiName        LDQuizQuestionUpload
	 * @apiGroup       Quiz
	 * @apiDescription Upload Quiz Question answers file.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {Number} [ question_id ] A unique numeric ID for the Quiz Question.
	 * @apiParam {File} [ attachment ] A file to upload for question answers.
	 */
	public function upload( $request ) {

		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/upload" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

	/**
	 * Save quiz question answers.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 * @since          0.1.0
	 * @apiPrivate
	 * @api            {POST} /wp-json/appboss/learndash/v1/quiz/:id/save Save quiz question answers.
	 * @apiName        LDQuizQuestionSave
	 * @apiGroup       Quiz
	 * @apiDescription Save quiz question answers.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 * @apiParam {array} [answer] Answer for quiz questions.
	 * @apiParam {array} [Form] Form details for quiz.
	 */
	public function save( $request ) {

		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];

		global $wp_rest_server;

		$request_curl = new \WP_REST_Request( 'POST', "/buddyboss-app/learndash/v1/quiz/{$quiz_id}/save" );
		$request_curl->set_body_params( $request->get_params() );
		$response = $this->dispatch( $request_curl );

		return $wp_rest_server->response_to_data( rest_ensure_response( $response ), isset( $request['_embed'] ) ? 1 : 0 );
	}

}