<?php

namespace BuddyBossApp\Migration\LearnDash\V1\Quiz;

use \WP_Error;
use \stdClass;
use \WpProQuiz_Controller_Statistics;
use \WpProQuiz_Model_CategoryMapper;
use \WpProQuiz_Model_QuestionMapper;
use \WpProQuiz_Model_QuizMapper;

class QuizQuestionsHelper {

	protected static $instance;

	/**
	 * singleton instance.
	 *
	 * @since 0.1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 0.1.0
	 */
	public function __construct() {

	}

	/**
	 * Get the Questions list.
	 *
	 * @param $quiz_pro
	 *
	 * @return array
	 */
	public function get_questions( $quiz_pro ) {

		$quizMapper     = new WpProQuiz_Model_QuizMapper();
		$questionMapper = new WpProQuiz_Model_QuestionMapper();

		$quiz = $quizMapper->fetch( $quiz_pro );

		/**
		 * Fetch the question according to setting
		 * - Show all quiz questions
		 * - Show predefine number of question [ LD Setting ]
		 * - Show predefine % of question [ LD Setting ]
		 * - Fetch All question Random order [ LD setting ]
		 */
		if ( $quiz->isShowMaxQuestion() && $quiz->getShowMaxQuestionValue() > 0 ) {

			$value = $quiz->getShowMaxQuestionValue();

			if ( $quiz->isShowMaxQuestionPercent() ) {
				$count = $questionMapper->count( $quiz_pro );

				$value = ceil( $count * $value / 100 );
			}

			$fetchQuestions = $questionMapper->fetchAll( $quiz, true, $value );

		} else {
			$isRendonFetch  = $quiz->isQuestionRandom();
			$fetchQuestions = $questionMapper->fetchAll( $quiz, $isRendonFetch );
		}

		$questions = array();
		foreach ( $fetchQuestions as $que_post ) {
			$que_data = $this->prepare_question_data( $que_post );

			/**
			 * Shuffle option if quiz setting is enabled.
			 */
			if ( $quiz->isAnswerRandom() ) {
				if ( ! empty( $que_data->options_1 ) ) {
					shuffle( $que_data->options_1 );
				} else {
					shuffle( $que_data->options );
				}
			}

			$questions[] = $que_data;
		}

		if ( $quiz->isSortCategories() ) {
			usort( $questions, array( $this, "SortCategories" ) );
		}

		return $questions;
	}

	/**
	 *
	 * Get the single Question
	 *
	 * @param $question_id
	 * @param $quiz_pro
	 *
	 * @return mixed
	 */
	public function get_guestion( $question_id, $quiz_pro ) {

		$quizMapper = new WpProQuiz_Model_QuizMapper();
		$quiz       = $quizMapper->fetch( $quiz_pro );

		$questionMapper = new WpProQuiz_Model_QuestionMapper();
		$que_post       = $questionMapper->fetchById( $question_id );

		return $this->prepare_question_data( $que_post );
	}

	/**
	 * Check Question Answer
	 *
	 * @param $answers
	 * @param $quiz_pro
	 * @param $quiz_id
	 *
	 * @return array|bool
	 */
	public function check_answers( $answers, $quiz_pro, $quiz_id ) {

		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Funcation: checkAnswers -> Ajax Request
		 */

		$user_id = get_current_user_id();

		/**
		 * On check endpoint we checking one question answer. we use loop because of we sending answer Array using question id so we need to use loop.
		 * End of loop we adding break statement so loop only work one time
		 */
		foreach ( $answers as $ques_id => $answer ) {

			$answer_data = $this->prepare_question_answer( $answer, $ques_id );
			$data        = array(
				'quizId'     => $quiz_pro,
				'quiz'       => $quiz_id,
				'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
				'responses'  => array(
					$ques_id => array( 'response' => $answer_data ),
				),
			);

			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}
			$responses = $this->ld_quiz_pro->checkAnswers( $data );
			break;
		}

		if ( ! empty( $responses ) ) {
			$questionMapper      = new WpProQuiz_Model_QuestionMapper();
			$resultData = array();
			$responses  = (array) json_decode( $responses );
			foreach ( $responses as $q_id => $response ) {
				$answer['id']                = $q_id;
				$que_post = $questionMapper->fetchById( $q_id );
				$resultData[ $q_id ]['data'] = apply_filters( 'bbapp_ld_prepare_quiz_question_answer',array(
					'id'               => $q_id,
					'question_type'    => $response->e->type,
					'isCorrect'        => $response->c,
					'message'          => $response->e->AnswerMessage,
					'point'            => $response->p,
					'possiblePoints'   => $response->e->possiblePoints,
					'sentItems'        => $answers[ $ques_id ],
					'sentItemsCorrect' => $this->get_Item_correct( $response ),
					'possibleItems'    => isset( $response->e->c ) ? $response->e->c : '',
				), $que_post );

				if ( 'essay' === $response->e->type ) {
					$resultData[ $q_id ]['data']['graded_id']     = $response->e->graded_id;
					$resultData[ $q_id ]['data']['graded_status'] = $response->e->graded_status;
				}
				$resultData[ $q_id ]['responce'] = $response;
			}

			return $resultData[ $q_id ];
		}

		return false;
	}

	/**
	 * Save quiz question answers
	 *
	 * @param $quiz_meta
	 * @param $quiz_id
	 *
	 * @return array|WP_Error
	 */
	public function save_quiz( $quiz_meta, $quiz_id ) {
		/**
		 * Ref_Files: sfwd-lms/includes/quiz/ld-quiz-pro.php
		 * Ref_Funcation: checkAnswers -> Ajax Request
		 */

		$user_id       = get_current_user_id();
		$quiz_settings = learndash_get_setting( $quiz_id );
		$quiz_pro      = $quiz_settings['quiz_pro'];

		$quizMapper = new WpProQuiz_Model_QuizMapper();
		$quiz       = $quizMapper->fetch( $quiz_pro );


		$questionMapper      = new WpProQuiz_Model_QuestionMapper();
		$statisticController = new WpProQuiz_Controller_Statistics();

		/**
		 * Prepare quiz result for learndash
		 */
		$quiz_result          = $quiz_answers = array();
		$total_awarded_points = $total_possible_points = $total_correct = 0;
		$catPoints            = array();

		$responses = $this->prepare_question_answer_response( $quiz_meta, $quiz_pro, $quiz_id );

		/**
		 * Either user answer is empty or CheckAnswer process returning some error
		 */
		if ( empty( $responses ) && is_wp_error( $responses ) ) {
			return QuizError::instance()->QuizCheckQuestionAnswersError();
		}

		$categories = $this->get_question_categories_list( $quiz_pro );

		/**
		 * Prepare Quiz answers & results data for response
		 */
		foreach ( $quiz_meta['questions'] as $index => $question ) {
			$ques_id  = $question['id'];
			$response = $responses[ $ques_id ];

			/**
			 * Prepare quiz question category total possible points count
			 */
			$que_post = $questionMapper->fetchById( $ques_id );
			if ( ! isset( $catPoints[ $que_post->getCategoryId() ] ) ) {
				$catPoints[ $que_post->getCategoryId() ] = 0;
			}
			$catPoints[ $que_post->getCategoryId() ] += $que_post->getPoints();

			/**
			 * Prepare quiz question category total awarded points count
			 */
			$categories[ $que_post->getCategoryId() ]['result'] += $response->p;

			/**
			 * Prepare quiz total correct answer count
			 */
			if ( $response->c ) {
				$total_correct += 1;
			}

			/**
			 * Prepare quiz total awarded points counts
			 */
			$total_awarded_points += $response->p;

			/**
			 * Prepare quiz total possible points counts
			 */
			$total_possible_points += $response->e->possiblePoints;

			/**
			 * Prepare User's answers
			 */
			$userResponse = array();
			if ( isset( $response->e->r ) ) {
				$userResponse = is_array( $response->e->r ) ? $response->e->r : array( $response->e->r );
				if ( ! empty( $userResponse ) ) {
					foreach ( $userResponse as $key => $value ) {
						if ( $value === true ) {
							$userResponse[ $key ] = 1;
						} else if ( $value === false ) {
							$userResponse[ $key ] = 0;
						}
					}
				}
			}

			/**
			 * Prepare Quiz Question Result data
			 */
			$quiz_result[ $question['id'] ] = array(
				"time"           => $question['end_time'] - $question['start_time'],
				"points"         => $response->p,
				"correct"        => $response->c,
				"data"           => $userResponse,
				"possiblePoints" => $response->e->possiblePoints,
				"a_nonce"        => $response->a_nonce,
				"p_nonce"        => $response->p_nonce,
			);

			/**
			 * Prepare & Update Quiz Question answers data.
			 */
			$quiz_answers[ $index ] = apply_filters( 'bbapp_ld_prepare_quiz_question_answer', array(
				'id'               => $ques_id,
				'question_type'    => $response->e->type,
				'isCorrect'        => $response->c,
				'message'          => $response->e->AnswerMessage,
				'point'            => $response->p,
				'possiblePoints'   => $response->e->possiblePoints,
				'sentItems'        => $question['answer'],
				'sentItemsCorrect' => $this->get_Item_correct( $response ),
				'possibleItems'    => isset( $response->e->c ) ? $response->e->c : '',
			), $que_post );

			if ( 'essay' === $response->e->type ) {
				$quiz_result[ $question['id'] ]['graded_id']     = $response->e->graded_id;
				$quiz_result[ $question['id'] ]['graded_status'] = $response->e->graded_status;
				$quiz_result[ $question['id'] ]['data']          = array( $response->e->graded_id );

				$quiz_answers[ $index ]['graded_id']     = $response->e->graded_id;
				$quiz_answers[ $index ]['graded_status'] = $response->e->graded_status;
			}
		}

		/**
		 * Calculate Quiz questrion categories percentage result.
		 */
		foreach ( $categories as $id => $value ) {
			if ( ! empty( $catPoints[ $id ] ) ) {
				$categories[ $id ]['result'] = round( $categories[ $id ]['result'] / $catPoints[ $id ] * 100 * 100 ) / 100;
			} else {
				$categories[ $id ]['result'] = 0;
			}
		}

		/**
		 * $total_possible_points is empty if quiz have only question which support answer point
		 */
		$result = 0;
		if ( ! empty( $total_possible_points ) ) {
			$result = ( $total_awarded_points / $total_possible_points * 100 * 100 ) / 100;
		}

		/**
		 * To do quiz from mobile we added trick to time limit fixed.
		 */
		$timelimit = $quiz->getTimeLimit();
		if ( ! empty( $timelimit ) ) {
			$diff = $quiz_meta['end_time'] - $quiz_meta['start_time'];
			if ( $diff > $timelimit ) {
				$quiz_meta['end_time'] = $quiz_meta['start_time'] + $timelimit;
			}
		}

		/**
		 * Prepare quiz result `comp` data.
		 */
		$quiz_result['comp'] = array(
			'points'             => $total_awarded_points,
			'correctQuestions'   => $total_correct,
			'quizTime'           => $quiz_meta['end_time'] - $quiz_meta['start_time'],
			'quizEndTimestamp'   => $quiz_meta['end_time'] * 1000,
			'quizStartTimestamp' => $quiz_meta['start_time'] * 1000,
			'result'             => $result,
			'cats'               => $categories,
		);

		/*******************************************
		 * Complete quiz using `completedQuiz`
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/controller/WpProQuiz_Controller_Admin.php
		 * Ref_Funcation: completedQuiz -> Ajax Request
		 *******************************************/

		/**
		 * To complete quiz we need to update GET, POST or REQUEST data so take copy before update
		 */
		$_temp_REQUEST = $_REQUEST;
		$_temp_POST    = $_POST;
		$_temp_GET     = $_GET;

		if ( empty( $_POST['results'] ) ) {

			$_REQUEST = $_POST = $_GET = array();

			$_POST['quizId']    = $quiz_pro;
			$_POST['quiz']      = $quiz_id;
			$_POST['results']   = $quiz_result;
			$_POST['timespent'] = $quiz_result['comp']['quizTime'];
			$_POST['forms']     = isset( $quiz_meta['form'] ) ? $quiz_meta['form'] : array();
			$_POST['course_id'] = isset( $_temp_REQUEST['course_id'] ) ? intval( $_temp_REQUEST['course_id'] ) : bbapp_learndash_get_course_id( $quiz_id );
			$_POST['lesson_id'] = isset( $_temp_REQUEST['lesson_id'] ) ? intval( $_temp_REQUEST['lesson_id'] ) : 0;
			$_POST['topic_id']  = isset( $_temp_REQUEST['topic_id'] ) ? intval( $_temp_REQUEST['topic_id'] ) : 0;

			$_REQUEST['quiz'] = $_POST['quiz'];
		}

		if ( empty( $_POST['quiz_nonce'] ) ) {
			$_POST['quiz_nonce'] = wp_create_nonce( 'sfwd-quiz-nonce-' . intval( $_POST['quiz'] ) . '-' . intval( $_POST['quizId'] ) . '-' . $user_id );
		}

		if ( ! isset( $_POST['results'] ) ) {
			return QuizError::instance()->invalidQuizResultData();
		}

		/**
		 * To stop "Uncanny LearnDash Toolkit" Redirection after complete event for rest endpoint
		 */
		$_REQUEST['doing_rest'] = 1;

		$is_submitted = $this->completedQuiz();

		/**
		 * If completedQuiz fails with some error.
		 */
		if ( is_wp_error( $is_submitted ) ) {
			return $is_submitted;
		}

		/*******************************************
		 *        Store leaderboard data
		 *******************************************/

		if ( $quiz->isToplistActivated() && $quiz->isToplistDataAddAutomatic() ) {
			LeaderboardRest::instance()->createitem( $quiz, array(
				'point'  => $total_awarded_points,
				'result' => $quiz_result['comp']['result'],
				'ip'     => ''
			) );
		}

		/**
		 * Make GET, POST or REQUEST reverted as quiz complete process is done.
		 */
		$_REQUEST = $_temp_REQUEST;
		$_POST    = $_temp_POST;
		$_GET     = $_temp_GET;


		/*******************************************
		 * Prepare rest response for quiz complete
		 *******************************************/

		$quiz_post = get_post( $quiz_id );

		$passingpercentage = intVal( $quiz_settings['passingpercentage'] );
		$pass              = ( $quiz_result['comp']['result'] >= $passingpercentage ) ? true : false;

		$reponse = array(
			'quiz'               => QuizRest::instance()->get_additional_data( $quiz_post ),
			'finished'           => $is_submitted,
			'is_pass'            => $pass,
			'result'             => $quiz_result['comp']['result'],
			'answers'            => $quiz_answers,
			'passing_percentage' => $passingpercentage,
		);

		/**
		 * Calculate average quiz result for users.
		 */
		if ( $quiz->isShowAverageResult() ) {
			$reponse['average_result'] = $statisticController->getAverageResult( $quiz_pro );
		}

		/**
		 * Prepare certificate if it's attached with quiz
		 */
		$certificate_post_id = intval( $quiz_settings['certificate'] );
		if ( ! empty( $certificate_post_id ) ) {
			$certificate_post = get_post( $certificate_post_id );
			if ( ! empty( $certificate_post ) ) {

				$certificate = learndash_certificate_details( $quiz_id );

				if ( ( $reponse['result'] / 100 >= $certificate['certificate_threshold'] ) ) {
					$reponse['certificate']['title']    = get_the_title( $quiz_id );
					$reponse['certificate']['filename'] = sanitize_file_name( $quiz_post->post_title ) . "-" . sanitize_file_name( $certificate_post->post_title ) . ".pdf";
					$reponse['certificate']['link']     = $certificate['certificateLink'];
				}
			}
		}

		/**
		 * Prepare categories point response
		 */
		if ( $quiz->isShowCategoryScore() ) {
			$reponse['cats'] = array_values( $quiz_result['comp']['cats'] );
		}

		/**
		 * Added quiz complete time
		 */
		if ( ! $quiz->isHideResultQuizTime() ) {
			$reponse['time'] = $quiz_result['comp']['quizTime'];
		}

		/**
		 * Added quiz points data
		 */
		if ( ! $quiz->isHideResultPoints() ) {
			$reponse['awarded_points']  = $total_awarded_points;
			$reponse['possible_points'] = $total_possible_points;
		}

		/**
		 * Manage Quiz grade result manage
		 */
		if ( $quiz->isResultGradeEnabled() ) {
			$_result     = $reponse['result'];
			$_resultText = $quiz->getResultText();
			$diff        = 99999;
			foreach ( $_resultText['prozent'] as $i => $value ) {
				if ( ( $_result >= $value ) && ( ( $_result - $value ) < $diff ) ) {
					$diff               = $_result - $value;
					$reponse['message'] = $_resultText['text'][ $i ];
				}
			}
			$reponse['message'] = apply_filters( 'comment_text', $reponse['message'], null, null );
		} else {
			$reponse['message'] = apply_filters( 'comment_text', $quiz->getResultText(), null, null );
		}

		return $reponse;
	}

	/**
	 * @param $que_post
	 *
	 * @return mixed
	 */
	private function prepare_question_data( $que_post ) {

		/**
		 * Ref_Files: sfwd-lms/includes/vendor/wp-pro-quiz/lib/view/WpProQuiz_View_FrontQuiz.php
		 * Ref_Funcation: showQuizBox
		 */

		$question           = new stdClass();
		$question->ID       = $que_post->getId();
		$question->title    = $que_post->getQuestion();
		$question->category = $que_post->getCategoryName();

		if ( ! $que_post->isAnswerPointsActivated() ) {
			$question->points = $que_post->getPoints();
		}

		$question->question_type = $que_post->getAnswerType();
		$question->hint          = ( $que_post->isTipEnabled() ) ? do_shortcode( apply_filters( 'comment_text', $que_post->getTipMsg(), null, null ) ) : '';
		$question->options       = array();

		if ( $question->question_type === 'matrix_sort_answer' ) {
			$question->options_1 = array();
		}

		if ( ! in_array( $question->question_type, array(
			'free_answer',
			'cloze_answer',
			'assessment_answer',
			'essay',
		) ) ) {
			$answers      = $que_post->getAnswerData();
			$answer_index = 0;
			foreach ( $answers as $answer_post ) {
				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points[ $answer_index ] = $answer_post->getPoints();
				}
				$answer        = new stdClass();
				$answer->title = $answer_post->getAnswer();
				$answer->value = $answer_post->getAnswer();

				if ( $question->question_type === 'sort_answer' ) {
					$answer->value = LdQuizpro::datapos( $question->ID, $answer_index );
				}

				if ( $question->question_type !== 'sort_answer' ) {
					$answer->pos = $answer_index;
				}

				if ( $question->question_type === 'matrix_sort_answer' ) {
					$answer_1              = new stdClass();
					$answer_1->title       = $answer_post->getSortString();
					$answer_1->value       = LdQuizpro::datapos( $question->ID, $answer_index );
					$question->options_1[] = $answer_1;
				}

				$question->options[] = $answer;

				$answer_index ++;
			}
		} else if ( 'cloze_answer' === $question->question_type ) {
			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$clozeData = bbapp_lms_fetch_cloze( $answer_post->getAnswer() );
				if ( $que_post->isAnswerPointsActivated() ) {
					$question->points = $clozeData['points'];
				}
				$answer              = new stdClass();
				$answer->title       = str_replace( '@@wpProQuizCloze@@', '{___}', $clozeData['replace'] );
				$answer->title       = nl2br( $answer->title ); // Convert nl to <br />
				$question->options[] = $answer;
			}
		} else if ( 'assessment_answer' === $question->question_type ) {

			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$getAnswer           = $answer_post->getAnswer();
				$question->options[] = $this->format_assessment_answer( $getAnswer );
			}

		} else if ( 'essay' === $question->question_type ) {
			$answers = $que_post->getAnswerData();
			foreach ( $answers as $answer_post ) {
				$question->gradedType = $answer_post->getGradedType();
			}
		}

		/**
		 * Shuffle Question answer if Question type is sort_answer or matrix_sort_answer.
		 */
		if ( $question->question_type === 'sort_answer' || $question->question_type === 'matrix_sort_answer' ) {
			if ( ! empty( $question->options_1 ) ) {
				shuffle( $question->options_1 );
			} else {
				shuffle( $question->options );
			}
		}

		return $question;
	}

	/**
	 * Formats the assessment answer according to learner format.
	 *
	 * @param $question
	 *
	 * @return array
	 */
	private function format_assessment_answer( $question ) {

		$field_text = $question;

		# Split all html from answer.
		$get_html = preg_split( "/{.*?}/", $field_text );

		# split all fields
		preg_match_all( "/{.*?}/", $field_text, $matched_field );
		if ( isset( $matched_field[0] ) ) {
			$matched_field = $matched_field[0];
		}

		$data = array();
		$i    = 0;

		foreach ( $get_html as $get_html ) {

			$data[] = array(
				"type"  => "html",
				"value" => bbapp_learndash_fix_html( $get_html ),
			);

			if ( isset( $matched_field[ $i ] ) ) {

				# format the values from fields into array.
				preg_match_all( "/\[(.*?)\]/", $matched_field[ $i ], $formated );
				if ( isset( $formated[1] ) ) {
					$formated = $formated[1];
				} else {
					$formated = array();
				}

				$data[] = array(
					"type"  => "radio",
					"value" => $formated,
				);

			}

			$i ++;
		}

		return $data;

	}

	/**
	 * Prepare Answer of question to check with learndash functions.
	 *
	 * @param $answer
	 * @param $ques_id
	 *
	 * @return array|int|mixed
	 */
	private function prepare_question_answer( $answer, $ques_id ) {

		$questionMapper = new WpProQuiz_Model_QuestionMapper();

		$que_post = $questionMapper->fetchById( $ques_id );
		$que_post = $this->prepare_question_data( $que_post );

		$answer_data = array();
		if ( ! empty( $que_post->options ) && 'assessment_answer' !== $que_post->question_type ) {
			$answer_index = 0;
			foreach ( $que_post->options as $option ) {
				if ( $que_post->question_type === 'single' ) {
					$answer_data[ $answer_index ] = in_array( $option->value, $answer );
				} else if ( $que_post->question_type === 'multiple' ) {
					$answer_data[ $answer_index ] = $answer[ $answer_index ] == 'true';
				} else if ( $que_post->question_type === 'cloze_answer' ) {
					$answer      = array_map( array( $this, 'cleanupCurlyQuotes' ), $answer );
					$answer_data = $answer;
				} else {
					$answer_data[ $answer_index ] = $answer[ $answer_index ];
				}
				$answer_index ++;
			}
		} elseif ( 'assessment_answer' == $que_post->question_type ) {
			$answer_data = 0;
			foreach ( $answer as $ans ) {
				$answer_data += $ans;
			}
		} else {
			$answer_data = $answer[0];
		}

		return $answer_data;
	}

	/**
	 * Prepare question answer check response.
	 *
	 * @param $quiz_meta
	 * @param $quiz_pro
	 * @param $quiz_id
	 *
	 * @return array
	 */
	private function prepare_question_answer_response( $quiz_meta, $quiz_pro, $quiz_id ) {

		$responses = array();
		$user_id   = get_current_user_id();

		$data = array(
			'quizId'     => $quiz_pro,
			'quiz'       => $quiz_id,
			'quiz_nonce' => wp_create_nonce( 'sfwd-quiz-nonce-' . $quiz_id . '-' . $quiz_pro . '-' . $user_id ),
			'responses'  => array(),
		);

		/**
		 * Check response is stroe in session data if not then prepare data to check answers.
		 */

		foreach ( $quiz_meta['questions'] as $question ) {
			$ques_id = $question['id'];

			if ( ! empty( $question['responce'] ) ) {
				/**
				 * if Question answer already check with check endpoint then use response from quiz session data
				 */
				$responses[ $ques_id ] = (object) $question['responce']['responce'];
			} else {
				$answer = $question['answer'];

				$answer_data = $this->prepare_question_answer( $answer, $ques_id );

				/**
				 * Add User's answer to check it's correct or not
				 */
				$data['responses'][ $ques_id ] = array( 'response' => $answer_data );
			}
		}

		/**
		 * Check question answer if quiz is submited with our checking answer.
		 */
		if ( ! empty( $data['responses'] ) ) {
			if ( ! isset( $this->ld_quiz_pro ) ) {
				$this->ld_quiz_pro = new LdQuizpro();
			}
			$ste_responses = $this->ld_quiz_pro->checkAnswers( $data );
			if ( empty( $ste_responses ) ) {

				/**
				 * Fails checkAnswers with some error
				 */
				return array();
			}
			$arr_responses = json_decode( $ste_responses );
			foreach ( $data['responses'] as $ques_id => $ans ) {
				$responses[ $ques_id ] = $arr_responses->$ques_id;
			}
		}

		return $responses;
	}

	/**
	 * Prepare Question categories list
	 *
	 * @param $quiz_pro
	 *
	 * @return array
	 */
	private function get_question_categories_list( $quiz_pro ) {

		$categories     = array();
		$categoryMapper = new WpProQuiz_Model_CategoryMapper();

		$fetchCategories = $categoryMapper->fetchByQuiz( $quiz_pro );
		foreach ( $fetchCategories as $catgory ) {
			/* @var $catgory WpProQuiz_Model_Category */

			if ( ! $catgory->getCategoryId() ) {
				$categories[0] = array(
					'name'   => __( 'Not categorized', 'wp-pro-quiz' ),
					'result' => 0,
				);
				continue;
			}

			$categories[ $catgory->getCategoryId() ] = array(
				'name'   => $catgory->getCategoryName(),
				'result' => 0,
			);
		}

		return $categories;
	}

	/**
	 * Sort question by quiz question category.
	 *
	 * @param $a
	 * @param $b
	 *
	 * @return int
	 */
	private function SortCategories( $a, $b ) {
		return strcmp( $b->category, $a->category );
	}

	/**
	 * Clean Up Extra space and make user's answers in lower case so we can support lower and upper both.
	 *
	 * @param $answer
	 *
	 * @return string
	 */
	private function cleanupCurlyQuotes( $answer ) {
		if ( version_compare( LEARNDASH_VERSION, '2.5', '<' ) ) {
			$answer = strtolower( $answer );
		}
		$answer = trim( $answer );

		return $answer;
	}

	/**
	 * Prepare Correct answers response
	 *
	 * @param $responce
	 *
	 * @return array
	 */
	private function get_Item_correct( $responce ) {
		switch ( $responce->e->type ) {
			case 'cloze_answer':
				$sentItemsCorrect = array_values( (array) $responce->s );
				break;
			case 'multiple':
				$sentItemsCorrect = array();
				foreach ( $responce->e->c as $index => $value ) {
					$sentItemsCorrect[] = ( $responce->e->r[ $index ] === (bool) $value ) ? true : false;
				}
				break;
			case 'sort_answer':
			case 'matrix_sort_answer':
				$sentItemsCorrect = array();
				foreach ( $responce->e->c as $index => $value ) {
					$sentItemsCorrect[] = ( $responce->e->r[ $index ] === $value ) ? true : false;
				}
				break;
			default:
				$sentItemsCorrect = $responce->c;
		}

		return $sentItemsCorrect;
	}

	/**
	 * Complete quiz
	 */
	private function completedQuiz() {
		/**
		 * Check if $_POST data is json encode then decode it
		 */
		if ( ( isset( $_POST['results'] ) ) && ( ! empty( $_POST['results'] ) ) && ( is_string( $_POST['results'] ) ) ) {
			$_POST['results'] = json_decode( stripslashes( $_POST['results'] ), true );
		}

		$user_id  = ( is_user_logged_in() ) ? get_current_user_id() : 0;
		$quiz_pro = ( isset( $_POST['quizId'] ) ) ? $_POST['quizId'] : 0;
		$quiz_id  = ( isset( $_POST['quiz'] ) ) ? $_POST['quiz'] : 0;

		/**
		 * If the results is not present then abort.
		 */
		if ( ! isset( $_POST['results'] ) ) {
			return QuizError::instance()->invalidQuizResultData();
		}

		/**
		 * LD 2.4.3 - Change in logic. Instead of accepting the values for points, correct etc from request we now result array [ question answer with none per answer ].
		 * Now we first verify that question nonce and than calculate points here
		 */
		$total_awarded_points = $total_correct = 0;

		/**
		 * Loop over the 'results' items. We verify and calculate the points+correct counts as well as the student response 'data'. When we get to the 'comp' results element
		 * we set the award points and correct as well as determine the total possible points.
		 */
		foreach ( $_POST['results'] as $r_idx => $result ) {
			if ( $r_idx === 'comp' ) {
				$_POST['results'][ $r_idx ]['points']           = intval( $total_awarded_points );
				$_POST['results'][ $r_idx ]['correctQuestions'] = intval( $total_correct );
				continue;
			}

			$points_array = array(
				'points'         => intval( $result['points'] ),
				'correct'        => intval( $result['correct'] ),
				'possiblePoints' => intval( $result['possiblePoints'] ),
			);

			$points_array['correct'] = ( $points_array['correct'] === true ) ? 1 : $points_array['correct'];
			$points_array['correct'] = ( $points_array['correct'] === false ) ? 0 : $points_array['correct'];
			$points_str              = maybe_serialize( $points_array );

			if ( ! wp_verify_nonce( $result['p_nonce'], 'ld_quiz_pnonce' . $user_id . '_' . $quiz_pro . '_' . $quiz_id . '_' . $r_idx . '_' . $points_str ) ) {
				$_POST['results'][ $r_idx ]['points']         = 0;
				$_POST['results'][ $r_idx ]['correct']        = 0;
				$_POST['results'][ $r_idx ]['possiblePoints'] = 0;
			}
			$total_awarded_points += intval( $_POST['results'][ $r_idx ]['points'] );
			$total_correct        += $_POST['results'][ $r_idx ]['correct'];
			$response_str         = maybe_serialize( $result['data'] );

			if ( ! wp_verify_nonce( $result['a_nonce'], 'ld_quiz_anonce' . $user_id . '_' . $quiz_pro . '_' . $quiz_id . '_' . $r_idx . '_' . $response_str ) ) {
				$_POST['results'][ $r_idx ]['data'] = array();
			}
		}

		$quiz = new QuizComplete();

		return $quiz->completedQuiz();
	}
}